#!/usr/bin/python3
'''
Fit-o-mat - a versatile program for nonlinear least-squares fitting
Copyright (C) 2017-2025  Andreas Moeglich, University of Bayreuth, Germany
contact me at andreas.moeglich@uni-bayreuth.de

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
'''
VERSION = '0.976'

# check for modules via importlib
try:
  from importlib.util import find_spec
  importItems = 'datetime,PyQt6,sys,pathlib,functools,copy,ast,time,os,webbrowser,zipfile,io'.split(',')
  importItems += 'matplotlib,numpy,scipy,warnings'.split(',')
  FAILED_IMPORTS = []
except ImportError:
  try:
    import ctypes
    ctypes.windll.user32.MessageBoxW(None, 'Fit-o-mat cannot run because importlib is missing.\n\nTry installing missing module importlib via:\n\tpython3 -m pip install importlib\n', 'Fit-o-mat -- missing module', 0)
  except:
    print('Fit-o-mat cannot run because importlib is missing.\n')
    print('\nTry installing missing module importlib via:\n\tpython3 -m pip install importlib\n')
  exit()
  
for item in importItems:
  if(find_spec(item) is None):
    FAILED_IMPORTS.append(item)

if(len(FAILED_IMPORTS)):
  try:
    from PyQt6 import QtWidgets, QtCore
  except:
    pass
  if('QtWidgets' in globals()):
    # create application
    app = QtWidgets.QApplication([])
    # display error dialogue
    msgBox = QtWidgets.QMessageBox()
    msgBox.setWindowTitle('Fit-o-mat -- missing module')
    text = 'Fit-o-mat cannot run because the following modules are missing:<br/>'
    for item in FAILED_IMPORTS:
      text += '\n' + '&nbsp;' * 3 + '<b>' + item + '</b><br/>'
    text += '\n<br/>Try installing missing module <b>xxx</b> via:\n<br/>' + '&nbsp;' * 3 + 'python3 -m pip install <b>xxx</b>'
    msgBox.setText(text)
    msgBox.setTextFormat(QtCore.Qt.TextFormat.RichText)
    msgBox.setStandardButtons(QtWidgets.QMessageBox.StandardButton.Abort)
    msgBox.setIcon(QtWidgets.QMessageBox.Icon.Critical)
    exit(msgBox.exec())
  else:
    try:
      import ctypes
      ctypes.windll.user32.MessageBoxW(None, 'Fit-o-mat cannot run because the following modules are missing\n\t' + '\n\t'.join(FAILED_IMPORTS) + '\n\nTry installing missing module xxx via:\n\tpython3 -m pip install xxx\n', 'Fit-o-mat -- missing module', 0)
    except:
      print('Fit-o-mat cannot run because the following modules are missing\n\t' + '\n\t'.join(FAILED_IMPORTS) + '\n')
      print('\nTry installing missing module xxx via:\n\tpython3 -m pip install xxx\n')
    exit()

# start proper
import datetime    # need this in rare instances when a datetime slipped through openpyxl import
from PyQt6 import QtCore, QtGui, QtWidgets
from sys import argv, exit, platform, exc_info, version_info
from pathlib import Path
from functools import partial, cmp_to_key
from copy import deepcopy
from ast import literal_eval
from time import asctime, time
from os.path import expanduser, isfile, exists, dirname, abspath
from os import getenv, makedirs
import webbrowser
import zipfile, io
try:
  from pycorn import pc_res3
  PYCORN_PRESENT = True
except:
  PYCORN_PRESENT = False

import matplotlib
matplotlib.use("Qt5Agg")
import matplotlib.pyplot as plt
import matplotlib.legend as Legend
from matplotlib.widgets import AxesWidget
from matplotlib.widgets import Lasso
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
from matplotlib import patheffects as PathEffects

try:
  import xlsxwriter
  XLSXWRITER_PRESENT = True
except:
  XLSXWRITER_PRESENT = False
try:
  import xlrd
  XLRD_PRESENT = True
except:
  XLRD_PRESENT = False
try:
  import openpyxl
  OPENPYXL_PRESENT = True
except:
  OPENPYXL_PRESENT = False
try:
  from pyexcel_ods3 import get_data
  ODS3_PRESENT = True
except:
  ODS3_PRESENT = False

import warnings
warnings.simplefilter('ignore')
import numpy as np
# this is to make numpy __repr__ work as it used to
# it is quite the ugly fudge but alternative would be a major revamp of state files and style sheets :(
np.set_printoptions(legacy='1.25')
import scipy.optimize as optim
import scipy.odr as odr
import scipy.stats as stats
import scipy.interpolate as interpolate


# reimplement FigureCanvas to optimize graphics refresh  
class MyFigureCanvas(FigureCanvas):
  def __init__(self, parent=None, matplotlibCanvas=None, name='superunknown'):
    super(MyFigureCanvas, self).__init__(parent)
    self.parent = parent
    self.matplotlibCanvas = matplotlibCanvas
    self.refreshCount = 0
    self.destructCounter = 0
    self.name = name
    
  def myRefresh(self):
    # wrapper function to globally set how draw updates are done
    self.refreshCount += 1
    
    # ready to destruct?
    if(self.destructCounter):
      self.destructCounter -= 1
      if(not self.destructCounter):
        self.matplotlibCanvas.destructAboutLogo()
    
    # do we have arrows to take care of?
    for entry in ['x', 'y', 'y2']:
      if(self.name == 'plot'):
        if((self.matplotlibCanvas.handleArrow[entry] != None) or (self.matplotlibCanvas.handleArrow2[entry] != None)):
          self.matplotlibCanvas.drawAxisArrow(axis=entry, redraw=False, target='plot')
      elif((self.name == 'resid') and (entry != 'y2')):
        if((self.matplotlibCanvas.handleArrowResid[entry] != None) or (self.matplotlibCanvas.handleArrowResid2[entry] != None)):
          self.matplotlibCanvas.drawAxisArrow(axis=entry, redraw=False, target='resid')
        
    # do we have split axis divider lines to take are of?
    # (the function call will also delete left-over previous divider lines)
    self.matplotlibCanvas.drawSplitAxisDivider(redraw=False, target=self.name)
    self.matplotlibCanvas.drawSplitYAxisDivider(redraw=False)
    
    # manually draw second x axis
    self.matplotlibCanvas.draw2ndXAxisTicks(axis='x', target=self.name)
    self.matplotlibCanvas.draw2ndXAxisTicks(axis='x2', target=self.name)

    # the actual draw command
    self.draw()
    
  def setDestructionCounter(self, counter=0):
    # sets up destruction of self.handlesAbout
    self.destructCounter = counter

  def getDestructionCounter(self):
    # returns destruction counter
    return self.destructCounter

# custom QTabWidget (to enable dynamic tooltips)
class myQTabWidget(QtWidgets.QTabWidget):
  def __init__(self, *args, **kwargs):
    super(myQTabWidget, self).__init__(*args, **kwargs)
    self.storedToolTips = []
    
  def toggleToolTips(self, state=True):
    # dynamically sets and unsets tab tool tips
    for tab in range(self.count()):
      if(state):
        if(len(self.storedToolTips) > tab):
          self.setTabToolTip(tab, self.storedToolTips[tab])
      else:
        if(len(self.storedToolTips) <= tab):
          self.storedToolTips.append(self.tabToolTip(tab))
        else:
          self.storedToolTips[tab] = self.tabToolTip(tab)
        self.setTabToolTip(tab, '')

# custom QTableWidgetItem for interactive display of tooltips
class QTableWidgetItemToolTip(QtWidgets.QTableWidgetItem):
  def __init__(self, *args, **kwargs):
    super(QTableWidgetItemToolTip, self).__init__(*args, **kwargs)

  # duh, QTableWidgetItem does not allow event filtering
  # have to hijack data()
  def data(self, role):
    if((role == QtCore.Qt.ItemDataRole.ToolTipRole) and (not myapp.toolTipsShow)):
      return ''
    else:
      return QtWidgets.QTableWidgetItem.data(self, role)

# custom window for picture preview
class PreviewWindow(QtWidgets.QMainWindow):
  def __init__(self, *args, **kwargs):
    super(PreviewWindow, self).__init__(*args, **kwargs)
    ###self.setWindowFlags(self.windowFlags() | QtCore.Qt.WindowType.WindowStaysOnTopHint)
    self.NUMBER_TILES = 15
    self.displayPreview = None

  def paintEvent(self, event):
    # draw rounded corners
    s = self.size()
    winWidth, winHeight = s.width(), s.height()
    offsetX, offsetY = 0, 0
    targetWidth, targetHeight = winWidth, winHeight
    # adjust this to fit the draw area
    if((self.displayPreview != None) and (hasattr(self.displayPreview, 'size'))):
      targetSize = self.displayPreview.size()
      targetWidth, targetHeight = targetSize.width(), targetSize.height()
      offsetX, offsetY = winWidth - targetWidth, winHeight - targetHeight
    # do the actual painting
    qp = QtGui.QPainter()
    if(qp is not None):
      qp.begin(self)
      qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
      qp.setPen(QtCore.Qt.PenStyle.NoPen)
      # draw background rectangle
      qp.setBrush(QtGui.QColor(*UI_BASE_COLOR))
      qp.drawRect(offsetX, offsetY, targetWidth, targetHeight)
      # draw checkered boxes
      qp.setBrush(QtGui.QColor(*UI_BRIGHT_COLOR))
      deltaX, deltaY = int(targetWidth / self.NUMBER_TILES), int(targetHeight / self.NUMBER_TILES)
      for dx in range(self.NUMBER_TILES):
        for dy in range(self.NUMBER_TILES):
          if((dx + dy) % 2):
            qp.drawRect(offsetX + dx * deltaX, offsetY + dy * deltaY, deltaX, deltaY)
      # fini
      qp.end()

  def setDisplayPreview(self, displayPreview):
    # tells the preview window where to paint the checker marks
    self.displayPreview = displayPreview
    if(hasattr(self.displayPreview, 'setObjectName')):
      self.displayPreview.setObjectName('preview')
    
  def keyPressEvent(self, event):
    # catches escape key
    if(event.key() == QtCore.Qt.Key.Key_Escape):
      self.close()
    else:
      # activate normal event handling
      QtWidgets.QMainWindow.keyPressEvent(self, event)

# custom QComboBox to fix Qt layout bug on Mac :(
# exploit Mac stupidity to also implement pass-through of Ctrl key presses
class QComboBoxMac(QtWidgets.QComboBox):
  def __init__(self, *args, **kwargs):
    super(QComboBoxMac, self).__init__(*args, **kwargs)
    self.setAttribute(QtCore.Qt.WidgetAttribute.WA_LayoutUsesWidgetRect)
    self.installEventFilter(self)

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def keyPressEvent(self, event):
    # pass through events with Ctrl modifier
    if(event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier):
      event.ignore()
      return
    else:
      # normal event processing
      QtWidgets.QComboBox.keyPressEvent(self, event)

# subclass delegate to format FontoBox
class FontoDelegate(QtWidgets.QStyledItemDelegate):
  def __init__(self, *args, **kwargs):
    super(FontoDelegate, self).__init__(*args, **kwargs)

  def paint(self, painter, option, index):
    if(index.row() == 0):
      painter.save()
      font = painter.font()
      font.setBold(True)
      painter.setFont(font)
      if(len(FontoBox.recentFonts)):
        painter.drawText(QtCore.QRectF(option.rect), 'Recently used fonts')
      else:
        painter.drawText(QtCore.QRectF(option.rect), 'All fonts')
      painter.restore()
    elif(len(FontoBox.recentFonts) and (index.row() == len(FontoBox.recentFonts) + 1)):
      painter.save()
      font = painter.font()
      font.setBold(True)
      painter.setFont(font)
      painter.drawText(QtCore.QRectF(option.rect), 'All fonts')
      painter.restore()
    else:
      if(index.row() < len(FontoBox.recentFonts) + 1):
        painter.save()
        if(DARK_MODE):
          brushColor = QtGui.QColor('#9999bb')
        else:
          brushColor = QtGui.QColor('#aaaaee')
        painter.setBrush(QtGui.QBrush(brushColor))
        painter.setPen(QtCore.Qt.PenStyle.NoPen)
        r = QtCore.QRect(option.rect)
        painter.drawRect(r)
        painter.restore()
    
      QtWidgets.QStyledItemDelegate.paint(self, painter, option, index)
    
# custom QComboBox for font selection
class FontoBox(QComboBoxMac):
  # set up class variable to allow communication between Font selectors
  recentFonts = []
  MAX_RECENT_FONTS = 5
  
  def __init__(self, *args, **kwargs):
    super(FontoBox, self).__init__(*args, **kwargs)
    self.fontoDelegate = FontoDelegate()
    self.setItemDelegate(self.fontoDelegate)
    self.recentItems, self.allItems, self.combinedItems = [], [], []
    self.activated.connect(self.catchActivation)
    self.doActivate = True
    
  def toggleActivate(self, state=True):
    # controls whether updating of QComboBox triggers update of recent font list
    self.doActivate = state
  
  def addItems(self, items=[]):
    # overrides normal addItem routine
    self.allItems = items
    
    # prepare combined list
    self.prepareCombinedItems()
      
    # call original function
    QtWidgets.QComboBox.addItems(self, self.combinedItems)
    
    # now disable certain items
    model = self.model()
    model.item(0).setEnabled(False)
    if(len(self.recentItems)):
      model.item(len(self.recentItems) + 1).setEnabled(False)
      
    self.setModel(model)
      
  def setMe(self, item=None):
    # sets QComboBox to correct entry
    if(item != None):
      defaultFont = 'DejaVu Sans'
      if(item in self.combinedItems):
        if(not (item in FontoBox.recentFonts)):
          self.updateRecentFonts(item=item)
        self.setCurrentText(item)
        return
      elif(defaultFont in self.combinedItems):
        self.setCurrentText(defaultFont)
        return

    # set to first entry by default
    if(len(self.combinedItems)):
      self.setCurrentIndex(1)

  def showPopup(self):
    # hijack showPopup() to update recent font list on the fly
    model = self.model()
    
    # first determine how many recent fonts we have at present
    if('All fonts' in self.combinedItems):
      index = self.combinedItems.index('All fonts')
      # strip out all recent fonts from combinedItems
      self.combinedItems = self.combinedItems[index:]
      
    # now loop through model and strip out all entries before 'All fonts'
    modelData = [model.data(model.index(i, 0)) for i in range(model.rowCount())]
    if('All fonts' in modelData):
      number = 0
      index = modelData.index('All fonts')
      while number < index:
        model.removeRow(0)
        number += 1
      
    if(len(FontoBox.recentFonts)):
      for item in sorted(FontoBox.recentFonts)[::-1]:
        self.combinedItems = [item] + self.combinedItems
        model.insertRow(0, QtGui.QStandardItem(item))
      
      self.combinedItems = ['Recently used fonts'] + self.combinedItems
      model.insertRow(0, QtGui.QStandardItem('Recently used fonts'))
      model.item(0).setEnabled(False)

      # set combo box to entry in recent fonts if it exists there
      if(self.currentText() in self.combinedItems):
        index = self.combinedItems.index(self.currentText())
        self.setCurrentIndex(index)

    self.setModel(model)

    # execute normal action
    QComboBoxMac.showPopup(self)

  def prepareCombinedItems(self):
    # helper function that updates combined font list
    if(len(FontoBox.recentFonts)):
      self.combinedItems = ['Recently used fonts'] + sorted(FontoBox.recentFonts)
    else:
      self.combinedItems = []
    self.combinedItems += ['All fonts'] + self.allItems
    
  def catchActivation(self):
    # tracks activation to update fonts accordingly
    if(self.doActivate):
      # check whether font exists at all
      if(self.currentText() in self.allItems):
        self.updateRecentFonts(self.currentText())
      else:
        pass
    
  def updateRecentFonts(self, item=None):
    # updates recent font list
    if(item != None):
      if(item in FontoBox.recentFonts):
        FontoBox.recentFonts = [item] + [i for i in FontoBox.recentFonts if (i != item)]
      else:
        FontoBox.recentFonts = [item] + FontoBox.recentFonts

      # truncate recent font list to desired length
      FontoBox.recentFonts = FontoBox.recentFonts[:FontoBox.MAX_RECENT_FONTS]
    
      # update combined list accordingly
      self.prepareCombinedItems()

# custom QRadioButton for interactive display of tooltips
class QRadioButtonToolTip(QtWidgets.QRadioButton):
  def __init__(self, *args, **kwargs):
    super(QRadioButtonToolTip, self).__init__(*args, **kwargs)
    self.installEventFilter(self)

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def paintEvent(self, event):
    if((UI_STYLE != 'default') and DARK_MODE):
      # trick Fusion style into changing outline color of radio indicator
      pal = self.palette()
      pal.setColor(QtGui.QPalette.ColorRole.Window, QtGui.QColor(*UI_TEXT_COLOR))
      self.setPalette(pal)
      
    # original draw event
    QtWidgets.QRadioButton.paintEvent(self, event)

# custom QLabel for interactive display of tooltips
class QLabelToolTip(QtWidgets.QLabel):
  def __init__(self, *args, **kwargs):
    super(QLabelToolTip, self).__init__(*args, **kwargs)
    self.installEventFilter(self)

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

# classes for drawing consistent lines
class HLine(QtWidgets.QFrame):
  # draws a horizontal line
  def __init__(self, *args, **kwargs):
    super(HLine, self).__init__(*args, **kwargs)
    self.setFrameShape(QtWidgets.QFrame.Shape.HLine)
    self.setFrameShadow(QtWidgets.QFrame.Shadow.Sunken)
    self.setFrameShadow(QtWidgets.QFrame.Shadow.Raised)
    self.setObjectName('hdivide')

class VLine(QtWidgets.QFrame):
  # draws a vertical line
  def __init__(self, *args, **kwargs):
    super(VLine, self).__init__(*args, **kwargs)
    self.setFrameShape(QtWidgets.QFrame.Shape.VLine)
    self.setFrameShadow(QtWidgets.QFrame.Shadow.Sunken)
    self.setObjectName('vdivide')

# a custom checkbox to deal with ui styles
class QCheckBoxToolTip(QtWidgets.QCheckBox):
  def __init__(self, *args, **kwargs):
    super(QCheckBoxToolTip, self).__init__(*args, **kwargs)
    self.installEventFilter(self)
    
  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def paintEvent(self, event):
    # original draw event
    QtWidgets.QCheckBox.paintEvent(self, event)

    if((UI_STYLE != 'default') and DARK_MODE):
      s = self.size()
      qp = QtGui.QPainter()
      if(qp is not None):
        qp.begin(self)
        qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
    
        pen = QtGui.QPen()
        pen.setStyle(QtCore.Qt.PenStyle.SolidLine)
        pen.setJoinStyle(QtCore.Qt.PenJoinStyle.RoundJoin)
        pen.setColor(QtGui.QColor(*UI_TEXT_COLOR))
        qp.setPen(pen)
    
        # overlay gradient
        if(self.isChecked()):
          grad = QtGui.QLinearGradient(0, s.height(), 0, 0)
        else:
          grad = QtGui.QLinearGradient(0, 0, 0, s.height())
        col1, col2 = [255, 255, 255], [180, 180, 180]
        grad.setColorAt(0, QtGui.QColor(*col1, 150))
        grad.setColorAt(0.2, QtGui.QColor(*col1, 0))
        grad.setColorAt(0.88, QtGui.QColor(*col2, 0))
        grad.setColorAt(1, QtGui.QColor(*col2, 120))
        qp.setBrush(QtGui.QBrush(grad))
    
        qp.drawRoundedRect(0, 3, s.height() - 6, s.height() - 6, 2, 2)
        qp.end()

# custom QPushButton to fix Qt layout bug on Mac :(
class QPushButtonMac(QtWidgets.QPushButton):
  def __init__(self, *args, **kwargs):
    super(QPushButtonMac, self).__init__(*args, **kwargs)
    self.setAttribute(QtCore.Qt.WidgetAttribute.WA_LayoutUsesWidgetRect)
    self.installEventFilter(self)

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

# subclass color button to have it work both with Fusion style and not mess up tool tips
class QPushButtonColor(QPushButtonMac):
  def __init__(self, *args, **kwargs):
    super(QPushButtonColor, self).__init__(*args, **kwargs)
    self.myColor = None
    
  def setMyColor(self, red=0, green=0, blue=0, alpha=255):
    # sets button color
    self.myColor = [red, green, blue, alpha]
    # issue update to ensure that color is displayed
    self.update()
    
  def getMyColor(self):
    # returns the current color value
    if(self.myColor == None):
      return [0, 0, 0, 1]
    
    return self.myColor

  def paintEvent(self, event):
    # original draw event
    #QPushButtonMac.paintEvent(self, event)
  
    # draws colored area in button
    if(self.myColor != None):
      s = self.size()
      qp = QtGui.QPainter()
      if(qp is not None):
        qp.begin(self)
        qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
        qp.setPen(QtCore.Qt.PenStyle.NoPen)
        qp.setBrush(QtGui.QBrush(QtGui.QColor(*self.myColor)))
        qp.drawRoundedRect(0, 0, s.width(), s.height(), 3, 3)
  
        # overlay gradient
        pen = QtGui.QPen()
        pen.setStyle(QtCore.Qt.PenStyle.SolidLine)
        pen.setJoinStyle(QtCore.Qt.PenJoinStyle.RoundJoin)
        pen.setColor(QtGui.QColor(171, 171, 171, 255))
        pen.setWidth(1)
        pen.setCapStyle(QtCore.Qt.PenCapStyle.SquareCap)
        qp.setPen(pen)
    
        grad = QtGui.QLinearGradient(0, 0, 0, s.height())
        grad.setColorAt(0, QtGui.QColor(255, 255, 255, 150))
        grad.setColorAt(0.2, QtGui.QColor(255, 255, 255, 0))
        grad.setColorAt(0.8, QtGui.QColor(0, 0, 0, 0))
        grad.setColorAt(1, QtGui.QColor(0, 0, 0, 80))
        qp.setBrush(QtGui.QBrush(grad))
    
        qp.drawRoundedRect(0, 0, s.width(), s.height(), 3, 3)
        qp.end()

# a funky new push button
class QPushButtonCheckable(QPushButtonMac):
  def __init__(self, *args, **kwargs):
    super(QPushButtonCheckable, self).__init__(*args, **kwargs)
    self.setCheckable(True)
    self.checkMe = False
    self.padMe = 0
    self.highlightMe = False
    self.installEventFilter(self)
    self.penColor = QtGui.QColor(0, 0, 0, 255)
    self.baseStyle = 'QPushButton:hover {color: #6258b8;}\n\
                    QPushButton:checked {background-color: $1$; color: black;}\n\
                    QPushButton:checked:hover {background-color: $2$; color: white;}\n\
                    QPushButton:checked:!enabled {background-color: $3$; color: #787878;}\n'
    
  def setCheckMe(self, state):
    # controls whether to draw check mark when checked
    self.checkMe = state
    
  def setHighlightMe(self, state):
    # controls whether to highlight (checked) pushbutton
    self.highlightMe = state
    if(state):
      styleString = self.baseStyle.replace('$1$', '#70ff66')
      styleString = styleString.replace('$2$', '#90ff88')
      styleString = styleString.replace('$3$', '#c0ffbb')
    else:
      styleString = self.baseStyle.replace('$1$', '#7066ff')
      styleString = styleString.replace('$2$', '#9088ff')
      styleString = styleString.replace('$3$', '#c0bbff')
    self.setStyleSheet(styleString)

  def setPadMe(self, value):
    # controls whether to pad button
    self.padMe = value
    # activate specific style sheet
    if(self.padMe):
      self.setObjectName('listos')
    else:
      self.setObjectName('')

  def paintEvent(self, event):
    # original draw event
    QPushButtonMac.paintEvent(self, event)
    
    s = self.size()
    qp = QtGui.QPainter()
    if(qp is not None):
      qp.begin(self)
      qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
  
      pen = QtGui.QPen()
      pen.setStyle(QtCore.Qt.PenStyle.SolidLine)
      pen.setJoinStyle(QtCore.Qt.PenJoinStyle.RoundJoin)
  
      # draw check mark?
      if(self.checkMe and self.isChecked()):
        pen.setColor(self.penColor)
        pen.setWidth(2)
        pen.setCapStyle(QtCore.Qt.PenCapStyle.RoundCap)
        qp.setPen(pen)
        qp.setBrush(QtCore.Qt.BrushStyle.NoBrush)
        
        path = QtGui.QPainterPath()
        path.moveTo(self.padMe + 5, self.padMe + 5)
        path.lineTo(s.width() - 5, s.height() - 5)
        path.moveTo(self.padMe + 5, s.height() - 5)
        path.lineTo(s.width() - 5, self.padMe + 5)
        qp.drawPath(path)
      
      # overlay gradient
      qp.setPen(QtCore.Qt.PenStyle.NoPen)
  
      if(self.isChecked()):
        grad = QtGui.QLinearGradient(self.padMe, s.height(), self.padMe, self.padMe)
      else:
        grad = QtGui.QLinearGradient(self.padMe, self.padMe, self.padMe, s.height())
      col1, col2 = [255, 255, 255], [180, 180, 180]
      grad.setColorAt(0, QtGui.QColor(*col1, 150))
      grad.setColorAt(0.2, QtGui.QColor(*col1, 0))
      grad.setColorAt(0.88, QtGui.QColor(*col2, 0))
      grad.setColorAt(1, QtGui.QColor(*col2, 120))
      qp.setBrush(QtGui.QBrush(grad))
  
      qp.drawRoundedRect(self.padMe, self.padMe, s.width(), s.height(), 2, 2)
      qp.end()
    
  def eventFilter(self, source, event):
    if(source == self):
      if(event.type() == QtCore.QEvent.Type.HoverEnter):
        self.hoverMe(True)
      elif(event.type() == QtCore.QEvent.Type.HoverLeave):
        self.hoverMe(False)
      elif(self.toolTip()):
        # this button has a tooltip alright
        if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
          # ignore this one
          return True
    return QtWidgets.QWidget.eventFilter(self, source, event)

  def hoverMe(self, toggle=True):
    if(toggle):
      self.penColor = QtGui.QColor(255, 255, 255, 255)
    else:
      self.penColor = QtGui.QColor(0, 0, 0, 255)
    
# a funky new push button with smaller label
class QPushButtonCheckableMini(QPushButtonCheckable):
  def __init__(self, *args, **kwargs):
    for item in ['parent', 'kind']:
      if(item in kwargs):
        self.__dict__[item] = kwargs[item]
        del kwargs[item]
      else:
        self.__dict__[item] = None
    super(QPushButtonCheckableMini, self).__init__(*args, **kwargs)
    
    # set size
    self.setMaximumSize(QtCore.QSize(int(11 * SCALEFONT + 1), int(15 * SCALEFONT + 1)))
    self.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), int(15 * SCALEFONT + 1)))
    
    # keep track of highlight
    self.isHigh = False
    
    # setPointSize is ignored, have to go via QSS :(
    self.setObjectName('mini')
    
    # install event filtering for right clicks
    self.removeEventFilter(self)
    self.installEventFilter(self)

  def setHigh(self, state=True):
    # function to change highlight status
    if(state != self.isHigh):
      self.isHigh = state
      useFont = self.font()
      useFont.setBold(self.isHigh)
      self.setFont(useFont)

  def paintEvent(self, event):
    # original draw event
    QtWidgets.QPushButton.paintEvent(self, event)

    # mark as active
    if(self.isHigh):
      s = self.size()
      penWidth = 1
      
      qp = QtGui.QPainter()
      if(qp is not None):
        qp.begin(self)
        qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
        pen = QtGui.QPen()
        pen.setStyle(QtCore.Qt.PenStyle.SolidLine)
        pen.setJoinStyle(QtCore.Qt.PenJoinStyle.RoundJoin)
        pen.setColor(QtGui.QColor(255, 0, 0, 255))
        pen.setWidth(penWidth)
        pen.setCapStyle(QtCore.Qt.PenCapStyle.SquareCap)
        qp.setPen(pen)
        qp.setBrush(QtCore.Qt.BrushStyle.NoBrush)
        qp.drawRoundedRect(int(penWidth / 2), int(penWidth / 2), s.width() - penWidth, s.height() - penWidth, 2, 2)
        qp.end()
    
  def eventFilter(self, source, event):
    if(source == self):
      if((event.type() == QtCore.QEvent.Type.MouseButtonPress) and (event.button() in [QtCore.Qt.MouseButton.RightButton, QtCore.Qt.MouseButton.MiddleButton])):
        if(self.parent != None):
          if(self.kind.startswith('master_')):
            if(event.button() == QtCore.Qt.MouseButton.RightButton):
              if(self.isChecked()):
                self.setChecked(False)
              splittor = self.kind.split('_')
              kind = splittor[-1]
              # activate button callback
              self.parent.toggloCallbackMaster(kind, self)
            elif(self.kind != 'master_extra'):
              kind2 = self.kind.split('_')[-1]
              self.parent.configureMeHereMyMaster(kind=kind2, callButton=self)
          else:
            try:
              if(self.kind == 'resid'):
                index = int(self.text()) - 1
              else:
                index = int(self.text())
            except:
              index = 0
            if(event.button() == QtCore.Qt.MouseButton.RightButton):
              self.parent.makeMeActive(kind=self.kind, index=index - 1)
            else:
              self.parent.configureMeHere(kind=self.kind, index=index - 1, callButton=self)
      elif(self.toolTip()):
        # this button has a tooltip alright
        if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
          # ignore this one
          return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

# custom QToolButton to fix Qt layout bug on Mac :(
class QToolButtonMac(QtWidgets.QToolButton):
  def __init__(self, *args, **kwargs):
    super(QToolButtonMac, self).__init__(*args, **kwargs)
    self.setAttribute(QtCore.Qt.WidgetAttribute.WA_LayoutUsesWidgetRect)
    self.installEventFilter(self)

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

# custom QWidget to fix Qt layout bug on Mac :(
class QWidgetMac(QtWidgets.QWidget):
  def __init__(self, *args, **kwargs):
    super(QWidgetMac, self).__init__(*args, **kwargs)
    self.setAttribute(QtCore.Qt.WidgetAttribute.WA_LayoutUsesWidgetRect)

  def paintEvent(self, event):
    # hijack paint event for customized background
    if(self.objectName() in ['out1', 'out2', 'out3']):
      # draw rounded corners
      s = self.size()
      qp = QtGui.QPainter()
      if(qp is not None):
        qp.begin(self)
        qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
        if(self.objectName() == 'out1'):
          qp.setBrush(QtCore.Qt.BrushStyle.NoBrush)
        elif(self.objectName() == 'out2'):
          if(myapp.ui.plotArea.themeUI == 'default'):
            qp.setBrush(QtGui.QColor(*UI_BASE_COLOR))
          else:
            qp.setBrush(QtGui.QColor(*UI_ALTERNATE_BASE_COLOR))
        else:
          # just for testing at present
          qp.setBrush(QtGui.QColor(255,0,0,255))
        qp.drawRoundedRect(0, 0, s.width(), s.height(), 2, 2)
        qp.end()
    else:
      QtWidgets.QWidget.paintEvent(self, event)
      
# custom QMenu for pass-through of Ctrl-key
class MuhMenu(QtWidgets.QMenu):
  def __init__(self, *args, **kwargs):
    super(MuhMenu, self).__init__(*args, **kwargs)

  def keyPressEvent(self, event):
    # pass through events with Ctrl modifier
    if(event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier):
      # somehow event pass-through not working
      # explicitly call keypressevent of main gui as pass through from QMenu fails for some reason
      myapp.keyPressEvent(event)
      event.ignore()
    else:
      # normal event processing
      QtWidgets.QMenu.keyPressEvent(self, event)

  # enable normal tab stops
  def focusNextPrevChild(self, next):
    return QtWidgets.QWidget.focusNextPrevChild(self, next)

# custom-styled QMenu
class KuhMenu(MuhMenu):
  def __init__(self, *args, **kwargs):
    super(KuhMenu, self).__init__(*args, **kwargs)
    self.borderRadius = 5
    
  def paintEvent(self, event):
    # draw rounded corners
    s = self.size()
    qp = QtGui.QPainter()
    if(qp is not None):
      qp.begin(self)
      qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
      qp.setBrush(QtGui.QColor(*UI_BASE_COLOR))
      qp.drawRoundedRect(0, 0, s.width(), s.height(), self.borderRadius, self.borderRadius)
      qp.end()

# custom QStatusbar for colored error messages
class KuhStatusBar(QtWidgets.QStatusBar):
  def __init__(self, *args, **kwargs):
    super(KuhStatusBar, self).__init__(*args, **kwargs)
    self.timer = None
    
  def showMessage(self, message='', timer=10000, color='red'):
    # parse color
    if(color == 'red'):
      color = 'rgba' + repr(UI_ALERT_COLOR)
    elif(color == 'blue'):
      color = 'rgba' + repr(UI_NOTIFICATION_COLOR)
    
    # set color
    self.setStyleSheet('QStatusBar{color:' + color + '; font-weight:bold;}')
    QtWidgets.QStatusBar.showMessage(self, message, timer)
    # set up timer to revert style sheet change (probably not needed b/c all messages should appear in same color)
    if(self.timer == None):
      self.timer = QtCore.QTimer()
      self.timer.timeout.connect(self.myTimeOut)
      self.timer.setSingleShot(True)
      self.timer.start(timer)
    else:
      # if timer already exists, reset timer
      self.timer.start(timer)
    
  def myTimeOut(self):
    # resets time sheet
    self.setStyleSheet('QStatusBar{color:black; font-weight:normal;}')
    self.timer = None
        
# a delegate for custom display of float numbers
class FloatFormatDelegate(QtWidgets.QStyledItemDelegate):
  def __init__(self):
    super(FloatFormatDelegate, self).__init__()

  def displayText(self, value, locale):
    if(type(value) == float):
      return QtWidgets.QStyledItemDelegate.displayText(self, value, locale).replace(',', '.')
    else:
      return QtWidgets.QStyledItemDelegate.displayText(self, value, locale)

  def paint(self, painter, option, index):
    # this part needed to be able to control tabel bg. independently
    if(index.row() % 2):
      # use the alternate color
      brushColor = QtGui.QColor(*UI_ALTERNATE_BASE_COLOR)
    else:
      # use the regular color
      brushColor = QtGui.QColor(*UI_BASE_COLOR)
    painter.save()
    painter.setBrush(QtGui.QBrush(brushColor))
    painter.setPen(QtCore.Qt.PenStyle.NoPen)
    r = QtCore.QRect(option.rect)
    painter.drawRect(r)
    painter.restore()

    QtWidgets.QStyledItemDelegate.paint(self, painter, option, index)

# subclass delegate to highlight certain cells
class FloatFormatDelegateColor(FloatFormatDelegate):
  def paint(self, painter, option, index):
    # this part needed to be able to control tabel bg. independently
    if(index.row() % 2):
      # use the alternate color
      brushColor = QtGui.QColor(*UI_ALTERNATE_BASE_COLOR)
    else:
      # use the regular color
      brushColor = QtGui.QColor(*UI_BASE_COLOR)
    painter.save()
    painter.setBrush(QtGui.QBrush(brushColor))
    painter.setPen(QtCore.Qt.PenStyle.NoPen)
    r = QtCore.QRect(option.rect)
    painter.drawRect(r)
    painter.restore()

    QtWidgets.QStyledItemDelegate.paint(self, painter, option, index)
    
    retv = index.data(QtCore.Qt.ItemDataRole.UserRole)
    if (retv):
      painter.save()
      if(retv == 1):
        # good rows
        brushColor = QtCore.Qt.GlobalColor.green
      else:
        # bad rows
        brushColor = QtCore.Qt.GlobalColor.red
      painter.setBrush(QtGui.QBrush(brushColor))
      painter.setOpacity(0.4)
      painter.setPen(QtCore.Qt.PenStyle.NoPen)
      r = QtCore.QRect(option.rect)
      painter.drawRect(r)
      painter.restore()

# custom table model for speeding up data loading (QTableView vs. QTableWidget)
class TableModel(QtCore.QAbstractTableModel):
  def __init__(self, data, parent=None):
    super(TableModel, self).__init__(parent)
    self.parent = parent
    if(hasattr(data, 'tolist')):
      # deal with numpy data
      self._data = data.tolist()
    else:
      self._data = data
    self.headers = [str(i + 1) for i in range(self.columnCount())]
    self.parent.setAlternatingRowColors(True)
    self.font = QtGui.QFont()
    self.columnNumbers, self.columnAverages, self.columnMinVal, self.columnMaxVal, self.columnSum =\
      [0] * self.columnCount(), [0] * self.columnCount(), ['n/a'] * self.columnCount(), ['n/a'] * self.columnCount(), ['n/a'] * self.columnCount()

  def getFont(self):
    return self.font

  def rowCount(self, parent=None):
    return len(self._data)

  def columnCount(self, parent=None):
    return len(self._data[0]) if self.rowCount() else 0

  def data(self, index, role=QtCore.Qt.ItemDataRole.DisplayRole):
    if role == QtCore.Qt.ItemDataRole.DisplayRole:
      row = index.row()
      if 0 <= row < self.rowCount():
        column = index.column()
        if 0 <= column < self.columnCount():
          return self._data[row][column]
    elif(role == QtCore.Qt.ItemDataRole.FontRole):
      return self.font
    elif(role == QtCore.Qt.ItemDataRole.TextAlignmentRole):
      row, column = index.row(), index.column()
      if(type(self._data[row][column]) in [float, int]):
        return QtCore.Qt.AlignmentFlag.AlignRight
      else:
        return QtCore.Qt.AlignmentFlag.AlignLeft
    elif(role == QtCore.Qt.ItemDataRole.ToolTipRole):
      column = index.column()
      if(self.columnNumbers[column] and (self.headers[column] != 'labels') and (self.headers[column] != 'labels2')):
        outstr = '<u>Stats ' + self.headers[column] + ' column</u><br/>\n'
        outstr += '&nbsp;' * 2 + 'n ' + str(self.columnNumbers[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'sum ' + self.formatNumber(self.columnSum[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'avg ' + self.formatNumber(self.columnAverages[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'min ' + self.formatNumber(self.columnMinVal[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'max ' + self.formatNumber(self.columnMaxVal[column])
        return outstr
      else:
        return
        
  def setColumnStats(self, index, number=0, csum=0, mean='n/a', minval='n/a', maxval='n/a'):
    # sets statistics information on columns
    if(index < self.columnCount()):
      self.columnNumbers[index] = number
      self.columnSum[index] = csum
      self.columnAverages[index] = mean
      self.columnMinVal[index] = minval
      self.columnMaxVal[index] = maxval
    
  def dataByIndices(self, row=0, column=0):
    if 0 <= row < self.rowCount():
      if 0 <= column < self.columnCount():
        return self._data[row][column]
        
  def headerData(self, section, orientation, role=QtCore.Qt.ItemDataRole.DisplayRole):
    if role == QtCore.Qt.ItemDataRole.DisplayRole and orientation == QtCore.Qt.Orientation.Horizontal:
      if(section < len(self.headers)):
        return self.headers[section]
      else:
        # needed for UI calls while table empty
        return ''
    return QtCore.QAbstractTableModel.headerData(self, section, orientation, role)        

  def setAllHeaders(self, headerData):
    maxIndex = min(self.columnCount(), len(headerData))
    self.headers[:maxIndex] = headerData[:maxIndex]

  def setSingleHeader(self, index, label):
    if(index < self.columnCount()):
      self.headers[index] = label
      
  def getDataRows(self, indices):
    retv = [self._data[index] for index in indices]
    return retv

  def getDataColumn(self, index):
    retv = [self._data[i][index] for i in range(self.rowCount())]
    return retv

  def getHeaders(self):
    return self.headers

  def getAllData(self):
    return self._data
  
  def setData(self, value, row, column):
    if((0 <= row < self.rowCount()) and (0 <= column < self.columnCount())):
      self._data[row][column] = value

  def pasteDataBlock(self, data, offsetRow=0, offsetColumn=0):
    for row in data:
      if(offsetRow < self.rowCount()):
        rowItems = len(row)
        self._data[offsetRow][offsetColumn:offsetColumn + rowItems] = row
      offsetRow += 1

  def formatNumber(self, number):
    # formats number for output
    NUMBER_SWITCH = 1e3
    FORMAT_DECIMAL = '{:.3f}'
    FORMAT_SCIENTIFIC = '{:.3e}'
    # determine return string
    if((self.isNumber(number)) and (np.isfinite(float(number)))):
      if((np.abs(number) > NUMBER_SWITCH) or (np.abs(number) < 1.0 / NUMBER_SWITCH)):
        numberstr = FORMAT_SCIENTIFIC.format(number)
        zerostr = FORMAT_SCIENTIFIC.format(0.0)
      else:
        numberstr = FORMAT_DECIMAL.format(number)
        zerostr = FORMAT_DECIMAL.format(0.0)
      if(numberstr == zerostr):
        numberstr = '0'
    else:
      numberstr = str(number)
    
    return numberstr

  def isNumber(self, s):
    # checks whether string is a number
    try:
      float(s)
      return True
    except ValueError:
      pass
   
    try:
      import unicodedata
      unicodedata.numeric(s)
      return True
    except (TypeError, ValueError):
      pass
    
    return False

# subclass custom table model for highlighting cells
class DataTableModel(TableModel):
  def __init__(self, data, parent=None):
    super(DataTableModel, self).__init__(data, parent)
    # cellhighlighting
    self.goodRows, self.badRows = [], []
    self.selCols = []
    self.timer = None
    self.defaultTimeOut = 5000
    self.parent.setObjectName('data')
    self.columnNumbers, self.columnAverages, self.columnMinVal, self.columnMaxVal, self.columnSum =\
      [0] * self.columnCount(), [0] * self.columnCount(), ['n/a'] * self.columnCount(), ['n/a'] * self.columnCount(), ['n/a'] * self.columnCount()

  def setFontProperty(self, prop, value):
    try:
      method = getattr(self.font, 'set' + prop)
      method(value)
    except:
      pass
    
  def data(self, index, role=QtCore.Qt.ItemDataRole.DisplayRole):
    if role == QtCore.Qt.ItemDataRole.DisplayRole:
      row = index.row()
      if 0 <= row < self.rowCount():
        column = index.column()
        if 0 <= column < self.columnCount():
          return self._data[row][column]
    elif(role == QtCore.Qt.ItemDataRole.UserRole):
      if(index.column() in self.selCols):
        if(index.row() in self.goodRows):
          return 1
        elif(index.row() in self.badRows):
          return 2
      return 0
    elif(role == QtCore.Qt.ItemDataRole.TextAlignmentRole):
      row, column = index.row(), index.column()
      if(type(self._data[row][column]) in [float, int]):
        return QtCore.Qt.AlignmentFlag.AlignRight
      else:
        return QtCore.Qt.AlignmentFlag.AlignLeft
    elif(role == QtCore.Qt.ItemDataRole.FontRole):
      return self.font
    elif(role == QtCore.Qt.ItemDataRole.ToolTipRole):
      column = index.column()
      if(self.columnNumbers[column]):
        outstr = '<u>Stats column ' + str(column + 1) + '</u><br/>\n'
        outstr += '&nbsp;' * 2 + 'n ' + str(self.columnNumbers[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'sum ' + self.formatNumber(self.columnSum[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'avg ' + self.formatNumber(self.columnAverages[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'min ' + self.formatNumber(self.columnMinVal[column]) + '<br/>\n'
        outstr += '&nbsp;' * 2 + 'max ' + self.formatNumber(self.columnMaxVal[column])
        return outstr
      else:
        return

  def setColumnStats(self, index, number=0, csum=0, mean='n/a', minval='n/a', maxval='n/a'):
    # sets statistics information on columns
    if(index < self.columnCount()):
      self.columnNumbers[index] = number
      self.columnSum[index] = csum
      self.columnAverages[index] = mean
      self.columnMinVal[index] = minval
      self.columnMaxVal[index] = maxval
    
  def setGoodBad(self, goodRows=[], badRows=[], selCols=[]):
    # updates good and bad rows
    self.goodRows, self.badRows, self.selCols = goodRows, badRows, selCols

    # set up timer to revert style sheet change (probably not needed b/c all messages should appear in same color)
    if(self.timer == None):
      self.timer = QtCore.QTimer()
      self.timer.timeout.connect(self.myTimeOut)
      self.timer.setSingleShot(True)
      self.timer.start(self.defaultTimeOut)
    else:
      # if timer already exists, reset timer
      self.timer.start(self.defaultTimeOut)
      
    # trigger update
    self.layoutChanged.emit()
    self.parent.setFocus()
    
  def myTimeOut(self):
    # resets coloring of cells
    self.goodRows, self.badRows, self.selCols = [], [], []
    self.timer = None
    # trigger update
    self.layoutChanged.emit()

# custom legend class
class MyLegend(Legend.Legend):
  def __init__(self, *args, **kwargs):
    super(MyLegend, self).__init__(*args, **kwargs)
    self.shadowDeltaX, self.shadowDeltaY = 2, -2
    self.shadowFaceColor = [0.5, 0.8, 0.9, 1.0]
    self.legendHatch, self.legendHatchMultiply, self.legendHatchLinewidth = '', 1, 1.
    self.legendEdgeFallbackColor = [0.5, 0.5, 0.5, 1.0]
    self.legendRounding = 0.2
    # ensure that _drawFrame is initialized
    self._drawFrame = True
    
  def additionalConfig(self, shadowDeltaX=None, shadowDeltaY=None, shadowFaceColor=None, legendHatch=None, legendHatchMultiply=None, legendHatchLinewidth=None, legendEdgeFallbackColor=None, legendRounding=None):
    # allows to set parameters for drop shadow
    if(shadowDeltaX != None):
      self.shadowDeltaX = shadowDeltaX
    if(shadowDeltaY != None):
      self.shadowDeltaY = shadowDeltaY
    if(shadowFaceColor != None):
      self.shadowFaceColor = shadowFaceColor
    if(legendHatch != None):
      self.legendHatch = legendHatch
    if(legendHatchMultiply != None):
      self.legendHatchMultiply = legendHatchMultiply
    if(legendHatchLinewidth != None):
      self.legendHatchLinewidth = legendHatchLinewidth
    if(legendEdgeFallbackColor != None):
      self.legendEdgeFallbackColor = legendEdgeFallbackColor
    if(legendRounding != None):
      self.legendRounding = legendRounding

  def draw(self, renderer):
    # the following is copied from matplotlib/legend.py and modified to allow customization of shadow patch
    "Draw everything that belongs to the legend."
    if not self.get_visible():
      return

    renderer.open_group('legend')
    fontsize = renderer.points_to_pixels(self._fontsize)

    # if mode == fill, set the width of the legend_box to the
    # width of the parent (minus pads)
    if self._mode in ["expand"]:
      pad = 2 * (self.borderaxespad + self.borderpad) * fontsize
      self._legend_box.set_width(self.get_bbox_to_anchor().width - pad)

    # update the location and size of the legend. This needs to
    # be done in any case to clip the figure right.
    bbox = self._legend_box.get_window_extent(renderer)
    self.legendPatch.set_bounds(bbox.x0, bbox.y0, bbox.width, bbox.height)
    self.legendPatch.set_mutation_scale(fontsize)

    if self._drawFrame:
      # this bit is modified
      useHatch = self.legendHatch * self.legendHatchMultiply
      self.legendPatch.set_hatch(useHatch)
      if(hasattr(self.legendPatch, 'set_hatch_linewidth')):
        self.legendPatch.set_hatch_linewidth(self.legendHatchLinewidth)
      if(np.isclose(sum(self.legendPatch.get_edgecolor()), 0.0)):
        # hatching is linked to edge color which is set to 0 when no edge
        # => need to trick legend here which is not a problem since we constantly generate new legend objects
        self.legendPatch.set_edgecolor(self.legendEdgeFallbackColor)
        self.legendPatch.set_linewidth(0)
      self.legendPatch.set_boxstyle("round", pad=0, rounding_size=self.legendRounding)
      # up to here
      
      if self.shadow:
        # this bit is modified
        shadow = matplotlib.patches.Shadow(self.legendPatch, self.shadowDeltaX, self.shadowDeltaY)
        shadow.set_facecolor(self.shadowFaceColor)
        if(len(self.shadowFaceColor) > 3):
          shadow.set_alpha(self.shadowFaceColor[3])
        # set edge color of shadow equal to facecolor
        shadow.set_edgecolor(self.shadowFaceColor)
        # until here
        shadow.draw(renderer)

      self.legendPatch.draw(renderer)

    self._legend_box.draw(renderer)

    renderer.close_group('legend')
    self.stale = False
   
# a custom cursor
class MyCursor(matplotlib.widgets.Cursor):
  def __init__(self, *args, **kwargs):
    super(MyCursor, self).__init__(*args, **kwargs)
    self.label = self.ax.text(1, 1, '', animated=True)
    # determine scale factor for matplotlib
    MPL_SCALEFACTOR = myapp.devicePixelRatio()
    self.label.set_fontsize(int(11 * SCALEFONT * MPL_SCALEFACTOR))
    # does not heed clip_on, mayhpas due to blitting
    self.label.set_clip_on(False)
    # need to store background for initial creation of cursor
    self.background = self.canvas.copy_from_bbox(self.ax.bbox)
    # initialize parent and ax2
    self.parent = None
    self.ax2 = None
    # initialize twin and slave
    self.twin = None
    self.twinVertical = None
    
  def setParent(self, parent=None):
    self.parent = parent
    
  def setAx2(self, ax2=None):
    self.ax2 = ax2

  def setTwin(self, twin=None):
    self.twin = twin

  def setTwinVertical(self, twinVertical=None):
    self.twinVertical = twinVertical

  def onmove(self, event):
    """on mouse motion draw the cursor if visible"""
    if self.ignore(event):
      return
    if not self.canvas.widgetlock.available(self):
      return
    if event.inaxes != self.ax:
      self.linev.set_visible(False)
      self.lineh.set_visible(False)
      self.label.set_visible(False)

      if self.needclear:
        self.canvas.draw()
        self.needclear = False
      return
    self.needclear = True
    if not self.visible:
      return
    
    self.refreshCrossHair(event)
    # check whether we are on canvas
    if ((event.xdata != None) and (event.ydata != None)):
      self._update()
      if(self.twin != None):
        self.twin._update()
      if(self.twinVertical != None):
        self.twinVertical._update()

  def refreshCrossHair(self, event):
    # update cross hair
    self.linev.set_xdata((event.xdata, event.xdata))
    self.lineh.set_ydata((event.ydata, event.ydata))
    self.linev.set_visible(self.visible and self.vertOn)
    self.lineh.set_visible(self.visible and self.horizOn)
    # update label size
    if(self.parent != None):
      currZoomLevel = self.parent.matplot.get_dpi()
      self.linev.set_linewidth(1.0 * 100.0 / currZoomLevel)
      self.lineh.set_linewidth(1.0 * 100.0 / currZoomLevel)
    # update label size
    if(self.parent != None):
      currZoomLevel = self.parent.matplot.get_dpi()
      # determine scale factor for matplotlib
      MPL_SCALEFACTOR = myapp.devicePixelRatio()
      self.label.set_fontsize(int(11 * SCALEFONT * MPL_SCALEFACTOR) * 100.0 / currZoomLevel)
    # check whether we are on canvas
    if ((event.xdata != None) and (event.ydata != None)):
      labelText = ' x: ' + self.formatNumber(event.xdata) + ' \n y: ' + self.formatNumber(event.ydata) + ' '
      if((self.parent != None) and (self.parent.isSecondAxesActive()) and (not self.parent.splitY)):
        secondX, secondY = self.parent.ax2.transData.inverted().transform((event.x, event.y))
        labelText += '\n y2: ' + self.formatNumber(secondY) + ' '
      self.label.set_text(labelText)
    # update color
    color = 'black'
    if(self.ax2 != None):
      canvasColor = self.ax2.patch.get_facecolor()
      # we need this check to deal with canvas having 'none' facecolor
      if(np.isclose(sum(canvasColor), 0.0)):
        figureColor = self.parent.matplot.get_facecolor()
        if(np.isclose(sum(figureColor), 0.0)):
          color = 'black'
        elif(sum(figureColor[0:3]) < 1.5):
          color = 'white'
      elif(sum(canvasColor[0:3]) < 1.5):
        color = 'white'
    self.label.set_color(color)
    self.lineh.set_color(color)
    self.linev.set_color(color)
    # check quadrant of plot
    if(event.x > ((self.ax.bbox.xmin + self.ax.bbox.xmax) / 2.0)):
      self.label.set_horizontalalignment('right')
    else:
      self.label.set_horizontalalignment('left')
    if(event.y > ((self.ax.bbox.ymin + self.ax.bbox.ymax) / 2.0)):
      self.label.set_verticalalignment('top')
    else:
      self.label.set_verticalalignment('bottom')
    # update label position
    self.label.set_x(event.xdata)
    self.label.set_y(event.ydata)
    self.label.set_visible(self.visible)
    # check twin
    if(self.twin != None):
      self.twin.lineh.set_visible(self.visible and self.twin.horizOn)
      self.twin.lineh.set_ydata((event.ydata, event.ydata))
      self.twin.lineh.set_color(color)
    # check twin vertical
    if(self.twinVertical != None):
      self.twinVertical.linev.set_visible(self.visible and self.twinVertical.vertOn)
      self.twinVertical.linev.set_xdata((event.xdata, event.xdata))
      self.twinVertical.linev.set_color(color)

  def getHandles(self):
    # returns handles to graphics elements
    handles = [self.linev, self.lineh, self.label]
    return handles

  def _update(self):
    if self.useblit:
      if self.background is not None:
        self.canvas.restore_region(self.background)
      # check whether axis object is visible (required for split x axis)
      if(self.ax.get_visible()):
        # include the drawing of the rectangle and lasso selectors here to get around vanishing selection box when blitting
        if(self.parent.rectangleSelector != None):
          self.parent.rectangleSelector.drawMe()
        if(self.parent.lassoSelector != None):
          self.parent.lassoSelector.drawMe()

        # draw crosshair and label last, ie in front
        self.ax.draw_artist(self.linev)
        self.ax.draw_artist(self.lineh)
        self.ax.draw_artist(self.label)
        self.canvas.blit(self.ax.bbox)
    else:
      self.canvas.draw_idle()

    return False

  def toggleVisibility(self, state=False, event=None):
    self.visible = state
    if(event != None):
      if(self.visible):
        self.refreshCrossHair(event)
        # cross hair was toggled on split axis -- should not display the cross hairs
        if(event.inaxes != self.ax):
          self.linev.set_visible(False)
          self.lineh.set_visible(False)
          self.label.set_visible(False)
      else:
        self.linev.set_visible(self.visible and self.vertOn)
        self.lineh.set_visible(self.visible and self.horizOn)
        self.label.set_visible(self.visible)
      # check whether we are on canvas
      if ((event.xdata != None) and (event.ydata != None)):
        self._update()
        if(self.twin != None):
          self.twin._update()
        if((self.twinVertical != None) and self.parent.splitY):
          self.twinVertical._update()

  def formatNumber(self, number):
    # formats number for output
    precision = 2
    NUMBER_SWITCH = 10 ** (precision + 1)
    FORMAT_DECIMAL = '{:.' + str(precision + 1) + 'f}'
    # determine return string
    if((self.isNumber(number)) and (np.isfinite(float(number)))):
      if((np.abs(number) > NUMBER_SWITCH) or (np.abs(number) < 1.0 / NUMBER_SWITCH)):
        # do this in mathtext
        # calculate exponent
        if(number != 0.0):
          try:
            exponent = int(np.floor(np.log10(np.abs(number))))
          except:
            exponent = 0
          # calculate preexponent
          try:
            pre = number / (10 ** exponent)
          except:
            pre, exponent = number, 0
        # assemble string
        if(number == 0.0):
          retstr = '{:.' + str(precision) + 'f}'
          retstr = retstr.format(0.0)
          numberstr = '$\\mathdefault{' + retstr + '}$' 
        elif(np.isclose(pre, 1.0)):
          numberstr = '$\\mathdefault{10^{' + str(exponent) + '}}$'
        elif(np.isclose(pre, -1.0)):
          numberstr = '$\\mathdefault{-10^{' + str(exponent) + '}}$'
        else:
          retstr = '{:.' + str(precision) + 'f}'
          retstr = retstr.format(pre)
          # cannot use \times here b/c of xkcd font which lacks the symbol
          numberstr = '$\\mathdefault{' + retstr + '\\ x\\ 10^{' + str(exponent) + '}}$'         
      else:
        numberstr = FORMAT_DECIMAL.format(number)
    else:
      numberstr = str(number)
    
    return numberstr

  def isNumber(self, s):
    # checks whether string is a number
    try:
      float(s)
      return True
    except ValueError:
      pass
   
    try:
      import unicodedata
      unicodedata.numeric(s)
      return True
    except (TypeError, ValueError):
      pass
    
    return False

# validators that take care of commata once and for all
class MyValidInt(QtGui.QIntValidator):
  def __init__(self, *args, **kwargs):
    super(MyValidInt, self).__init__(*args, **kwargs)
    
  def validate(self, s, pos):
    if(',' in s):
      # plainly ignores commata
      s = s.replace(',', '')
      # correct pos for the character we deleted
      if(pos):
        pos -= 1
      
    # call the validator with US locale to get decimal points right
    valid = QtGui.QIntValidator()
    valid.setLocale(QtCore.QLocale(QtCore.QLocale.Language.English))
    return valid.validate(s, pos)

class MyValidFloat(QtGui.QDoubleValidator):
  def __init__(self, *args, **kwargs):
    super(MyValidFloat, self).__init__(*args, **kwargs)

  def validate(self, s, pos):
    if(',' in s):
      # plainly ignores commata
      s = s.replace(',', '.')
      
    # call the validator with US locale to get decimal points right
    valid = QtGui.QDoubleValidator()
    valid.setLocale(QtCore.QLocale(QtCore.QLocale.Language.English))
    return valid.validate(s, pos)

# a QSpinBox that finally behaves as I want
class QSpinClick(QtWidgets.QSpinBox):
  def __init__(self, argument=None):
    super(QSpinClick, self).__init__(argument)
    self._gainedFocus = False
    self._originalMousePressEvent = QtWidgets.QSpinBox.mousePressEvent
    self._originalFocusInEvent = QtWidgets.QSpinBox.focusInEvent
    self.setAlignment(QtCore.Qt.AlignmentFlag.AlignRight)
    self.installEventFilter(self)
    self.setAccelerated(True)
    # implement your own adaptive increment as this only becomes standard in later PyQt versions
    self.adaptive = True
    # set up event filter
    self.thisLineEdit = self.lineEdit()
    self.thisLineEdit.installEventFilter(self)
    
  def setAdaptive(self, value=True):
    # toggles on/off adaptive size adjustment of increments
    self.adaptive = value

  def eventFilter(self, source, event):
    # filter events of child widget
    if(source == self.thisLineEdit):
      # hijack mouse button press on line edit
      if(event.type() == QtCore.QEvent.Type.MouseButtonPress):
        try:
          # this call throws error under older Python versions due to accessing protected member
          self.thisLineEdit.mousePressEvent(event)
        except:
          self.thisLineEdit.deselect()
        if(self._gainedFocus):
          self.selectAll()
          self._gainedFocus = False
        return True

    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def paintEvent(self, event):
    # original draw event
    QtWidgets.QSpinBox.paintEvent(self, event)
  
    # draws colored area in button
    if(UI_STYLE != 'default'):
      s = self.size()
      qp = QtGui.QPainter()
      if(qp is not None):
        qp.begin(self)
        qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
        qp.setBrush(QtCore.Qt.BrushStyle.NoBrush)
        pen = QtGui.QPen()
        pen.setStyle(QtCore.Qt.PenStyle.SolidLine)
        pen.setJoinStyle(QtCore.Qt.PenJoinStyle.RoundJoin)
        pen.setColor(QtGui.QColor(*UI_TEXT_COLOR))
        pen.setWidth(1)
        pen.setCapStyle(QtCore.Qt.PenCapStyle.SquareCap)
        qp.setPen(pen)
        qp.drawRoundedRect(0, 0, s.width(), s.height(), 3, 3)
        qp.end()

  def focusInEvent(self, event):
    self._originalFocusInEvent(self, event)
    
    # determine how we got focus
    if(event.reason() == QtCore.Qt.FocusReason.MouseFocusReason):
      self._gainedFocus = True
      
  # reimplement stepBy() to fire event upon activation of arrow buttons
  def stepBy(self, step):
    # adjust increment if needed
    if(self.adaptive):
      increment = self.singleStep()
      value = self.value()
      # don't change anything if value is zero
      if(value):
        if(step > 0):
          increment = 10 ** np.floor(np.log10(np.abs(value) / 2.0))
          if(increment * 14 <= np.abs(value)):
            increment *= 2
          elif(increment * 3.0 > np.abs(value)):
            increment /= 2
        else:
          increment = 10 ** (np.ceil(np.log10(np.abs(value) / 2.00001)) - 1.0)
          if(increment * 14 < np.abs(value)):
            increment *= 2
          elif(increment * 3.0 >= np.abs(value)):
            increment /= 2
      increment = max(increment, 1)
      increment = int(increment)
      self.setSingleStep(increment)
    # normal event first
    QtWidgets.QSpinBox.stepBy(self, step)
    # then fire editing finished event
    self.editingFinished.emit()

# a QSpinBox that finally behaves as I want
class QDoubleSpinClick(QtWidgets.QDoubleSpinBox):
  def __init__(self, argument=None):
    super(QDoubleSpinClick, self).__init__(argument)
    self._gainedFocus = False
    self._originalMousePressEvent = QtWidgets.QDoubleSpinBox.mousePressEvent
    self._originalFocusInEvent = QtWidgets.QDoubleSpinBox.focusInEvent
    self.setAlignment(QtCore.Qt.AlignmentFlag.AlignRight)
    self.installEventFilter(self)
    self.setDecimals(2)
    self.setAccelerated(True)
    # implement your own adaptive increment as this only becomes standard in later PyQt versions
    self.adaptive = True
    # set up event filter
    self.thisLineEdit = self.lineEdit()
    self.thisLineEdit.installEventFilter(self)
    self.setLocale(QtCore.QLocale(QtCore.QLocale.Language.English))
    
  def setAdaptive(self, value=True):
    # toggles on/off adaptive size adjustment of increments
    self.adaptive = value

  def eventFilter(self, source, event):
    # filter events of child widget
    if(source == self.thisLineEdit):
      # hijack mouse button press on line edit
      if(event.type() == QtCore.QEvent.Type.MouseButtonPress):
        try:
          # this call throws error under older Python versions due to accessing protected member
          self.thisLineEdit.mousePressEvent(event)
        except:
          self.thisLineEdit.deselect()
        if(self._gainedFocus):
          self.selectAll()
          self._gainedFocus = False
        return True
    elif(event.type() == QtCore.QEvent.Type.KeyPress):
      # capture the pesky comma to accept it as valid decimal delimiter
      if(event.text() == ','):
        nuEvent = QtGui.QKeyEvent(QtCore.QEvent.Type.KeyPress, 46, event.modifiers(), 0, 0, 0, text='.')
        try:
          # this call throws error under older Python versions due to accessing protected member
          self.thisLineEdit.keyPressEvent(nuEvent)
        except:
          pass
        return True

    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def paintEvent(self, event):
    # original draw event
    QtWidgets.QDoubleSpinBox.paintEvent(self, event)
  
    # draws colored area in button
    if(UI_STYLE != 'default'):
      s = self.size()
      qp = QtGui.QPainter()
      if(qp is not None):
        qp.begin(self)
        qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
        qp.setBrush(QtCore.Qt.BrushStyle.NoBrush)
        pen = QtGui.QPen()
        pen.setStyle(QtCore.Qt.PenStyle.SolidLine)
        pen.setJoinStyle(QtCore.Qt.PenJoinStyle.RoundJoin)
        pen.setColor(QtGui.QColor(*UI_TEXT_COLOR))
        pen.setWidth(1)
        pen.setCapStyle(QtCore.Qt.PenCapStyle.SquareCap)
        qp.setPen(pen)
        qp.drawRoundedRect(0, 0, s.width(), s.height(), 3, 3)
        qp.end()

  def focusInEvent(self, event):
    self._originalFocusInEvent(self, event)
    
    # determine how we got focus
    if(event.reason() == QtCore.Qt.FocusReason.MouseFocusReason):
      self._gainedFocus = True

  # reimplement stepBy() to fire event upon activation of arrow buttons
  def stepBy(self, step):
    # adjust increment if needed
    if(self.adaptive):
      increment = self.singleStep()
      value = self.value()
      # don't change anything if value is zero
      if(value):
        if(step > 0):
          increment = 10 ** np.floor(np.log10(np.abs(value) / 2.0))
          if(increment * 14 <= np.abs(value)):
            increment *= 2
          elif(increment * 3.0 > np.abs(value)):
            increment /= 2
        else:
          increment = 10 ** (np.ceil(np.log10(np.abs(value) / 2.00001)) - 1.0)
          if(increment * 14 < np.abs(value)):
            increment *= 2
          elif(increment * 3.0 >= np.abs(value)):
            increment /= 2
      increment = max(increment, 0.1)
      self.setSingleStep(increment)
    # normal event first
    QtWidgets.QDoubleSpinBox.stepBy(self, step)
    # then fire editing finished event
    self.editingFinished.emit()

# a QLineEdit that finally behaves as I want
class QLineEditClick(QtWidgets.QLineEdit):
  def __init__(self, argument=None):
    super(QLineEditClick, self).__init__(argument)
    self._gainedFocus = False
    self._originalMousePressEvent = QtWidgets.QLineEdit.mousePressEvent
    self._originalFocusInEvent = QtWidgets.QLineEdit.focusInEvent
    self.setAlignment(QtCore.Qt.AlignmentFlag.AlignRight)
    self.installEventFilter(self)
    # force English formatting ... but it is not quite as easy, also need to modify validator and setText()
    self.setLocale(QtCore.QLocale(QtCore.QLocale.Language.English))

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def focusInEvent(self, event):
    self._originalFocusInEvent(self, event)
    self.selectAll()
    
    # determine how we got focus
    if(event.reason() == QtCore.Qt.FocusReason.MouseFocusReason):
      self._gainedFocus = True

  def setText(self, text):
    # kill the comma encore
    if(self.validator() != None):
      text = text.replace(',', '.')
    QtWidgets.QLineEdit.setText(self, text)
    # ensure that long entries are scrolled to beginning
    self.setCursorPosition(0)
    
  def focusOutEvent(self, event):
    # ensure that long entries are scrolled to beginning
    QtWidgets.QLineEdit.focusOutEvent(self, event)
    self.setCursorPosition(0)

  def mousePressEvent(self, event):
    self._originalMousePressEvent(self, event)
    if(self._gainedFocus):
      self.selectAll()
      self._gainedFocus = False

# a QLineEdit that rounds numbers for display purposes
class QLineEditClickExtra(QLineEditClick):
  def __init__(self, argument=None):
    super(QLineEditClickExtra, self).__init__(argument)
    # additionally define formatter function
    self.formatString = None
    # and variable that holds the actual numerical value of the entry
    self.value = None
    
  def setFormatString(self, s):
    self.formatString = s
    
  def setValue(self, value):
    self.value = value
    self.setText(self.formatValue())

  # override focusIn event
  def focusInEvent(self, event):
    QtWidgets.QLineEdit.setText(self, str(self.value))
    self._originalFocusInEvent(self, event)
    self.selectAll()
    
    # determine how we got focus
    if(event.reason() == QtCore.Qt.FocusReason.MouseFocusReason):
      self._gainedFocus = True

  # override focusOut event
  def focusOutEvent(self, event):
    QtWidgets.QLineEdit.focusOutEvent(self, event)
    # assign current value
    try:
      self.value = float(self.text())
    except:
      pass
    if(self.formatString == None):
      QtWidgets.QLineEdit.setText(self, str(self.value))
    else:
      QtWidgets.QLineEdit.setText(self, self.formatValue())
    # ensure that long entries are scrolled to beginning
    self.setCursorPosition(0)
    
  def formatValue(self):
    # formats value according to format string
    # also strips away trailing zeros
    if(self.formatString == None):
      valueString = str(self.value)
    else:
      valueString = self.formatString.format(self.value)
      # check for scientific format
      if(('e' in valueString) or ('E' in valueString)):
        if('e' in valueString):
          pre, exp = valueString.split('e')
        else:
          pre, exp = valueString.split('e')
  
        while((len(pre) > 1) and (pre.endswith('0') or pre.endswith('.'))):
          pre = pre[:-1]
          
        valueString = pre + 'e' + exp
      else:
        while((len(valueString) > 1) and ('.' in valueString) and (valueString.endswith('0') or valueString.endswith('.'))):
          valueString = valueString[:-1]
        
    return valueString

# menu for transforming columns in the data table
class TransformerMenu(MuhMenu):
  def __init__(self, parent=None, col=1, formula=''):
    super(TransformerMenu, self).__init__()
    self.parent = parent
    self.col = col
    self.formula = formula
      
    # set up GUI
    self.buildRessource()

  def mousePressEvent(self, event):
    # check whether click event is within widget
    if(not self.underMouse()):
      # pass through event (and thereby close the QMenu)
      MuhMenu.mousePressEvent(self, event)

  def buildRessource(self):
    # build gui for setting options
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(*[2]*4)
    self.vLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignTop)
    self.vLayout.setSpacing(2)
    
    # heading
    self.transformHeadingLabel = QtWidgets.QLabel('Transform column')
    useFont = self.transformHeadingLabel.font()
    useFont.setBold(True)
    self.transformHeadingLabel.setFont(useFont)
    self.vLayout.addWidget(self.transformHeadingLabel)
    
    # formula entry
    self.transformGroup = QtWidgets.QWidget()
    self.vLayout.addWidget(self.transformGroup)
    
    self.transformLayout = QtWidgets.QHBoxLayout(self.transformGroup)
    self.transformLayout.setContentsMargins(*[2]*4)
    self.transformLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    self.transformLabel = QtWidgets.QLabel('C' + str(self.col + 1) + ' = ')
    self.transformLabel.setMaximumHeight(int(BASE_SIZE))
    self.transformLabel.setMinimumHeight(int(BASE_SIZE))
    self.transformLayout.addWidget(self.transformLabel)

    self.transformEntry = QLineEditClick()
    self.transformEntry.setToolTip('Formula for column transformation.\nUse C1, C2, .. to denote 1st, 2nd, .. column in the data sheet.\nUse ROW to denote row index.')
    self.transformEntry.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.transformEntry.setMaximumSize(QtCore.QSize(int(150 * SCALEFONT), int(BASE_SIZE)))
    if(self.formula == ''):
      self.transformEntry.setText('C' + str(self.col + 1) + ' + ROW')
    else:
      self.transformEntry.setText(self.formula)
    self.transformLayout.addWidget(self.transformEntry)
    
    # button
    self.transformButton = QPushButtonMac()
    self.transformButton.setText('Transform!')
    self.transformButton.setToolTip('Perform column transformation\n(previous cell contents will be overwritten)')
    self.transformButton.setMaximumSize(QtCore.QSize(int(100 * SCALEFONT), int(BASE_SIZE)))
    self.transformButton.setMinimumSize(QtCore.QSize(int(100 * SCALEFONT), int(BASE_SIZE)))
    self.transformButton.clicked.connect(self.doTransform)
    self.vLayout.addWidget(self.transformButton)

    self.setFocus()    
    self.focusNextChild()

  def doTransform(self):
    # starts transformation in parent object
    formula = self.transformEntry.text()
    self.parent.doTransformer(self.col, formula)
    self.close()

# menu for tabulating current function
class TabulaRasaMenu(MuhMenu):
  def __init__(self, parent=None, col=1, factor=0):
    super(TabulaRasaMenu, self).__init__()
    self.parent = parent
    self.col = col
    self.factor = factor
    self.validFloat = MyValidFloat()
      
    # set up GUI
    self.buildRessource()

  def mousePressEvent(self, event):
    # check whether click event is within widget
    if(not self.underMouse()):
      # pass through event (and thereby close the QMenu)
      MuhMenu.mousePressEvent(self, event)

  def buildRessource(self):
    # build gui for setting options
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(*[2]*4)
    self.vLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignTop)
    self.vLayout.setSpacing(2)
    
    # heading
    self.tabulateHeadingLabel = QtWidgets.QLabel('Tabulate function')
    useFont = self.tabulateHeadingLabel.font()
    useFont.setBold(True)
    self.tabulateHeadingLabel.setFont(useFont)
    self.vLayout.addWidget(self.tabulateHeadingLabel)
    
    # formula entry
    self.tabulateGroup = QtWidgets.QWidget()
    self.vLayout.addWidget(self.tabulateGroup)
    
    self.tabulateLayout = QtWidgets.QHBoxLayout(self.tabulateGroup)
    self.tabulateLayout.setContentsMargins(*[2]*4)
    self.tabulateLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    self.tabulateLabel = QtWidgets.QLabel('Noise')
    self.tabulateLabel.setMaximumHeight(int(BASE_SIZE))
    self.tabulateLabel.setMinimumHeight(int(BASE_SIZE))
    self.tabulateLayout.addWidget(self.tabulateLabel)

    self.tabulateRandomEntry = QLineEditClick()
    self.tabulateRandomEntry.setToolTip('Amplitude of normally distributed noise added to data.')
    self.tabulateRandomEntry.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.tabulateRandomEntry.setMaximumSize(QtCore.QSize(int(150 * SCALEFONT), int(BASE_SIZE)))
    self.tabulateRandomEntry.setText(str(self.factor))
    self.tabulateRandomEntry.setValidator(self.validFloat)
    self.tabulateRandomEntry.editingFinished.connect(self.sanityCheckEntry)
    self.tabulateLayout.addWidget(self.tabulateRandomEntry)
    
    # button
    self.tabulateButton = QPushButtonMac()
    self.tabulateButton.setText('Tabulate!')
    self.tabulateButton.setToolTip('Tabulate currently active function\n(previous cell contents will be overwritten)')
    self.tabulateButton.setMaximumSize(QtCore.QSize(int(100 * SCALEFONT), int(BASE_SIZE)))
    self.tabulateButton.setMinimumSize(QtCore.QSize(int(100 * SCALEFONT), int(BASE_SIZE)))
    self.tabulateButton.clicked.connect(self.doTabulate)
    self.vLayout.addWidget(self.tabulateButton)

    self.setFocus()    
    self.focusNextChild()
    
  def sanityCheckEntry(self):
    # ensures that only float numbers are entered
    value = self.tabulateRandomEntry.text()
    try:
      factor = float(value)
    except:
      factor = 0
      
    if(factor <= 0):
      self.tabulateRandomEntry.setText('0.0')

  def doTabulate(self):
    # starts transformation in parent object
    try:
      factor = float(self.tabulateRandomEntry.text())
    except:
      factor = 0
    self.parent.doTabulate(self.col, factor)
    self.close()

# the data table widget
class DataTable(QtWidgets.QTableView):
  def __init__(self, parent=None):
    super(DataTable, self).__init__(parent)
    self.setEditTriggers(QtWidgets.QAbstractItemView.EditTrigger.NoEditTriggers)
    self.parent = parent
    self.tableModel = None
    self.currentRow, self.currentCol = 0, 0
    self.pageStep = 20
    self.minColWidth = 35
    self.storeFontMetrics = False
    self.selectedRows = []

    # set up and connect table header
    hheader = self.horizontalHeader()
    self.storedToolTip = '- Click left to assign role of column\n- Click right to sort table by column values\n- Ctrl key and click left to transform column\n- Ctrl key and click right to tabulate function for chosen x values'
    hheader.setToolTip(self.storedToolTip)
    hheader.sectionClicked.connect(self.changeRole)
    hheader.setContextMenuPolicy(QtCore.Qt.ContextMenuPolicy.CustomContextMenu)
    hheader.customContextMenuRequested.connect(self.sortitsch)

    btn = self.findChild(QtWidgets.QAbstractButton)
    self.storedCornerTip = 'Click to select entire table'
    btn.setToolTip(self.storedCornerTip)

    # set delegate
    self.setItemDelegate(FloatFormatDelegateColor())

    # set up namespace
    # import numpy again
    import numpy as np
    # import common functions from numpy for ease of access
    from numpy import abs, arccos, arcsin, arctan, exp, cos, cosh, log, log2, log10, power, sin, sinh, sqrt, tan, tanh
    self.mySpace = locals()
    self.storeFormula, self.storeTabulateFactor = '', 0.0
    
    # settings for zoom of data table
    self.zoomLevels = [3, 4, 6, 8, 10, 12, 14, 16, 18, 20, 24, 28, 32, 36, 40, 48, 56, 64, 72, 80, 90, 100]
    self.currZoom = -1

  def selectionChanged(self, selected=None, deselected=None):
    # override selectionChanged() to streamline tracking selected rows
    # reset selectedRows
    self.selectedRows = []
    # pass through event
    QtWidgets.QTableView.selectionChanged(self, selected, deselected)

  def toggleToolTips(self, state=True):
    # dynamically sets and unsets tool tips
    hheader = self.horizontalHeader()
    btn = self.findChild(QtWidgets.QAbstractButton)
    if(state):
      hheader.setToolTip(self.storedToolTip)
      btn.setToolTip(self.storedCornerTip)
    else:
      hheader.setToolTip('')
      btn.setToolTip('')

  def mySizeHintForColumn(self, index):
    # custom size hint for column sizing to eke out some speed
    # INIT_SEARCH ensures that initially visible cells are used
    INIT_SEARCH, MAX_SEARCH, SPACER = 20, 50, 10
    # for size calculation, truncate float numbers after 6 significant places
    PRECISION_STRING = '%.6g'
    if(not self.storeFontMetrics):
      self.storeFontMetrics = self.fontMetrics()
    
    # analyze column contents
    data = self.tableModel.getDataColumn(index)
    if(len(data) < MAX_SEARCH):
      # search through entire table
      sizes = [self.storeFontMetrics.horizontalAdvance(PRECISION_STRING % i) if type(i) == type(2.3) else self.storeFontMetrics.horizontalAdvance(str(i)) for i in data]
      if(len(sizes)):
        return max(sizes) + SPACER
      else:
        return SPACER
    else:
      # search through MAX_SEARCH randomly chosen cells
      sizes = data[:INIT_SEARCH] + np.random.choice(data[INIT_SEARCH:], size=MAX_SEARCH - INIT_SEARCH, replace=False).tolist()
      if(len(sizes)):
        maxelement = max(sizes, key = lambda i: len(PRECISION_STRING % i) if type(i) == type(2.3) else len(str(i)))
        if(type(maxelement) == type(2.3)):
          maxelement = PRECISION_STRING % maxelement
        return self.storeFontMetrics.horizontalAdvance(str(maxelement)) + SPACER
      else:
        return SPACER

  def readjustSize(self):
    # fudge function to prevent PyQt5 from uncontrollably resizing row height
    vheader = self.verticalHeader()
    vheader.setSectionResizeMode(QtWidgets.QHeaderView.ResizeMode.Fixed)
    vheader.setDefaultSectionSize(self.rowHeight)

  def updateColumnStatistics(self, column=-1):
    # helper function to calculate column statistics
    self.columnAverages = ['n/a'] * self.tableModel.columnCount()
    if(self.tableModel.rowCount()):
      if(column + 1):
        userange = range(column, column + 1)
      else:
        userange = range(self.tableModel.columnCount())
      for index in userange:
        temp = self.tableModel.getDataColumn(index)
        temp = [i for i in temp if type(i) in [int, float]]
        if(len(temp)):
          self.tableModel.setColumnStats(index, number=len(temp), csum=np.sum(temp), mean=np.mean(temp), minval=np.min(temp), maxval=np.max(temp))
        else:
          self.tableModel.setColumnStats(index, number=0, csum=0, mean='n/a', minval='n/a', maxval='n/a')
    
  def configTable(self, dimx, dimy, retainRoles=False, retainSelection=False, init=False):
    # helper function called by different file importers
    # set row height and prevent from resizing
    self.rowHeight = int(self.fontMetrics().height() + 2)
    if(init):
      self.rowHeight = int(BASE_SIZE - 5)
    vheader = self.verticalHeader()
    vheader.setDefaultSectionSize(self.rowHeight)
    for entry in range(dimy):
      vheader.setSectionResizeMode(entry, QtWidgets.QHeaderView.ResizeMode.Fixed)
      
    # set col width
    self.colWidth = int(self.size().width() / 4.5)
    self.colWidth = min(120, self.colWidth)
    hheader = self.horizontalHeader()
    if(not init):
      maxSize = int(0.5 * self.size().width())
      # limit resizing to first 15 columns as it becomes mortally slow otherwise
      maxDimX = min(dimx, 15)
      for entry in range(maxDimX):
        hheader.setSectionResizeMode(entry, QtWidgets.QHeaderView.ResizeMode.Interactive)
        useSize = max(self.mySizeHintForColumn(entry) + 2, self.minColWidth)
        # restrict column size to max. 50% of visible area (to allow easy resizing)
        useSize = min(useSize, maxSize)
        hheader.resizeSection(entry, useSize)
    
    # set font size
    if(self.currZoom >= 0):
      font = self.tableModel.getFont()
      font.setPointSize(self.zoomLevels[self.currZoom])
    
    # set selection mode
    self.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectionBehavior.SelectRows)
    
    # select all
    if(not retainSelection):
      self.selectAll()
    
    # connect event for selection update
    self.selectionModel().selectionChanged.connect(partial(self.parent.updateData, True))
    self.selectionModel().currentChanged.connect(self.currentCellChanged)
    
    # calculate statistics on columns
    self.updateColumnStatistics()
    
    # assign roles to columns (-1 means role is undefined)
    if(retainRoles):
      for item in ['x', 'y', 'xerr', 'yerr', 'labels', 'labels2']:
        if(item in self.roles):
          # check whether target column is available
          if(self.roles[item] >= dimx):
            self.roles[item] = -1
        else:
          self.roles[item] = -1
    else:
      self.roles = {'x': -1, 'y': -1, 'xerr': -1, 'yerr': -1, 'labels': -1, 'labels2': -1}
      if (dimx):
        self.roles['x'] = 0
        if (dimx == 2):
          self.roles['y'] = 1
        elif(dimx >= 3):
          self.roles['y'] = 1
    self.rolestr = {'x':'x', 'y':'y', 'xerr':u'\N{GREEK CAPITAL LETTER DELTA}x', 'yerr':u'\N{GREEK CAPITAL LETTER DELTA}y', 'labels':'labels', 'labels2': 'labels2'}
    # asisgn numbered column headers
    headerData = [str(i + 1) for i in range(dimx)]
    # update headers for roles
    for key in self.roles:
      if(self.roles[key] + 1):
        headerData[self.roles[key]] = str(self.roles[key] + 1) + ' (' + self.rolestr[key] + ')'
    self.tableModel.setAllHeaders(headerData)
    self.setModel(self.tableModel)
    self.setFocus()

  def killTheComma(self):
    # processes sheet data and replaces all commata by period
    if(len(self.sheetData)):
      dimx, dimy = len(self.sheetData[0]), len(self.sheetData)
      nuData = []
      for row in self.sheetData:
        nuData.append([self.killHelper(i) for i in row])
        
      self.sheetData = nuData
      self.tableModel = DataTableModel(self.sheetData, self)
      self.configTable(dimx, dimy, retainRoles=True, retainSelection=False)
      self.selectAll()

      nuIndex = self.model().index(self.currentRow, self.currentCol)
      self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
      self.tableModel.layoutChanged.emit()
      
      # for some strange reason we have to connect the event again, grrrr! (should've been done in self.configTable())
      self.selectionModel().currentChanged.connect(self.currentCellChanged)
      
  def killHelper(self, item):
    if(type(item) in [int, float]):
      return item
    elif((type(item) == str) and (',' in item)):
      convItem = item.replace(',', '.')
      try:
        convItem = float(convItem)
        return convItem
      except:
        return item
    else:
      return item

  def generateEmptyTable(self, columnCount=4, rowCount=20):
    # intializes blank table
    blankData = [[''] * columnCount for i in range(rowCount)]
    self.sheetData = blankData
    self.tableModel = DataTableModel(self.sheetData, self)
    self.setModel(self.tableModel)
    self.configTable(columnCount, rowCount, init=True)

    self.clearSelection()
    if(rowCount):
      self.selectRow(0)
    nuIndex = self.model().index(self.currentRow, self.currentCol)
    self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    self.tableModel.layoutChanged.emit()

    # turn off multiple sheet option
    self.parent.resetSheetSpinBox(currVal=1, maxVal=1, currName='')

  def resizeTable(self, columnCount=4, rowCount=20):
    # grows/shrinks data table as needed
    if(self.tableModel != None):
      currData = self.tableModel.getAllData()
      
      # prepare new data, initialize empty
      nuData = [[''] * columnCount for i in range(rowCount)]
      cycleRow = min(len(currData), rowCount)
      if(len(currData)):
        cycleColumn = min(len(currData[0]), columnCount)
        
        for row in range(cycleRow):
          nuData[row][:cycleColumn] = currData[row][:cycleColumn]
      
        self.tableModel = DataTableModel(nuData, self)
        self.setModel(self.tableModel)
        self.sheetData = nuData
        
        # configure table
        self.configTable(columnCount, rowCount, retainRoles=True)

        # turn off multiple sheet option
        self.parent.resetSheetSpinBox(currVal=1, maxVal=1, currName='')

  def getDimension(self):
    return self.tableModel.rowCount(), self.tableModel.columnCount()

  def restoreTable(self, tableData=[]):
    # used by loadState fxn
    if(len(tableData)):
      self.currentRow, self.currentCol = 0, 0
      self.sheetData = tableData
      self.tableModel = DataTableModel(self.sheetData, self)
      self.setModel(self.tableModel)
      
      dimx, dimy = len(tableData[0]), len(tableData)
      self.configTable(dimx, dimy)
      
      nuIndex = self.model().index(self.currentRow, self.currentCol)
      self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
      self.tableModel.layoutChanged.emit()    

  def convertDatetime(self, item=None):
    # helper function to weed out obnoxious datetime objects
    try:
      if(type(item) == datetime.date):
        return '%02d/%02d/%4s' % (int(item.month), int(item.day), item.year)
      elif(type(item) == datetime.time):
        return '%02d:%02d:%02d' % (int(item.hour), int(item.minute), int(item.second))
      else:
        outstr = ''
        if(int(item.year) + int(item.month) + int(item.day)):
          # date information is present
          outstr = '%02d/%02d/%4s' % (int(item.month), int(item.day), item.year)
        if(int(item.hour) + int(item.minute) + int(item.second)):
          if(len(outstr)):
            outstr += ' '
          outstr += '%02d:%02d:%02d' % (int(item.hour), int(item.minute), int(item.second))
        return outstr
    except:
      return ''

  def loadODS(self, filename, transpose=False):
    # open ODS sheet
    QtWidgets.QApplication.setOverrideCursor(QtGui.QCursor(QtCore.Qt.CursorShape.WaitCursor))
    QtCore.QCoreApplication.processEvents()
    
    # check for presence of pyexcel_ods3
    if(not ODS3_PRESENT):
      self.parent.parent.statusbar.showMessage('Try installing pyexcel_ods3 module to load data file ' + filename, self.parent.parent.STATUS_TIME)
    else:
      # read data into ordered dict
      continueFlag = False
      try:
        # use read_only to dramatically speed up loading of sheet
        self.wb = get_data(filename)
      except:
        self.parent.parent.statusbar.showMessage('Cannot load data file ' + filename, self.parent.parent.STATUS_TIME)
      else:
        continueFlag = True
        
      if(continueFlag):
        # update number of available sheets and current sheet
        self.sheetNames = list(self.wb.keys())
        self.parent.resetSheetSpinBox(currVal=1, maxVal=len(self.sheetNames), currName=self.sheetNames[0], silent=True)
        
        # initally assume data is on first sheet
        self.sheetData = []
        self.sheet = self.wb[self.sheetNames[0]]
  
        # ensure minimum size of table
        if(not len(self.sheet)):
          self.sheetData = [['']]
          dimx, dimy = 1, 1
        else:
          (dimx, dimy) = (len(self.sheet[0]), len(self.sheet))
        
          # populate the table
          maxLength = 0
          for entry in range(dimy):
            self.sheetData.append(self.sheet[entry])
            maxLength = max(maxLength, len(self.sheet[entry]))
            
          # adjust rows to same length
          for entry in range(dimy):
            addItems = (maxLength - len(self.sheetData[entry])) * ['']
            self.sheetData[entry].extend(addItems)
            
          # transpose data?
          if((transpose) and (len(self.sheetData))):
            transposedData = []
            for entry1 in range(len(self.sheetData[0])):
              row = []
              for entry2 in range(len(self.sheetData)):
                row.append(self.sheetData[entry2][entry1])
              transposedData.append(row)
            self.sheetData = transposedData
            dimx, dimy = dimy, dimx
        
        self.tableModel = DataTableModel(self.sheetData, self)
        self.setModel(self.tableModel)
        self.odsFlag, self.xlsxFlag, self.xlsxFilename = True, False, None
        
        # configure table
        self.configTable(dimx, dimy)

    QtWidgets.QApplication.restoreOverrideCursor()

  def loadXLS(self, filename, transpose=False):
    # open XLS sheet
    QtWidgets.QApplication.setOverrideCursor(QtGui.QCursor(QtCore.Qt.CursorShape.WaitCursor))
    QtCore.QCoreApplication.processEvents()
    # first figure out whether this is a classic xls or a xml-based xlsx file
    splitName = filename.split('.')
    continueFlag, self.xlsxFlag, self.xlsxFilename = False, False, None
    if((len(splitName) > 1) and (splitName[-1].lower() == 'xlsx')):
      # use openpyxl
      if(not OPENPYXL_PRESENT):
        # can try to use good old xlrd for people whose Python version still supports it
        try:
          self.wb = xlrd.open_workbook(filename)
        except:
          # this exception will also catch errors due to missing xlrd (which is intended here)
          # better refer people to installing openpyxl
          self.parent.parent.statusbar.showMessage('Try installing openpyxl module to load data file ' + filename, self.parent.parent.STATUS_TIME)
        else:
          self.sheetNames = self.wb.sheet_names()
          continueFlag = True
      else:
        self.xlsxFlag = True
        try:
          # use read_only to dramatically speed up loading of sheet
          self.wb = openpyxl.load_workbook(filename, data_only=True, read_only=True)
        except:
          self.parent.parent.statusbar.showMessage('Cannot load data file ' + filename, self.parent.parent.STATUS_TIME)
        else:
          self.sheetNames = self.wb.sheetnames
          continueFlag = True
          self.xlsxFilename = filename
    else:
      if(XLRD_PRESENT):
        try:
          self.wb = xlrd.open_workbook(filename)
        except:
          self.parent.parent.statusbar.showMessage('Cannot load data file ' + filename, self.parent.parent.STATUS_TIME)
        else:
          self.sheetNames = self.wb.sheet_names()
          continueFlag = True
      else:
        self.parent.parent.statusbar.showMessage('To open old-style .xls files, install xlrd. Cannot read ' + filename, self.parent.parent.STATUS_TIME)
    
    if(continueFlag):
      # update number of available sheets and current sheet
      self.parent.resetSheetSpinBox(currVal=1, maxVal=len(self.sheetNames), currName=self.sheetNames[0], silent=True)
      
      # initally assume data is on first sheet
      self.sheetData = []
      if(self.xlsxFlag):
        # need to escape the following call in case xlsx file is not open -- unclear?
        try:
          self.sheet = self.wb.worksheets[0]
          (dimx, dimy) = (self.sheet.max_column, self.sheet.max_row)
          
          # populate the table
          for rowRead in self.sheet.values:
            row = [self.convertDatetime(i) if (type(i) in [datetime.datetime, datetime.date, datetime.time]) else i if (i != None) else '' for i in rowRead]
            self.sheetData.append(row)
        except:
          self.parent.parent.statusbar.showMessage('Experienced some problem with access to ' + filename, self.parent.parent.STATUS_TIME)
      else:
        self.sheet = self.wb.sheet_by_index(0)
        (dimx, dimy) = (self.sheet.ncols, self.sheet.nrows)
      
        # populate the table
        for entry in range(dimy):
          row = self.sheet.row_values(entry)
          self.sheetData.append(row)
        
      # transpose data?
      if((transpose) and (len(self.sheetData))):
        transposedData = []
        for entry1 in range(len(self.sheetData[0])):
          row = []
          for entry2 in range(len(self.sheetData)):
            row.append(self.sheetData[entry2][entry1])
          transposedData.append(row)
        self.sheetData = transposedData
        dimx, dimy = dimy, dimx
      
      # ensure minimum size of table
      if(not len(self.sheetData)):
        self.sheetData = [['']]
        dimx, dimy = 1, 1
  
      self.tableModel = DataTableModel(self.sheetData, self)
      self.setModel(self.tableModel)
      self.odsFlag = False
      
      # configure table
      self.configTable(dimx, dimy)
      
      # close workbook in case of openpyxl
      if(hasattr(self.wb, 'close')):
        self.wb.close()
        self.wb, self.sheet = None, None

    QtWidgets.QApplication.restoreOverrideCursor()
  
  def changeSheet(self, currVal=1, transpose=False):
    # update number of available sheets and current sheet
    self.parent.resetSheetSpinBox(currVal=currVal, maxVal=len(self.sheetNames), currName=self.sheetNames[currVal - 1])

    # changes sheet in multi-sheet Excel file
    continueFlag = False
    if(self.odsFlag):
      # initally assume data is on first sheet
      self.sheetData = []
      self.sheet = self.wb[self.sheetNames[currVal - 1]]

      # ensure minimum size of table
      if(not len(self.sheet)):
        self.sheetData = [['']]
        dimx, dimy = 1, 1
      else:
        (dimx, dimy) = (len(self.sheet[0]), len(self.sheet))
      
        # populate the table
        maxLength = 0
        for entry in range(dimy):
          self.sheetData.append(self.sheet[entry])
          maxLength = max(maxLength, len(self.sheet[entry]))
          
        # adjust rows to same length
        for entry in range(dimy):
          addItems = (maxLength - len(self.sheetData[entry])) * ['']
          self.sheetData[entry].extend(addItems)
          
        continueFlag = True
    elif(self.xlsxFlag):
      # beware that we closed the file to prevent openpyxl from locking it for other applications
      # so, we need to open it again (or, die trying)
      try:
        self.wb = openpyxl.load_workbook(self.xlsxFilename, data_only=True, read_only=True)
        self.sheet = self.wb.worksheets[currVal - 1]
        (dimx, dimy) = (self.sheet.max_column, self.sheet.max_row)
      except:
        self.parent.parent.statusbar.showMessage('Cannot load data file ' + self.xlsxFilename, self.parent.parent.STATUS_TIME)
      else:
        # populate the table
        self.sheetData, continueFlag = [], True
        for rowRead in self.sheet.values:
          row = [self.convertDatetime(i) if (type(i) in [datetime.datetime, datetime.date, datetime.time]) else i if (i != None) else '' for i in rowRead]
          self.sheetData.append(row)
        self.wb.close()
        self.wb, self.sheet = None, None
    else:
      self.sheetData, continueFlag = [], True
      self.sheet = self.wb.sheet_by_index(currVal - 1)
      (dimx, dimy) = (self.sheet.ncols, self.sheet.nrows)
      
      # populate the table
      for entry in range(dimy):
        row = self.sheet.row_values(entry)
        self.sheetData.append(row)
    
    if(continueFlag):
      # transpose data?
      if((transpose) and (len(self.sheetData))):
        transposedData = []
        for entry1 in range(len(self.sheetData[0])):
          row = []
          for entry2 in range(len(self.sheetData)):
            row.append(self.sheetData[entry2][entry1])
          transposedData.append(row)
        self.sheetData = transposedData
        dimx, dimy = dimy, dimx
        
      # ensure minimum size of table
      if(not len(self.sheetData)):
        self.sheetData = [['']]
        dimx, dimy = 1, 1
  
      self.tableModel = DataTableModel(self.sheetData, self)
      self.setModel(self.tableModel)
      
      # configure table
      self.configTable(dimx, dimy, retainRoles=True)

  def transposeTable(self):
    # takes current data table and transposes contents
    if(self.tableModel != None):
      currData = self.tableModel.getAllData()
      
      # transpose data
      if(len(currData)):
        transposedData = []
        for entry1 in range(len(currData[0])):
          row = []
          for entry2 in range(len(currData)):
            row.append(currData[entry2][entry1])
          transposedData.append(row)
        self.sheetData = transposedData
        dimx, dimy = len(self.sheetData[0]), len(self.sheetData)
        
      self.tableModel = DataTableModel(self.sheetData, self)
      self.setModel(self.tableModel)
  
      # configure table
      self.configTable(dimx, dimy, retainRoles=True)

  def loadUnicornFile(self, filename, transpose=False):
    # opens a Unicorn file
    QtWidgets.QApplication.setOverrideCursor(QtGui.QCursor(QtCore.Qt.CursorShape.WaitCursor))
    QtCore.QCoreApplication.processEvents()
    try:
      filecontent = pc_res3(filename)
      filecontent.load()
    except:
      self.parent.parent.statusbar.showMessage('Cannot load data file ' + filename, self.parent.parent.STATUS_TIME)
    else:
      tempSheetData = []
      dimx, dimy = 0, 0
      for entry in ['UV', 'Cond', 'pH', 'Pressure', 'Temp', 'Conc']:
        if((entry in filecontent) and ('data' in filecontent[entry])):
          if(len(filecontent[entry]['data'])):
            if(len(tempSheetData)):
              # write header
              tempSheetData[0].extend([''] * (len(filecontent[entry]['data'][0]) + 1))
              tempSheetData[0][-len(filecontent[entry]['data'][0])] = entry
              # write data
              curry = len(filecontent[entry]['data'])
              # is new entry larger?
              if(curry > dimy):
                for tilt in range(curry - dimy):
                  tempSheetData.append([''] * dimx)
                dimy = curry
              # now append new columns
              for row in range(dimy - 1):
                if(row < curry):
                  tempSheetData[row + 1].extend([''] + list(filecontent[entry]['data'][row]))
                else:
                  tempSheetData[row + 1].extend([''] * (len(filecontent[entry]['data'][0]) + 1))
              # adjust dimensions
              dimx += 1 + len(filecontent[entry]['data'][0])
            else:
              # write header
              tempSheetData = [[''] * len(filecontent[entry]['data'][0])]
              tempSheetData[0][0] = entry
              # write data
              tempSheetData.extend([list(i) for i in filecontent[entry]['data']])
              dimx, dimy = len(filecontent[entry]['data'][0]), len(filecontent[entry]['data']) + 1

      # did we find any data to import?
      if(len(tempSheetData)):
        self.sheetData = tempSheetData
  
        # transpose data?
        if((transpose) and (len(self.sheetData))):
          transposedData = []
          for entry1 in range(len(self.sheetData[0])):
            row = []
            for entry2 in range(len(self.sheetData)):
              row.append(self.sheetData[entry2][entry1])
            transposedData.append(row)
          self.sheetData = transposedData
          dimx, dimy = dimy, dimx
  
        # ensure minimum size of table
        if(not len(self.sheetData)):
          self.sheetData = [['']]
          dimx, dimy = 1, 1

        self.tableModel = DataTableModel(self.sheetData, self)
        self.setModel(self.tableModel)
  
        # configure table
        self.configTable(dimx, dimy)            

    QtWidgets.QApplication.restoreOverrideCursor()

  def loadTextFile(self, filename, delimiter='\t', transpose=False):
    # open a text file
    QtWidgets.QApplication.setOverrideCursor(QtGui.QCursor(QtCore.Qt.CursorShape.WaitCursor))
    QtCore.QCoreApplication.processEvents()
    try:
      try:
        # try opening text file, first w/ UTF-16 (use the exception to discriminate)
        with open(filename, 'r', encoding='utf-16') as readhandle:
          filecontent = readhandle.readlines()
      except:
        # only if this fails, use regular UTF-8 but ignore any errors
        with open(filename, 'r', encoding='utf-8', errors='ignore') as readhandle:
          filecontent = readhandle.readlines()
    except:
      self.parent.parent.statusbar.showMessage('Cannot load data file ' + filename, self.parent.parent.STATUS_TIME)
    else:
      # determine row and col count
      filecontent = [i.rstrip() for i in filecontent]
      dimy = len(filecontent)
      #dimx = max([i.count(delimiter) for i in filecontent]) + 1
      dimx = max([len(i.split(delimiter)) for i in filecontent])

      # turn off multiple sheet option
      self.parent.resetSheetSpinBox(currVal=1, maxVal=1, currName='')
      
      # populate the table
      maxNumberItems = 0
      self.sheetData = []
      for row in filecontent:
        splitline = row.split(delimiter)
        splitline = [float(i) if self.isNumber(i) else i for i in splitline]
        self.sheetData.append(splitline)
        maxNumberItems = np.max((maxNumberItems, len(splitline)))
      
      # fill sheetData with empty cells to make square (otherwise will have problems in TableModel)
      for entry in self.sheetData:
        while(len(entry) < maxNumberItems):
          entry.append('')

      # transpose data?
      if((transpose) and (len(self.sheetData))):
        transposedData = []
        for entry1 in range(len(self.sheetData[0])):
          row = []
          for entry2 in range(len(self.sheetData)):
            row.append(self.sheetData[entry2][entry1])
          transposedData.append(row)
        self.sheetData = transposedData
        dimx, dimy = dimy, dimx

      # ensure minimum size of table
      if(not len(self.sheetData)):
        self.sheetData = [['']]
        dimx, dimy = 1, 1

      self.tableModel = DataTableModel(self.sheetData, self)
      self.setModel(self.tableModel)

      # configure table
      self.configTable(dimx, dimy)

    QtWidgets.QApplication.restoreOverrideCursor()

  def sortitsch(self, position):
    # first check whether control key pressed -- if so, redirect to column transformations
    # allows sorting by column
    col = self.horizontalHeader().logicalIndexAt(position)
    modifiers = QtWidgets.QApplication.queryKeyboardModifiers()
    if(modifiers & QtCore.Qt.KeyboardModifier.ControlModifier):
      self.tabulateFunction(col)
    else:
      # open context menu
      self.menu = MuhMenu(self)
      self.menu.setTitle('Sort by column ' + str(col + 1))
      
      for index, entry in enumerate(['sort ascending', 'sort descending']):
        action = QtGui.QAction(entry, self)
        action.triggered.connect(partial(self.sortitschHelper, col, index==1))
        self.menu.addAction(action)
        
      # apply styles to popup window
      if(QSTYLE != None):
        self.menu.setStyle(QSTYLE)
      if(QSTYLESHEET != None):
        self.menu.setStyleSheet(QSTYLESHEET)
        
      # display menu at current column
      menuX, menuY = self.horizontalHeader().sectionViewportPosition(col) + self.verticalHeader().width(), self.horizontalHeader().height()
      menuPos = self.mapToGlobal(QtCore.QPoint(int(menuX), int(menuY)))
      self.menu.popup(menuPos)

  def sortitschHelper(self, col, mode):
    # helper function for sort routine
    def mixedSort(col, alist, blist):
      # first analyze types
      a, b = alist[col], blist[col]
      #a, b = alist, blist
      if((type(a) == type(b)) or (isinstance(a, (int, float, complex)) and isinstance(b, (int, float, complex)))):
        return (a > b) - (a < b)
        #return cmp(a, b)
      else:
        if(isinstance(a, (int, float, complex))):
          return -1
        elif(isinstance(b, (int, float, complex))):
          return 1
        else:
          return 0

    # code continues -- generate sorted data
    currData = self.tableModel.getAllData()
    self.sheetData = sorted(currData, key=cmp_to_key(partial(mixedSort, col)))
    if(mode):
      self.sheetData = list(reversed(self.sheetData))
      
    # update table
    self.tableModel = DataTableModel(self.sheetData, self)
    self.setModel(self.tableModel)

    # configure table
    dimx, dimy = len(self.sheetData[0]), len(self.sheetData)
    self.configTable(dimx, dimy, retainRoles=True)

  def changeRole(self, col):
    # first check whether control key pressed -- if so, redirect to column transformations
    modifiers = QtWidgets.QApplication.queryKeyboardModifiers()
    if(modifiers & QtCore.Qt.KeyboardModifier.ControlModifier):
      self.transformColumn(col)
    else:
      # open context menu to assign roles
      options = ['x', 'xerr', 'y', 'yerr', 'labels', 'labels2']
      self.menu = MuhMenu(self)
      self.menu.setTitle('Assign role')
      
      for entry in options:
        action = QtGui.QAction(self.rolestr[entry], self)
        action.triggered.connect(partial(self.changeRoleHelper, col, entry))
        self.menu.addAction(action)
        
      action = QtGui.QAction('none', self)
      action.triggered.connect(partial(self.clear_role, col))
      self.menu.addAction(action)
  
      # apply styles to popup window
      if(QSTYLE != None):
        self.menu.setStyle(QSTYLE)
      if(QSTYLESHEET != None):
        self.menu.setStyleSheet(QSTYLESHEET)
        
      # display menu at current column
      menuX, menuY = self.horizontalHeader().sectionViewportPosition(col) + self.verticalHeader().width(), self.horizontalHeader().height()
      menuPos = self.mapToGlobal(QtCore.QPoint(int(menuX), int(menuY)))
      self.menu.popup(menuPos)
  
  def changeRoleHelper(self, col, role):
    # actually assigns the new role
    # did this column have any role assigned?
    for key in self.roles:
      if(self.roles[key] == col):
        self.roles[key] = -1
    
    # is the new role already taken?
    if (self.roles[role] + 1):
      # reset old label
      self.tableModel.setSingleHeader(self.roles[role], str(self.roles[role] + 1))
      
    # assign new role
    self.roles[role] = col
    self.tableModel.setSingleHeader(col, str(col + 1) + ' (' + self.rolestr[role] + ')')
    self.tableModel.layoutChanged.emit()
    
    # trigger data update
    self.parent.updateData(docheck=True)

  def clear_role(self, col):
    # unassigns role of column
    # did this column have any role assigned?
    for key in self.roles:
      if(self.roles[key] == col):
        self.roles[key] = -1
        
    # reset label
    self.tableModel.setSingleHeader(col, str(col + 1))
    self.tableModel.layoutChanged.emit()

    # trigger data update
    self.parent.updateData(docheck=True)

  def tabulateFunction(self, col):
    # opens context menu to allow transformation of column contents
    self.menu = TabulaRasaMenu(self, col, self.storeTabulateFactor)
    self.menu.setTitle('Tabulate function')
    
    # apply styles to popup window
    if(QSTYLE != None):
      self.menu.setStyle(QSTYLE)
    if(QSTYLESHEET != None):
      self.menu.setStyleSheet(QSTYLESHEET)
      
    # display menu at current column
    menuX, menuY = self.horizontalHeader().sectionViewportPosition(col) + self.verticalHeader().width(), self.horizontalHeader().height()
    menuPos = self.mapToGlobal(QtCore.QPoint(int(menuX), int(menuY)))
    self.menu.popup(menuPos)
    
  def doTabulate(self, col, factor):
    # performs the actual function tabulation
    self.storeTabulateFactor = factor
    targetCol, useCol = col, self.roles['x']
    if(not (self.roles['x'] + 1)):
      self.parent.parent.statusbar.showMessage('Cannot tabulate function as no x column is assigned in data table.', self.parent.parent.STATUS_TIME)
    else:
      # do some reporting
      goodRows, badRows = [], []
        
      # determine selected rows
      if(len(self.selectedRows)):
        selind = self.selectedRows
      else:
        selind = self.selectionModel().selectedRows()
        selind = sorted([i.row() for i in selind])
        self.selectedRows = selind
      if (len(selind)):
        # get all selected data rows
        selectedData = self.tableModel.getDataRows(selind)
        
        # reduce data to x column
        xValues = [i[useCol] for i in selectedData]
        
        # process data to weed out problematic rows
        prunedValues, prunedIndices = [], []
        for index, value in enumerate(xValues):
          # only process rows where x value is numeric and finite
          if((type(value) in [float, int]) and (not np.isnan(value)) and (not np.isinf(value))):
            prunedValues.append(value)
            prunedIndices.append(selind[index])
          else:
            badRows.append(selind[index])
        
        # prepare data tabulation
        if(len(prunedValues)):
          prunedValues = np.array(prunedValues)
          
          # do the concrete tabulation
          x, fVal = self.parent.parent.fit[self.parent.parent.activeFit].evaluateFunc(x=prunedValues)
          if(factor):
            fVal += np.random.normal(0, factor, len(fVal))
  
          # everything has worked, so assign transformed values to data table
          for index, value in enumerate(fVal):
            if((not np.isnan(value)) and (not np.isinf(value))):
              self.tableModel.setData(float(value), prunedIndices[index], targetCol)
              goodRows.append(prunedIndices[index])
            else:
              self.tableModel.setData('', prunedIndices[index], targetCol)
              badRows.append(prunedIndices[index])
            # refresh table view
            cellIndex = self.tableModel.index(prunedIndices[index], targetCol)
            self.tableModel.dataChanged.emit(cellIndex, cellIndex, [QtCore.Qt.ItemDataRole.DisplayRole])
            
          # calculate statistics on columns
          self.updateColumnStatistics(column=targetCol)
  
        self.tableModel.setGoodBad(goodRows=goodRows, badRows=badRows, selCols=[targetCol, useCol])

  def transformColumn(self, col):
    # opens context menu to allow transformation of column contents
    self.menu = TransformerMenu(self, col, self.storeFormula)
    self.menu.setTitle('Transform column')
    
    # apply styles to popup window
    if(QSTYLE != None):
      self.menu.setStyle(QSTYLE)
    if(QSTYLESHEET != None):
      self.menu.setStyleSheet(QSTYLESHEET)
      
    # display menu at current column
    menuX, menuY = self.horizontalHeader().sectionViewportPosition(col) + self.verticalHeader().width(), self.horizontalHeader().height()
    menuPos = self.mapToGlobal(QtCore.QPoint(int(menuX), int(menuY)))
    self.menu.popup(menuPos)
    
  def doTransformer(self, col, formula):
    # performs the actual column transformation
    targetCol = col
    if(formula == 'C' + str(col)):
      self.storeFormula = ''
    else:
      self.storeFormula = formula
    
    # analyze formula to determine which columns to use
    useCol = []
    for col in range(self.tableModel.columnCount()):
      probeCol = 'C' + str(col + 1)
      # check for presence of probeCol in formula
      if(probeCol in formula):
        flag = False
        splitty = formula.split(probeCol)
        for entry in splitty[1:]:
          # next character in formula string is not a number
          if((entry == '') or (not entry[0].isdigit())):
            flag = True
        if(flag):
          useCol.append(col)
          
    # analyze whether we should use row
    useRow = False
    if('ROW' in formula):
      useRow = True
    
    # define transformer function
    try:
      # check whether we have at least one input column
      if(len(useCol)):
        funcstr = 'def transformThis(self, inputs'
        if(useRow):
          funcstr += ', ROW):'
        else:
          funcstr += '):'
        for index, entry in enumerate(useCol):
          funcstr += '\n\tC' + str(entry + 1) + ' = inputs[:, ' + str(index) + ']'
        funcstr += '\n\toutput = ' + formula + '\n\treturn output'
      else:
        funcstr = 'def transformThis(self, zeroInput'
        if(useRow):
          funcstr += ', ROW):'
        else:
          funcstr += '):'
        funcstr += '\n\toutput = zeroInput + ' + formula + '\n\treturn output'
        
      # generate ffunc in local namespace (this is needed for Python3 vs. Python2, bummer)
      namespace = self.mySpace
      exec(funcstr, namespace)
      # now define the new function in the object scope
      setattr(DataTable, 'transformThis', namespace['transformThis'])
    except:
      self.parent.parent.statusbar.showMessage('Error when setting transformation for column ' + str(targetCol), self.parent.parent.STATUS_TIME)
    else:
      # do the actual transform
      # do some reporting
      goodRows, badRows = [], []
        
      # determine selected rows
      if(len(self.selectedRows)):
        selind = self.selectedRows
      else:
        selind = self.selectionModel().selectedRows()
        selind = sorted([i.row() for i in selind])
        self.selectedRows = selind
      if (len(selind)):
        # get all selected data rows
        selectedData = self.tableModel.getDataRows(selind)
        
        # reduce data to columns we are interested in
        for index, entry in enumerate(selectedData):
          selectedData[index] = [entry[i] for i in useCol]
        
        # process data to weed out problematic rows
        prunedData, prunedIndices = [], []
        for index, row in enumerate(selectedData):
          # only process rows where all required cells are numeric
          checkRow = [0 if(type(i) in [float, int]) else 1 for i in row]
          if(sum(checkRow) == 0):
            prunedData.append(row)
            prunedIndices.append(selind[index])
          else:
            badRows.append(selind[index])
        
        # do the concrete transform
        if(len(prunedData)):
          prunedData = np.array(prunedData)
          try:
            # check whether we have at least one input column
            if(len(useCol)):
              if(useRow):
                ROW = np.array(prunedIndices) + 1
                transCol = self.transformThis(prunedData, ROW=ROW)
              else:
                transCol = self.transformThis(prunedData)
            else:
              zeroInput = np.array([0] * len(prunedIndices))
              if(useRow):
                ROW = np.array(prunedIndices) + 1
                transCol = self.transformThis(zeroInput, ROW=ROW)
              else:
                transCol = self.transformThis(zeroInput)

            # everything has worked, so assign transformed values to data table
            for index, value in enumerate(transCol):
              if((not np.isnan(value)) and (not np.isinf(value))):
                self.tableModel.setData(float(value), prunedIndices[index], targetCol)
                goodRows.append(prunedIndices[index])
              else:
                self.tableModel.setData('', prunedIndices[index], targetCol)
                badRows.append(prunedIndices[index])
              # refresh table view
              cellIndex = self.tableModel.index(prunedIndices[index], targetCol)
              self.tableModel.dataChanged.emit(cellIndex, cellIndex, [QtCore.Qt.ItemDataRole.DisplayRole])
          except:
            self.parent.parent.statusbar.showMessage('Error when applying transformation for column ' + str(targetCol), self.parent.parent.STATUS_TIME)

          # calculate statistics on columns
          self.updateColumnStatistics(column=targetCol)
  
        self.tableModel.setGoodBad(goodRows=goodRows, badRows=badRows, selCols=[targetCol])

  def hasComma(self):
    # cycles through selected cells and checks for presence of comma
    selind = self.selectionModel().selectedIndexes()
    retv = False
    index = 0
    while((not retv) and (index < len(selind))):
      if((type(selind[index].data()) == str) and (',' in selind[index].data())):
        retv = True
      index += 1

    return retv
    
  def getData(self, needXErr=True, needYErr=True):
    # returns selected data as numpy array
    # argl, selectionModel is super slow ... we should maybe store the selected rows somewhere in the object and
    # update it on selection changed
    # determine selected rows
    if(len(self.selectedRows)):
      selind = self.selectedRows
    else:
      selind = self.selectionModel().selectedRows()
      selind = sorted([i.row() for i in selind])
      self.selectedRows = selind
     
    # retrieve data from table
    retv, roles = [], []
    # check whether at least x and y assigned
    if (((self.roles['x'] + 1) or (self.roles['labels'] + 1)) and (self.roles['y'] + 1)):
      if (len(selind)):
        # get all selected data rows
        selectedData = self.tableModel.getDataRows(selind)

        # deal with labels separately to allow non-numerical entries here
        if(self.roles['labels'] > -1):
          selectedLabels = []
          index = self.roles['labels']
          for entry in selectedData:
            selectedLabels.append(entry[index])
        if(self.roles['labels2'] > -1):
          selectedLabels2 = []
          index = self.roles['labels2']
          for entry in selectedData:
            selectedLabels2.append(entry[index])
        
        # reduce data to columns we are interested in
        activeKeys = ['x', 'xerr', 'y', 'yerr']
        if(not needXErr):
          activeKeys.pop(activeKeys.index('xerr'))
        if(not needYErr):
          activeKeys.pop(activeKeys.index('yerr'))
        activeKeys = [key for key in activeKeys if (self.roles[key] > -1)]
        indices = [self.roles[key] for key in activeKeys]
        for index, entry in enumerate(selectedData):
          selectedData[index] = [entry[i] for i in indices]
        
        # do some reporting
        goodRows, badRows = [], []
        
        # prepare list for numpy array
        prunedData, no_items = [], len(activeKeys)
        for index, row in enumerate(selectedData):
          types = [1 if type(i) in [int, float] else 0 for i in row]
          if(np.sum(types) == no_items):
            # only numerical entries on row
            if(self.roles['labels'] > -1):
              row.append(selectedLabels[index])
            if(self.roles['labels2'] > -1):
              row.append(selectedLabels2[index])
            prunedData.append(row)
            goodRows.append(selind[index])
          elif(np.sum(types) > 0):
            # at least one numerical entry on row
            lengths = [len(i) for index, i in enumerate(row) if(not types[index])]
            if(np.sum(lengths) == 0):
              # all other cells empty => replace by zero
              row = [i if types[index] else 0.0 for index, i in enumerate(row)]
              if(self.roles['labels'] > -1):
                row.append(selectedLabels[index])
              if(self.roles['labels2'] > -1):
                row.append(selectedLabels2[index])
              prunedData.append(row)
              goodRows.append(selind[index])
            else:
              badRows.append(selind[index])
          else:
            # check for completely empty row
            lengths = [len(i) for index, i in enumerate(row) if(not types[index])]
            if(np.sum(lengths) != 0):
              # only spaces in line
              badRows.append(selind[index])
              
        # can we retrieve table headers?
        xlabel, ylabel = '', ''
        if(len(goodRows) and ((min(goodRows) - 1) in badRows)):
          if((self.roles['x'] + 1) or (self.roles['labels'] + 1)):
            if((min(goodRows) - 1) in selind):
              if('x' in activeKeys):
                xlabel = str(selectedData[selind.index(min(goodRows)) - 1][activeKeys.index('x')])
              else:
                xlabel = str(selectedLabels[selind.index(min(goodRows)) - 1])
            else:
              xlabel = ''
            if(len(xlabel)):
              for target in ['plot', 'resid']:
                self.parent.parent.plotArea.setAxisLabel(xlabel, axis='x', redraw=False, target=target)
                if(target=='plot'):
                  self.parent.parent.graphicsarea.configXName.setText(xlabel)

          if('y' in activeKeys):
            if((min(goodRows) - 1) in selind):
              ylabel = str(selectedData[selind.index(min(goodRows)) - 1][activeKeys.index('y')])
              if(len(ylabel)):
                for target in ['plot', 'resid']:
                  self.parent.parent.plotArea.setAxisLabel(ylabel, axis='y', redraw=False, target=target)
                  if(target=='plot'):
                    self.parent.parent.graphicsarea.configYName.setText(ylabel)

        # color table temporarily
        useKeys = ['x', 'xerr', 'y', 'yerr', 'labels', 'labels2']
        if(self.parent.errorModel or (not self.parent.errorSwitch)):
          useKeys.remove('yerr')
        if(self.parent.errorXModel or (not self.parent.errorXSwitch)):
          useKeys.remove('xerr')
        selCols = [self.roles[key] for key in useKeys if (self.roles[key] > -1)]
        self.tableModel.setGoodBad(goodRows=goodRows, badRows=badRows, selCols=selCols)
        
        # convert nested list to numpy array
        retv = prunedData
        roles = activeKeys
        if(self.roles['labels'] > -1):
          roles.append('labels')
        if(self.roles['labels2'] > -1):
          roles.append('labels2')
    
        # sort data by ascending x values
        #if(len(retv)):
          # will sort according to first entry w/in nested list which should be 'x'
        #  retv = sorted(retv)
        # turned off this feature as it causes problems for cyclic data (e.g., voltammetry)
    return retv, roles, [xlabel, ylabel]

  def wheelEvent(self, event):
    # adjusts zoom level when Ctrl is pressed also
    modifiers = QtWidgets.QApplication.queryKeyboardModifiers()
    if(modifiers & QtCore.Qt.KeyboardModifier.ControlModifier):
      if(event.angleDelta().y() > 0):
        self.zoomIn()
      else:
        self.zoomOut()
      # issue update
      self.update()
      QtCore.QCoreApplication.processEvents()
    else:
      # normal event handling
      QtWidgets.QTableView.wheelEvent(self, event)

  def zoomIn(self):
    self.zoom(increment=1)
    
  def zoomOut(self):
    self.zoom(increment=-1)

  def zoom(self, increment=1):
    # general zoom function
    font = self.tableModel.getFont()
    # check whether we run zoom() the first time
    if(self.currZoom < 0):
      ###fontSize = font.pointSize()
      ###temp = np.abs(np.array(self.zoomLevels) - fontSize)
      # better use WIDGET_FONTSIZE to properly heed stylesheet (which somehow overrides other font properties)
      temp = np.abs(np.array(self.zoomLevels) - float(WIDGET_FONTSIZE))
      self.currZoom = temp.argmin()
      
    if(((self.currZoom < len(self.zoomLevels) - 1) and (increment > 0)) or ((self.currZoom > 0) and (increment < 0))):
      # alter font size in cells
      self.currZoom += increment
      self.currZoom = int(self.currZoom)
      font.setPointSize(self.zoomLevels[self.currZoom])
      
      # alter row height
      fm = QtGui.QFontMetrics(font)
      self.rowHeight = int(fm.height() + 2)

      # alter font in horizontal header
      # make the header fonts a tad larger than the cell font
      self.verticalHeader().setStyleSheet("QHeaderView { font-size: " + str(self.zoomLevels[self.currZoom] + 2) + "pt; }")
      self.horizontalHeader().setStyleSheet("QHeaderView { font-size: " + str(self.zoomLevels[self.currZoom] + 2) + "pt; }")
      
      # resize column width proportionally
      if(len(self.sheetData)):
        dimx = min(15, len(self.sheetData[0]))
        hheader = self.horizontalHeader()
        scale = self.zoomLevels[self.currZoom] / self.zoomLevels[self.currZoom - increment]
        for entry in range(dimx):
          nuSize = int(hheader.sectionSize(entry) * scale)
          hheader.resizeSection(entry, nuSize)

      # finally readjust size
      self.readjustSize()

  def keyPressEvent(self, event):
    if event.matches(QtGui.QKeySequence.StandardKey.Copy):
      # prepare output
      if(len(self.selectedRows)):
        selind = self.selectedRows
      else:
        selind = self.selectionModel().selectedRows()
        selind = sorted([i.row() for i in selind])
        self.selectedRows = selind
      # get data
      selectedData = self.tableModel.getDataRows(selind)
      output = []
      for row in selectedData:
        row = [str(i) for i in row]
        output.append('\t'.join(row))
      output = '\n'.join(output)
      clipboard = QtWidgets.QApplication.clipboard()
      clipboard.setText(output)
    elif event.matches(QtGui.QKeySequence.StandardKey.Paste):
      clipboard = QtWidgets.QApplication.clipboard()
      clipMime = clipboard.mimeData()
      # check wether clip object contains text
      if(clipMime.hasText()):
        clipContent = clipboard.text()
        self.pasteText(pastedText=clipContent)
    elif event.matches(QtGui.QKeySequence.StandardKey.SelectAll):
      self.selectAll()
    elif(event.key() == QtCore.Qt.Key.Key_Down):
      if(self.currentRow < self.tableModel.rowCount() - 1):
        if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          self.clearSelection()
        self.currentRow += 1
        self.selectTo(self.currentRow, self.currentRow)
        nuIndex = self.model().index(self.currentRow, self.currentCol)
        self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() == QtCore.Qt.Key.Key_Up):
      if(self.currentRow > 0):
        if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          self.clearSelection()
        self.currentRow -= 1
        self.selectTo(self.currentRow, self.currentRow)
        nuIndex = self.model().index(self.currentRow, self.currentCol)
        self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() == QtCore.Qt.Key.Key_PageDown):
      if(self.currentRow < self.tableModel.rowCount() - 1):
        if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          self.clearSelection()
        origRow = self.currentRow
        self.currentRow += self.pageStep
        self.currentRow = min(self.currentRow, self.tableModel.rowCount() - 1)
        if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          origRow = self.currentRow
        self.selectTo(origRow, self.currentRow)
        nuIndex = self.model().index(self.currentRow, self.currentCol)
        self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() == QtCore.Qt.Key.Key_PageUp):
      if(self.currentRow > 0):
        if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          self.clearSelection()
        origRow = self.currentRow
        self.currentRow -= self.pageStep
        self.currentRow = max(self.currentRow, 0)
        if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          origRow = self.currentRow
        self.selectTo(origRow, self.currentRow)
        nuIndex = self.model().index(self.currentRow, self.currentCol)
        self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() in [QtCore.Qt.Key.Key_Left, QtCore.Qt.Key.Key_Right]):
      # ignore event such that we can capture the left/right keys
      event.ignore()
      flag = False
      if(event.key() == QtCore.Qt.Key.Key_Left):
        if(self.currentCol > 0):
          self.currentCol -= 1
          flag = True
      elif(self.currentCol < self.tableModel.columnCount() - 1):
        self.currentCol += 1
        flag = True
      if(flag):
        nuIndex = self.model().index(self.currentRow, self.currentCol)
        self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() in [QtCore.Qt.Key.Key_Home, QtCore.Qt.Key.Key_End]):
      if(event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier):
        flag = False
        if(event.key() == QtCore.Qt.Key.Key_Home):
          if(self.currentRow > 0):
            origRow = self.currentRow
            self.currentRow = 0
            flag = True
        elif(self.currentRow < self.tableModel.rowCount() - 1):
          origRow = self.currentRow
          self.currentRow = self.tableModel.rowCount() - 1
          flag = True
        if(flag):
          if(not (event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier)):
            self.clearSelection()
            origRow = self.currentRow
          self.selectTo(origRow, self.currentRow)
          nuIndex = self.model().index(self.currentRow, self.currentCol)
          self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
      else:
        flag = False
        if(event.key() == QtCore.Qt.Key.Key_Home):
          if(self.currentCol > 0):
            self.currentCol = 0
            flag = True
        elif(self.currentCol < self.tableModel.columnCount() - 1):
          self.currentCol = self.tableModel.columnCount() - 1
          flag = True
        if(flag):
          nuIndex = self.model().index(self.currentRow, self.currentCol)
          self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() in [QtCore.Qt.Key.Key_Tab, QtCore.Qt.Key.Key_Backtab]):
      # advance cell on tab
      if(event.key() == QtCore.Qt.Key.Key_Backtab):
        self.currentCol -= 1
        if(self.currentCol < 0):
          if(self.currentRow > 0):
            self.currentRow -= 1
            self.currentCol = self.tableModel.columnCount() - 1
          else:
            self.currentCol = 0
      else:
        self.currentCol += 1
        if(self.currentCol >= self.tableModel.columnCount()):
          if(self.currentRow < self.tableModel.rowCount() - 1):
            self.currentRow += 1
            self.currentCol = 0
          else:
            self.currentCol = self.tableModel.columnCount() - 1
      self.clearSelection()
      self.selectTo(self.currentRow, self.currentRow)
      nuIndex = self.model().index(self.currentRow, self.currentCol)
      self.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)
    elif(event.key() in [QtCore.Qt.Key.Key_Backspace, QtCore.Qt.Key.Key_Delete]):
      # check how many rows are selected -- if more than one, clear all contents
      selectedRows = self.selectionModel().selectedRows()
      if(len(selectedRows) > 1):
        # generate empty data block
        emptyRow = [[''] * self.tableModel.columnCount()]
        # cycle over rows and paste stuff
        for i in selectedRows:
          self.tableModel.pasteDataBlock(data=emptyRow, offsetRow=i.row(), offsetColumn=0)
        # calculate statistics on all columns
        for i in range(self.tableModel.columnCount()):
          self.updateColumnStatistics(column=i)
        # update table
        self.tableModel.layoutChanged.emit()
      else:
        # clear current cell
        self.tableModel.setData('', self.currentRow, self.currentCol)
        # calculate statistics on columns
        self.updateColumnStatistics(column=self.currentCol)
        # refresh table view
        cellIndex = self.tableModel.index(self.currentRow, self.currentCol)
        self.tableModel.dataChanged.emit(cellIndex, cellIndex, [QtCore.Qt.ItemDataRole.DisplayRole])
    elif (event.matches(QtGui.QKeySequence.StandardKey.ZoomIn)):
      self.zoomIn()
      event.accept()
    elif (event.matches(QtGui.QKeySequence.StandardKey.ZoomOut)):
      self.zoomOut()
      event.accept()
    elif (event.matches(QtGui.QKeySequence.StandardKey.Save) or event.matches(QtGui.QKeySequence.StandardKey.Open) or event.matches(QtGui.QKeySequence.StandardKey.HelpContents)\
          or event.matches(QtGui.QKeySequence.StandardKey.Print) or event.matches(QtGui.QKeySequence.StandardKey.Quit) or event.matches(QtGui.QKeySequence.StandardKey.Italic)\
          or event.matches(QtGui.QKeySequence.StandardKey.Find) or event.matches(QtGui.QKeySequence.StandardKey.New) or event.matches(QtGui.QKeySequence.StandardKey.Refresh) or event.matches(QtGui.QKeySequence.StandardKey.FindNext)):
      # pass event to main ui
      event.ignore()
    elif(not (event.key() in [QtCore.Qt.Key.Key_Escape])):
      openDialog = False
      if(event.key() in [QtCore.Qt.Key.Key_Enter, QtCore.Qt.Key.Key_Return]):
        openDialog = True
        initialEdit = ''
      elif(len(event.text())):
        openDialog = True
        initialEdit = event.text()
      if(openDialog):
        # ensure that cell is visible
        indexAt = self.model().index(self.currentRow, self.currentCol)
        self.scrollTo(indexAt)
        # open edit QMenu at cell position
        rowViewport, colViewport = self.rowViewportPosition(self.currentRow), self.columnViewportPosition(self.currentCol) + self.verticalHeader().width()
        menuPos = self.mapToGlobal(QtCore.QPoint(int(colViewport), int(rowViewport)))
        if((indexAt.row() != -1) and (indexAt.column() != -1)):
          if(self.currZoom >= 0):
            fontSize = self.zoomLevels[self.currZoom]
          else:
            fontSize = float(WIDGET_FONTSIZE)
          self.menu = EditDataMenu(parent=self, tableModel=self.tableModel, indexAt=indexAt, initialEdit=initialEdit, fontSize=fontSize)
          # apply styles to popup window
          if(QSTYLE != None):
            self.menu.setStyle(QSTYLE)
          if(QSTYLESHEET != None):
            self.menu.setStyleSheet(QSTYLESHEET)
          self.menu.popup(menuPos)    

  def selectTo(self, startRow=0, endRow=0):
    # selects target rows
    lowRow, hiRow = min(startRow, endRow), max(startRow, endRow)
    columnCount = self.tableModel.columnCount()
    
    topLeft = self.model().index(lowRow, 0)
    bottomRight = self.model().index(hiRow, columnCount - 1)
    itemSelection = QtCore.QItemSelection(topLeft, bottomRight)
    self.selectionModel().select(itemSelection, QtCore.QItemSelectionModel.SelectionFlag.Select)

  def pasteText(self, pastedText=''):
    # display busy pointer as this can take a long time for big data chunks
    QtWidgets.QApplication.setOverrideCursor(QtGui.QCursor(QtCore.Qt.CursorShape.WaitCursor))
    # store target cell
    offsetRow, offsetColumn = self.currentRow, self.currentCol
    # determine size of text to be pasted
    clipRows = pastedText.split('\n')
    clipCols = [i.split('\t') for i in clipRows]
    clipColCount = [len(i) for i in clipCols]
    # determine new data sheet dimensions
    nuRowCount, nuColCount = len(clipRows), max(clipColCount)
    # check for trailing all-empty columns which we won't copy
    index = nuColCount - 1
    trailCol = nuColCount
    while(index >= 0):
      for entry in clipCols:
        if(0 < index < len(entry)):
          if(len(entry[index])):
            trailCol = index + 1
            index = -1
      index -= 1
    # make clipped text square and truncate after trailCol
    clipCols = [i + ([''] * nuColCount) for i in clipCols]
    clipCols = [i[:trailCol] for i in clipCols]
    # convert to number where possible
    clipCols = [[float(j) if self.isNumber(j) else j for j in i] for i in clipCols]
    # prepare pasting of text -- resize if needed
    dimy, dimx = self.tableModel.rowCount(), self.tableModel.columnCount()
    if(((offsetColumn + trailCol) > dimx) or ((offsetRow + nuRowCount) > dimy)):
      # store current data
      currData = self.tableModel.getAllData()
      # blank data
      dimx, dimy = max(dimx, offsetColumn + trailCol), max(dimy, offsetRow + nuRowCount)
      blankData = [[''] * dimx for i in range(dimy)]
      self.tableModel = DataTableModel(blankData, self)
      self.setModel(self.tableModel)
      # restore original data
      self.tableModel.pasteDataBlock(data=currData, offsetRow=0, offsetColumn=0)
    
    # paste new data
    self.tableModel.pasteDataBlock(data=clipCols, offsetRow=offsetRow, offsetColumn=offsetColumn)
    self.configTable(offsetColumn + trailCol, offsetRow + nuRowCount, retainRoles=True, retainSelection=True)
    self.tableModel.layoutChanged.emit()
    
    # update self.sheetData as well
    self.sheetData = self.tableModel.getAllData()

    # restore cursor
    QtWidgets.QApplication.restoreOverrideCursor()
    
  def mouseDoubleClickEvent(self, event):
    # allow editing of cell on double click
    # perform original event
    QtWidgets.QTableView.mouseDoubleClickEvent(self, event)
    
    # determine indices of clicked cell and scroll to ensure visibility
    indexAt = self.indexAt(event.pos())
    self.scrollTo(indexAt)
    row, col = indexAt.row(), indexAt.column()
    rowViewport, colViewport = self.rowViewportPosition(row), self.columnViewportPosition(col)
    
    # open edit QMenu at cell position
    menuPos = self.mapToGlobal(QtCore.QPoint(int(colViewport + self.verticalHeader().width()), int(rowViewport)))

    if((indexAt.row() != -1) and (indexAt.column() != -1)):
      if(self.currZoom >= 0):
        fontSize = self.zoomLevels[self.currZoom]
      else:
        fontSize = float(WIDGET_FONTSIZE)
      self.menu = EditDataMenu(parent=self, tableModel=self.tableModel, indexAt=indexAt, fontSize=fontSize)
      # apply styles to popup window
      if(QSTYLE != None):
        self.menu.setStyle(QSTYLE)
      if(QSTYLESHEET != None):
        self.menu.setStyleSheet(QSTYLESHEET)
      self.menu.popup(menuPos)    

  def currentCellChanged(self, current):
    # keeps tabs on current cell
    self.currentRow, self.currentCol = current.row(), current.column()

  def isNumber(self, s):
    # checks whether string is a number
    try:
      float(s)
      return True
    except ValueError:
      pass
   
    try:
      import unicodedata
      unicodedata.numeric(s)
      return True
    except (TypeError, ValueError):
      pass
    
    return False

# subclass edit to better capture key presses
class EditDataEdit(QtWidgets.QLineEdit):
  def __init__(self, parent=None):
    super(EditDataEdit, self).__init__(parent)
    self.parent = parent
  
  def keyPressEvent(self, event):
    # ignore alt keys as they would close the QMenu
    if(event.key() in [QtCore.Qt.Key.Key_Alt, QtCore.Qt.Key.Key_AltGr]):
      return

    # capture enter and arrow keys
    if(event.key() in [QtCore.Qt.Key.Key_Return, QtCore.Qt.Key.Key_Enter]):
      if(event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier):
        self.parent.advanceCell(-1, 0)
      else:
        self.parent.advanceCell(1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Up):
      self.parent.advanceCell(-1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Down):
      self.parent.advanceCell(1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Left):
      if((event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier) or (self.cursorPosition() == 0)):
        self.parent.advanceCell(0, -1)
        return
    elif(event.key() == QtCore.Qt.Key.Key_Right):
      if((event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier) or (self.cursorPosition() == len(self.text()))):
        self.parent.advanceCell(0, 1)
        return
      
    # normal event processing
    QtWidgets.QLineEdit.keyPressEvent(self, event)

class EditDataMenu(KuhMenu):
  def __init__(self, parent=None, tableModel=None, indexAt=None, initialEdit='', fontSize=10):
    super(EditDataMenu, self).__init__()
    self.finalUpdate = False
    if(None in [parent, tableModel, indexAt]):
      self.close()
    # check for click outside cells
    elif((indexAt.row() == -1) or (indexAt.column() == -1)):
      self.close()
    else:
      self.parent = parent
      self.tableModel = tableModel
      self.maxRow, self.maxCol = self.tableModel.rowCount(), self.tableModel.columnCount()
      self.indexAt = indexAt
      self.row, self.col = self.indexAt.row(), self.indexAt.column()
      self.minWidth = int(100 * SCALEFONT)
      self.finalUpdate = True
      self.fontSize = fontSize
        
      # set up GUI
      self.buildRessource()
      
      # set QMenu position
      self.adjustWindowPosition(initialEdit=initialEdit)

  def adjustWindowPosition(self, initialEdit=''):
    # update label
    labelText = 'Edit cell ' + str(self.col + 1) + '/' + str(self.row + 1)
    useFont = self.editDataLabel.font()
    useFont.setBold(True)
    self.editDataLabel.setFont(useFont)
    self.editDataLabel.setText(labelText)
    self.editDataLabel.setStyleSheet("QLabel { font-size: " + str(self.fontSize) + "pt; }")

    # update QlineEdit
    if(len(initialEdit)):
      self.initValue = ''
      self.editData.setText(initialEdit)
    else:
      self.initValue = self.tableModel.dataByIndices(self.row, self.col)
      self.editData.setText(str(self.initValue))
    self.editData.setStyleSheet("QLineEdit { font-size: " + str(self.fontSize) + "pt; }")

    self.editData.selectAll()
    self.editData.setFocus()

    # ensure that cell is visible
    cellIndex = self.tableModel.index(self.row, self.col)
    self.parent.scrollTo(cellIndex)
    
    # adjust width of edit window -- use min/max size as resize not properly heeded
    cellWidth = int(max(self.minWidth, self.parent.columnWidth(self.col)))
    ###self.editData.setMaximumSize(QtCore.QSize(cellWidth, int(BASE_SIZE)))
    ###self.editData.setMinimumSize(QtCore.QSize(cellWidth, int(BASE_SIZE)))
    self.setMaximumWidth(cellWidth)
    self.setMinimumWidth(cellWidth)

    # adjusts window position to currently edited cell
    rowViewport = self.parent.rowViewportPosition(self.row)
    colViewport = self.parent.columnViewportPosition(self.col) + self.parent.verticalHeader().width()
    menuPos = self.parent.mapToGlobal(QtCore.QPoint(int(colViewport), int(rowViewport)))
    self.move(menuPos)
    
    if(len(initialEdit)):
      self.editData.deselect()
      self.editData.setCursorPosition(1)

  def buildRessource(self):
    # build gui
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(0, 0, 0, 0)
    
    # QlineEdit for data modification
    self.editDataLabel = QtWidgets.QLabel()
    self.vLayout.addWidget(self.editDataLabel)
    
    self.editData = EditDataEdit(self)
    self.vLayout.addWidget(self.editData)
    self.editData.setFocus()
    
  def updateData(self):
    # updates data table if required
    currValue = self.editData.text()
    if(currValue != self.initValue):
      try:
        currValue = float(currValue)
      except:
        pass
      self.parent.tableModel.setData(currValue, self.row, self.col)

      # calculate statistics on columns
      self.parent.updateColumnStatistics(column=self.col)
  
      # refresh table view
      cellIndex = self.tableModel.index(self.row, self.col)
      self.tableModel.dataChanged.emit(cellIndex, cellIndex, [QtCore.Qt.ItemDataRole.DisplayRole])
      
  def keyPressEvent(self, event):
    # process tab keys and escape
    if(event.key() == QtCore.Qt.Key.Key_Backtab):
      self.advanceCell(0, -1)
    elif(event.key() == QtCore.Qt.Key.Key_Tab):
      self.advanceCell(0, 1)
    elif(event.key() == QtCore.Qt.Key.Key_Escape):
      self.finalUpdate = False
      self.close()
      
  def advanceCell(self, deltaRow=0, deltaCol=0):
    # update previous data
    self.updateData()
    
    # move edit window to new position
    # adjust cell indices
    self.deltaRow, self.deltaCol = deltaRow, deltaCol
    
    # apply column shift
    self.col += self.deltaCol
    if(self.col >= self.maxCol):
      if(self.row < self.maxRow - 1):
        self.col = 0; self.row += 1
      else:
        self.col = self.maxCol - 1
    elif(self.col < 0):
      if(self.row > 0):
        self.col = self.maxCol - 1; self.row -= 1
      else:
        self.col = 0

    # apply row shift
    self.row += deltaRow
    self.row = max(self.row, 0)
    self.row = min(self.row, self.maxRow - 1)

    # set cursor in data table to currently edited cell
    self.parent.currentCol, self.parent.currentRow = self.col, self.row
    self.parent.clearSelection()
    self.parent.selectTo(self.row, self.row)
    nuIndex = self.parent.model().index(self.row, self.col)
    self.parent.selectionModel().setCurrentIndex(nuIndex, QtCore.QItemSelectionModel.SelectionFlag.Select)

    # move window
    self.adjustWindowPosition()
    
  def closeEvent(self, event):
    # perform final update
    if(self.finalUpdate):
      self.updateData()
    # perform original close event
    QtWidgets.QMenu.closeEvent(self, event)

# subclass edit to better capture key presses
class EditResultsEdit(QtWidgets.QLineEdit):
  def __init__(self, parent=None):
    super(EditResultsEdit, self).__init__(parent)
    self.parent = parent
  
  def keyPressEvent(self, event):
    # ignore alt keys as they would close the QMenu
    if(event.key() in [QtCore.Qt.Key.Key_Alt, QtCore.Qt.Key.Key_AltGr]):
      return

    # capture enter and arrow keys
    if(event.key() in [QtCore.Qt.Key.Key_Return, QtCore.Qt.Key.Key_Enter]):
      if(event.modifiers() & QtCore.Qt.KeyboardModifier.ShiftModifier):
        self.parent.advanceCell(-1, 0)
      else:
        self.parent.advanceCell(1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Up):
      self.parent.advanceCell(-1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Down):
      self.parent.advanceCell(1, 0)
#    elif(event.key() == QtCore.Qt.Key.Key_Left):
#      if((event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier) or (self.cursorPosition() == 0)):
#        self.parent.advanceCell(0, -1)
#        return
#    elif(event.key() == QtCore.Qt.Key.Key_Right):
#      if((event.modifiers() & QtCore.Qt.KeyboardModifier.ControlModifier) or (self.cursorPosition() == len(self.text()))):
#        self.parent.advanceCell(0, 1)
#        return
      
    # normal event processing
    QtWidgets.QLineEdit.keyPressEvent(self, event)

class EditResultsMenu(KuhMenu):
  def __init__(self, parent=None, tableModel=None, indexAt=None, initialEdit='', fontSize=10, validColumns=[]):
    super(EditResultsMenu, self).__init__()
    self.finalUpdate = False
    if(None in [parent, tableModel, indexAt]):
      self.close()
    # check for click outside cells
    elif((indexAt.row() == -1) or (indexAt.column() == -1)):
      self.close()
    else:
      self.parent = parent
      self.tableModel = tableModel
      self.maxRow, self.maxCol = self.tableModel.rowCount(), self.tableModel.columnCount()
      self.indexAt = indexAt
      self.row, self.col = self.indexAt.row(), self.indexAt.column()
      self.minWidth = int(100 * SCALEFONT)
      self.finalUpdate = True
      self.fontSize = fontSize
        
      # set up GUI
      self.buildRessource()
      
      # set QMenu position
      self.adjustWindowPosition(initialEdit=initialEdit)

  def adjustWindowPosition(self, initialEdit=''):
    # update label
    labelText = 'Edit label ' + str(self.row + 1)
    useFont = self.editDataLabel.font()
    useFont.setBold(True)
    self.editDataLabel.setFont(useFont)
    self.editDataLabel.setText(labelText)
    self.editDataLabel.setStyleSheet("QLabel { font-size: " + str(self.fontSize) + "pt; }")

    # update QlineEdit
    if(len(initialEdit)):
      self.initValue = ''
      self.editData.setText(initialEdit)
    else:
      self.initValue = self.tableModel.dataByIndices(self.row, self.col)
      self.editData.setText(str(self.initValue))
    self.editData.setStyleSheet("QLineEdit { font-size: " + str(self.fontSize) + "pt; }")

    self.editData.selectAll()
    self.editData.setFocus()

    # ensure that cell is visible
    cellIndex = self.tableModel.index(self.row, self.col)
    self.parent.scrollTo(cellIndex)
    
    # adjust width of edit window -- use min/max size as resize not properly heeded
    cellWidth = int(max(self.minWidth, self.parent.columnWidth(self.col)))
    self.setMaximumWidth(cellWidth)
    self.setMinimumWidth(cellWidth)

    # adjusts window position to currently edited cell
    rowViewport = self.parent.rowViewportPosition(self.row)
    colViewport = self.parent.columnViewportPosition(self.col) + self.parent.verticalHeader().width()
    menuPos = self.parent.mapToGlobal(QtCore.QPoint(int(colViewport), int(rowViewport)))
    self.move(menuPos)
    
    if(len(initialEdit)):
      self.editData.deselect()
      self.editData.setCursorPosition(1)

  def buildRessource(self):
    # build gui
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(0, 0, 0, 0)
    
    # QlineEdit for data modification
    self.editDataLabel = QtWidgets.QLabel()
    self.vLayout.addWidget(self.editDataLabel)
    
    self.editData = EditResultsEdit(self)
    self.vLayout.addWidget(self.editData)
    self.editData.setFocus()
    
  def updateData(self):
    # updates data table if required
    currValue = self.editData.text()
    if(currValue != self.initValue):
      try:
        currValue = float(currValue)
      except:
        pass
      self.parent.tableModel.setData(currValue, self.row, self.col)

      # calculate statistics on columns
      self.parent.updateColumnStatistics(column=self.col)
  
      # refresh table view
      cellIndex = self.tableModel.index(self.row, self.col)
      self.tableModel.dataChanged.emit(cellIndex, cellIndex, [QtCore.Qt.ItemDataRole.DisplayRole])
      
  def keyPressEvent(self, event):
    # process tab keys and escape
    if(event.key() == QtCore.Qt.Key.Key_Backtab):
      self.advanceCell(-1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Tab):
      self.advanceCell(1, 0)
    elif(event.key() == QtCore.Qt.Key.Key_Escape):
      self.finalUpdate = False
      self.close()
      
  def advanceCell(self, deltaRow=0, deltaCol=0):
    # update previous data
    self.updateData()
    
    # move edit window to new position
    # adjust cell indices
    self.deltaRow, self.deltaCol = deltaRow, deltaCol
    
    # apply column shift
    self.col += self.deltaCol
    if(self.col >= self.maxCol):
      if(self.row < self.maxRow - 1):
        self.col = 0; self.row += 1
      else:
        self.col = self.maxCol - 1
    elif(self.col < 0):
      if(self.row > 0):
        self.col = self.maxCol - 1; self.row -= 1
      else:
        self.col = 0

    # apply row shift
    self.row += deltaRow
    self.row = max(self.row, 0)
    self.row = min(self.row, self.maxRow - 1)

    # set cursor in data table to currently edited cell
    self.parent.currentCol, self.parent.currentRow = self.col, self.row
    self.parent.clearSelection()
    self.parent.selectRow(self.row)

    # move window
    self.adjustWindowPosition()
    
  def closeEvent(self, event):
    # perform final update
    if(self.finalUpdate):
      self.updateData()
    # perform original close event
    QtWidgets.QMenu.closeEvent(self, event)
    # trigger update of labels in parent object
    self.parent.updateLabels()

# the results table widget
class ResultsTable(QtWidgets.QTableView):
  def __init__(self, parent=None):
    super(ResultsTable, self).__init__(parent)
    self.setEditTriggers(QtWidgets.QAbstractItemView.EditTrigger.NoEditTriggers)
    self.setItemDelegate(FloatFormatDelegate())
    self.setEditTriggers(QtWidgets.QAbstractItemView.EditTrigger.NoEditTriggers)
    self.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectionBehavior.SelectRows)
    btn = self.findChild(QtWidgets.QAbstractButton)
    self.storedCornerTip = 'Click to select entire table'
    btn.setToolTip(self.storedCornerTip)

    self.parent = parent
    self.tableModel = None
    self.rowHeight = 10

    # settings for zoom of data table
    self.zoomLevels = [3, 4, 6, 8, 10, 12, 14, 16, 18, 20, 24, 28, 32, 36, 40, 48, 56, 64, 72, 80, 90, 100]
    self.currZoom = -1

  def wheelEvent(self, event):
    # adjusts zoom level when Ctrl is pressed also
    modifiers = QtWidgets.QApplication.queryKeyboardModifiers()
    if((self.tableModel != None) and self.tableModel.rowCount() and (modifiers & QtCore.Qt.KeyboardModifier.ControlModifier)):
      if(event.angleDelta().y() > 0):
        self.zoomIn()
      else:
        self.zoomOut()
      # issue update
      self.update()
      QtCore.QCoreApplication.processEvents()
    else:
      # normal event handling
      QtWidgets.QTableView.wheelEvent(self, event)

  def keyPressEvent(self, event):
    # captures zoom keys
    if(event.matches(QtGui.QKeySequence.StandardKey.ZoomIn)):
      self.zoomIn()
      event.accept()
    elif(event.matches(QtGui.QKeySequence.StandardKey.ZoomOut)):
      self.zoomOut()
      event.accept()
    elif(event.matches(QtGui.QKeySequence.StandardKey.SelectAll)):
      self.selectAll()
    else:
      event.ignore()

  def mouseDoubleClickEvent(self, event):
    # allow editing of cell on double click
    # perform original event
    QtWidgets.QTableView.mouseDoubleClickEvent(self, event)
    
    # determine indices of clicked cell and scroll to ensure visibility
    indexAt = self.indexAt(event.pos())
    self.scrollTo(indexAt)
    row, col = indexAt.row(), indexAt.column()
    rowViewport, colViewport = self.rowViewportPosition(row), self.columnViewportPosition(col)
    
    # open edit QMenu at cell position
    menuPos = self.mapToGlobal(QtCore.QPoint(int(colViewport + self.verticalHeader().width()), int(rowViewport)))

    if((indexAt.row() != -1) and (indexAt.column() != -1)):
      # determine if we have a label column and if we clicked on it
      validColumn = []
      for item in ['labels', 'labels2']:
        if(item in self.tableModel.getHeaders()):
          validColumn.append(self.tableModel.getHeaders().index(item))
      if(indexAt.column() in validColumn):
        if(self.currZoom >= 0):
          fontSize = self.zoomLevels[self.currZoom]
        else:
          fontSize = float(WIDGET_FONTSIZE)
        self.menu = EditResultsMenu(parent=self, tableModel=self.tableModel, indexAt=indexAt, fontSize=fontSize, validColumns=[validColumn])
        # apply styles to popup window
        if(QSTYLE != None):
          self.menu.setStyle(QSTYLE)
        if(QSTYLESHEET != None):
          self.menu.setStyleSheet(QSTYLESHEET)
        self.menu.popup(menuPos)
          
  def updateLabels(self):
    # callback from EditResultsMenu => update the labeli
    for item in ['labels', 'labels2']:
      if(item in self.tableModel.getHeaders()):
        validColumn = self.tableModel.getHeaders().index(item)
        labels = self.tableModel.getDataColumn(validColumn)
      
        # determine index value of data set on display
        currIndex = self.parent.dataSetSpinBox.value()
        # update corresponding data set
        self.parent.parent.data[currIndex - 1].addLabeli(labels=labels, labels2=item == 'labels2')

  def zoomIn(self):
    self.zoom(increment=1)
    
  def zoomOut(self):
    self.zoom(increment=-1)

  def zoom(self, increment=1):
    # general zoom function
    if(self.tableModel != None):
      font = self.tableModel.getFont()
      # check whether we run zoom() the first time
      if(self.currZoom < 0):
        ###fontSize = font.pointSize()
        ###temp = np.abs(np.array(self.zoomLevels) - fontSize)
        # better use WIDGET_FONTSIZE to properly heed stylesheet (which somehow overrides other font properties)
        temp = np.abs(np.array(self.zoomLevels) - float(WIDGET_FONTSIZE))
        self.currZoom = temp.argmin()
        
      if(((self.currZoom < len(self.zoomLevels) - 1) and (increment > 0)) or ((self.currZoom > 0) and (increment < 0))):
        # alter font size in cells
        self.currZoom += increment
        self.currZoom = int(self.currZoom)
        font.setPointSize(self.zoomLevels[self.currZoom])
        
        # alter row height
        fm = QtGui.QFontMetrics(font)
        self.rowHeight = int(fm.height() + 2)
  
        # alter font in horizontal header
        # make the header fonts a tad larger than the cell font
        self.verticalHeader().setStyleSheet("QHeaderView { font-size: " + str(self.zoomLevels[self.currZoom] + 2) + "pt; }")
        self.horizontalHeader().setStyleSheet("QHeaderView { font-size: " + str(self.zoomLevels[self.currZoom] + 2) + "pt; }")
        
        # resize column width proportionally
        if(self.tableModel.rowCount()):
          dimx = min(15, self.tableModel.columnCount())
          hheader = self.horizontalHeader()
          scale = self.zoomLevels[self.currZoom] / self.zoomLevels[self.currZoom - increment]
          for entry in range(dimx):
            nuSize = int(hheader.sectionSize(entry) * scale)
            hheader.resizeSection(entry, nuSize)
  
        # finally readjust size
        # fudge function to prevent PyQt5 from uncontrollably resizing row height
        vheader = self.verticalHeader()
        vheader.setSectionResizeMode(QtWidgets.QHeaderView.ResizeMode.Fixed)
        vheader.setDefaultSectionSize(self.rowHeight)

  def updateColumnStatistics(self, column=-1):
    # helper function to calculate column statistics
    self.tableModel.columnAverages = ['n/a'] * self.tableModel.columnCount()
    if(self.tableModel.rowCount()):
      if(column + 1):
        userange = range(column, column + 1)
      else:
        userange = range(self.tableModel.columnCount())
      for index in userange:
        temp = self.tableModel.getDataColumn(index)
        temp = [i for i in temp if type(i) in [int, float]]
        if(len(temp)):
          self.tableModel.setColumnStats(index, number=len(temp), csum=np.sum(temp), mean=np.mean(temp), minval=np.min(temp), maxval=np.max(temp))
        else:
          self.tableModel.setColumnStats(index, number=0, csum=0, mean='n/a', minval='n/a', maxval='n/a')

class ResultsArea(QWidgetMac):
  def __init__(self, parent = None):
    super(ResultsArea, self).__init__()
    self.parent = parent
    self.rolestr = {'x':'x', 'y':'y', 'xerr':u'\N{GREEK CAPITAL LETTER DELTA}x', \
      'yerr':u'\N{GREEK CAPITAL LETTER DELTA}y', 'fit':'fit', 'resid':'resid'}
    self.descriptors = []
    self.buildRessource()
    self.minColWidth = 35
    
    # initialize filename
    self.currExportFile = None
    
  def buildRessource(self):
    # set up results table
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(0, 0, 0, 0)

    # allow change of data set on display
    self.dataSetBox = QWidgetMac()
    self.vLayout.addWidget(self.dataSetBox)
    self.dataSetLayout = QtWidgets.QHBoxLayout(self.dataSetBox)
    self.dataSetLayout.setContentsMargins(0, 0, 0, 0)
    self.dataSetLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    self.dataSetLabel = QtWidgets.QLabel()
    useFont = self.dataSetLabel.font()
    useFont.setBold(True)
    self.dataSetLabel.setFont(useFont)
    self.dataSetLabel.setText('Data Set')
    self.dataSetLabel.setMaximumSize(int(50 * SCALEFONT), int(BASE_SIZE))
    self.dataSetLabel.setMinimumSize(int(50 * SCALEFONT), int(BASE_SIZE))
    self.dataSetLayout.addWidget(self.dataSetLabel)
    
    self.dataSetSpinBox = QSpinClick()
    self.dataSetSpinBox.setAlignment(QtCore.Qt.AlignmentFlag.AlignRight)
    self.dataSetSpinBox.setToolTip('Select data set to display in table')
    self.dataSetSpinBox.setMinimum(1)
    self.dataSetSpinBox.setMaximum(1)
    self.dataSetSpinBox.setValue(1)
    self.dataSetSpinBox.setMinimumSize(int(50 * SCALEFONT), int(BASE_SIZE))
    self.dataSetSpinBox.setMaximumSize(int(50 * SCALEFONT), int(BASE_SIZE))
    self.dataSetSpinBox.editingFinished.connect(self.changeDataSet)
    self.dataSetSpinBox.setEnabled(False)
    self.dataSetLayout.addWidget(self.dataSetSpinBox)
    
    self.dataSetName = QtWidgets.QLabel()
    self.dataSetName.setText(self.parent.data[self.parent.activeData].name)
    self.dataSetLayout.addWidget(self.dataSetName)
    
    self.dataSetLayout.addStretch()
    self.addLabelsButton = QPushButtonMac()
    self.addLabelsButton.setText('Add labels')
    self.addLabelsButton.setToolTip('Adds label column to data set (if not existing already)')
    self.addLabelsButton.setMinimumSize(int(70 * SCALEFONT), int(BASE_SIZE))
    self.addLabelsButton.setMaximumSize(int(70 * SCALEFONT), int(BASE_SIZE))
    self.addLabelsButton.clicked.connect(self.addLabeli)
    self.addLabelsButton.setEnabled(False)
    self.dataSetLayout.addWidget(self.addLabelsButton)

    self.resultstable = ResultsTable(parent=self)
    self.resultstable.setObjectName('data')
    self.vLayout.addWidget(self.resultstable)

    self.rowHeight = int(BASE_SIZE - 5)
    ###self.rowHeight = int(self.resultstable.fontMetrics().height() + 2)
    vheader = self.resultstable.verticalHeader()
    vheader.setDefaultSectionSize(self.rowHeight)
    vheader.setSectionResizeMode(QtWidgets.QHeaderView.ResizeMode.Fixed)

    self.buttonBox = QtWidgets.QWidget()
    self.vLayout.addWidget(self.buttonBox)
    self.buttonBoxLayout = QtWidgets.QHBoxLayout(self.buttonBox)
    self.buttonBoxLayout.setContentsMargins(0, 0, 0, 0)
    self.buttonBoxLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    self.exportButton = QPushButtonMac()
    self.exportButton.setText(' Export Results')
    self.exportButton.setToolTip('Compile report of current graphics and fit results')
    self.exportButton.setMaximumHeight(int(BASE_SIZE))
    self.exportButton.setMinimumSize(QtCore.QSize(100, int(BASE_SIZE)))
    self.exportButton.clicked.connect(self.exportWrapper)
    self.exportButton.setIcon(FOM_ICON_ARROW_RIGHT)
    self.buttonBoxLayout.addWidget(self.exportButton, stretch=1)

    self.selectionButton = QPushButtonMac()
    self.selectionButton.setText(' Update Selection')
    self.selectionButton.setToolTip('Update data selection')
    self.selectionButton.setMaximumHeight(int(BASE_SIZE))
    self.selectionButton.setMinimumSize(QtCore.QSize(100, int(BASE_SIZE)))
    self.selectionButton.clicked.connect(self.updateSelection)
    self.selectionButton.setIcon(FOM_ICON_BROWSER_RELOAD)
    self.buttonBoxLayout.addWidget(self.selectionButton, stretch=1)

    self.clearButton = QPushButtonMac()
    self.clearButton.setText(' Clear Selection')
    self.clearButton.setToolTip('Clear data selection')
    self.clearButton.setMaximumHeight(int(BASE_SIZE))
    self.clearButton.setMinimumSize(QtCore.QSize(100, int(BASE_SIZE)))
    self.clearButton.clicked.connect(self.clearSelection)
    self.clearButton.setIcon(FOM_ICON_DIALOG_CANCEL_BUTTON)
    self.buttonBoxLayout.addWidget(self.clearButton, stretch=1)
    self.clearButton.setEnabled(False)

    self.spawnButton = QPushButtonMac()
    self.spawnButton.setText(' Spawn Data')
    self.spawnButton.setToolTip('Generate new data set from data selection')
    self.spawnButton.setMaximumHeight(int(BASE_SIZE))
    self.spawnButton.setMinimumSize(QtCore.QSize(100, int(BASE_SIZE)))
    self.spawnButton.clicked.connect(self.spawnMe)
    self.spawnButton.setIcon(FOM_ICON_COMMAND_LINK)
    self.buttonBoxLayout.addWidget(self.spawnButton, stretch=1)
    self.spawnButton.setEnabled(False)

  def addLabeli(self, labels2=False):
    # add labels to data set unless already existing
    # determine index value of data set on display
    currIndex = self.dataSetSpinBox.value()
    # update corresponding data set
    self.parent.data[currIndex - 1].addLabeli([], labels2=labels2)
    # and update table
    self.changeDataSet()
  
  def spawnMe(self):
    # generates a new data set from current data selection
    values, roles = self.parent.selectedData.getData_n_Fit()
    name = self.parent.data[self.parent.activeData].name
    self.parent.objectsarea.dataSetCreate()
    self.parent.data[-1].setData(data=values, roles=roles)
    self.parent.data[-1].setName('spawned from ' + name)
    # display new data set
    self.parent.data[-1].drawMe(redraw=False)
    # need to update various tables
    self.parent.objectsarea.refreshDataTable()
    self.parent.globalarea.updateDataSetTable()
    # update results table
    self.setDataSet(currIndex=len(self.parent.data) - 1, maxIndex=len(self.parent.data) - 1)
    # update legend if needed
    self.parent.objectsarea.updateLegend(redraw=True)

  def updateSelection(self):
    # uses table selection to select data points
    currIndex = self.dataSetSpinBox.value()
    if(self.parent.activeData != currIndex - 1):
      self.parent.statusbar.showMessage('Data set ' + str(currIndex) + ' is on display, but data set ' + str(self.parent.activeData + 1) + ' is active => no selection done!', self.parent.STATUS_TIME)
      return
    
    if(self.resultstable.tableModel != None):
      # for good measure try/except this
      try:
        selind = self.resultstable.selectionModel().selectedRows()
        self.parent.selectedData.tableSelectIndices(selind)
        return
      except:
        pass
      
    self.parent.statusbar.showMessage('Results table appears to be empty, cannot update data set!', self.parent.STATUS_TIME)

  def clearSelection(self):
    # clears selection of data points
    self.resultstable.clearSelection()
    
    # clear selected data points (this will also control the display of the clear buttons)
    self.parent.selectedData.clearMe()

  def selectTheseRows(self, selind=[]):
    # selects rows in resultstable
    self.resultstable.clearSelection()
    if(len(selind)):
      # ensure that currently selected data set is on display
      if(self.parent.activeData + 1 != self.dataSetSpinBox.value()):
        self.dataSetSpinBox.setValue(self.parent.activeData + 1)
      selectionModel = self.resultstable.selectionModel()
      for index in selind:
        cellItem = selectionModel.model().index(index, 0)
        selectionModel.select(cellItem, QtCore.QItemSelectionModel.SelectionFlag.Select | QtCore.QItemSelectionModel.SelectionFlag.Rows)

  def toggleToolTips(self, state=True):
    # dynamically sets and unsets tool tips
    btn = self.resultstable.findChild(QtWidgets.QAbstractButton)
    if(state):
      btn.setToolTip(self.resultstable.storedCornerTip)
    else:
      btn.setToolTip('')

  def exportWrapper(self, modeHTMLOnly=False):
    # writes results to output file
    global REMEMBERDIR
    if(modeHTMLOnly):
      filter_options = ['HTML Files (*.html)']
    else:
      filter_options = ['HTML Files (*.html)','Excel Files (*.xlsx)']
    filterstring = ';;'.join(filter_options)
    usedir = REMEMBERDIR
    if(self.currExportFile != None):
      usedir = self.currExportFile
    if(modeHTMLOnly):
      caption = 'Compile Report'
    else:
      caption = 'Export Results'
    filename, filter_ = QtWidgets.QFileDialog.getSaveFileName(self, filter=filterstring, directory=usedir, caption=caption)
    filename = str(filename)
    if(PATH_SEPARATOR in filename):
      REMEMBERDIR = filename.split(PATH_SEPARATOR)[:-1]
      REMEMBERDIR = PATH_SEPARATOR.join(REMEMBERDIR)
    elif('/' in filename):
      REMEMBERDIR = filename.split('/')[:-1]
      REMEMBERDIR = PATH_SEPARATOR.join(REMEMBERDIR)
    if(len(filename) > 0):
      mode = filter_options.index(filter_)
      if(modeHTMLOnly or (mode == 0)):
        self.writeHTML(filename=filename)
      elif(mode == 1):
        if(XLSXWRITER_PRESENT):
          self.writeXLS_old(filename=filename)
        elif(OPENPYXL_PRESENT):
          self.writeXLS(filename=filename)
        else:
          self.parent.statusbar.showMessage('Results cannot be written as either xlsxwriter or openpyxl are required!', self.parent.STATUS_TIME)

  def writeHTML(self, filename=None):
    # writes Results table to HTML file
    if(filename != None):
      # set SVG export to path to ensure correct display in browser
      prevSVGSetting = None
      if('svg.fonttype' in matplotlib.rcParams):
        prevSVGSetting = matplotlib.rcParams['svg.fonttype']
      matplotlib.rcParams.update({'svg.fonttype': 'path'})
      # save current figure temporarily under new filename
      if(self.parent.plotArea.figureFill):
        useFaceColor = self.parent.plotArea.figureColor
      else:
        useFaceColor = 'none'
      if(self.parent.plotArea.frameDraw):
        useFrameColor = self.parent.plotArea.frameColor
      else:
        useFrameColor = 'none'
      try:
        self.parent.plotArea.matplot.savefig(filename, format='svg', dpi=600, facecolor=useFaceColor, edgecolor=useFrameColor)
        flag = True
      except:
        flag = False
      # store contents of SVG file in memory
      if(flag):
        with open(filename, 'r', encoding='utf-8') as readhandle:
          svg_plot = readhandle.readlines()
      else:
        svg_plot = []
      
      # save current residuals temporarily under new filename
      try:
        self.parent.plotArea.residplot.savefig(filename, format='svg', dpi=600, facecolor=useFaceColor, edgecolor=useFrameColor)
        flag2 = True
      except:
        flag2 = False

      # store contents of SVG file in memory
      if(flag2):
        with open(filename, 'r', encoding='utf-8') as readhandle:
          svg_resid = readhandle.readlines()
      else:
        svg_resid = []

      # restore SVG export setting to previous value
      if(prevSVGSetting != None):
        matplotlib.rcParams.update({'svg.fonttype': prevSVGSetting})
        
      # generate actual HTML file
      with open(filename, 'w', encoding='utf-8') as writehandle:
        if(writehandle):
          # write header
          writehandle.write('<html xmlns="http://www.w3.org/1999/xhtml">\n<head>\n')
          writehandle.write('<title>Fit-o-mat Results</title>\n')
          writehandle.write('<meta charset="UTF-8">\n')
          writehandle.write('<meta author="Moeglich laboratory, University of Bayreuth">\n')
          writehandle.write('<meta description="Fit-o-mat Fit Results">\n')
          writehandle.write('</head>\n<body>\n')
          
          writehandle.write('<h2>Fit-o-mat Results</h2>\n')
          writehandle.write(asctime() + '\n')
          # check whether current fit results are available
          if(self.parent.lastFitType != 'none'):
            # write heading and div
            writehandle.write('<div style="padding: 5px;">\n')
            writehandle.write('<a href="javascript:;" onclick="togglitsch(\'results\')" class="toggle">\n')
            writehandle.write('<h3 id="results_ctrl">&#9744; Fit Results</h3>\n')
            writehandle.write('</a>\n')
            writehandle.write('<div id="results" style="display:none;">\n')
            writehandle.write('<button id="results_button" class="little">Copy Results</button>\n')
            writehandle.write('<div id="results_content">\n')
            # further check whether these are local or global results
            writehandle.write('<span class="mypre">\n')
            if(self.parent.lastFitType == 'local'):
              if('<br/>' in self.parent.fitarea.outstring.lower()):
                fitresults = self.parent.fitarea.outstring.split('<br/>')
              else:
                fitresults = self.parent.fitarea.outstring.splitlines()
              if(self.parent.fitarea.bootString != ''):
                if('<br/>' in self.parent.fitarea.bootString.lower()):
                  fitresults += self.parent.fitarea.bootString.split('<br/>')
                else:
                  fitresults += self.parent.fitarea.bootString.splitlines()
            else:
              if('<br/>' in self.parent.globalarea.globalOutstring.lower()):
                fitresults = self.parent.globalarea.globalOutstring.split('<br/>')
              else:
                fitresults = self.parent.globalarea.globalOutstring.splitlines()
              writehandle.write('<h4>Global fit</h4>')
            writehandle.write('\n'.join(fitresults[1:]) + '\n')
            writehandle.write('</span>\n')

            # export covariances if available
            if(self.parent.lastFitType == 'local'):
              covar, covarParameters = self.parent.fitarea.covar, self.parent.fitarea.covarParameters
            else:
              covar, covarParameters = self.parent.globalarea.globalCovar, self.parent.globalarea.globalCovarParameters
            if(len(covar)):
              writehandle.write('<u>Covariances</u>\n')
              writehandle.write('<table class="covar">\n<thead>\n<tr>\n')
              theader = '<th>&nbsp;</th>'
              for param in covarParameters:
                theader += '\n<th>' + param + '</th>'
              writehandle.write(theader + '\n')
              writehandle.write('</tr>\n</thead>\n<tbody>\n')
              for index, entry in enumerate(covarParameters):
                if(index + 1 == len(covarParameters)):
                  trow = '<tr class="last">'
                else:
                  trow = '<tr>'
                trow += '<td class="global">' + entry + '</td>'
                for index2 in range(len(covarParameters)):
                  if(index == index2):
                    trow += '\n<td class="diag">'
                  elif(abs(covar[index][index2]) > .9):
                    trow += '\n<td class="excessive">'
                  else:
                    trow += '\n<td>'
                  trow += self.parent.formatNumber(covar[index][index2]) + '</td>'
                trow += '\n</tr>\n'
                writehandle.write(trow)
              writehandle.write('</tbody>\n</table>\n')

            writehandle.write('</div>\n')
            writehandle.write('</div>\n')
            writehandle.write('</div>\n')
          else:
            pass
              
          writehandle.write('<div class="container">\n')
  
          # write graphics
          if(flag or flag2):
            writehandle.write('<div class="flexmatic2">\n')
            writehandle.write('<a href="javascript:;" onclick="togglitsch(\'plot\')" class="toggle">\n')
            writehandle.write('<h3 id="plot_ctrl">&#9746; Plot and Residuals</h3>\n')
            writehandle.write('</a>\n')
            writehandle.write('<div id="plot" style="display:block;">\n')
            writehandle.write('<button id="plot_button" class="little">Copy Plots</button>\n')
            writehandle.write('<div id="plot_content">\n')
            max_width = 0
    
            # write plot figure if available
            if(flag):
              output = False
              for entry in svg_plot:
                if('<svg' in entry):
                  output = True
                  # extract width of SVG item
                  if('width' in entry):
                    red = entry.split('width')[1]
                    max_width = red.split('"')[1]
                if(output):
                  writehandle.write(entry)
                if('</svg' in entry):
                  output = False
                
            # write residuals figure if available
            if(flag2):
              output = False
              for entry in svg_resid:
                if('<svg' in entry):
                  output = True
                if(output):
                  writehandle.write(entry)
                if('</svg' in entry):
                  output = False
            writehandle.write('</div>\n')
            writehandle.write('</div>\n')
            writehandle.write('</div>\n')
  
          # count how many curves are on display currently
          curvesOnDisplay = [i for i in self.parent.fit if i.visibility]
          
          # write parameters
          if(len(curvesOnDisplay)):
            writehandle.write('<div class="flexmatic">\n')
            writehandle.write('<a href="javascript:;" onclick="togglitsch(\'param\')" class="toggle">\n')
            writehandle.write('<h3 id="param_ctrl">&#9746; Parameter Values</h3>\n')
            writehandle.write('</a>\n')
            writehandle.write('<div id="param" style="display:block;">\n')
            writehandle.write('<button id="param_button" class="little">Copy Parameters</button>\n')
            writehandle.write('<div id="param_content">\n')
    
            # function to write parameter values and statistics for one curve
            def writeParamTable(paramList, param_active, param, confidence, confidence_apriori, chisq, redchisq, globalparam):
              writehandle.write('<table class="param">\n<thead>\n<tr>\n')
              writehandle.write('<th>Name</th>\n<th>Value</th>\n<th>Confidence</th>\n<th>Conf. <i>a priori</i></th>\n')
              writehandle.write('</tr>\n</thead>\n')
      
              writehandle.write('<tbody>\n')
              for i, j, k, l, m, n in zip(paramList, param, confidence, confidence_apriori, param_active, globalparam):
                if(i == paramList[-1]):
                  writehandle.write('<tr class="last">\n')
                else:
                  writehandle.write('<tr>\n')
                if(m):
                  if(n):
                    writehandle.write('<td class="global">' + str(i) + '</td>\n')
                  else:
                    writehandle.write('<td>' + str(i) + '</td>\n')
                else:
                  if(n):
                    writehandle.write('<td class="fixed global">' + str(i) + '</td>\n')
                  else:
                    writehandle.write('<td class="fixed">' + str(i) + '</td>\n')
                if(m):
                  if(n):
                    tdString = '<td class="rite global">'
                  else:
                    tdString = '<td class="rite">'
                else:
                  if(n):
                    tdString = '<td class="rite fixed global">'
                  else:
                    tdString = '<td class="rite fixed">'
                writehandle.write(tdString + self.parent.formatNumber(j) + '</td>\n')
                writehandle.write(tdString + self.parent.formatNumber(k) + '</td>\n')
                writehandle.write(tdString + self.parent.formatNumber(l) + '</td>\n')
                writehandle.write('</tr>\n')
              writehandle.write('</tbody>\n</table>\n')
              if(1):
              ###if(self.parent.lastFitType != 'none'):
                try:
                  useChi = self.parent.formatNumber(chisq)
                except:
                  useChi = str(chisq)
                writehandle.write('<i>&#x1D6D8;</i><sup>2</sup> = ' + useChi + '\n')
                try:
                  useRedChi = self.parent.formatNumber(redchisq)
                except:
                  useRedChi = str(redchisq)
                writehandle.write('<br/>reduced <i>&#x1D6D8;</i><sup>2</sup> = ' + useRedChi + '\n')
    
            # check whether last fit local or global
            if(self.parent.lastFitType != 'global'):
              for index, entry in enumerate(self.parent.fit):
                if(entry.visibility):
                  tempLineString = '<span style=\'color:rgb(' + ', '.join([str(i * 255) for i in entry.style['color'][0:3]]) + ');\nfont-size:150%;\'>&#x2015;</span>'
                  writehandle.write('<h4>' + tempLineString + ' ' + str(index) + ' -- '  + entry.name + '</h4>\n')
                  paramList, param_active, param, confidence, confidence_apriori, chisq, redchisq = entry.retrieveInfoReport()
                  globalParam = [False for i in param]
                  writeParamTable(paramList, param_active, param, confidence, confidence_apriori, chisq, redchisq, globalParam)
                  writehandle.write('<br/>(fixed parameters in <i>italics</i>)\n')
                  writehandle.write('<br/>\n')
            else:
              writehandle.write('<h4>Global fit</h4>\n')
              paramList, param_active, param, confidence, confidence_apriori, chisq, redchisq, globalParam = self.parent.globalarea.reportGlobalParam()
              writeParamTable(paramList, param_active, param, confidence, confidence_apriori, chisq, redchisq, globalParam)
              writehandle.write('<br/>(fixed parameters in <i>italics</i>)\n')
              writehandle.write('<br/>(global parameters in <b>bold</b>)\n')
    
            writehandle.write('</div>\n')
            writehandle.write('</div>\n')
            writehandle.write('</div>\n')
  
          # define HTML translation of plot symbols
          translatedStyles = {'.': '&#x2022;', ',': '&#x2015;', 'o': '&#x25cf;', 'v': '&#x25bc;', '^': '&#x25b2;', '<': '&#x25c0;', '>': '&#x25b6;',\
                              '1': 'Y', '2': 'Y', '3': 'Y', '4': 'Y', '8': '&#x25cf;', 's': '&#x25fc;', 'p': '&#x2b1f;', '*': '&#x2605;',\
                              'h': '&#x2b22;', 'H': '&#x2b22;', '+': '<b>+</b>', 'x': '&#x00d7;', 'D': '&#x25c6;', 'd': '&#x25c6;',\
                              '|': '|', '_': '-', 'P': '&#x2b1f;', 'X': '&#x00d7;', 0: '-', 1: '-', 2: '|', 3: '|', 4: '&#x25c0;', 5: '&#x25b6;',\
                              6: '&#x25b2;', 7: '&#x25bc;', 8: '&#x25c0;', 9: '&#x25b6;', 10: '&#x25b2;', 11: '&#x25bc;',\
                              'None': '&#x2015;', None: '&#x2015;', ' ': '&#x2015;', '': '&#x2015;'}
          filledStyles = ['.', 'o', 'v', '^', '<', '>', '8', 's', 'p', 'P', '*', 'h', 'H', 'd', 'D', 'X', 4, 5, 6, 7, 8, 9, 10, 11]
          translatedOpenStyles = {'.': '&#x25cb;', 'o': '&#x25cb;', 'v': '&#x25bd;', '^': '&#x25b3;', '<': '&#x25c1;', '>': '&#x25b7;',\
                              '8': '&#x25cb;', 's': '&#x25fb;', 'p': '&#x2b20;', '*': '&#x2606;',\
                              'h': '&#x2b21;', 'H': '&#x2b21;', 'x': '&#x00d7;', 'D': '&#x25c7;', 'd': '&#x25c7;',\
                              'P': '&#x2b20;', 'X': '&#x00d7;', 4: '&#x25c1;', 5: '&#x25b7;',\
                              6: '&#x25b3;', 7: '&#x25bd;', 8: '&#x25c1;', 9: '&#x25b7;', 10: '&#x25b3;', 11: '&#x25bd;'}
          noneStyles = ['None', None, ' ', '']
  
          # write data
          dataFlag, dataButtons = False, []
          if(self.parent.lastFitType != 'global'):
            # collate and order data sets
            useDatasets = [i for i in self.parent.data if (i.visibility)]
            useDatasets = sorted(useDatasets, key=lambda k: k.zorder)
            if(len(useDatasets)):
              writehandle.write('<div class="flexmatic">\n')
              for entry in useDatasets:
                index = useDatasets.index(entry)
                dataButtons.append(index)
                values, roles = entry.getData_n_Fit()
                writehandle.write('<a href="javascript:;" onclick="togglitsch(\'data_' + str(index) + '\')" class="toggle">\n')
                useSymbol = entry.style['marker']
                toggledColor = False
                if((useSymbol in filledStyles) or (useSymbol in CUSTOM_MARKERS)):
                  useColor = entry.style['markerfacecolor']
                  if(((not entry.style['doFill']) or (sum(useColor[0:3]) > 2.8)) and (useSymbol in translatedOpenStyles)):
                    useColor = entry.style['markeredgecolor']
                    toggledColor = True
                elif(useSymbol in noneStyles):
                  useColor = entry.style['color']
                else:
                  useColor = entry.style['markeredgecolor']
                if(useSymbol in translatedStyles):
                  translatedSymbol = translatedStyles[useSymbol]
                  if(((not entry.style['doFill']) or toggledColor) and (useSymbol in translatedOpenStyles)):
                    translatedSymbol = translatedOpenStyles[useSymbol]
                elif(useSymbol in CUSTOM_MARKERS):
                  # replace funky symbols by default circle
                  translatedSymbol = translatedStyles['o']
                  if(not entry.style['doFill']):
                    translatedSymbol = translatedOpenStyles['o']
                else:
                  translatedSymbol = translatedStyles['None']
                tempLineString = '<span style=\'color:rgb(' + ', '.join([str(i * 255) for i in useColor[0:3]]) + ');\nfont-size:150%;\'>' + translatedSymbol + '</span>'
                writehandle.write('<h3 id="data_' + str(index) + '_ctrl">&#9744; ' + tempLineString + ' Data set ' + entry.name + '</h3>\n')
                writehandle.write('</a>\n')
                writehandle.write('<div id="data_' + str(index) + '" style="display:none;">\n')
                writehandle.write('<button id="data_' + str(index) + '_button" class="little">Copy Data</button>\n')
                if(hasattr(entry, 'notes') and (entry.notes != '')):
                  writehandle.write('<span class="notes"><span style="text-decoration:underline;margin-left:-5px;">Notes</span>\n')
                  writehandle.write(entry.notes.replace('"', '\"'))
                  writehandle.write('</span>\n')
                writehandle.write('<div id="data_' + str(index) + '_content">\n')
                writehandle.write('<table class="misc">\n<thead>\n<tr>\n')
                for header in roles:
                  writehandle.write('<th>' + str(header) + '</th>\n')
                writehandle.write('</tr>\n</thead>\n')
                # write data table
                writehandle.write('<tbody>\n')
                resultsData = values
                # check for selection
                if(self.parent.selectedData.isLive and (entry == self.parent.data[self.parent.activeData])):
                  indices = self.parent.selectedData.getIndices()
                else:
                  indices = []
                for count, row in enumerate(resultsData):
                  # deal with selection
                  if(len(roles) and (count < len(indices)) and indices[count]):
                    if(count + 1 < len(resultsData)):
                      writehandle.write('<tr class="selected">\n')
                    else:
                      writehandle.write('<tr class="selected last">\n')
                  elif(count + 1 < len(resultsData)):
                    writehandle.write('<tr>\n')
                  else:
                    writehandle.write('<tr class="last">\n')
                  row = [self.parent.formatNumber(i) for i in row]
                  rowstring = '</td><td>'.join(row)
                  writehandle.write('<td>' + rowstring + '</td>')
                  writehandle.write('\n</tr>\n')
      
                writehandle.write('</tbody>\n</table>\n')
                # did we have selected data?
                if(len(indices)):
                  writehandle.write('<br/>(selected data in <span style="background-color:#aaaaee;">blue</span>)\n')
                writehandle.write('</div>\n')
                writehandle.write('</div>\n')
  
              writehandle.write('</div>\n')
          else:
            if(len(self.parent.lastFitGlobalDataSets)):
              for index in self.parent.lastFitGlobalDataSets:
                if(index < len(self.parent.data)):
                  dataFlag = True
                  
              if(dataFlag):
                writehandle.write('<div class="flexmatic">\n')
                for index in self.parent.lastFitGlobalDataSets:
                  if(index < len(self.parent.data)):
                    dataButtons.append(index)
                    values, roles = self.parent.data[index].getData_n_Fit()
                    writehandle.write('<a href="javascript:;" onclick="togglitsch(\'data_' + str(index) + '\')" class="toggle">\n')
                    useSymbol = self.parent.data[index].style['marker']
                    toggledColor = False
                    if((useSymbol in filledStyles) or (useSymbol in CUSTOM_MARKERS)):
                      useColor = self.parent.data[index].style['markerfacecolor']
                      if(((not self.parent.data[index].style['doFill']) or (sum(useColor[0:3]) > 2.8)) and (useSymbol in translatedOpenStyles)):
                        useColor = self.parent.data[index].style['markeredgecolor']
                        toggledColor = True
                    elif(useSymbol in noneStyles):
                      useColor = self.parent.data[index].style['color']
                    else:
                      useColor = self.parent.data[index].style['markeredgecolor']
                    if(useSymbol in translatedStyles):
                      translatedSymbol = translatedStyles[useSymbol]
                      if(((not self.parent.data[index].style['doFill']) or toggledColor) and (useSymbol in translatedOpenStyles)):
                        translatedSymbol = translatedOpenStyles[useSymbol]
                    elif(useSymbol in CUSTOM_MARKERS):
                      # replace funky symbols by default circle
                      translatedSymbol = translatedStyles['o']
                      if(not self.parent.data[index].style['doFill']):
                        translatedSymbol = translatedOpenStyles['o']
                    else:
                      translatedSymbol = translatedStyles['None']
                    tempLineString = '<span style=\'color:rgb(' + ', '.join([str(i * 255) for i in useColor[0:3]]) + ');\nfont-size:150%;\'>' + translatedSymbol + '</span>'
                    writehandle.write('<h3 id="data_' + str(index) + '_ctrl">&#9744; ' + tempLineString + ' Data set ' + self.parent.data[index].name + '</h3>\n')
                    writehandle.write('</a>\n')
                    writehandle.write('<div id="data_' + str(index) + '" style="display:none;">\n')
                    writehandle.write('<button id="data_' + str(index) + '_button" class="little">Copy Data</button>\n')
                    if(hasattr(self.parent.data[index], 'notes') and (self.parent.data[index].notes != '')):
                      writehandle.write('<span class="notes"><span style="text-decoration:underline;margin-left:-5px;">Notes</span>\n')
                      writehandle.write(self.parent.data[index].notes.replace('"', '\"'))
                      writehandle.write('</span>\n')
                    writehandle.write('<div id="data_' + str(index) + '_content">\n')
                    writehandle.write('<table class="misc">\n<thead>\n<tr>\n')
                    for header in roles:
                      writehandle.write('<th>' + str(header) + '</th>\n')
                    writehandle.write('</tr>\n</thead>\n')
                    # write data table
                    writehandle.write('<tbody>\n')
                    resultsData = values
                    for count, row in enumerate(resultsData):
                      if(count + 1 < len(resultsData)):
                        writehandle.write('<tr>\n')
                      else:
                        writehandle.write('<tr class="last">\n')
                      row = [self.parent.formatNumber(i) for i in row]
                      rowstring = '</td><td>'.join(row)
                      writehandle.write('<td>' + rowstring + '</td>')
                      writehandle.write('\n</tr>\n')
      
                    writehandle.write('</tbody>\n</table>\n')
                    writehandle.write('</div>\n')
                    writehandle.write('</div>\n')
                
                writehandle.write('</div>\n')
  
          # write simulated data as well
          functionFlag, functionButtons = False, []
          if(self.parent.lastFitType != 'global'):
            # collate and order curves
            useCurves = [i for i in self.parent.fit if (i.visibility)]
            useCurves = sorted(useCurves, key=lambda k: k.zorder)
            if(len(useCurves)):
              writehandle.write('<div class="flexmatic">\n')
              for entry in useCurves:
                index = useCurves.index(entry)
                functionButtons.append(index)
                simX, simY = entry.x, entry.y
                writehandle.write('<a href="javascript:;" onclick="togglitsch(\'fxn_' + str(index) + '\')" class="toggle">\n')
                tempLineString = '<span style=\'color:rgb(' + ', '.join([str(i * 255) for i in entry.style['color'][0:3]]) + ');\nfont-size:150%;\'>&#x2015;</span>'
                writehandle.write('<h3 id="fxn_' + str(index) + '_ctrl">&#9744; ' + tempLineString + ' Curve ' + entry.name + '</h3>\n')
                writehandle.write('</a>\n')
                writehandle.write('<div id="fxn_' + str(index) + '" style="display:none;">\n')
                writehandle.write('<span class="mypre">\n' + entry.ffuncstr_base + '</span>\n<br/><br/>\n')
                writehandle.write('<button id="fxn_' + str(index) + '_button" class="little">Copy Curve</button>\n')
                if(hasattr(entry, 'notes') and (entry.notes != '')):
                  writehandle.write('<span class="notes"><span style="text-decoration:underline;margin-left:-5px;">Notes</span>\n')
                  writehandle.write(entry.notes.replace('"', '\"'))
                  writehandle.write('</span>\n')
                writehandle.write('<div id="fxn_' + str(index) + '_content">\n')
                writehandle.write('<table class="misc">\n<thead>\n<tr>\n')
                writehandle.write('<th>x</th>\n<th>f(x)</th>\n')
                writehandle.write('<tbody>\n')
                for count, row in enumerate(range(len(simX))):
                  if(count + 1 < len(simX)):
                    writehandle.write('<tr>\n')
                  else:
                    writehandle.write('<tr class="last">\n')
                  writehandle.write('<td>' + self.parent.formatNumber(simX[row]) + '</td>')
                  writehandle.write('<td>' + self.parent.formatNumber(simY[row]) + '</td>')
                  writehandle.write('\n</tr>\n')
                writehandle.write('</tbody>\n</table>\n')
                writehandle.write('</div>\n')
                writehandle.write('</div>\n')
  
              writehandle.write('</div>\n')
          else:
            if(len(self.parent.lastFitGlobalCurves)):
              for index in self.parent.lastFitGlobalCurves:
                if(index < len(self.parent.fit)):
                  functionFlag = True
                  
            if(functionFlag):
              writehandle.write('<div class="flexmatic">\n')
              for index in self.parent.lastFitGlobalCurves:
                if(index < len(self.parent.fit)):
                  functionButtons.append(index)
                  simX, simY = self.parent.fit[index].x, self.parent.fit[index].y
                  writehandle.write('<a href="javascript:;" onclick="togglitsch(\'fxn_' + str(index) + '\')" class="toggle">\n')
                  tempLineString = '<span style=\'color:rgb(' + ', '.join([str(i * 255) for i in self.parent.fit[index].style['color'][0:3]]) + ');\nfont-size:150%;\'>&#x2015;</span>'
                  writehandle.write('<h3 id="fxn_' + str(index) + '_ctrl">&#9744; ' + tempLineString + ' Curve ' + self.parent.fit[index].name + '</h3>\n')
                  writehandle.write('</a>\n')
                  writehandle.write('<div id="fxn_' + str(index) + '" style="display:none;">\n')
                  writehandle.write('<span class="mypre">\n' + self.parent.fit[index].ffuncstr_base + '</span>\n<br/><br/>\n')
                  writehandle.write('<button id="fxn_' + str(index) + '_button" class="little">Copy Curve</button>\n')
                  if(hasattr(self.parent.fit[index], 'notes') and (self.parent.fit[index].notes != '')):
                    writehandle.write('<span class="notes"><span style="text-decoration:underline;margin-left:-5px;">Notes</span>\n')
                    writehandle.write(self.parent.fit[index].notes.replace('"', '\"'))
                    writehandle.write('</span>\n')
                  writehandle.write('<div id="fxn_' + str(index) + '_content">\n')
                  writehandle.write('<table class="misc">\n<thead>\n<tr>\n')
                  writehandle.write('<th>x</th>\n<th>f(x)</th>\n')
                  writehandle.write('<tbody>\n')
                  for count, row in enumerate(range(len(simX))):
                    if(count + 1 < len(simX)):
                      writehandle.write('<tr>\n')
                    else:
                      writehandle.write('<tr class="last">\n')
                    writehandle.write('<td>' + self.parent.formatNumber(simX[row]) + '</td>')
                    writehandle.write('<td>' + self.parent.formatNumber(simY[row]) + '</td>')
                    writehandle.write('\n</tr>\n')
                  writehandle.write('</tbody>\n</table>\n')
                  writehandle.write('</div>\n')
                  writehandle.write('</div>\n')
  
              writehandle.write('</div>\n')
            
          writehandle.write('</div>\n')
          writehandle.write('<div class="disclaimer">generated by fit-o-mat v' + VERSION + ' by @MoeglichLab</div>\n')
  
          # add style definitions
          writehandle.write('</body>\n')
          writehandle.write('<style type="text/css">\n')
          writehandle.write('body {\nfont-family: Calibri,Helvetica,Arial,sans-serif;\n}\n')
          writehandle.write('.container {\npadding: 0;\nmargin: 0;\ndisplay: flex;\nflex-direction: row;\
            \nalign-items: flex-start;\n}\n')
          writehandle.write('.flexmatic {\npadding: 5px;\nmargin: 0;\nflex: 0 0 auto;\n}\n')
          writehandle.write('.flexmatic2 {\npadding: 5px;\nmargin: 0;\nflex: 1 1 auto;\n')
          if(max_width != 0):
            writehandle.write('max-width: ' + max_width + ';\n')
          writehandle.write('min-width: 250pt;\n}\n')
          writehandle.write('svg {\nwidth: 100%;\nheight: auto;\n}\n')
          writehandle.write('h2 {\ncolor: rgb(13, 78, 179);\nfont-size: 250%;\nfont-weight: normal;\n}\n')
          writehandle.write('h3 {\ntext-align: left;\nwhite-space: nowrap;\n}\n')
          writehandle.write('h3:hover {\nbackground-color: rgba(255, 243, 185, 255);\ndisplay: table;\n}\n')
          writehandle.write('.disclaimer {\nposition: fixed;\nbottom: 0px;\nright: 0px;\nfont-size: 125%;\
                                          \ncolor: rgb(13, 78, 179);\nbackground-color: rgba(255, 255, 255, 0.5);\
                                          \nborder: 1px;\nborder-style: solid;\nborder-radius: 2px;\
                                          \nborder-color: #333333;\npadding: 2px 10px 2px 10px;\n}\n')
          writehandle.write('td.rite {\ntext-align: right;\n}\n')
          writehandle.write('td.fixed {\ncolor: #555555;\nfont-style: italic;\n}\n')
          writehandle.write('td.global {\nfont-weight: bold;\n}\n')
          writehandle.write('td.diag {\nbackground-color: #aaaaaa;\n}\n')
          writehandle.write('td.excessive {\nbackground-color: #ffaabb;\n}\n')
          writehandle.write('tr.last > td {\nborder-bottom: 1px solid #333333;\n}\n')
          writehandle.write('tr.selected > td {\nbackground-color: #aaaaee !important;\n}\n')
          writehandle.write('th {\nborder-bottom: 1px solid #333333;\n}\n')
          writehandle.write('table.param > tbody > tr:nth-child(odd) > td {\nbackground-color: #e5e5e5;\n}\n')
          writehandle.write('table.misc > tbody > tr:nth-child(odd) > td {\nbackground-color: #e5e5e5;\n}\n')
          writehandle.write('a.toggle {\ntext-decoration: none;\ncolor: inherit;\n}\n')
          writehandle.write('button.little {\nfont-size: 75%;\n}\n')
          writehandle.write('.mypre {\nwhite-space: pre;\n}\n')
          writehandle.write('.notes {\nbackground-color: fffcb8;\ndisplay: block;\nmargin: 10px;\npadding: 5px;\nwhite-space:pre;\n}\n')
          writehandle.write('</style>\n\n')
  
          # add script controls
          writehandle.write('<script language="javascript">\n')
          writehandle.write('function togglitsch(id){\n')
          writehandle.write('\tvar content;\n')
          writehandle.write('\tif(document.getElementById(id).style.display == \'none\'){\n')
          writehandle.write('\t\tdocument.getElementById(id).style.display = \'block\';\n')
          writehandle.write('\t\tcontent = document.getElementById(id + \'_ctrl\').innerHTML;\n')
          writehandle.write('\t\tdocument.getElementById(id + \'_ctrl\').innerHTML = \'&#9746;\' + content.slice(1)\n')
          writehandle.write('\t} else {\n')
          writehandle.write('\t\tdocument.getElementById(id).style.display = \'none\';\n')
          writehandle.write('\t\tcontent = document.getElementById(id + \'_ctrl\').innerHTML;\n')
          writehandle.write('\t\tdocument.getElementById(id + \'_ctrl\').innerHTML = \'&#9744;\' + content.slice(1)\n')
          writehandle.write('\t}\n}\n\n')
  
          writehandle.write('// controls for copy to clipboard\n')
          writehandle.write('function copyHelper(item, mode){\n')
          writehandle.write('\t// selects target range and then restores previous selection\n')
          writehandle.write('\tvar sel, backup = [];\n\n')
          writehandle.write('\t// backup ranges\n')
          writehandle.write('\tsel = window.getSelection();\n')
          writehandle.write('\tfor(let i = 0; i < sel.rangeCount; i++) {\n')
          writehandle.write('\t\tbackup[i] = sel.getRangeAt(i);\n')
          writehandle.write('\t}\n\n')
          writehandle.write('\t// delete previous ranges\n')
          writehandle.write('\tsel.removeAllRanges();\n')
          writehandle.write('\tif(mode){\n')
          writehandle.write('\t\tsel.addRange(item);\n')
          writehandle.write('\t} else {\n')
          writehandle.write('\t\titem.focus();\n')
          writehandle.write('\t\titem.select();\n')
          writehandle.write('\t}\n')
          writehandle.write('\t// copy selection\n')
          writehandle.write('\tdocument.execCommand(\'copy\');\n')
          writehandle.write('\t// restore previous selection\n')
          writehandle.write('\tsel.removeAllRanges();\n')
          writehandle.write('\tfor(let i = 0; i < backup.length; i++) {\n')
          writehandle.write('\t\tsel.addRange(backup[i]);\n')
          writehandle.write('\t}\n}\n\n')
  
          writehandle.write('function copyThis(targetId){\n')
          writehandle.write('\tvar body = document.body, target, range;\n')
          writehandle.write('\ttarget = document.getElementById(targetId);\n\n')
          writehandle.write('\tif (document.createRange && window.getSelection) {\n')
          writehandle.write('\t\t// select target stuff\n')
          writehandle.write('\t\trange = document.createRange();\n')
          writehandle.write('\t\ttry {\n')
          writehandle.write('\t\t\trange.selectNodeContents(target);\n')
          writehandle.write('\t\t} catch (e) {\n')
          writehandle.write('\t\t\trange.selectNode(target);\n')
          writehandle.write('\t\t}\n')
          writehandle.write('\t\tcopyHelper(range, true);\n')
          writehandle.write('\t}\n}\n\n')
          
          writehandle.write('// enable copy of SVG plots\n')
          writehandle.write('function copyThisInnerHTML(targetId){\n')
          writehandle.write('\tvar target, textarea, scrollX, scrollY;\n')
          writehandle.write('\ttarget = document.getElementById(targetId);\n')
          writehandle.write('\t// create dummy text area to temporarily paste innerHTML data\n')
          writehandle.write('\ttextarea = document.createElement(\'textarea\');\n')
          writehandle.write('\ttextarea.style.height = 0;\n')
          writehandle.write('\tscrollX = window.scrollX;\n')
          writehandle.write('\tscrollY = window.scrollY;\n')
          writehandle.write('\tdocument.body.appendChild(textarea);\n')
          writehandle.write('\t// now fill in textarea\n')
          writehandle.write('\ttextarea.value = target.innerHTML;\n')
          writehandle.write('\t// now call copyThis() on the temp object\n')
          writehandle.write('\tcopyHelper(textarea, false);\n')
          writehandle.write('\t// clean up the mess\n')
          writehandle.write('\tdocument.body.removeChild(textarea);\n')
          writehandle.write('\twindow.scroll(scrollX, scrollY);\n')
          writehandle.write('}\n\n')
  
          writehandle.write('// set up ctrl buttons\n')
          if(self.parent.lastFitType != 'none'):
            items = ['results']
          else:
            items = []
          for entry in dataButtons:
            items.append('data_' + str(entry))
          if(len(curvesOnDisplay)):
            items.append('param')
          for entry in functionButtons:
            items.append('fxn_' + str(entry))
          for entry in items:
            writehandle.write('var btn_' + entry + ' = document.getElementById(\'' + entry + '_button\');\n')
            writehandle.write('btn_' + entry + '.addEventListener(\'click\', function(event) {\n')
            writehandle.write('\tcopyThis(\'' + entry + '_content\');\n')
            writehandle.write('});\n\n')
        
          writehandle.write('var btn_plot = document.getElementById(\'plot_button\');\n')
          writehandle.write('btn_plot.addEventListener(\'click\', function(event) {\n')
          writehandle.write('\tcopyThisInnerHTML(\'plot_content\');\n')
          writehandle.write('});\n\n')
  
          writehandle.write('</script>\n')
          writehandle.write('</html>\n')
        
        # save filename for future use
        self.currExportFile = filename
      
  def writeXLS(self, filename=None):
    # writes Results table to Excel file
    # this function uses openpyxl which, however, has awful documentation
    # hence, the output is somewhat rudimentary
    # (sorry, but do not have time to dive deeper into the inane .xlsx XML format)
    # (once again, it goes to show that Microsoft are a bunch of jerks)
    if(filename != None):
      workbook = openpyxl.Workbook()
      availableSheets = workbook.get_sheet_names()
      if(not len(availableSheets)):
        workbook.create_sheet()
      worksheet = workbook.active
      worksheet.title = 'Fit-o-mat forever'

      # check whether current fit results are available
      index = 0
      if(self.parent.lastFitType != 'none'):
        # further check whether these are local or global results
        if(self.parent.lastFitType == 'local'):
          fitresults = self.parent.fitarea.outstring.splitlines()
        else:
          fitresults = self.parent.globalarea.globalOutstring.splitlines()
        for index, entry in enumerate(fitresults):
          worksheet.cell(row=1, column=index + 1).value = entry
      if(not index):
        offset = index + 3
      else:
        offset = 1
        
      # write header
      if((self.resultstable.tableModel != None) and (self.resultstable.tableModel.columnCount() > 0)):
        resultsHeader = self.resultstable.tableModel.getHeaders()
        for col, header in enumerate(resultsHeader):
          worksheet.cell(row=offset, column=col + 1).value = header
        # write data
        resultsData = self.resultstable.tableModel.getAllData()
        for rowIndex, row in enumerate(resultsData):
          row = [float(self.parent.formatNumber(i)) if (not j in ['labels', 'labels2']) else i for i, j in zip(row, resultsHeader)]
          for colIndex, entry in enumerate(row):
            worksheet.cell(row=rowIndex + offset + 1, column=colIndex + 1).value = entry
        coloffset = self.resultstable.tableModel.columnCount() + 2
      else:
        coloffset = 1
        
      # write simulated data as well
      simX, simY = self.parent.fit[self.parent.activeFit].x, self.parent.fit[self.parent.activeFit].y
      worksheet.cell(offset, coloffset).value = 'x'
      worksheet.cell(offset, coloffset + 1).value = 'f(x)'
      for row in range(len(simX)):
        worksheet.cell(row=row + offset + 1, column=coloffset).value = simX[row]
        worksheet.cell(row=row + offset + 1, column=coloffset + 1).value = simY[row]

      # # write graphics
      chart = openpyxl.chart.ScatterChart(marker=False)
      chart.title = 'Fit-o-mat Results'

      # plot fit
      xval = openpyxl.chart.Reference(worksheet, min_col=coloffset, min_row=offset + 1, max_col=coloffset, max_row=offset + len(simX))
      yval = openpyxl.chart.Reference(worksheet, min_col=coloffset + 1, min_row=offset + 1, max_col=coloffset + 1, max_row=offset + len(simX))
      # seriously Microsoft, you utter jerks
      # putting y values before x values? what the hell, complete morons!
      series1 = openpyxl.chart.Series(yval, xval, title='fit')
      # set props and add to chart
      series1.graphicalProperties.line.solidFill = 'FF0000'
      series1.marker.symbol = None
      chart.series.append(series1)
      
      # plot data (if present)
      if(('x' in self.descriptors) and ('y' in self.descriptors)):
        xcol, ycol = self.descriptors.index('x'), self.descriptors.index('y')
        rowcount = self.resultstable.tableModel.rowCount()
        xval2 = openpyxl.chart.Reference(worksheet, min_col=xcol + 1, min_row=offset + 1, max_col=xcol + 1, max_row=offset + rowcount)
        yval2 = openpyxl.chart.Reference(worksheet, min_col=ycol + 1, min_row=offset + 1, max_col=ycol + 1, max_row=offset + rowcount)
        series2 = openpyxl.chart.Series(yval2, xval2, title='data')
        # set props and add to chart
        series2.graphicalProperties.line.solidFill = None
        series2.graphicalProperties.line.noFill =True
        series2.marker.symbol = 'circle'
        series2.marker.graphicalProperties.solidFill = '000000'

        # error bars are not supported for time being b/c:
        # 1. documentation of openpyxl positively atrocious
        # 2. .xlsx XML format absolutely insane
        chart.series.append(series2)

      # requisite chart formatting
      chart.x_axis.title = self.parent.plotArea.labelX
      chart.y_axis.title = self.parent.plotArea.labelX
      chart.x_axis.delete = False
      chart.y_axis.delete = False
      chart.x_axis.minorGridlines = None
      chart.x_axis.majorGridlines = None
      chart.y_axis.minorGridlines = None
      chart.y_axis.majorGridlines = None

      # place chart on worksheet
      worksheet.add_chart(chart, chr(65 + 1 + coloffset) + str(offset))

      try:
        workbook.save(filename)
      except:
        self.parent.statusbar.showMessage('Cannot write data file ' + filename, self.parent.STATUS_TIME)

  # this is the original implementation using xlsxwriter which, however, seems to be going stale
  # switch to openpyxl instead but retain this code for time being as it may become useful again, tbd
  # actually, retain xlsxwriter as preferred option as openpyxl is a major mess, phew!
  def writeXLS_old(self, filename=None):
    # writes Results table to Excel file
    if(filename != None):
      try:
        workbook = xlsxwriter.Workbook(filename, options={'nan_inf_to_errors': True})
        worksheet = workbook.add_worksheet()
      except:
        self.parent.statusbar.showMessage('Cannot write data file ' + filename, self.parent.STATUS_TIME)
      else:
        # check whether current fit results are available
        index = 0
        if(self.parent.lastFitType != 'none'):
          # further check whether these are local or global results
          if(self.parent.lastFitType == 'local'):
            fitresults = self.parent.fitarea.outstring.splitlines()
          else:
            fitresults = self.parent.globalarea.globalOutstring.splitlines()
          for index, entry in enumerate(fitresults):
            worksheet.write(index, 0, entry)
        offset = index + 2
        # write header
        if((self.resultstable.tableModel != None) and (self.resultstable.tableModel.columnCount() > 0)):
          resultsHeader = self.resultstable.tableModel.getHeaders()
          worksheet.write_row(offset, 0, resultsHeader)
          # write data
          resultsData = self.resultstable.tableModel.getAllData()
          for rowIndex, row in enumerate(resultsData):
            row = [float(self.parent.formatNumber(i)) if (not j in ['labels', 'labels2']) else i for i, j in zip(row, resultsHeader)]
            worksheet.write_row(rowIndex + offset + 1, 0, row)
          coloffset = self.resultstable.tableModel.columnCount() + 1
        else:
          coloffset = 0
        # write simulated data as well
        simX, simY = self.parent.fit[self.parent.activeFit].x, self.parent.fit[self.parent.activeFit].y
        worksheet.write(offset, coloffset, 'x')
        worksheet.write(offset, coloffset + 1, 'f(x)')
        for row in range(len(simX)):
          worksheet.write(row + offset + 1, coloffset, simX[row])
          worksheet.write(row + offset + 1, coloffset + 1, simY[row])
        # write graphics
        chart = workbook.add_chart({'type': 'scatter'})
        worksheet.insert_chart(chr(coloffset + 1 + 67) + str(offset + 2), chart)
        # write fit
        chart.add_series({'categories': ['Sheet1', offset + 1, coloffset, offset + row + 1, coloffset],\
                          'values': ['Sheet1', offset + 1, coloffset + 1, offset + row + 1, coloffset + 1],\
                          'line': {'color': 'red'},\
                          'name': 'fit',\
                          'marker': {'type': 'none'}})
        # write data (if present)
        if(('x' in self.descriptors) and ('y' in self.descriptors)):
          xcol, ycol = self.descriptors.index('x'), self.descriptors.index('y')
          rowcount = self.resultstable.tableModel.rowCount()
          chartdict = {'categories': ['Sheet1', offset+1, xcol, offset + rowcount, xcol],\
                            'values': ['Sheet1', offset+1, ycol, offset + rowcount, ycol],\
                            'name': 'data',\
                            'marker': {'type': 'diamond'}}
          # include x-errors
          if('xerr' in self.descriptors):
            xerrcol = self.descriptors.index('xerr')
            rangestring = 'Sheet1!$' + chr(xerrcol + 65) + '$' + str(offset + 2) + ':$' + chr(xerrcol + 65) + '$' + str(offset + 1 + rowcount)
            chartdict['x_error_bars'] = {'type': 'custom',\
                     'plus_values': rangestring,\
                     'minus_values': rangestring}
          # include y-errors
          if('yerr' in self.descriptors):
            yerrcol = self.descriptors.index('yerr')
            rangestring = 'Sheet1!$' + chr(yerrcol + 65) + '$' + str(offset + 2) + ':$' + chr(yerrcol + 65) + '$' + str(offset + 1 + rowcount)
            chartdict['y_error_bars'] = {'type': 'custom',\
                     'plus_values': rangestring,\
                     'minus_values': rangestring}
          chart.add_series(chartdict)
    
        workbook.close()

  def setDataSet(self, currIndex=0, maxIndex=None):
    # updates spinbox for dataset selection
    if(maxIndex != None):
      self.dataSetSpinBox.setMaximum(maxIndex + 1)
      if(maxIndex):
        self.dataSetSpinBox.setEnabled(True)
      else:
        self.dataSetSpinBox.setEnabled(False)
    # set spin box
    self.dataSetSpinBox.blockSignals(True)
    self.dataSetSpinBox.setValue(currIndex + 1)
    self.dataSetSpinBox.blockSignals(False)
    # trigger update
    self.changeDataSet()

  def changeDataSet(self):
    # change dataset to display
    currIndex = self.dataSetSpinBox.value()
    values, descriptors = self.parent.data[currIndex - 1].getData_n_Fit()
    labels = self.parent.data[currIndex - 1].getLabels(labels2=False)
    labels2 = self.parent.data[currIndex - 1].getLabels(labels2=True)
    self.dataSetName.setText(self.parent.data[currIndex - 1].name)
    # now set table content
    self.updateResults(values=values, descriptors=descriptors, labels=labels, labels2=labels2)
    # now check for presence of selectedData and update row selection
    if(self.parent.selectedData.isLive and (currIndex == self.parent.activeData + 1)):
      selectIndices = [i for i, j in enumerate(self.parent.selectedData.getIndices()) if j]
      self.selectTheseRows(selectIndices)
    # check whether data set has labels and update button accordingly
    if(len(labels)):
      self.addLabelsButton.setEnabled(False)
    else:
      self.addLabelsButton.setEnabled(True)
    
  def updateResults(self, values=[], descriptors=[], labels=[], labels2=[]):
    # updates results table
    if(len(values)):
      # prepare table
      if(len(labels) or len(labels2)):
        values = values.tolist()
        if(len(labels)):
          descriptors.append('labels')
          values = [i + [str(j)] for i, j in zip(values, labels)]
        if(len(labels2)):
          descriptors.append('labels2')
          values = [i + [str(j)] for i, j in zip(values, labels2)]
      self.descriptors = descriptors
      self.resultstable.tableModel = TableModel(values, self.resultstable)
      self.resultstable.setModel(self.resultstable.tableModel)
      self.resultstable.tableModel.setAllHeaders(self.descriptors)
      self.resultstable.updateColumnStatistics()

      # need to update font size correctly
      if(self.resultstable.currZoom >= 0):
        font = self.resultstable.tableModel.getFont()
        font.setPointSize(self.resultstable.zoomLevels[self.resultstable.currZoom])

      # set col width
      self.colWidth = int(self.resultstable.size().width() / 4.5)
      hheader = self.resultstable.horizontalHeader()
      hheader.setDefaultSectionSize(self.colWidth)

      # sponsored by recent PyQt5 version, we have to reset the row height
      vheader = self.resultstable.verticalHeader()
      vheader.setDefaultSectionSize(self.rowHeight)
      vheader.setSectionResizeMode(QtWidgets.QHeaderView.ResizeMode.Fixed)
    else:
      self.resultstable.tableModel = None
      nullModel = TableModel([], self.resultstable)
      self.resultstable.setModel(nullModel)

    # set col widths
    if(len(descriptors)):
      colWidth = int(self.resultstable.size().width() / 4.5)
      colWidth = min(120, self.colWidth)
      hheader = self.resultstable.horizontalHeader()
      maxSize = int(0.5 * self.size().width())
      # limit resizing to first 15 columns as it becomes mortally slow otherwise
      maxDimX = min(len(descriptors), 15)
      for entry in range(maxDimX):
        hheader.setSectionResizeMode(entry, QtWidgets.QHeaderView.ResizeMode.Interactive)
        useSize = max(self.mySizeHintForColumn(entry) + 2, self.minColWidth)
        # restrict column size to max. 50% of visible area (to allow easy resizing)
        useSize = min(useSize, maxSize)
        hheader.resizeSection(entry, useSize)
    
  def mySizeHintForColumn(self, index):
    # custom size hint for column sizing to eke out some speed
    # INIT_SEARCH ensures that initially visible cells are used
    INIT_SEARCH, MAX_SEARCH, SPACER = 20, 50, 10
    # for size calculation, truncate float numbers after 6 significant places
    PRECISION_STRING = '%.6g'
    storeFontMetrics = self.fontMetrics()
    
    # analyze column contents
    data = self.resultstable.tableModel.getDataColumn(index)
    if(len(data) < MAX_SEARCH):
      # search through entire table
      sizes = [storeFontMetrics.horizontalAdvance(PRECISION_STRING % i) if type(i) == type(2.3) else storeFontMetrics.horizontalAdvance(str(i)) for i in data]
      if(len(sizes)):
        return max(sizes) + SPACER
      else:
        return SPACER
    else:
      # search through MAX_SEARCH randomly chosen cells
      sizes = data[:INIT_SEARCH] + np.random.choice(data[INIT_SEARCH:], size=MAX_SEARCH - INIT_SEARCH, replace=False).tolist()
      if(len(sizes)):
        maxelement = max(sizes, key = lambda i: len(PRECISION_STRING % i) if type(i) == type(2.3) else len(str(i)))
        if(type(maxelement) == type(2.3)):
          maxelement = PRECISION_STRING % maxelement
        return storeFontMetrics.horizontalAdvance(str(maxelement)) + SPACER
      else:
        return SPACER

  def keyPressEvent(self, event):
    if event.matches(QtGui.QKeySequence.StandardKey.Copy):
      # prepare output
      selind = self.resultstable.selectionModel().selectedRows()
      selind = sorted([i.row() for i in selind])      
      # get data
      selectedData = self.resultstable.tableModel.getDataRows(selind)
      output = ''
      for row in selectedData:
        row = [str(i) for i in row]
        output += '\t'.join(row) + '\n'
      
      clipboard = QtWidgets.QApplication.clipboard()
      clipboard.setText(output)
    else:
      QtWidgets.QWidget.keyPressEvent(self, event)

class BlankResizeMenu(KuhMenu):
  def __init__(self, parent = None, tableWidget = None):
    super(BlankResizeMenu, self).__init__()
    self.parent = parent
    self.tableWidget = tableWidget
    self.nrow, self.ncol = self.tableWidget.getDimension()
      
    # int validator
    self.validInt = MyValidInt()

    # set up GUI
    self.buildRessource()

  def mousePressEvent(self, event):
    # check whether click event is within widget
    if(not self.underMouse()):
      # pass through event (and thereby close the QMenu)
      KuhMenu.mousePressEvent(self, event)

  def buildRessource(self):
    # build gui for setting options
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(*[2]*4)
    self.vLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignTop)
    self.vLayout.setSpacing(2)
    
    # heading
    self.blankResizeLabel = QtWidgets.QLabel()
    useFont = self.blankResizeLabel.font()
    useFont.setBold(True)
    self.blankResizeLabel.setFont(useFont)
    self.blankResizeLabel.setText('Blank/resize table')
    self.vLayout.addWidget(self.blankResizeLabel)
    
    # rows
    self.setRowGroup = QWidgetMac()
    self.vLayout.addWidget(self.setRowGroup)
    self.hLayout = QtWidgets.QHBoxLayout(self.setRowGroup)
    self.hLayout.setContentsMargins(0, 0, 0, 0)
    self.hLayout.setSpacing(int(-2 + 8 * SCALEFONT))
    self.hLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.setRowLabel = QtWidgets.QLabel('# rows')
    self.setRowLabel.setMaximumSize(QtCore.QSize(int(35 * SCALEFONT), int(BASE_SIZE)))
    self.setRowLabel.setMinimumSize(QtCore.QSize(int(35 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout.addWidget(self.setRowLabel)

    self.setRowEntry = QLineEditClick()
    self.setRowEntry.setToolTip('Number of rows in data table')
    self.setRowEntry.setText(str(self.nrow))
    self.setRowEntry.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.setRowEntry.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.setRowEntry.editingFinished.connect(partial(self.setDimension, self.setRowEntry, 'row', 1, 1e6))
    self.setRowEntry.setValidator(self.validInt)
    self.hLayout.addWidget(self.setRowEntry)

    # cols
    self.setColGroup = QWidgetMac()
    self.vLayout.addWidget(self.setColGroup)
    self.hLayout2 = QtWidgets.QHBoxLayout(self.setColGroup)
    self.hLayout2.setContentsMargins(0, 0, 0, 0)
    self.hLayout2.setSpacing(int(-2 + 8 * SCALEFONT))
    self.hLayout2.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.setColLabel = QtWidgets.QLabel('# cols')
    self.setColLabel.setMaximumSize(QtCore.QSize(int(35 * SCALEFONT), int(BASE_SIZE)))
    self.setColLabel.setMinimumSize(QtCore.QSize(int(35 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout2.addWidget(self.setColLabel)

    self.setColEntry = QLineEditClick()
    self.setColEntry.setToolTip('Number of columns in data table')
    self.setColEntry.setText(str(self.ncol))
    self.setColEntry.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.setColEntry.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.setColEntry.editingFinished.connect(partial(self.setDimension, self.setColEntry, 'col', 1, 1e6))
    self.setColEntry.setValidator(self.validInt)
    self.hLayout2.addWidget(self.setColEntry)
    
    # resize button
    self.resizeButton = QPushButtonMac()
    self.resizeButton.setText('Resize table')
    self.resizeButton.setToolTip('Resize data table to new dimensions\n(cell contents outside new limits will irretrievably be deleted)')
    self.resizeButton.clicked.connect(self.resizeTable)
    self.resizeButton.setMaximumHeight(int(BASE_SIZE))
    self.resizeButton.setMinimumHeight(int(BASE_SIZE))
    self.vLayout.addWidget(self.resizeButton)

    # blank button
    self.blankButton = QPushButtonMac()
    self.blankButton.setText('Blank table')
    self.blankButton.setToolTip('Create blank data table with new dimensions\n(previous cell contents will irretrievably be deleted)')
    self.blankButton.clicked.connect(self.blankTable)
    self.blankButton.setMaximumHeight(int(BASE_SIZE))
    self.blankButton.setMinimumHeight(int(BASE_SIZE))
    self.vLayout.addWidget(self.blankButton)

    self.setFocus()    
    self.focusNextChild()

  def setDimension(self, entryfield=None, target=None, minval=1, maxval=100):
    # sets number of rows/cols in data table
    if((entryfield != None) and (target != None)):
      # check paramter boundaries
      try:
        value = int(entryfield.text())
        originalvalue = value
      except:
        value = 0
        originalvalue = 1
      value = min(value, maxval)
      value = max(value, minval)
      # update parameters
      if (value != originalvalue):
        entryfield.setText(str(value))

      # update row/column count
      if(target == 'row'):
        self.nrow = value
      else:
        self.ncol = value
          
  def blankTable(self):
    # resets table to empty new table
    self.tableWidget.generateEmptyTable(self.ncol, self.nrow)
    self.close()
    
  def resizeTable(self):
    # resizes table to new dimensions
    self.tableWidget.resizeTable(self.ncol, self.nrow)
    self.close()

class RecentFileMenu(KuhMenu):
  def __init__(self, parent = None):
    super(RecentFileMenu, self).__init__()
    self.parent = parent
    self.aboutToShow.connect(self.updateMe)
    self.installEventFilter(self)

    # set up GUI
    self.buildRessource()

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def mousePressEvent(self, event):
    # check whether click event is within widget
    if(not self.underMouse()):
      # pass through event (and thereby close the QMenu)
      KuhMenu.mousePressEvent(self, event)

  def buildRessource(self):
    # build gui for setting options
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(*[2]*4)
    self.vLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignTop)
    self.vLayout.setSpacing(2)
    
    # heading
    self.recentFilesLabel = QtWidgets.QLabel()
    useFont = self.recentFilesLabel.font()
    useFont.setBold(True)
    self.recentFilesLabel.setFont(useFont)
    self.recentFilesLabel.setText('Recent data files')
    self.vLayout.addWidget(self.recentFilesLabel)

    # set up table view
    self.tableView = QtWidgets.QTableView()
    self.tableView.resizeColumnsToContents()
    self.tableView.horizontalHeader().hide()
    self.tableView.verticalHeader().hide()
    self.tableView.setShowGrid(False)

    self.tableView.clicked.connect(self.clickHandler)
    self.tableView.setSelectionMode(self.tableView.SelectionMode.NoSelection)
    self.tableView.setEditTriggers(self.tableView.EditTrigger.NoEditTriggers)
    self.vLayout.addWidget(self.tableView)
    ###self.tableView.setFocus()
    
  def keyPressEvent(self, event):
    # capture enter key to allow selection by this approach
    if(event.key() in [QtCore.Qt.Key.Key_Enter, QtCore.Qt.Key.Key_Return]):
      selectionModel = self.tableView.selectionModel()
      if(selectionModel != None):
        index = selectionModel.currentIndex()
        if(index != None):
          # okay, we can call clickHandler now
          self.clickHandler(index=index)
    else:
      # pass through event
      KuhMenu.keyPressEvent(self, event)

  def clickHandler(self, index):
    # callback for click on table
    global RECENTFILES
  
    selectedRow = index.row()
    if(self.parent != None):
      if(selectedRow < len(RECENTFILES)):
        filename, mode = RECENTFILES[selectedRow]
        if(isfile(filename)):
          if(mode == 0):
            self.parent.tableWidget.loadXLS(filename=filename, transpose=self.parent.transposeData)
          elif(mode == 1):
            self.parent.tableWidget.loadODS(filename=filename, transpose=self.parent.transposeData)
          elif(mode == 2):
            self.parent.tableWidget.loadTextFile(filename=filename, delimiter='\t', transpose=self.parent.transposeData)
          elif(mode == 3):
            self.parent.tableWidget.loadTextFile(filename=filename, delimiter=',', transpose=self.parent.transposeData)
          elif(mode == 4):
            self.parent.tableWidget.loadTextFile(filename=filename, delimiter=None, transpose=self.parent.transposeData)
          elif((mode == 5) and PYCORN_PRESENT):
            self.parent.tableWidget.loadUnicornFile(filename=filename, transpose=self.parent.transposeData)
          # store this for use in recent files
          if(mode in [0, 1, 2, 3, 4, 5]):
            # loop through list and remove previous entry
            index = len(RECENTFILES) - 1
            for entry in RECENTFILES[::-1]:
              if(entry[0] == filename):
                RECENTFILES.pop(index)
              index -= 1
            RECENTFILES.insert(0, [filename, mode])
            # truncate to 20 entries
            RECENTFILES = RECENTFILES[:20]
        else:
          self.parent.parent.statusbar.showMessage('Cannot locate data file ' + filename + '!', self.parent.parent.STATUS_TIME)
    
    # close QMenu
    self.close()

  def updateMe(self):
    # apply styles
    if(QSTYLE != None):
      self.setStyle(QSTYLE)
    if(QSTYLESHEET != None):
      self.setStyleSheet(QSTYLESHEET)

    styleString = "QTableView {background-color: rgba" + UI_ALTERNATE_BASE_STRING + ";}\n\
      QTableView {border-style: solid; border-radius: 4px;}"
    if(len(RECENTFILES)):
      styleString += "\nQTableView::item:hover {background-color: #0078d7; color: white; border-style: none; border-radius: 4px;}\n\
        QTableView::item {background-color: transparent; text-align: center; border-radius: 4px;}\n\
        QTableView::item:focus {color: white; background-color: #0078d7; border-style: none; border-radius: 4px;}\n\
        QTableView::item:hover:focus {color: white; background-color: #0078d7; border-style: none; border-radius: 4px;}"
    else:
      styleString += "\nQTableView::item {background-color: transparent; text-align: center; border-radius: 4px; font-style: italic; color: rgba" + UI_BRIGHT_STRING + ";}"
    self.tableView.setStyleSheet(styleString)
      
    # update list
    self.model = QtGui.QStandardItemModel()
    if(len(RECENTFILES)):
      for index, entry in enumerate(RECENTFILES):
        filename = entry[0].split('/')[-1]
        item = QtGui.QStandardItem(str(index + 1) + ' - ' + filename)
        item.setTextAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignVCenter)
        self.model.setItem(index, item)
    else:
      self.model.setItem(0, QtGui.QStandardItem('<empty>'))
    self.tableView.setModel(self.model)
    
    # now have to resize everything to altered contents
    self.tableView.resizeColumnToContents(0)
    width, height = self.tableView.viewportSizeHint().width(), self.tableView.viewportSizeHint().height()
    # need to add some extra pixels b/c on repeat calls viewportSizeHint() screws up :( 
    # => what an ugly fudge
    self.tableView.setFixedSize(QtCore.QSize(width + int(4 * SCALEFONT), height + int(4 * SCALEFONT)))
    self.adjustSize()

class RecentStateMenu(KuhMenu):
  def __init__(self, parent = None):
    super(RecentStateMenu, self).__init__()
    self.parent = parent
    self.aboutToShow.connect(self.updateMe)
    self.installEventFilter(self)

    # set up GUI
    self.buildRessource()

  def eventFilter(self, source, event):
    # filters events to allow dynamic display of tooltips
    if(self.toolTip()):
      # this button has a tooltip alright
      if((event.type() == QtCore.QEvent.Type.ToolTip) and (not myapp.toolTipsShow)):
        # ignore this one
        return True

    return QtWidgets.QWidget.eventFilter(self, source, event)

  def showEvent(self, event):
    # show menu above button
    position = self.pos()
    position.setY(int(position.y() - self.height() - int(BASE_SIZE)))
    self.move(position)

  def mousePressEvent(self, event):
    # check whether click event is within widget
    if(not self.underMouse()):
      # pass through event (and thereby close the QMenu)
      KuhMenu.mousePressEvent(self, event)
      
  def keyPressEvent(self, event):
    # capture enter key to allow selection by this approach
    if(event.key() in [QtCore.Qt.Key.Key_Enter, QtCore.Qt.Key.Key_Return]):
      selectionModel = self.tableView.selectionModel()
      if(selectionModel != None):
        index = selectionModel.currentIndex()
        if(index != None):
          # okay, we can call clickHandler now
          self.clickHandler(index=index)
    else:
      # pass through event
      KuhMenu.keyPressEvent(self, event)

  def buildRessource(self):
    # build gui for setting options
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(*[2]*4)
    self.vLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignTop)
    self.vLayout.setSpacing(2)
    
    # heading
    self.recentFilesLabel = QtWidgets.QLabel()
    useFont = self.recentFilesLabel.font()
    useFont.setBold(True)
    self.recentFilesLabel.setFont(useFont)
    self.recentFilesLabel.setText('Recent state files')
    self.vLayout.addWidget(self.recentFilesLabel)

    # set up table view
    self.tableView = QtWidgets.QTableView()
    self.tableView.resizeColumnsToContents()
    self.tableView.horizontalHeader().hide()
    self.tableView.verticalHeader().hide()
    self.tableView.setShowGrid(False)

    self.tableView.clicked.connect(self.clickHandler)
    self.tableView.setSelectionMode(self.tableView.SelectionMode.NoSelection)
    self.tableView.setEditTriggers(self.tableView.EditTrigger.NoEditTriggers)
    self.vLayout.addWidget(self.tableView)
    ###self.tableView.setFocus()
    
  def clickHandler(self, index):
    # callback for click on table
    selectedRow = index.row()
    if(self.parent != None):
      if(selectedRow < len(RECENTSTATES)):
        filename = RECENTSTATES[selectedRow]
        if(isfile(filename)):
          self.parent.loadState(stateFile=filename, dropEvent=True, appendState=True)
        else:
          self.parent.statusbar.showMessage('Cannot locate state file ' + filename + '!', self.parent.STATUS_TIME)
    
    # close QMenu
    self.close()
  
  def updateMe(self):
    # apply styles
    if(QSTYLE != None):
      self.setStyle(QSTYLE)
    if(QSTYLESHEET != None):
      self.setStyleSheet(QSTYLESHEET)
      
    styleString = "QTableView {background-color: rgba" + UI_ALTERNATE_BASE_STRING + ";}\n\
      QTableView {border-style: solid; border-radius: 4px;}"
    if(len(RECENTSTATES)):
      styleString += "\nQTableView::item:hover {background-color: #0078d7; color: white; border-style: none; border-radius: 4px;}\n\
        QTableView::item {background-color: transparent; text-align: center; border-radius: 4px;}\n\
        QTableView::item:focus {color: white; background-color: #0078d7; border-style: none; border-radius: 4px;}\n\
        QTableView::item:hover:focus {color: white; background-color: #0078d7; border-style: none; border-radius: 4px;}"
    else:
      styleString += "\nQTableView::item {background-color: transparent; text-align: center; border-radius: 4px; font-style: italic; color: rgba" + UI_BRIGHT_STRING + ";}"
    self.tableView.setStyleSheet(styleString)
      
    # update list
    self.model = QtGui.QStandardItemModel()
    if(len(RECENTSTATES)):
      for index, entry in enumerate(RECENTSTATES):
        filename = entry.split('/')[-1]
        item = QtGui.QStandardItem(str(index + 1) + ' - ' + filename)
        item.setTextAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignVCenter)
        self.model.setItem(index, item)
    else:
      self.model.setItem(0, QtGui.QStandardItem('<empty>'))
    self.tableView.setModel(self.model)
    
    # now have to resize everything to altered contents
    self.tableView.resizeColumnToContents(0)
    width, height = self.tableView.viewportSizeHint().width(), self.tableView.viewportSizeHint().height()
    # need to add some extra pixels b/c on repeat calls viewportSizeHint() screws up :( 
    # => what an ugly fudge
    self.tableView.setFixedSize(QtCore.QSize(width + int(4 * SCALEFONT), height + int(4 * SCALEFONT)))
    self.adjustSize()
  
class DataArea(QWidgetMac):
  def __init__(self, parent = None):
    super(DataArea, self).__init__()
    self.parent = parent
    self.errorSwitch, self.errorXSwitch = True, False
    self.errorModel, self.errorXModel = 0, 1
    self.errorConst, self.errorXConst = 1.0, 1.0
    self.errorPercent, self.errorXPercent = 5.0, 5.0
    self.errorMultiply, self.errorXMultiply = 1.0, 1.0
    self.errorPropagate = True
    self.reductionSwitch, self.reductionModel = False, 1
    self.reductionSkip = 1
    self.reductionAvg = 2
    self.reductionMovAvg = 2
    self.reductionLog = 100
    self.sheetNumber = 1
    self.transposeData = False
    self.importFilter = ''
    self.firstLoad = True

    self.buildRessource()
    self.tableWidget.generateEmptyTable(3, 50)
    
    # set up namespace
    # import numpy again
    import numpy as np
    # import common functions from numpy for ease of access
    from numpy import abs, arccos, arcsin, arctan, exp, cos, cosh, log, log2, log10, power, sin, sinh, sqrt, tan, tanh
    self.mySpace = locals()

  def buildRessource(self):
    # set up GUI
    self.validFloat = MyValidFloat()
    self.validInt = MyValidInt()
    self.validInt.setBottom(1)

    # set up data table
    self.vLayout = QtWidgets.QVBoxLayout(self)
    self.vLayout.setContentsMargins(0, 0, 0, 0)
    
    self.importBox = QWidgetMac()
    self.vLayout.addWidget(self.importBox)
    self.importLayout = QtWidgets.QHBoxLayout(self.importBox)
    self.importLayout.setContentsMargins(0, 0, 0, 0)
    self.importLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    
    self.importButton = QToolButtonMac()
    self.importButton.setText(' Open File')
    if(PYCORN_PRESENT):
      self.importButton.setToolTip('Import Excel, text or unicorn data\n(Ctrl-I)')
    else:
      self.importButton.setToolTip('Import Excel or text data\n(Ctrl-I)')
    self.importButton.clicked.connect(self.loadData)
    self.importButton.setMaximumHeight(int(BASE_SIZE))
    self.importButton.setMinimumSize(int(90 * SCALEFONT), int(BASE_SIZE))
    self.importButton.setIcon(FOM_ICON_OPEN)
    self.importButton.setToolButtonStyle(QtCore.Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
    # QToolButton does not heed Alignment -- need to subclass button, argl
    ###self.importButton.setAlignment(QtCore.Qt.AlignmentFlag.AlignCenter)
    self.importLayout.addWidget(self.importButton, stretch=1)
    
    # set up recent file list menu
    self.recentMenu = RecentFileMenu(self)
    self.importButton.setPopupMode(QtWidgets.QToolButton.ToolButtonPopupMode.MenuButtonPopup)
    self.importButton.setMenu(self.recentMenu)

    self.resizeButton = QPushButtonMac()
    self.resizeButton.setText(' Blank/Resize \u25be')
    self.resizeButton.setToolTip('Clear the data sheet and/or alter its dimensions')
    self.resizeButton.clicked.connect(partial(self.blankResizeTable, self.resizeButton))
    self.resizeButton.setMaximumHeight(int(BASE_SIZE))
    self.resizeButton.setMinimumSize(int(90 * SCALEFONT), int(BASE_SIZE))
    self.resizeButton.setIcon(FOM_ICON_FILE)
    self.importLayout.addWidget(self.resizeButton, stretch=1)

    self.killCommaButton = QPushButtonMac()
    self.killCommaButton.setText(' Replace Comma')
    self.killCommaButton.setToolTip('Replace all commata in data sheet by periods')
    self.killCommaButton.clicked.connect(self.killTheComma)
    self.killCommaButton.setMaximumHeight(int(BASE_SIZE))
    self.killCommaButton.setMinimumSize(int(90 * SCALEFONT), int(BASE_SIZE))
    self.killCommaButton.setIcon(FOM_ICON_DIALOG_CLOSE_BUTTON)
    self.importLayout.addWidget(self.killCommaButton, stretch=1)

    self.transposeCheck = QPushButtonCheckable()
    self.transposeCheck.setText(' Transpose')
    self.transposeCheck.setToolTip('Swap columns and rows in data sheet')
    self.transposeCheck.setChecked(False)
    self.transposeCheck.setMaximumHeight(int(BASE_SIZE))
    self.transposeCheck.setMinimumSize(int(90 * SCALEFONT), int(BASE_SIZE))
    self.transposeCheck.clicked.connect(self.dataTransposition)
    self.transposeCheck.setIcon(FOM_ICON_MESSAGE_BOX_QUESTION)
    self.importLayout.addWidget(self.transposeCheck, stretch=1)
    
    self.sheetBox = QWidgetMac()
    self.vLayout.addWidget(self.sheetBox)
    self.sheetLayout = QtWidgets.QHBoxLayout(self.sheetBox)
    self.sheetLayout.setContentsMargins(0, 0, 0, 0)
    self.sheetLayout.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    self.importSheetLabel = QtWidgets.QLabel('sheet')
    self.importSheetLabel.setMaximumSize(int(28 * SCALEFONT), int(BASE_SIZE))
    self.importSheetLabel.setMinimumSize(int(28 * SCALEFONT), int(BASE_SIZE))
    self.sheetLayout.addWidget(self.importSheetLabel)
    self.importSheetLabel.setEnabled(False)
    
    self.importSheetSpinBox = QSpinClick()
    self.importSheetSpinBox.setToolTip('Select data sheet within Excel file')
    self.importSheetSpinBox.setAlignment(QtCore.Qt.AlignmentFlag.AlignRight)
    self.importSheetSpinBox.setMinimum(1)
    self.importSheetSpinBox.setMaximum(self.sheetNumber)
    self.importSheetSpinBox.setValue(1)
    self.importSheetSpinBox.setMinimumSize(int(50 * SCALEFONT), int(BASE_SIZE))
    self.importSheetSpinBox.setMaximumSize(int(50 * SCALEFONT), int(BASE_SIZE))
    self.importSheetSpinBox.editingFinished.connect(self.changeSheet)
    self.importSheetSpinBox.setEnabled(False)
    self.importSheetSpinBox.setAccelerated(True)
    self.importSheetSpinBox.setAdaptive(False)
    self.sheetLayout.addWidget(self.importSheetSpinBox)
    
    self.importSheetName = QtWidgets.QLabel()
    self.sheetLayout.addWidget(self.importSheetName)
    self.sheetBox.hide()

    self.tableWidget = DataTable(self)
    self.vLayout.addWidget(self.tableWidget)
    
    # import options
    blah = HLine()
    self.vLayout.addWidget(blah)
    self.dataImportLabel = QtWidgets.QLabel()
    useFont = self.dataImportLabel.font()
    useFont.setBold(True)
    self.dataImportLabel.setFont(useFont)
    self.dataImportLabel.setText('Data Import Options')
    self.vLayout.addWidget(self.dataImportLabel)

    self.outerBox1 = QWidgetMac()
    self.outerBox1.setObjectName('out1')
    self.vLayout.addWidget(self.outerBox1)
    self.Layout_outerBox1 = QtWidgets.QVBoxLayout(self.outerBox1)
    self.Layout_outerBox1.setContentsMargins(2, 4, 2, 4)
    self.Layout_outerBox1.setSpacing(2)
    self.Layout_outerBox1.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    # set up box for x-error specification
    self.errorXSelectorBox = QWidgetMac()
    self.errorXSelectorBox.setContentsMargins(0, 0, 0, 0)
    self.Layout_outerBox1.addWidget(self.errorXSelectorBox)
    self.errorXSelectorLayout = QtWidgets.QHBoxLayout(self.errorXSelectorBox)
    self.errorXSelectorLayout.setContentsMargins(0, 0, 0, 0)
###    self.vLayout.setSpacing(2)
    
    self.errorXSelectorLabel = QPushButtonCheckable()
    self.errorXSelectorLabel.setChecked(self.errorXSwitch)
    self.errorXSelectorLabel.setText('error x')
    self.errorXSelectorLabel.setToolTip('Set error \u0394x for x values')
    self.errorXSelectorLabel.clicked.connect(partial(self.toggleErrorSwitch, 'x'))
    self.errorXSelectorLabel.setMaximumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.errorXSelectorLabel.setMinimumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.errorXSelectorLayout.addWidget(self.errorXSelectorLabel)
    
    self.errorXSelectorGroup = QtWidgets.QGroupBox()
    self.errorXSelectorGroup.setMinimumHeight(int(BASE_SIZE + 4))
    self.errorXSelectorGroup.setMaximumHeight(int(BASE_SIZE + 4))
    self.errorXSelectorLayout.addWidget(self.errorXSelectorGroup)

    self.errorXGroupLayout = QtWidgets.QHBoxLayout()
    self.errorXSelectorGroup.setLayout(self.errorXGroupLayout)
    
    self.errorXSelectorButtons = []
    self.errorXSelectorButtons.append(QRadioButtonToolTip(self.errorXSelectorGroup))
    self.errorXSelectorButtons[-1].setGeometry(QtCore.QRect(int(240 * SCALEFONT), 2, int(42 * SCALEFONT), int(BASE_SIZE)))
    self.errorXSelectorButtons[-1].setChecked(False)
    self.errorXSelectorButtons[-1].toggled.connect(partial(self.toggleErrorXModel, 0))
    self.errorXSelectorButtons[-1].setText('\u0394x \u00D7')
    self.errorXSelectorButtons[-1].setToolTip('Use assigned \u0394x error column')
    self.errorXDeltaEntry = QLineEditClick(self.errorXSelectorGroup)
    self.errorXDeltaEntry.setGeometry(QtCore.QRect(int(288 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.errorXDeltaEntry.setText(str(self.errorXMultiply))
    self.errorXDeltaEntry.setToolTip('Multiply assigned \u0394x by factor')
    self.errorXDeltaEntry.setValidator(self.validFloat)
    self.errorXDeltaEntry.editingFinished.connect(partial(self.validateErrorEntry, self.errorXDeltaEntry, 'errorXMultiply'))
    self.errorXDeltaEntry.focusOutEvent = partial(self.lostFocus, self.errorXDeltaEntry, 'errorXMultiply', self.errorXDeltaEntry.focusOutEvent)
    self.errorXDeltaEntry.focusInEvent = partial(self.gainFocus, self.errorXSelectorButtons[-1], self.errorXDeltaEntry.focusInEvent)
    
    self.errorXSelectorButtons.append(QRadioButtonToolTip(self.errorXSelectorGroup))
    self.errorXSelectorButtons[-1].setGeometry(QtCore.QRect(int(2 * SCALEFONT), 2, int(44 * SCALEFONT), int(BASE_SIZE)))
    self.errorXSelectorButtons[-1].setChecked(True)
    self.errorXSelectorButtons[-1].toggled.connect(partial(self.toggleErrorXModel, 1))
    self.errorXSelectorButtons[-1].setText('const')
    self.errorXSelectorButtons[-1].setToolTip('Use constant \u0394x error')
    self.errorXConstEntry = QLineEditClick(self.errorXSelectorGroup)
    self.errorXConstEntry.setGeometry(QtCore.QRect(int(56 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.errorXConstEntry.setText(str(self.errorXConst))
    self.errorXConstEntry.setToolTip('Use constant \u0394x error')
    self.errorXConstEntry.setValidator(self.validFloat)
    self.errorXConstEntry.editingFinished.connect(partial(self.validateErrorEntry, self.errorXConstEntry, 'errorXConst'))
    self.errorXConstEntry.focusOutEvent = partial(self.lostFocus, self.errorXConstEntry, 'errorXConst', self.errorXConstEntry.focusOutEvent)
    self.errorXConstEntry.focusInEvent = partial(self.gainFocus, self.errorXSelectorButtons[-1], self.errorXConstEntry.focusInEvent)
    
    self.errorXSelectorButtons.append(QRadioButtonToolTip(self.errorXSelectorGroup))
    self.errorXSelectorButtons[-1].setGeometry(QtCore.QRect(int(126 * SCALEFONT), 2, int(42 * SCALEFONT), int(BASE_SIZE)))
    self.errorXSelectorButtons[-1].setChecked(False)
    self.errorXSelectorButtons[-1].toggled.connect(partial(self.toggleErrorXModel, 2))
    self.errorXSelectorButtons[-1].setText('prop')
    self.errorXSelectorButtons[-1].setToolTip('Use \u0394x error proportional to x value')
    self.errorXPercentEntry = QLineEditClick(self.errorXSelectorGroup)
    self.errorXPercentEntry.setGeometry(QtCore.QRect(int(172 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.errorXPercentEntry.setText(str(self.errorXPercent))
    self.errorXPercentEntry.setToolTip('Use \u0394x error proportional to x value')
    self.errorXPercentEntry.editingFinished.connect(partial(self.validateErrorEntry, self.errorXPercentEntry, 'errorXPercent'))
    self.errorXPercentEntry.focusOutEvent = partial(self.lostFocus, self.errorXPercentEntry, 'errorXPercent', self.errorXPercentEntry.focusOutEvent)
    self.errorXPercentEntry.focusInEvent = partial(self.gainFocus, self.errorXSelectorButtons[-1], self.errorXPercentEntry.focusInEvent)
    self.errorXPercentEntry.setValidator(self.validFloat)
    self.errorXPercentLabel = QtWidgets.QLabel(self.errorXSelectorGroup)
    self.errorXPercentLabel.setGeometry(QtCore.QRect(int(216 * SCALEFONT), 2, int(18 * SCALEFONT), int(BASE_SIZE)))
    self.errorXPercentLabel.setText('%')

    # set up controls for y error
    self.errorSelectorBox = QWidgetMac()
    self.errorSelectorBox.setContentsMargins(0, 0, 0, 0)
    self.Layout_outerBox1.addWidget(self.errorSelectorBox)
    self.errorSelectorLayout = QtWidgets.QHBoxLayout(self.errorSelectorBox)
    self.errorSelectorLayout.setContentsMargins(0, 0, 0, 0)
    
    self.errorSelectorLabel = QPushButtonCheckable()
    self.errorSelectorLabel.setChecked(self.errorSwitch)
    self.errorSelectorLabel.setText('error y')
    self.errorSelectorLabel.setToolTip('Set error \u0394y for y values')
    self.errorSelectorLabel.clicked.connect(partial(self.toggleErrorSwitch, 'y'))
    self.errorSelectorLabel.setMaximumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.errorSelectorLabel.setMinimumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.errorSelectorLayout.addWidget(self.errorSelectorLabel)

    self.errorSelectorGroup = QtWidgets.QGroupBox()
    self.errorSelectorGroup.setMinimumHeight(int(BASE_SIZE + 4))
    self.errorSelectorGroup.setMaximumHeight(int(BASE_SIZE + 4))
    self.errorSelectorLayout.addWidget(self.errorSelectorGroup)

    self.errorGroupLayout = QtWidgets.QHBoxLayout()
    self.errorSelectorGroup.setLayout(self.errorGroupLayout)
    
    self.errorSelectorButtons = []
    self.errorSelectorButtons.append(QRadioButtonToolTip(self.errorSelectorGroup))
    self.errorSelectorButtons[-1].setGeometry(QtCore.QRect(int(240 * SCALEFONT), 2, int(42 * SCALEFONT), int(BASE_SIZE)))
    self.errorSelectorButtons[-1].setChecked(True)
    self.errorSelectorButtons[-1].toggled.connect(partial(self.toggleErrorModel, 0))
    self.errorSelectorButtons[-1].setText('\u0394y \u00D7')
    self.errorSelectorButtons[-1].setToolTip('Use assigned \u0394y error column')
    self.errorDeltaEntry = QLineEditClick(self.errorSelectorGroup)
    self.errorDeltaEntry.setGeometry(QtCore.QRect(int(288 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.errorDeltaEntry.setText(str(self.errorMultiply))
    self.errorDeltaEntry.setToolTip('Multiply assigned \u0394y by factor')
    self.errorDeltaEntry.setValidator(self.validFloat)
    self.errorDeltaEntry.editingFinished.connect(partial(self.validateErrorEntry, self.errorDeltaEntry, 'errorMultiply'))
    self.errorDeltaEntry.focusOutEvent = partial(self.lostFocus, self.errorDeltaEntry, 'errorMultiply', self.errorDeltaEntry.focusOutEvent)
    self.errorDeltaEntry.focusInEvent = partial(self.gainFocus, self.errorSelectorButtons[-1], self.errorDeltaEntry.focusInEvent)
    
    self.errorSelectorButtons.append(QRadioButtonToolTip(self.errorSelectorGroup))
    self.errorSelectorButtons[-1].setGeometry(QtCore.QRect(int(2 * SCALEFONT), 2, int(44 * SCALEFONT), int(BASE_SIZE)))
    self.errorSelectorButtons[-1].setChecked(False)
    self.errorSelectorButtons[-1].toggled.connect(partial(self.toggleErrorModel, 1))
    self.errorSelectorButtons[-1].setText('const')
    self.errorSelectorButtons[-1].setToolTip('Use constant \u0394y error')
    self.errorConstEntry = QLineEditClick(self.errorSelectorGroup)
    self.errorConstEntry.setGeometry(QtCore.QRect(int(56 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.errorConstEntry.setText(str(self.errorConst))
    self.errorConstEntry.setToolTip('Use constant \u0394y error')
    self.errorConstEntry.setValidator(self.validFloat)
    self.errorConstEntry.editingFinished.connect(partial(self.validateErrorEntry, self.errorConstEntry, 'errorConst'))
    self.errorConstEntry.focusOutEvent = partial(self.lostFocus, self.errorConstEntry, 'errorConst', self.errorConstEntry.focusOutEvent)
    self.errorConstEntry.focusInEvent = partial(self.gainFocus, self.errorSelectorButtons[-1], self.errorConstEntry.focusInEvent)
    
    self.errorSelectorButtons.append(QRadioButtonToolTip(self.errorSelectorGroup))
    self.errorSelectorButtons[-1].setGeometry(QtCore.QRect(int(126 * SCALEFONT), 2, int(42 * SCALEFONT), int(BASE_SIZE)))
    self.errorSelectorButtons[-1].setChecked(False)
    self.errorSelectorButtons[-1].toggled.connect(partial(self.toggleErrorModel, 2))
    self.errorSelectorButtons[-1].setText('prop')
    self.errorSelectorButtons[-1].setToolTip('Use \u0394y error proportional to y value')
    self.errorPercentEntry = QLineEditClick(self.errorSelectorGroup)
    self.errorPercentEntry.setGeometry(QtCore.QRect(int(172 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.errorPercentEntry.setText(str(self.errorPercent))
    self.errorPercentEntry.setToolTip('Use \u0394y error proportional to y value')
    self.errorPercentEntry.editingFinished.connect(partial(self.validateErrorEntry, self.errorPercentEntry, 'errorPercent'))
    self.errorPercentEntry.focusOutEvent = partial(self.lostFocus, self.errorPercentEntry, 'errorPercent', self.errorPercentEntry.focusOutEvent)
    self.errorPercentEntry.focusInEvent = partial(self.gainFocus, self.errorSelectorButtons[-1], self.errorPercentEntry.focusInEvent)
    self.errorPercentEntry.setValidator(self.validFloat)
    self.errorPercentLabel = QtWidgets.QLabel(self.errorSelectorGroup)
    self.errorPercentLabel.setGeometry(QtCore.QRect(int(216 * SCALEFONT), 2, int(18 * SCALEFONT), int(BASE_SIZE)))
    self.errorPercentLabel.setText('%')
    
    # set up controls for error propagation
    self.errorPropagateBox = QWidgetMac()
    self.errorPropagateBox.setContentsMargins(0, 0, 0, 0)
    self.Layout_outerBox1.addWidget(self.errorPropagateBox)
    self.errorPropagateLayout = QtWidgets.QHBoxLayout(self.errorPropagateBox)
    self.errorPropagateLayout.setContentsMargins(0, 0, 0, 0)
    
    self.errorPropagateCheck = QPushButtonCheckable()
    self.errorPropagateCheck.setToolTip('Propagate data errors through transformation')
    self.errorPropagateCheck.setMaximumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.errorPropagateCheck.setMinimumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.errorPropagateCheck.setChecked(self.errorPropagate)
    self.errorPropagateCheck.setText('propagate?')
    self.errorPropagateCheck.clicked.connect(self.toggleErrorPropagation)
    self.errorPropagateLayout.addWidget(self.errorPropagateCheck)
    self.errorPropagateLayout.addStretch()
    ###self.vLayout.addWidget(HLine())

    self.outerBox2 = QWidgetMac()
    self.outerBox2.setObjectName('out1')
    self.vLayout.addWidget(self.outerBox2)
    self.Layout_outerBox2 = QtWidgets.QVBoxLayout(self.outerBox2)
    self.Layout_outerBox2.setContentsMargins(2, 4, 2, 4)
    self.Layout_outerBox2.setSpacing(2)
    # self.Layout_outerBox2.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    # set up box for data reduction
    self.dataReductionBox = QWidgetMac()
    self.dataReductionBox.setContentsMargins(0, 0, 0, 0)
    self.Layout_outerBox2.addWidget(self.dataReductionBox)
    self.dataReductionLayout = QtWidgets.QHBoxLayout(self.dataReductionBox)
    self.dataReductionLayout.setContentsMargins(0, 0, 0, 0)
    
    self.dataReductionLabel = QPushButtonCheckable()
    self.dataReductionLabel.setChecked(self.reductionSwitch)
    self.dataReductionLabel.setText('reduce')
    self.dataReductionLabel.setToolTip('Reduction of data upon import')
    self.dataReductionLabel.clicked.connect(self.toggleReductionSwitch)
    self.dataReductionLabel.setMaximumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.dataReductionLabel.setMinimumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.dataReductionLayout.addWidget(self.dataReductionLabel)
    
    self.dataReductionGroup = QtWidgets.QGroupBox()
    self.dataReductionGroup.setMinimumHeight(int(2 * BASE_SIZE + 8))
    self.dataReductionGroup.setMaximumHeight(int(2 * BASE_SIZE + 8))
    self.dataReductionLayout.addWidget(self.dataReductionGroup)
    
    self.dataReductionButtons = []
    self.dataReductionButtons.append(QRadioButtonToolTip(self.dataReductionGroup))
    self.dataReductionButtons[-1].setGeometry(QtCore.QRect(int(2 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.dataReductionButtons[-1].setChecked(True)
    self.dataReductionButtons[-1].toggled.connect(partial(self.toggleDataReduction, 1))
    self.dataReductionButtons[-1].setText('skip')
    self.dataReductionButtons[-1].setToolTip('Skip n data points')

    self.dataSkipEntry = QLineEditClick(self.dataReductionGroup)
    self.dataSkipEntry.setGeometry(QtCore.QRect(int(56 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.dataSkipEntry.setText(str(self.reductionSkip))
    self.dataSkipEntry.setToolTip('Skip n data points')
    self.dataSkipEntry.editingFinished.connect(partial(self.validateReductionEntry, self.dataSkipEntry, 'reductionSkip'))
    self.dataSkipEntry.focusOutEvent = partial(self.lostFocusInt, self.dataSkipEntry, 'reductionSkip', self.dataSkipEntry.focusOutEvent)
    self.dataSkipEntry.focusInEvent = partial(self.gainFocus, self.dataReductionButtons[-1], self.dataSkipEntry.focusInEvent)
    self.dataSkipEntry.setValidator(self.validInt)
    
    self.dataSkipLabel = QtWidgets.QLabel(self.dataReductionGroup)
    self.dataSkipLabel.setGeometry(QtCore.QRect(int(100 * SCALEFONT), 2, int(18 * SCALEFONT), int(BASE_SIZE)))
    self.dataSkipLabel.setText('pts')

    self.dataReductionButtons.append(QRadioButtonToolTip(self.dataReductionGroup))
    self.dataReductionButtons[-1].setGeometry(QtCore.QRect(int(126 * SCALEFONT), 2, int(36 * SCALEFONT), int(BASE_SIZE)))
    self.dataReductionButtons[-1].setChecked(False)
    self.dataReductionButtons[-1].toggled.connect(partial(self.toggleDataReduction, 2))
    self.dataReductionButtons[-1].setText('avg')
    self.dataReductionButtons[-1].setToolTip('Average n data points')

    self.dataAvgEntry = QLineEditClick(self.dataReductionGroup)
    self.dataAvgEntry.setGeometry(QtCore.QRect(int(172 * SCALEFONT), 2, int(40 * SCALEFONT), int(BASE_SIZE)))
    self.dataAvgEntry.setText(str(self.reductionAvg))
    self.dataAvgEntry.setToolTip('Average n data points')
    self.dataAvgEntry.editingFinished.connect(partial(self.validateReductionEntry, self.dataAvgEntry, 'reductionAvg'))
    self.dataAvgEntry.focusOutEvent = partial(self.lostFocusInt, self.dataAvgEntry, 'reductionAvg', self.dataAvgEntry.focusOutEvent)
    self.dataAvgEntry.focusInEvent = partial(self.gainFocus, self.dataReductionButtons[-1], self.dataAvgEntry.focusInEvent)
    self.dataAvgEntry.setValidator(self.validInt)
    
    self.dataAvgLabel = QtWidgets.QLabel(self.dataReductionGroup)
    self.dataAvgLabel.setGeometry(QtCore.QRect(int(216 * SCALEFONT), 2, int(15 * SCALEFONT), int(BASE_SIZE)))
    self.dataAvgLabel.setText('pts')

    self.dataReductionButtons.append(QRadioButtonToolTip(self.dataReductionGroup))
    self.dataReductionButtons[-1].setGeometry(QtCore.QRect(int(2 * SCALEFONT), int(BASE_SIZE + 4), int(50 * SCALEFONT), int(BASE_SIZE)))
    self.dataReductionButtons[-1].setChecked(False)
    self.dataReductionButtons[-1].toggled.connect(partial(self.toggleDataReduction, 3))
    self.dataReductionButtons[-1].setText('mvavg')
    self.dataReductionButtons[-1].setToolTip('Moving average of n data points')

    self.dataMovAvgEntry = QLineEditClick(self.dataReductionGroup)
    self.dataMovAvgEntry.setGeometry(QtCore.QRect(int(56 * SCALEFONT), int(BASE_SIZE + 4), int(40 * SCALEFONT), int(BASE_SIZE)))
    self.dataMovAvgEntry.setText(str(self.reductionMovAvg))
    self.dataMovAvgEntry.setToolTip('Moving average of n data points')
    self.dataMovAvgEntry.editingFinished.connect(partial(self.validateReductionEntry, self.dataMovAvgEntry, 'reductionMovAvg'))
    self.dataMovAvgEntry.focusOutEvent = partial(self.lostFocusInt, self.dataMovAvgEntry, 'reductionMovAvg', self.dataMovAvgEntry.focusOutEvent)
    self.dataMovAvgEntry.focusInEvent = partial(self.gainFocus, self.dataReductionButtons[-1], self.dataMovAvgEntry.focusInEvent)
    self.dataMovAvgEntry.setValidator(self.validInt)
    
    self.dataMovAvgLabel = QtWidgets.QLabel(self.dataReductionGroup)
    self.dataMovAvgLabel.setGeometry(QtCore.QRect(int(100 * SCALEFONT), int(BASE_SIZE + 4), int(18 * SCALEFONT), int(BASE_SIZE)))
    self.dataMovAvgLabel.setText('pts')

    self.dataReductionButtons.append(QRadioButtonToolTip(self.dataReductionGroup))
    self.dataReductionButtons[-1].setGeometry(QtCore.QRect(int(126 * SCALEFONT), int(BASE_SIZE + 4), int(36 * SCALEFONT), int(BASE_SIZE)))
    self.dataReductionButtons[-1].setChecked(False)
    self.dataReductionButtons[-1].toggled.connect(partial(self.toggleDataReduction, 4))
    self.dataReductionButtons[-1].setText('log')
    self.dataReductionButtons[-1].setToolTip('Logarithmically reduce data to ~ n data points')

    self.dataLogEntry = QLineEditClick(self.dataReductionGroup)
    self.dataLogEntry.setGeometry(QtCore.QRect(int(172 * SCALEFONT), int(BASE_SIZE + 4), int(40 * SCALEFONT), int(BASE_SIZE)))
    self.dataLogEntry.setText(str(self.reductionLog))
    self.dataLogEntry.setToolTip('Logarithmically reduce data to ~ n data points')
    self.dataLogEntry.editingFinished.connect(partial(self.validateReductionEntry, self.dataLogEntry, 'reductionLog'))
    self.dataLogEntry.focusOutEvent = partial(self.lostFocusInt, self.dataLogEntry, 'reductionLog', self.dataLogEntry.focusOutEvent)
    self.dataLogEntry.focusInEvent = partial(self.gainFocus, self.dataReductionButtons[-1], self.dataLogEntry.focusInEvent)
    self.dataLogEntry.setValidator(self.validInt)

    self.dataLogLabel = QtWidgets.QLabel(self.dataReductionGroup)
    self.dataLogLabel.setGeometry(QtCore.QRect(int(216 * SCALEFONT), int(BASE_SIZE + 4), int(42 * SCALEFONT), int(BASE_SIZE)))
    self.dataLogLabel.setText('pts (ca.)')
    ###self.vLayout.addWidget(HLine())

    self.outerBox3 = QWidgetMac()
    self.outerBox3.setObjectName('out1')
    self.vLayout.addWidget(self.outerBox3)
    self.Layout_outerBox3 = QtWidgets.QVBoxLayout(self.outerBox3)
    self.Layout_outerBox3.setContentsMargins(2, 4, 2, 4)
    self.Layout_outerBox3.setSpacing(4)
    self.Layout_outerBox3.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)

    # set up box for data transform
    self.dataTransformBox = QWidgetMac()
    self.dataTransformBox.setContentsMargins(0, 0, 0, 0)
    self.Layout_outerBox3.addWidget(self.dataTransformBox)
    self.dataTransformLayout = QtWidgets.QHBoxLayout(self.dataTransformBox)
    self.dataTransformLayout.setContentsMargins(0, 0, 0, 0)
    
    self.dataTransformXCheck = QPushButtonCheckable()
    self.dataTransformXCheck.setMaximumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformXCheck.setMinimumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformXCheck.setChecked(False)
    self.dataTransformXCheck.setText('transform x')
    self.dataTransformLayout.addWidget(self.dataTransformXCheck)
    self.dataTransformXCheck.setToolTip('Transformation of x values upon import')

    self.dataTransformLabel = QtWidgets.QLabel('x = ')
    self.dataTransformLabel.setMaximumSize(int(20 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformLabel.setMinimumSize(int(20 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformLayout.addWidget(self.dataTransformLabel)

    self.dataTransformXEntry = QLineEditClick()
    self.dataTransformXEntry.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.dataTransformXEntry.setMaximumHeight(int(BASE_SIZE))
    self.dataTransformXEntry.setMinimumSize(int(300 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformXEntry.setText('x')
    self.dataTransformXEntry.setToolTip('Formula for transforming x values')
    self.dataTransformXEntry.textChanged.connect(partial(self.dataTransformXCheck.setChecked, True))
    self.dataTransformLayout.addWidget(self.dataTransformXEntry)

    self.dataTransformYBox = QWidgetMac()
    self.dataTransformYBox.setContentsMargins(0, 0, 0, 0)
    self.Layout_outerBox3.addWidget(self.dataTransformYBox)
    self.dataTransformYLayout = QtWidgets.QHBoxLayout(self.dataTransformYBox)
    self.dataTransformYLayout.setContentsMargins(0, 0, 0, 0)

    self.dataTransformYCheck = QPushButtonCheckable()
    self.dataTransformYCheck.setMaximumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformYCheck.setMinimumSize(int(64 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformYCheck.setChecked(False)
    self.dataTransformYCheck.setText('transform y')
    self.dataTransformYLayout.addWidget(self.dataTransformYCheck)
    self.dataTransformYCheck.setToolTip('Transformation of y values upon import')

    self.dataTransformYLabel = QtWidgets.QLabel('y = ')
    self.dataTransformYLabel.setMaximumSize(int(20 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformYLabel.setMinimumSize(int(20 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformYLayout.addWidget(self.dataTransformYLabel)

    self.dataTransformYEntry = QLineEditClick()
    self.dataTransformYEntry.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.dataTransformYEntry.setMaximumHeight(int(BASE_SIZE))
    self.dataTransformYEntry.setMinimumSize(int(300 * SCALEFONT), int(BASE_SIZE))
    self.dataTransformYEntry.setText('y')
    self.dataTransformYEntry.setToolTip('Formula for transforming y values')
    self.dataTransformYEntry.textChanged.connect(partial(self.dataTransformYCheck.setChecked, True))
    self.dataTransformYLayout.addWidget(self.dataTransformYEntry)

    # set up data import controls
    blah = HLine()
    ###self.vLayout.addWidget(blah)
    
    self.refreshBox = QWidgetMac()
    self.refreshBox.setContentsMargins(0, 0, 0, 0)
    self.vLayout.addWidget(self.refreshBox)
    self.refreshLayout = QtWidgets.QHBoxLayout(self.refreshBox)
    self.refreshLayout.setContentsMargins(0, 0, 0, 0)
    self.refreshButton = QPushButtonMac()
    self.refreshButton.setText(' Import Data')
    self.refreshButton.setToolTip('Import the currently selected cells in the data table')
    self.refreshButton.setMaximumHeight(int(BASE_SIZE))
    self.refreshButton.setMinimumHeight(int(BASE_SIZE))
    self.refreshButton.clicked.connect(partial(self.updateData, False, True, False, False, False))
    self.refreshButton.setIcon(FOM_ICON_MEDIA_PLAY)
    self.refreshLayout.addWidget(self.refreshButton, stretch=2)

    self.refreshCreateButton = QPushButtonMac()
    self.refreshCreateButton.setText(' Import && Create')
    self.refreshCreateButton.setToolTip('Generate a new data object and import the currently selected cells in the data table')
    self.refreshCreateButton.setMaximumHeight(int(BASE_SIZE))
    self.refreshCreateButton.setMinimumHeight(int(BASE_SIZE))
    self.refreshCreateButton.clicked.connect(partial(self.updateData, False, True, False, True, False))
    self.refreshCreateButton.setIcon(FOM_ICON_MEDIA_PLAY)
    self.refreshLayout.addWidget(self.refreshCreateButton, stretch=1)

    self.refreshAddButton = QPushButtonMac()
    self.refreshAddButton.setText(' Import && Add')
    self.refreshAddButton.setToolTip('Import the currently selected cells in the data table and add to current data object')
    self.refreshAddButton.setMaximumHeight(int(BASE_SIZE))
    self.refreshAddButton.setMinimumHeight(int(BASE_SIZE))
    self.refreshAddButton.clicked.connect(partial(self.updateData, False, True, False, False, True))
    self.refreshAddButton.setIcon(FOM_ICON_MEDIA_PLAY)
    self.refreshLayout.addWidget(self.refreshAddButton, stretch=1)

    self.dataSeriesButton = QPushButtonMac()
    self.dataSeriesButton.setText(' Import Series')
    self.dataSeriesButton.setToolTip('Import y column and all data columns to its right')
    self.dataSeriesButton.setMaximumHeight(int(BASE_SIZE))
    self.dataSeriesButton.setMinimumHeight(int(BASE_SIZE))
    self.dataSeriesButton.clicked.connect(self.importDataSeries)
    self.dataSeriesButton.setIcon(FOM_ICON_MEDIA_SEEK_FORWARD)
    self.refreshLayout.addWidget(self.dataSeriesButton, stretch=1)
    
    self.refreshCheck = QPushButtonCheckable()
    self.refreshCheck.setText('Auto Import?')
    self.refreshCheck.setChecked(False)
    self.refreshCheck.clicked.connect(partial(self.updateData, True, True, False, False, False))
    #self.refreshLayout.addWidget(self.refreshCheck)

  def reportState(self):
    # reports data content for saveState function
    retv = self.tableWidget.tableModel.getAllData()
    return retv

  def restoreState(self, data):
    # restores data content from loadState function
    success = False
    try:
      tableData = literal_eval(data[0])
      success = True
    except:
      # literal_eval fails in rare cases, e.g., when encountering datetime objects (thank you, openpyxl ...)
      # try eval() -- yes, it's unsafe but as we exec() fit functions elsewhere, it does not really matter
      try:
        tableData = eval(data[0])
        success = True
      except:
        print('Failed to restore data table ' + data[0])
    
    if(success):
      self.resetSheetSpinBox(currVal=1, maxVal=1, currName='')
      self.transposeCheck.setChecked(False)
      self.tableWidget.restoreTable(tableData=tableData)

  def killTheComma(self):
    # scour data table for commata and replace them
    self.tableWidget.killTheComma()

  def dataTransposition(self):
    # set data transposition flag
    self.transposeData = self.transposeCheck.isChecked()
    # trigger data transposition
    self.tableWidget.transposeTable()

  def resetSheetSpinBox(self, currVal=1, maxVal=1, currName='', silent=False):
    # updates import spin box
    self.importSheetSpinBox.setMaximum(maxVal)
    self.sheetNumber = currVal
    if(silent):
      # prevent autoupdating of sheet contents when called from loadXLS
      self.importSheetSpinBox.blockSignals(True)
      self.importSheetSpinBox.setValue(self.sheetNumber)
      self.importSheetSpinBox.blockSignals(False)
    else:
      self.importSheetSpinBox.setValue(self.sheetNumber)
    
    if(maxVal > 1):
      self.sheetBox.show()
      self.importSheetSpinBox.setEnabled(True)
      self.importSheetLabel.setEnabled(True)
      self.importSheetName.setText(currName)
    else:
      self.importSheetSpinBox.setEnabled(False)
      self.importSheetLabel.setEnabled(False)
      self.importSheetName.setText('')
      self.sheetBox.hide()

  def changeSheet(self):
    # change current sheet in Excel files with several sheets
    self.sheetNumber = self.importSheetSpinBox.value()
    self.tableWidget.changeSheet(self.sheetNumber, transpose=self.transposeData)
    self.importSheetSpinBox.selectAll()

  def gainFocus(self, toggleOption=None, defaultHandler=None, event=None):
    # entry field gained focus
    # select corresponding option
    if(toggleOption != None):
      toggleOption.setChecked(True)

    # pass signal to original handler
    if(defaultHandler != None):
      defaultHandler(event)

  def toggleDataReduction(self, mode=0):
    # change error model
    self.reductionModel = mode
    # check switch button
    self.dataReductionLabel.setChecked(True)
    self.reductionSwitch = True

  def toggleReductionSwitch(self):
    # do data reduction?
    value = self.dataReductionLabel.isChecked()
    self.reductionSwitch = value

  def toggleErrorModel(self, mode=0):
    # change error model
    self.errorModel = mode
    # check switch button
    self.errorSelectorLabel.setChecked(True)
    self.errorSwitch = True

  def toggleErrorXModel(self, mode=0):
    # change error model
    self.errorXModel = mode
    # check switch button
    self.errorXSelectorLabel.setChecked(True)
    self.errorXSwitch = True

  def toggleErrorSwitch(self, axis='y'):
    # use errors?
    if(axis in ['x', 'y']):
      if(axis == 'y'):
        value = self.errorSelectorLabel.isChecked()
        self.errorSwitch = value
      else:
        value = self.errorXSelectorLabel.isChecked()
        self.errorXSwitch = value

  def toggleErrorPropagation(self):
    # toggles error propagation
    self.errorPropagate = self.errorPropagateCheck.isChecked()

  def lostFocusInt(self, entryobject=None, quantity=None, defaultHandler=None, event=None):
    # entry field lost focus, perform sanity check
    if(entryobject != None):
      entrytext = entryobject.text()
      if(self.parent.isNumber(entrytext)):
        self.__dict__[quantity] = int(entrytext)
      else:
        # restore previous value
        entryobject.setText(str(self.__dict__[quantity]))
    # pass signal to original handler
    if(defaultHandler != None):
      defaultHandler(event)
    
  def validateReductionEntry(self, entryobject=None, quantity=None):
    # validates entryfield
    if(entryobject != None):
      entrytext = entryobject.text()
      if(self.parent.isNumber(entrytext)):
        newnumber = int(entrytext)
        self.__dict__[quantity] = np.abs(newnumber)
        if(newnumber < 0):
          entryobject.setText(str(np.abs(newnumber)))
      else:
        # restore previous value
        entryobject.setText(str(self.__dict__[quantity]))

  def lostFocus(self, entryobject=None, quantity=None, defaultHandler=None, event=None):
    # entry field lost focus, perform sanity check
    if(entryobject != None):
      entrytext = entryobject.text()
      if(self.parent.isNumber(entrytext)):
        self.__dict__[quantity] = float(entrytext)
      else:
        # restore previous value
        entryobject.setText(str(self.__dict__[quantity]))
    # pass signal to original handler
    if(defaultHandler != None):
      defaultHandler(event)
    
  def validateErrorEntry(self, entryobject=None, quantity=None):
    # validates entryfield
    if(entryobject != None):
      entrytext = entryobject.text()
      if(self.parent.isNumber(entrytext)):
        newnumber = float(entrytext)
        self.__dict__[quantity] = np.abs(newnumber)
        if(newnumber < 0):
          entryobject.setText(str(np.abs(newnumber)))
      else:
        # restore previous value
        entryobject.setText(str(self.__dict__[quantity]))

  def importDataSeries(self):
    # greedy import of data
    cycleColors = self.parent.objectsarea.cycleColors
    # check whether any data has been loaded
    if((self.tableWidget.tableModel == None) or (self.tableWidget.tableModel.rowCount() == 0)):
      self.parent.statusbar.showMessage('Open a data file first!', self.parent.STATUS_TIME)
    else:
      # can do this by repeatedly calling updateData
      roles = self.tableWidget.roles
      columnCount = self.tableWidget.tableModel.columnCount()
      
      # check for presence of x and y
      if((not 'x' in roles) or (not 'y' in roles)):
        self.parent.statusbar.showMessage('Assign x and y columns!', self.parent.STATUS_TIME)
      else:
        # remember original y column
        originalY = roles['y']
        
        # change dataset style to line and no symbol
        style = self.parent.data[self.parent.activeData].getStyle()
        if(style['linestyle'] == 'None'):
          self.parent.data[self.parent.activeData].setStyle('linestyle', 'solid', redraw=False)
        if(style['marker'] != 'None'):
          self.parent.data[self.parent.activeData].setStyle('marker', 'None', redraw=False)
        # at least temporarily set dataset visibility to True to speed up import (this obviates a call to
        # setVisibility in updateData())
        if(self.parent.data[self.parent.activeData].visibility):
          invisible = False
        else:
          invisible, originalDataset = True, self.parent.activeData
          self.parent.data[self.parent.activeData].setVisibility(True, redraw=False)
        
        # loop over columns
        successes = 0
        currY = originalY
        while((currY < columnCount) and (successes < 100)):
          # check whether current column is already assigned to sth. else
          if((currY == originalY) or (not (currY in list(roles.values())))):
            # update status message as this can take a while
            self.parent.statusbar.showMessage('Now processing column ' + str(currY + 1) + '!', self.parent.STATUS_TIME, color='blue')
            # set current color
            currColor = (currY - originalY) % len(cycleColors)
            self.parent.data[self.parent.activeData].setStyle('color', cycleColors[currColor], redraw=False)
            # assign new y column
            self.tableWidget.roles['y'] = currY
            # get the new data
            if((currY + 1 == columnCount) or (successes == 99)):
              # need to update togglo sidebar before issuing updateData() in regular mode
              self.parent.plotArea.updateToggloContainer()
              # this is the last item to import => updateData() must behave differently
              axis_labels = self.updateData(docheck=False, redraw=False, quiet=True, create=False, add_data=False, series=False)
            else:
              axis_labels = self.updateData(docheck=False, redraw=False, quiet=True, create=False, add_data=False, series=True)
            # check the new data
            nuData = self.parent.data[self.parent.activeData].value()
            if(('x' in nuData) and (len(nuData['x']))):
              # assign name to data set
              if((len(axis_labels) > 1) and len(axis_labels[1])):
                self.parent.data[-1].setName(axis_labels[1])
              else:
                self.parent.data[-1].setName('Data_' + str(len(self.parent.data)))
              # keep track of successes
              successes += 1
              if((currY + 1 < columnCount) and (successes < 100)):
                # generate a new data set (if needed)
                self.parent.data.append(DataObject(self.parent))
                # need to copy contents of original object
                self.parent.data[-1].spawned(self.parent.data[self.parent.activeData])
                # reinit values (as they were copied by spawned)
                self.parent.data[-1].initValues()
                # set new data object as active
                self.parent.activeData = (len(self.parent.data) - 1)
                # need to update dataset table, otherwise updateData() will fail
                self.parent.objectsarea.refreshDataTable()
          currY += 1

        # restore original roles
        self.tableWidget.roles['y'] = originalY
        
        # was original data set invisible?
        if(invisible):
          self.parent.data[originalDataset].setVisibility(False, redraw=False)
        # for safe measure, always update legend -- should be fine with no redraw issued
        self.updateLegend(redraw=False)
        
        # update objects area
        self.parent.objectsarea.refreshDataTable()
        self.parent.objectsarea.refreshResidTable()
        self.parent.objectsarea.refreshCurvesTable()
        self.parent.objectsarea.refreshExtrasTable()
        self.parent.globalarea.updateDataSetTable()
        # for good measure, issue another update of togglo sidebar
        self.parent.plotArea.updateToggloContainer()
        
        # update results table
        self.parent.resultsarea.setDataSet(currIndex=len(self.parent.data) - 1, maxIndex=len(self.parent.data) - 1)

        # issue refresh of plots
        self.parent.plotArea.dataplotwidget.myRefresh()
        # do we need this call? (theoretically, original dataset [with residuals] could be thus reset -- so, likely needed)
        self.parent.plotArea.residplotwidget.myRefresh()

  def updateData(self, docheck=False, redraw=True, quiet=False, create=False, add_data=False, series=False):
    # initialize return value for importDataSeries
    axis_labels = ['', '']
    # check whether autoimport enabled
    if ((not docheck) or (self.refreshCheck.isChecked())):
      # check whether any data has been loaded
      if((self.tableWidget.tableModel == None) or (self.tableWidget.tableModel.rowCount() == 0)):
        self.parent.statusbar.showMessage('Open a data file first!', self.parent.STATUS_TIME)
      else:
        QtWidgets.QApplication.setOverrideCursor(QtGui.QCursor(QtCore.Qt.CursorShape.WaitCursor))
        QtCore.QCoreApplication.processEvents()

        # if selected dataset exists, clear it
        if(self.parent.selectedData.isLive):
          self.parent.selectedData.clearMe(redraw=False)
        
        # check whether we need to retrieve x-err and y-err columns at all
        needXErr, needYErr = self.errorXSwitch and (not self.errorXModel), self.errorSwitch and (not self.errorModel)
        # get and process data
        useLabelFlag = False
        new_data, roles, axis_labels = self.tableWidget.getData(needXErr=needXErr, needYErr=needYErr)
        if(needXErr and ('xerr' in roles)):
          # apply factor to x errors
          errXCol = roles.index('xerr')
          new_data = [[k if (j != errXCol) else (k * self.errorXMultiply) for j, k in enumerate(i)] for i in new_data]
        if(needYErr and ('yerr' in roles)):
          # apply factor to x errors
          errYCol = roles.index('yerr')
          new_data = [[k if (j != errYCol) else (k * self.errorMultiply) for j, k in enumerate(i)] for i in new_data]
        labels, labels2 = [], []
        labelCols = []
        if('labels' in roles):
          # separate numerical data from labels
          labelCols.append(roles.index('labels'))
          labels = [str(i[labelCols[-1]]) for i in new_data]
        if('labels2' in roles):
          # separate numerical data from labels
          labelCols.append(roles.index('labels2'))
          labels2 = [str(i[labelCols[-1]]) for i in new_data]
        if(len(labelCols)):
          new_data = [[k for j, k in enumerate(i) if j not in labelCols] for i in new_data]
          newRoles = [i for i in roles if i not in ['labels', 'labels2']]
          # check whether x values in roles
          # if no x-values present (but labels), generate pseudo-x categories on the fly
          if(not 'x' in roles):
            newRoles.append('x')
            roles.append('x')
            for index, row in enumerate(new_data):
              row.append(index)
            useLabelFlag = True
        else:
          newRoles = roles
        labels = np.array(labels)
        labels2 = np.array(labels2)
        new_data = np.array(new_data)

        # check for presence of x and y
        if((not 'x' in roles) or (not 'y' in roles)):
          if(not quiet):
            self.parent.statusbar.showMessage('Assign x (or, labels) and y columns!', self.parent.STATUS_TIME)
        else:
          # check whether the new selection makes sense
          array_dim = new_data.shape
          if(len(array_dim) == 1):
            if(array_dim[0] > 0):
              if(not quiet):
                self.parent.statusbar.showMessage('Select at least two data rows!', self.parent.STATUS_TIME)
            elif(self.tableWidget.hasComma()):
              if(not quiet):
                self.parent.statusbar.showMessage('Select some data to import them -- try replacing comma in data!', self.parent.STATUS_TIME)
            else:
              if(not quiet):
                self.parent.statusbar.showMessage('Select some data to import them!', self.parent.STATUS_TIME)
          elif ((len(array_dim) > 1) and (array_dim[1] > 1)):
            # make selected import buttons flash to remind user that they have been active
            self.highlightButtons = [self.errorSelectorLabel, self.errorXSelectorLabel, self.errorPropagateCheck,
                                     self.dataReductionLabel, self.dataTransformXCheck, self.dataTransformYCheck]
            self.timers = []
            for index, button in enumerate(self.highlightButtons):
              if(button.isChecked()):
                button.setHighlightMe(True)
                self.timers.append(QtCore.QTimer())
                self.timers[-1].timeout.connect(partial(self.timerHelper, index))
                self.timers[-1].setSingleShot(True)
                self.timers[-1].start(5000)
          
            # process the error model for y
            if(self.errorSwitch):
              if(self.errorModel):
                if(self.errorModel == 1):
                  # use const
                  errors = [self.errorConst] * array_dim[0]
                  errors = np.array(errors)
                elif(self.errorModel == 2):
                  # use percentage of y
                  if('y' in newRoles):
                    index = newRoles.index('y')
                    errors = [abs(self.errorPercent / 100.0 * i) for i in new_data[:, index]]
                    errors = np.array(errors)
                    #errors = errors.transpose()
                  else:
                    if(not quiet):
                      self.parent.statusbar.showMessage('Cannot locate y values for percentage error calculation!', self.parent.STATUS_TIME, color='blue')
                    
                # check if y-error column already exists
                if('yerr' in newRoles):
                  index = newRoles.index('yerr')
                  new_data[:, index] = errors
                else:
                  newRoles.append('yerr')
                  # repackage errors to enable hstacking
                  errors = [[i] for i in errors]
                  errors = np.array(errors)
                  new_data = np.hstack((new_data, errors))
            else:
              # no errors => possibly delete yerr column
              if('yerr' in newRoles):
                index = newRoles.index('yerr')
                newRoles.pop(index)
                new_data = np.delete(new_data, index, 1)
              
            # process the error model for x
            if(self.errorXSwitch):
              if(self.errorXModel):
                if(self.errorXModel == 1):
                  # use const
                  errors = [self.errorXConst] * array_dim[0]
                  errors = np.array(errors)
                elif(self.errorXModel == 2):
                  # use percentage of x
                  if('x' in newRoles):
                    index = newRoles.index('x')
                    errors = [abs(self.errorXPercent / 100.0 * i) for i in new_data[:, index]]
                    errors = np.array(errors)
                  else:
                    if(not quiet):
                      self.parent.statusbar.showMessage('Cannot locate x values for percentage error calculation!', self.parent.STATUS_TIME, color='blue')
                    
                # check if x-error column already exists
                if('xerr' in newRoles):
                  index = newRoles.index('xerr')
                  new_data[:,index] = errors
                else:
                  newRoles.append('xerr')
                  # repackage errors to enable hstacking
                  errors = [[i] for i in errors]
                  errors = np.array(errors)
                  new_data = np.hstack((new_data, errors))
            else:
              # no x errors => possibly delete xerr column
              if('xerr' in newRoles):
                index = newRoles.index('xerr')
                newRoles.pop(index)
                new_data = np.delete(new_data, index, 1)
              
             # process the data reduction model
            if(self.reductionSwitch):
              if(self.reductionModel == 1):
                # skip data points by numpy slicing
                new_data = new_data[::self.reductionSkip + 1]
                if('labels' in roles):
                  labels = labels[::self.reductionSkip + 1]
                if('labels2' in roles):
                  labels2 = labels2[::self.reductionSkip + 1]
              elif(self.reductionModel == 2):
                # average with error propagation
                if(('labels' in roles) or ('labels2' in roles)):
                  new_data, labels, labels2 = self.movingAverage(sourceData=new_data, roles=newRoles, average=self.reductionAvg, stepsize=self.reductionAvg, labels=labels, labels2=labels2)
                else:
                  new_data = self.movingAverage(sourceData=new_data, roles=newRoles, average=self.reductionAvg, stepsize=self.reductionAvg)
              elif(self.reductionModel == 3):
                # moving average with error propagation
                if(('labels' in roles) or ('labels2' in roles)):
                  new_data, labels, labels2 = self.movingAverage(sourceData=new_data, roles=newRoles, average=self.reductionMovAvg, stepsize=1, labels=labels, labels2=labels2)
                else:
                  new_data = self.movingAverage(sourceData=new_data, roles=newRoles, average=self.reductionMovAvg, stepsize=1)
              elif(self.reductionModel == 4):
                # logarithmic data reduction
                if(('labels' in roles) or ('labels2' in roles)):
                  new_data, labels, labels2 = self.logAverage(sourceData=new_data, roles=newRoles, targetpoints=self.reductionLog, labels=labels, labels2=labels2)
                else:
                  new_data = self.logAverage(sourceData=new_data, roles=newRoles, targetpoints=self.reductionLog)
                
            # do data transform if necessary
            # make copy of original data in case we also have to transform y axis
            if(self.dataTransformYCheck.isChecked()):
              new_data2 = deepcopy(new_data)
            if(self.dataTransformXCheck.isChecked()):
              formula = str(self.dataTransformXEntry.text())
              if(len(formula) > 0):
                # transformer
                formula = 'x = ' + formula
                new_data, _ = self.transformer(sourceData=new_data, roles=newRoles, formula=formula, axis='x')
              else:
                if(not quiet):
                  self.parent.statusbar.showMessage('Enter formula for x transformation!', self.parent.STATUS_TIME)
                
            if(self.dataTransformYCheck.isChecked()):
              formula = str(self.dataTransformYEntry.text())
              if(len(formula) > 0):
                # transformer
                formula = 'y = ' + formula
                new_data2, _ = self.transformer(sourceData=new_data2, roles=newRoles, formula=formula, axis='y')
                # replace y and yerr columns in new_data
                ycol = newRoles.index('y')
                new_data[:, ycol] = new_data2[:, ycol]
                # do error propagation?
                if(('yerr' in newRoles) and self.errorPropagate):
                  yerrcol = newRoles.index('yerr')
                  new_data[:, yerrcol] = new_data2[:, yerrcol]
              else:
                if(not quiet):
                  self.parent.statusbar.showMessage('Enter formula for y transformation!', self.parent.STATUS_TIME)
                
            # delete all rows with non-numerical content
            for role in newRoles:
              index = newRoles.index(role)
              if('labels' in roles):
                labels = labels[np.isfinite(new_data[:, index])]
              if('labels2' in roles):
                labels2 = labels2[np.isfinite(new_data[:, index])]
              new_data = new_data[np.isfinite(new_data[:, index])]
              
            # should we add data to existing data here?
            if(add_data):
              sourceData = self.parent.data[self.parent.activeData].value()
              sourceRoles = list(sourceData.keys())
              sourceLabels = [i for i in self.parent.data[self.parent.activeData].labels]
              sourceLabels2 = [i for i in self.parent.data[self.parent.activeData].labels2]

              # first consider labels?
              if(len(labels) or len(sourceLabels)):
                # expand labels to correct size
                # first convert labels to list
                labels = list(labels)
                i = 0
                while(len(sourceLabels) < len(sourceData[sourceRoles[0]])):
                  sourceLabels.append(i)
                  i += 1
                while(len(labels) < len(new_data[0])):
                  labels.append(i)
                  i += 1
              combinedLabels = sourceLabels + labels

              # first consider labels?
              if(len(labels2) or len(sourceLabels2)):
                # expand labels to correct size
                # first convert labels to list
                labels2 = list(labels2)
                i = 0
                while(len(sourceLabels2) < len(sourceData[sourceRoles[0]])):
                  sourceLabels2.append(i)
                  i += 1
                while(len(labels2) < len(new_data[0])):
                  labels2.append(i)
                  i += 1
              combinedLabels2 = sourceLabels2 + labels2
              
              # then deal with data
              allRoles = list(set(sourceRoles + newRoles))
              # expand data sets prior to merger?
              defaultValues = {'xerr': 0.0, 'yerr': 0.0, 'labels': '', 'labels2': '', 'fval': 0.0, 'resid': 0.0, 'x': 0.0, 'y': 0.0}
              missingRoles = [i for i in allRoles if (not i in sourceRoles)]
              # expand source data set by needed columns
              length = len(sourceData[sourceRoles[0]])
              for entry in missingRoles:
                sourceData[entry] = [defaultValues[entry] for i in range(length)]
              # also expand merge data set by needed columns
              length = len(new_data)
              for entry in sourceData:
                if(entry in newRoles):
                  sourceData[entry] = np.hstack((sourceData[entry], [item[newRoles.index(entry)] for item in new_data]))
                else:
                  sourceData[entry] = np.hstack((sourceData[entry], np.array([defaultValues[entry] for i in range(length)])))
              
              # prepare combined data so that it can be gobbled up by updateData()
              new_data, newRoles, labels, labels2 = [], [], combinedLabels, combinedLabels2
              for key in sourceData:
                newRoles.append(key)
                if(len(new_data)):
                  new_data = np.vstack((new_data, sourceData[key]))
                else:
                  new_data = sourceData[key]
              new_data = np.transpose(new_data)
  
            # assign new data
            if(create):
              self.parent.objectsarea.dataSetCreate()
              self.parent.activeData = len(self.parent.data) - 1
            if(('labels' in roles) or ('labels2' in roles)):
              labels = list(labels)
              self.parent.data[self.parent.activeData].setData(new_data, newRoles, labels=labels, labels2=labels2)
            else:
              self.parent.data[self.parent.activeData].setData(new_data, newRoles)
            
            # make data object visibile
            self.parent.objectsarea.dataSetTable.cellWidget(self.parent.activeData, 0).setChecked(True)
            if(not series):
              self.parent.data[self.parent.activeData].setVisibility(True, redraw=False)
              # update toggle buttons in side bar
              self.parent.plotArea.checkToggloItem(kind='data', index=self.parent.activeData, state=True)
            
            # here we should update the plot
            if(series):
              self.parent.data[self.parent.activeData].handleData, self.parent.data[self.parent.activeData].handleErr, self.parent.data[self.parent.activeData].handleErrShady,\
                self.parent.data[self.parent.activeData].handleBar, self.parent.data[self.parent.activeData].handleStack, self.parent.data[self.parent.activeData].handleStackNeg,\
                self.parent.data[self.parent.activeData].handleText, self.parent.data[self.parent.activeData].handleViolin, self.parent.data[self.parent.activeData].handleScatter = \
                self.parent.plotArea.plotData(self.parent.data[self.parent.activeData].value(), dataobject = self.parent.data[self.parent.activeData], \
                handleData = self.parent.data[self.parent.activeData].handleData, handleErr = self.parent.data[self.parent.activeData].handleErr, handleErrShady=self.parent.data[self.parent.activeData].handleErrShady,\
                handleBar = self.parent.data[self.parent.activeData].handleBar, handleStack = self.parent.data[self.parent.activeData].handleStack,\
                handleStackNeg = self.parent.data[self.parent.activeData].handleStackNeg, handleText = self.parent.data[self.parent.activeData].handleText, handleViolin = self.parent.data[self.parent.activeData].handleViolin,\
                handleScatter = self.parent.data[self.parent.activeData].handleScatter, redraw=False, rescale=False, autoIgnoreCurrentY=True)
              self.parent.data[self.parent.activeData].handleData_div, self.parent.data[self.parent.activeData].handleErr_div, self.parent.data[self.parent.activeData].handleErrShady_div,\
                self.parent.data[self.parent.activeData].handleBar_div, self.parent.data[self.parent.activeData].handleStack_div, self.parent.data[self.parent.activeData].handleStackNeg_div,\
                self.parent.data[self.parent.activeData].handleText_div, self.parent.data[self.parent.activeData].handleViolin_div, self.parent.data[self.parent.activeData].handleScatter_div = \
                self.parent.plotArea.plotData(self.parent.data[self.parent.activeData].value(), dataobject = self.parent.data[self.parent.activeData], \
                handleData = self.parent.data[self.parent.activeData].handleData_div, handleErr = self.parent.data[self.parent.activeData].handleErr_div, handleErrShady=self.parent.data[self.parent.activeData].handleErrShady_div,\
                handleBar = self.parent.data[self.parent.activeData].handleBar_div, handleStack = self.parent.data[self.parent.activeData].handleStack_div,\
                handleStackNeg = self.parent.data[self.parent.activeData].handleStackNeg_div, handleText = self.parent.data[self.parent.activeData].handleText_div, handleViolin = self.parent.data[self.parent.activeData].handleViolin_div,\
                handleScatter = self.parent.data[self.parent.activeData].handleScatter_div, redraw=False, splitX=True, rescale=False, autoIgnoreCurrentY=True)
            else:
              self.parent.data[self.parent.activeData].handleData, self.parent.data[self.parent.activeData].handleErr, self.parent.data[self.parent.activeData].handleErrShady,\
                self.parent.data[self.parent.activeData].handleBar, self.parent.data[self.parent.activeData].handleStack, self.parent.data[self.parent.activeData].handleStackNeg,\
                self.parent.data[self.parent.activeData].handleText, self.parent.data[self.parent.activeData].handleViolin, self.parent.data[self.parent.activeData].handleScatter = \
                self.parent.plotArea.plotData(self.parent.data[self.parent.activeData].value(), dataobject = self.parent.data[self.parent.activeData], \
                handleData = self.parent.data[self.parent.activeData].handleData, handleErr = self.parent.data[self.parent.activeData].handleErr, handleErrShady=self.parent.data[self.parent.activeData].handleErrShady,\
                handleBar = self.parent.data[self.parent.activeData].handleBar, handleStack = self.parent.data[self.parent.activeData].handleStack,\
                handleStackNeg = self.parent.data[self.parent.activeData].handleStackNeg, handleText = self.parent.data[self.parent.activeData].handleText, handleViolin = self.parent.data[self.parent.activeData].handleViolin,\
                handleScatter = self.parent.data[self.parent.activeData].handleScatter, redraw=False, autoIgnoreCurrentY=True)
              self.parent.data[self.parent.activeData].handleData_div, self.parent.data[self.parent.activeData].handleErr_div, self.parent.data[self.parent.activeData].handleErrShady_div,\
                self.parent.data[self.parent.activeData].handleBar_div, self.parent.data[self.parent.activeData].handleStack_div, self.parent.data[self.parent.activeData].handleStackNeg_div,\
                self.parent.data[self.parent.activeData].handleText_div, self.parent.data[self.parent.activeData].handleViolin_div, self.parent.data[self.parent.activeData].handleScatter_div = \
                self.parent.plotArea.plotData(self.parent.data[self.parent.activeData].value(), dataobject = self.parent.data[self.parent.activeData], \
                handleData = self.parent.data[self.parent.activeData].handleData_div, handleErr = self.parent.data[self.parent.activeData].handleErr_div, handleErrShady=self.parent.data[self.parent.activeData].handleErrShady_div,\
                handleBar = self.parent.data[self.parent.activeData].handleBar_div, handleStack = self.parent.data[self.parent.activeData].handleStack_div,\
                handleStackNeg = self.parent.data[self.parent.activeData].handleStackNeg_div, handleText = self.parent.data[self.parent.activeData].handleText_div, handleViolin = self.parent.data[self.parent.activeData].handleViolin_div,\
                handleScatter = self.parent.data[self.parent.activeData].handleScatter_div, redraw=False, splitX=True, autoIgnoreCurrentY=True)
              # and we should redraw the fit function to cover new x-range
              self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.parent.plotArea.plotFunction(\
                fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot,\
                handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False, doAutoZoom=False)
              self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.parent.plotArea.plotFunction(\
                fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div,\
                handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True, doAutoZoom=False)
              # copy in case split axes are shown
              curve = self.parent.fit[self.parent.activeFit]
              if(self.parent.plotArea.splitY and curve.onBothAxes):
                curve.duplicateForSplit()
              # check whether we should set category labels on x-axis
              if(useLabelFlag):
                self.parent.plotArea.setDataAxisTicks(dataSet=self.parent.activeData, redraw=False, target='plot')
                self.parent.plotArea.setDataAxisTicks(dataSet=self.parent.activeData, redraw=False, target='resid')
              elif((self.parent.plotArea.ticksXDataSet == self.parent.activeData) and (not self.parent.graphicsarea.configTickXAuto.isChecked())):
                # only clear tick labels if inmporting to same object
                self.parent.plotArea.ticksXDataSet = -1
                nuTicks = self.parent.plotArea.setAutoTicks(axis='x', redraw=False, target='plot')
                self.parent.plotArea.setAutoTicks(axis='x', redraw=False, target='resid')
                # manually check button on graphicsarea
                self.parent.graphicsarea.configTickXAuto.setChecked(True)
                self.parent.plotArea.ticksXAuto = True
                tickstr = self.parent.graphicsarea.magicTickstring(nuTicks)
                self.parent.graphicsarea.configTickXEntry.setText(tickstr)
              # and we should update the legend
              self.updateLegend(redraw=redraw)
              # and we should update the corresponding residuals
              self.parent.data[self.parent.activeData].handleResid, self.parent.plotArea.handleResidZero,\
                self.parent.data[self.parent.activeData].handleResidBar, self.parent.data[self.parent.activeData].handleResidStack,\
                self.parent.data[self.parent.activeData].handleResidStackNeg, self.parent.data[self.parent.activeData].handleResidText,\
                self.parent.data[self.parent.activeData].handleResidScatter = self.parent.plotArea.plotResid(\
                dataobject = self.parent.data[self.parent.activeData], handleResid = self.parent.data[self.parent.activeData].handleResid,\
                handleResidZero = self.parent.plotArea.handleResidZero, handleResidBar = self.parent.data[self.parent.activeData].handleResidBar,\
                handleResidStack = self.parent.data[self.parent.activeData].handleResidStack, handleResidStackNeg = self.parent.data[self.parent.activeData].handleResidStackNeg,\
                handleResidText = self.parent.data[self.parent.activeData].handleResidText, handleResidScatter=self.parent.data[self.parent.activeData].handleResidScatter, redraw=False)
              self.parent.data[self.parent.activeData].handleResid_div, self.parent.plotArea.handleResidZero_div,\
                self.parent.data[self.parent.activeData].handleResidBar_div, self.parent.data[self.parent.activeData].handleResidStack_div,\
                self.parent.data[self.parent.activeData].handleResidStackNeg_div, self.parent.data[self.parent.activeData].handleResidText_div,\
                self.parent.data[self.parent.activeData].handleResidScatter_div = self.parent.plotArea.plotResid(\
                dataobject = self.parent.data[self.parent.activeData], handleResid = self.parent.data[self.parent.activeData].handleResid_div,\
                handleResidZero = self.parent.plotArea.handleResidZero_div, handleResidBar = self.parent.data[self.parent.activeData].handleResidBar_div,\
                handleResidStack = self.parent.data[self.parent.activeData].handleResidStack_div, handleResidStackNeg = self.parent.data[self.parent.activeData].handleResidStackNeg_div,\
                handleResidText = self.parent.data[self.parent.activeData].handleResidText_div, handleResidScatter=self.parent.data[self.parent.activeData].handleResidScatter_div, redraw=False, splitX=True)
              # and we should update the resid plot (as x-axis will most likely have rescaled)
              self.parent.plotArea.setAxisLimits(lower=self.parent.plotArea.minX, upper=self.parent.plotArea.maxX, axis='x',\
                updateLabel=False, target='resid', redraw=False, updateGrid=True)
              self.parent.plotArea.setAxisLimits(lower=self.parent.plotArea.minX_div, upper=self.parent.plotArea.maxX_div, axis='x2',\
                updateLabel=False, target='resid', redraw=False, updateGrid=True)
              # draw resid line (again) to ensure coverage of entire x range
              self.parent.plotArea.handleResidZero = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero, redraw=False)
              self.parent.plotArea.handleResidZero_div = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero_div, redraw=redraw, splitX=True)
              
              # and we should update the results table
              self.parent.resultsarea.setDataSet(currIndex=self.parent.activeData)

        QtWidgets.QApplication.restoreOverrideCursor()
        if(True):
          return axis_labels

  def timerHelper(self, index=-1):
    # helper function to control button highlighting
    if(index + 1):
      self.highlightButtons[index].setHighlightMe(False)

  def transformer(self, sourceData=None, roles=None, formula='', axis='x'):
    self.EPSILON, success = 1e-9, False
    # does axis transform
    if(axis in ['x', 'y']):
      if(type(sourceData) != type(None)):
        if(axis in roles):
          # try defining transformation function
          try:
            funcstr = 'def transformThis(self, x, y):'
            funcstr += '\n\t' + formula + '\n\treturn ' + axis
            # generate ffunc in local namespace (this is needed for Python3 vs. Python2, bummer)
            namespace = self.mySpace
            exec(funcstr, namespace)
            # now define the new function in the object scope
            setattr(DataArea, 'transformThis', namespace['transformThis'])
          except:
            self.parent.statusbar.showMessage('Error when setting transformation for ' + axis, self.parent.STATUS_TIME)
          else:
            # do the actual transform
            index = roles.index(axis) #val = sourceData[:,index]
            xindex = roles.index('x'); xval = deepcopy(sourceData[:,xindex])
            yindex = roles.index('y'); yval = deepcopy(sourceData[:,yindex])
            try:
              newVal = self.transformThis(xval, yval)
              # now copy transformed data to data matrix
              sourceData[:,index] = newVal
              success = True
            except:
              self.parent.statusbar.showMessage('Error when calculating transform for ' + axis, self.parent.STATUS_TIME)
            else:
              # deal with data errors
              errname = axis + 'err'
              if(errname in roles):
                errindex = roles.index(errname)
                # numerically determine derivatives in x and y
                # consider x
                if('xerr' in roles):
                  xerrindex = roles.index('xerr'); xerrval = sourceData[:,xerrindex]; xerrval = xerrval ** 2
                  try:
                    xderiv = self.transformThis(xval + self.EPSILON, yval)
                    xderiv = (xderiv - newVal) / self.EPSILON
                    xderiv = xderiv ** 2
                    newErr = xerrval * xderiv
                  except:
                    pass
                else:
                  newErr = np.array([0] * len(newVal))
                
                # consider y
                if('yerr' in roles):
                  yerrindex = roles.index('yerr'); yerrval = sourceData[:,yerrindex]; yerrval = yerrval ** 2
                  try:
                    yderiv = self.transformThis(xval, yval + self.EPSILON)
                    yderiv = (yderiv - newVal) / self.EPSILON
                    yderiv = yderiv ** 2
                    newErr = newErr + (yerrval * yderiv)
                  except:
                    pass
                  
                # calculate root of error
                newErr = newErr ** 0.5
                sourceData[:,errindex] = newErr

    return sourceData, success

  def logAverage(self, sourceData=None, roles=None, targetpoints=100, labels=np.array([]), labels2=np.array([])):
    # reduces data logarithmically to (approx.) target no. of points
    if(type(sourceData) != type(None)):
      if(('x' in roles) and ('y' in roles)):
        # locate x values
        xcol = roles.index('x')
        xval = sourceData[:,xcol]

        # get positive x values
        posXval = xval[xval > 0]
        if(len(posXval) > 0):
          # calculate target x values on log-spaced scale
          logXval = np.linspace(np.log(np.min(posXval)), np.log(np.max(posXval)), targetpoints)
          targetXval = np.exp(logXval)
          targetBoundary = [0]
          targetBoundary.extend([(targetXval[i] + targetXval[i+1])/2 for i in range(len(targetXval)-1)])
          targetBoundary.append(targetXval[-1])

          # cycle through boundary list
          output, avgLabels, avgLabels2 = np.array([]), [], []
          for index, entry in enumerate(targetBoundary[:-1]):
            targetRows = sourceData[sourceData[:,xcol] > targetBoundary[index]]
            labelIndex = len(sourceData[sourceData[:,xcol] <= targetBoundary[index]])
            targetRows = targetRows[targetRows[:,xcol] <= targetBoundary[index+1]]
          
            # check current entry
            if(targetRows.size > 0):
              if((len(targetRows.shape) > 1) and (targetRows.shape[0] > 1)):
                # calculate averages
                newRow = []
                for index2, entry2 in enumerate(roles):
                  if(entry2 in ['x', 'y']):
                    # numerically average x and y values
                    newRow.append(np.average(targetRows[:,index2]))
                  elif(entry2 in ['xerr', 'yerr']):
                    errVal = (targetRows[:,index2] / len(targetRows[:,index2])) ** 2
                    errVal = np.sum(errVal) ** 0.5
                    newRow.append(errVal)
                    
                targetRows = np.array(newRow)
                
              if(labels.size):
                avgLabels.append(labels[labelIndex])
              if(labels2.size):
                avgLabels2.append(labels2[labelIndex])

              # append current line to output
              if(len(output) > 0):
                output = np.vstack((output, targetRows))
              else:
                output = targetRows
          
          if(labels.size or labels2.size):
            return output, np.array(avgLabels), np.array(avgLabels2)
          else:
            return output
  
        else:
          self.parent.statusbar.showMessage('No positive x values, cannot do any reduction!', self.parent.STATUS_TIME)
          return sourceData

  def movingAverage(self, sourceData=None, roles=None, average=1, stepsize=1, labels=np.array([]), labels2=np.array([])):
    # calculate a moving average with error propagation
    if(type(sourceData) != type(None)):
      if(('x' in roles) and ('y' in roles)):
        # locate x and y values
        xcol, ycol = roles.index('x'), roles.index('y')
        xval, yval = sourceData[:, xcol], sourceData[:, ycol]
        # moving average
        avgXval = np.array([np.average(xval[i: i + average]) for i in range(0, len(xval) - average + 1, stepsize)])
        avgYval = np.array([np.average(yval[i: i + average]) for i in range(0, len(yval) - average + 1, stepsize)])
        # check for presence of error values
        if('yerr' in roles):
          # need to do error propagation
          yerrcol = roles.index('yerr')
          yerrval = sourceData[:, yerrcol]
          yerrval = (yerrval / average) ** 2
          # error propagation
          avgYerrval = np.array([np.sum(yerrval[i: i + average]) for i in range(0, len(yerrval) - average + 1, stepsize)])
          avgYerrval = avgYerrval ** 0.5
        if('xerr' in roles):
          # need to do error propagation
          xerrcol = roles.index('xerr')
          xerrval = sourceData[:, xerrcol] 
          xerrval = (xerrval / average) ** 2
          # error propagation
          avgXerrval = np.array([np.sum(xerrval[i: i + average]) for i in range(0, len(xerrval) - average + 1, stepsize)])
          avgXerrval = avgXerrval ** 0.5
          
        # deal with data labels
        if(labels.size):
          # use label of first data point to be averaged
          avgLabel = np.array([labels[i] for i in range(0, len(xval) - average + 1, stepsize)])
        if(labels2.size):
          # use label of first data point to be averaged
          avgLabel2 = np.array([labels2[i] for i in range(0, len(xval) - average + 1, stepsize)])

        # now need to assemble columns according to roles
        procData = np.array([])
        for entry in roles:
          # get current column
          if(entry == 'x'):
            addItem = avgXval
          elif(entry == 'y'):
            addItem = avgYval
          elif(entry == 'xerr'):
            addItem = avgXerrval
          elif(entry == 'yerr'):
            addItem = avgYerrval
          
          # assemble output data
          if(procData.size > 0):
            procData = np.vstack((procData, addItem))
          else:
            procData = addItem

        if(labels.size or labels2.size):
          return procData.transpose(), avgLabel, avgLabel2
        else:
          return procData.transpose()
        
  def loadData(self):
    global REMEMBERDIR
    global RECENTFILES
    filter_options = ['Excel Files (*.xls *.xlsx)', 'Open Office (*.ods)', 'Text Tab Delimited (*.txt)', 'Text Comma Separated (*.txt *.csv)', 'Text Whitespace Delimited (*.txt)']
    if(PYCORN_PRESENT):
      filter_options.append('Unicorn Files (*.res)')
    filterstring = ';;'.join(filter_options)
    filename, filter_ = QtWidgets.QFileDialog.getOpenFileName(self, filter=filterstring, directory = REMEMBERDIR, caption='Open Data', initialFilter=self.importFilter)
    self.importFilter = filter_
    filename = str(filename)
    if(PATH_SEPARATOR in filename):
      REMEMBERDIR = filename.split(PATH_SEPARATOR)[:-1]
      REMEMBERDIR = PATH_SEPARATOR.join(REMEMBERDIR)
    elif('/' in filename):
      REMEMBERDIR = filename.split('/')[:-1]
      REMEMBERDIR = PATH_SEPARATOR.join(REMEMBERDIR)
    if(len(filename) > 0):
      mode = filter_options.index(filter_)
      if(mode == 0):
        self.tableWidget.loadXLS(filename=filename, transpose=self.transposeData)
      elif(mode == 1):
        self.tableWidget.loadODS(filename=filename, transpose=self.transposeData)
      elif(mode == 2):
        self.tableWidget.loadTextFile(filename=filename, delimiter='\t', transpose=self.transposeData)
      elif(mode == 3):
        self.tableWidget.loadTextFile(filename=filename, delimiter=',', transpose=self.transposeData)
      elif(mode == 4):
        self.tableWidget.loadTextFile(filename=filename, delimiter=None, transpose=self.transposeData)
      elif(mode == 5):
        self.tableWidget.loadUnicornFile(filename=filename, transpose=self.transposeData)
      # store this for use in recent files
      if(mode in [0, 1, 2, 3, 4, 5]):
        # loop through list and remove previous entry
        index = len(RECENTFILES) - 1
        for entry in RECENTFILES[::-1]:
          if(entry[0] == filename):
            RECENTFILES.pop(index)
          index -= 1
        RECENTFILES.insert(0, [filename, mode])
        # truncate to 20 entries
        RECENTFILES = RECENTFILES[:20]
    else:
      return False
    
    # mayhaps display tool tip (only do this on first time after program start)
    if(self.firstLoad):
      self.firstLoad = False
      QtWidgets.QToolTip.showText(self.refreshButton.mapToGlobal(QtCore.QPoint(50, -20)), '\u2199 After selecting data, click here to import and plot them', self.refreshButton)

    return True

  def blankResizeTable(self, callButton=None):
    # opens a QMenu to allow blanking and resizing of data table
    self.menu = BlankResizeMenu(self, self.tableWidget)
    # apply styles to popup window
    if(QSTYLE != None):
      self.menu.setStyle(QSTYLE)
    if(QSTYLESHEET != None):
      self.menu.setStyleSheet(QSTYLESHEET)

    # now move window to new position
    if(callButton != None):
      qPoint = callButton.mapToGlobal(QtCore.QPoint(0, 0))
      menuX, menuY = qPoint.x(), qPoint.y()
      menuY += callButton.geometry().height()
    else:
      menuX, menuY = QtGui.QCursor.pos().x(), QtGui.QCursor.pos().y()

    # open at mouse pointer
    self.menu.popup(QtCore.QPoint(int(menuX), int(menuY)))

  def updateLegend(self, redraw=True):
    # does legend need to be updated?
    value = self.parent.plotArea.legendVisible
    if(value or redraw):
      self.parent.plotArea.setLegend(value=value, redraw=redraw)
      
class ShrinkoWidget(QWidgetMac):
  def __init__(self, notify=None, notifyResid=False, parent=None):
    super(ShrinkoWidget, self).__init__()
    self.notify = notify
    self.notifyResid = notifyResid
    self.parent = parent
    
  def resizeEvent(self, event):
    # do the regular resize event
    QtWidgets.QWidget.resizeEvent(self, event)
    # is this the container with the matplot inside?
    if(self.notify != None):
      if(self.notifyResid):
        # can probably skip the horizontal resize as this is already done by the partner shrink-o-widget
        self.notify.verticalRulerResid.updateRuler()
        # also need to update tick entry fields and axis grid
        if(self.parent != None):
          self.parent.updateTickEntryField(axis='resid')
          for axis in ['x', 'x2', 'y']:
            self.parent.drawAxisGrid(axis=axis, target='resid', redraw=False)
      else:
        self.notify.horizontalRuler.updateRuler()
        self.notify.verticalRuler.updateRuler()
        self.parent.destructAboutLogo()
        # also need to update tick entry fields and axis grid
        # still does not fix the issue on split axis toggling
        if(self.parent != None):
          for axis in ['x', 'x2', 'y', 'y2']:
            self.parent.updateTickEntryField(axis=axis)
            self.parent.drawAxisGrid(axis=axis, target='plot', redraw=False)

class Ruler(QtWidgets.QFrame):
  def __init__(self, parent=None, mode=0, resid=False, numTicks=5):
    super(Ruler, self).__init__()
    self.parent = parent
    self.mode = mode
    self.resid = resid
    self.ticks = {}
    self.LARGE_RECTANGLE = 1000
    self.NUMBER_TICKS = numTicks
    self.ROUNDNESS = 3
    self.currwidth, self.currheight, self.currheight_resid = 4.0, 4.0, 1.0
    
    # init appearance
    self.setFrameShape(QtWidgets.QFrame.Shape.Box)
    self.setFrameShadow(QtWidgets.QFrame.Shadow.Sunken)

  def paintEvent(self, event):
    # draw ruler
    s = self.size()
    width, height = s.width(), s.height()
    qp = QtGui.QPainter()
    if(qp is not None):
      qp.begin(self)
      qp.setPen(QtGui.QColor('black'))
      qp.setBrush(QtGui.QColor('white'))
      qp.setPen(QtGui.QColor(*UI_TEXT_COLOR))
      qp.setBrush(QtGui.QColor(*UI_BASE_COLOR))
      qp.setRenderHint(QtGui.QPainter.RenderHint.Antialiasing, True)
      qp.drawRoundedRect(0, 0, width, height, int(self.ROUNDNESS), int(self.ROUNDNESS))
      
      # draw ticks
      if(self.mode):
        # vertical ruler
        qp.rotate(90)
        for entry in self.ticks:
          posY = int(entry * height + 0.5 - self.LARGE_RECTANGLE / 2)
          label = self.ticks[entry]
          if(entry > 0):
            qp.drawText(posY, -int(BASE_SIZE) - 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignVCenter, label)
            
        # draw dots in between
        labelPos = sorted(list(self.ticks.keys()))
        if(len(labelPos) > 1):
          # extend by one to draw dots beyond last number
          labelPos.append(2.0 * labelPos[-1] - labelPos[-2])
        for index in range(len(labelPos) - 1):
          posY = ((labelPos[index] + labelPos[index + 1]) / 2.0)
          posY = int(posY * height + 0.5 - self.LARGE_RECTANGLE / 2)
          qp.drawText(posY, -int(BASE_SIZE) - 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignVCenter, '\u2022')
          ###if((self.resid and (self.currheight_resid >= 2.5)) or ((not self.resid) and (self.currheight > 2.5))):
          # never draw small dots
          if(0):
            # small dots
            posY = ((3.0 * labelPos[index] + labelPos[index + 1]) / 4.0)
            posY = int(posY * height + 0.5 - self.LARGE_RECTANGLE / 2)
            qp.drawText(posY, -int(BASE_SIZE) - 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignVCenter, '\u00b7')
            posY = ((labelPos[index] + 3.0 * labelPos[index + 1]) / 4.0)
            posY = int(posY * height + 0.5 - self.LARGE_RECTANGLE / 2)
            qp.drawText(posY, -int(BASE_SIZE) - 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignVCenter, '\u00b7')
        qp.rotate(-90)
      else:
        # horizontal ruler
        for entry in self.ticks:
          posX = int(entry * width + 0.5 - self.LARGE_RECTANGLE / 2)
          label = self.ticks[entry]
          if(entry > 0):
            # for alignment of text, need to specify a bounding box (which can be fairly large)
            qp.drawText(posX, 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop, label)
            
        # draw dots in between
        labelPos = sorted(list(self.ticks.keys()))
        if(len(labelPos) > 1):
          # extend by one to draw dots beyond last number
          labelPos.append(2.0 * labelPos[-1] - labelPos[-2])
        for index in range(len(labelPos) - 1):
          posX = ((labelPos[index] + labelPos[index + 1]) / 2.0)
          posX = int(posX * width + 0.5 - self.LARGE_RECTANGLE / 2)
          qp.drawText(posX, 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop, '\u2022')
          # never draw small dots
          if(0):
            posX = ((3.0 * labelPos[index] + labelPos[index + 1]) / 4.0)
            posX = int(posX * width + 0.5 - self.LARGE_RECTANGLE / 2)
            qp.drawText(posX, 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop, '\u00b7')
            posX = ((labelPos[index] + 3.0 * labelPos[index + 1]) / 4.0)
            posX = int(posX * width + 0.5 - self.LARGE_RECTANGLE / 2)
            qp.drawText(posX, 2, self.LARGE_RECTANGLE, int(BASE_SIZE), QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop, '\u00b7')
        
      qp.end()

  def resizeEvent(self, event):
    # custom resize event
    QtWidgets.QFrame.resizeEvent(self, event)
    ###self.updateRuler()
    ### need to get the synchronization right, has to trigger after canvas resize
    ### we better trigger the update in the parental shrink-o-widget

  def updateRuler(self):
    # updates the ruler
    # first delete all previous ticks
    self.ticks = {}

    # caluclate new ones
    if(self.resid):
      currwidth, currheight = self.parent.residplot.get_size_inches()
      self.currheight_resid = currheight
    else:
      currwidth, currheight = self.parent.matplot.get_size_inches()
      self.currwidth, self.currheight = currwidth, currheight
    
    if(self.mode):
      # vertical ruler
      useTicks = self.NUMBER_TICKS
      if(currheight < 2.5):
        useTicks = 2
      incrementV = int(currheight / useTicks)
      # check for large zoom scale
      escalate = 1
      while(not incrementV):
        escalate *= 2.0
        incrementV = int(escalate * currheight / useTicks)
      incrementV *= 1.0 / escalate
      # and the actual calculation
      currPos, incPos = 0, 1.0 * incrementV / currheight
      while((currPos * incPos) < 1):
        posY = currPos * incPos
        label = currPos * incrementV
        self.ticks[posY] = '{0:g}'.format(label)
        currPos += 1
    else:
      # horizontal ruler
      useTicks = self.NUMBER_TICKS
      if(currwidth < 2.5):
        useTicks = 2
      incrementH = int(currwidth / useTicks)
      # check for large zoom scale
      escalate = 1
      while(not incrementH):
        escalate *= 2.0
        incrementH = int(escalate * currwidth / useTicks)
      incrementH *= 1.0 / escalate
      # and the actual calculation
      currPos, incPos = 0, 1.0 * incrementH / currwidth
      while((currPos * incPos) < 1):
        posX = currPos * incPos
        label = currPos * incrementH
        self.ticks[posX] = '{0:g}'.format(label)
        currPos += 1
      
    # issue paint event
    self.update()

class MyAutoMinorLocator(matplotlib.ticker.AutoMinorLocator):
  # custom minor locator to heed missing major ticks
  def __init__(self, n=None):
    super(MyAutoMinorLocator, self).__init__(n=n)

  def __call__(self):
    'Return the locations of the ticks'
    majorlocs = self.axis.get_majorticklocs()
    # are we on a log scale?
    mode = self.axis.get_scale()

    if(mode != 'log'):
      try:
        majorstep = majorlocs[1] - majorlocs[0]
      except IndexError:
        return []
    else:
      try:
        majorstep = majorlocs[1] / majorlocs[0]
      except IndexError:
        return []
      except ZeroDivisionError:
        return []

    if self.ndivs is None:
      ndivs = 9
    else:
      ndivs = self.ndivs

    # view interval
    vmin, vmax = self.axis.get_view_interval()
    if vmin > vmax:
      vmin, vmax = vmax, vmin

    # calculate minor ticks
    locs = np.array([])
    for index in range(len(majorlocs) - 1):
      # can treat linear and log axes the same
      nulocs = np.linspace(majorlocs[index], majorlocs[index + 1], ndivs + 1)
      locs = np.hstack((locs, nulocs[1: -1]))
    
    # run through conditions
    cond1 = locs >= vmin
    cond2 = locs <= vmax
    locs = locs.compress(cond1 & cond2)

    return self.raise_if_exceeds(np.array(locs))

class MyLassoli(Lasso):
  def __init__(self, *args, **kwargs):
    item = 'parent'
    if(item in kwargs):
      self.__dict__[item] = kwargs[item]
      del kwargs[item]
    else:
      self.__dict__[item] = None
    # normal constructor
    super(MyLassoli, self).__init__(*args, **kwargs)
    # ensure compatibility with matplotlib 3.5.0 and above
    if(not hasattr(self, 'line')):
      self.line = self._selection_artist
    self.line.set_linewidth(2.0)
    self.line.set_solid_capstyle('round')
    self.line.set_color([0.2, 0.15, 0.9, 1.0])
    # to coordinate crosstalk with rectangle selector
    # adjust line size acording to zoom level
    try:
      if(self.parent != None):
        currZoomLevel = self.parent.matplot.get_dpi()
        linewidth = 2.0 * 100.0 / currZoomLevel
        self.line.set_linewidth(linewidth)
        if(linewidth >= 4.0):
          self.line.set_path_effects([PathEffects.withStroke(linewidth=12.0, foreground=[1.0, 1.0, 1.0, 1.0])])
        elif(linewidth >= 2.0):
          self.line.set_path_effects([PathEffects.withStroke(linewidth=5.0, foreground=[1.0, 1.0, 1.0, 1.0])])
        elif(linewidth >= 1.0):
          self.line.set_path_effects([PathEffects.withStroke(linewidth=2.0, foreground=[1.0, 1.0, 1.0, 1.0])])
        else:
          self.line.set_path_effects([PathEffects.withStroke(linewidth=1.0, foreground=[1.0, 1.0, 1.0, 1.0])])
    except:
      pass

  # need to override onmove to get crosstalk with crosshair cursor right
  # other than that, copied from widgets.py
  def onmove(self, event):
    if (self.ignore(event)
        or self.verts is None
        or event.button != 1
        or not self.ax.contains(event)[0]):
      return
    # allow for different matplotlib versions
    try:
      self.verts.append(self._get_data_coords(event))
    except:
      self.verts.append((event.xdata, event.ydata))
    self.line.set_data(list(zip(*self.verts)))

    # only draw if cursor is hidden
    if((self.parent != None) and (not self.parent.cursorVisible)):
      if self.useblit:
        self.canvas.restore_region(self.background)
        self.ax.draw_artist(self.line)
        self.canvas.blit(self.ax.bbox)
      else:
        self.canvas.draw_idle()
    
  def drawMe(self):
    # draws the artist
    self.ax.draw_artist(self.line)
  
  def cleanUp(self):
    # remove lassoli after use
    # don't blit here to avoid crosstalk with rectangle selector
    #if(self.useblit):
    if(False):
      self.canvas.restore_region(self.background)
      self.canvas.blit(self.ax.bbox)
    else:
      self.canvas.draw_idle()

class MyRectangli(AxesWidget):
  # adapted from Lasso in matplotlib.widgets.py
  def __init__(self, ax, xy, callback, useblit=True, parent=None):
    super().__init__(ax)

    self.parent = parent
    self.useblit = useblit and self.canvas.supports_blit
    if self.useblit:
      self.background = self.canvas.copy_from_bbox(self.ax.bbox)

    x, y = xy
    self.verts = [(x, y)]
    self.rectangle = matplotlib.patches.Rectangle((0, 0), 0, 1, visible=True)
    self.ax.add_patch(self.rectangle)
    self.rectangle.set_linewidth(4.0)
    self.rectangle.set_linestyle('solid')
    self.rectangle.set_edgecolor([0, 0, 0, 1])
    self.rectangle.set_facecolor([0, 0, 1, .4])
    self.callback = callback
    self.connect_event('button_release_event', self.onrelease)
    self.connect_event('motion_notify_event', self.onmove)

    # adjust line size acording to zoom level
    try:
      if(self.parent != None):
        currZoomLevel = self.parent.matplot.get_dpi()
        linewidth = 2.0 * 100.0 / currZoomLevel
        self.rectangle.set_linewidth(linewidth)
        if(linewidth >= 4.0):
          self.rectangle.set_path_effects([PathEffects.withStroke(linewidth=12.0, foreground=[1.0, 1.0, 1.0, 1.0])])
        elif(linewidth >= 2.0):
          self.rectangle.set_path_effects([PathEffects.withStroke(linewidth=5.0, foreground=[1.0, 1.0, 1.0, 1.0])])
        elif(linewidth >= 1.0):
          self.rectangle.set_path_effects([PathEffects.withStroke(linewidth=2.0, foreground=[1.0, 1.0, 1.0, 1.0])])
        else:
          self.rectangle.set_path_effects([PathEffects.withStroke(linewidth=1.0, foreground=[1.0, 1.0, 1.0, 1.0])])
    except:
      pass

  def onrelease(self, event):
    if self.ignore(event):
        return
    if self.verts is not None:
      # allow for different matplotlib versions
      try:
        self.verts.append(self._get_data_coords(event))
      except:
        self.verts.append((event.xdata, event.ydata))
      if len(self.verts) > 2:
        self.callback(self.verts)
      self.rectangle.remove()
    self.verts = None
    self.disconnect_events()

  def onmove(self, event):
    if (self.ignore(event)
      or self.verts is None
      or event.button != 1
      or not self.ax.contains(event)[0]):
        return
    # allow for different matplotlib versions
    try:
      self.verts.append(self._get_data_coords(event))
    except:
      self.verts.append((event.xdata, event.ydata))
    if(len(self.verts) > 1):
      xy = (np.min((self.verts[0][0], self.verts[-1][0])), np.min((self.verts[0][1], self.verts[-1][1])))
      width, height = np.abs((self.verts[0][0] - self.verts[-1][0])), np.abs((self.verts[0][1] - self.verts[-1][1]))
      self.rectangle.set_xy(xy)
      self.rectangle.set_height(height)
      self.rectangle.set_width(width)

      # only draw if cursor is hidden
      if((self.parent != None) and (not self.parent.cursorVisible)):
        if self.useblit:
          self.canvas.restore_region(self.background)
          self.ax.draw_artist(self.rectangle)
          self.canvas.blit(self.ax.bbox)
        else:
          self.canvas.draw_idle()
        
  def drawMe(self):
    # draws the rectangle
    self.ax.draw_artist(self.rectangle)

  def cleanUp(self):
    # remove rectangli after use
    # don't blit here to avoid crosstalk
    #if(self.useblit):
    if(False):
      self.canvas.restore_region(self.background)
      self.canvas.blit(self.ax.bbox)
    else:
      self.canvas.draw_idle()

  def getMyExtents(self):
    # returns first and last element of self.vert
    if(len(self.verts) > 2):
      return self.verts[0], self.verts[-1]
    else:
      return None, None

class MatplotlibCanvas(QWidgetMac):
  def __init__(self, parent=None):
    super(MatplotlibCanvas, self).__init__()
    self.parent = parent
    
    # initialize param values
    self.initParam()

    # set initial matplotlib style
    originalFont = matplotlib.rcParams.get('mathtext.fontset')
    matplotlib.style.use(self.stylemodel)
    # ensure that Dejavu Sans is used for mathtext (rather use original font as dejavusans is platform/version-depedent)
    matplotlib.rc('mathtext', fontset=originalFont)
    
    plt.rcParams['axes.unicode_minus'] = False
    
    # a validator
    self.validFloat = MyValidFloat()
    
    # generate GUI elements
    self.buildRessource()
    
    # initialize plot
    self.initPlot(initialize=True)

    # toggle split axes
    self.toggleSplit(redraw=False, toggled=False)
    
    # update rulers
    self.horizontalRuler.updateRuler()
    self.verticalRuler.updateRuler()
    self.verticalRulerResid.updateRuler()
    
  def initParam(self):
    # allow some spacing around data when autoscaling
    self.data_spacer = 0.025
    
    # initialize some values
    self.x, self.y, self.y2 = np.array([]), np.array([]), np.array([])
    self.minX, self.maxX = 0.0, 1.0
    self.minY, self.maxY = 0.0, 1.0
    self.storeCoord = []
    self.minResidY, self.maxResidY = -0.5, 0.5
    self.modeX, self.modeY = 'linear', 'linear'
    self.EPSILON = 1e-6
    self.DATAPOINTS_SIMULATION = 2000
    self.autoScaleX, self.autoScaleY = True, True
    self.labelX = 'x'
    self.labelY = 'y'
    self.labelXShow, self.labelYShow = True, True
    self.labelXColor, self.labelYColor = [0.2, 0.2, 0.2, 1.0], [0.2, 0.2, 0.2, 1.0]
    self.labelXSize, self.labelYSize = 14.0, 14.0
    self.labelXWeight, self.labelYWeight = 'normal', 'normal'
    self.labelXStyle, self.labelYStyle = 'italic', 'italic'
    self.labelXVariant, self.labelYVariant = 'normal', 'normal'
    self.labelXPad, self.labelYPad = 4.0, 4.0
    self.labelXLinespacing, self.labelYLinespacing = 1., 1.
    self.labelXPos, self.labelYPos = 0.5, 0.5
    self.labelXAngle, self.labelYAngle = 0.0, 90.0
    self.labelXAlignment, self.labelYAlignment = 'center', 'center'
    self.labelXAlignmentVertical, self.labelYAlignmentVertical = 'top', 'baseline'
    self.axisVisible = {'left':True, 'right':True, 'bottom':True, 'top':True, 'left2':True, 'right2':True}
    self.axisWidth = {'left':1.0, 'right':1.0, 'bottom':1.0, 'top':1.0, 'left2':1.0, 'right2':1.0}
    self.axisStyle = {'left':'solid', 'right':'solid', 'bottom':'solid', 'top':'solid', 'left2':'solid', 'right2':'solid'}
    self.axisDashStyle = {'left':'butt', 'right':'butt', 'bottom':'butt', 'top':'butt', 'left2':'butt', 'right2':'butt'}
    self.axisColor = {'left':[0.2, 0.2, 0.2, 1.0], 'right':[0.2, 0.2, 0.2, 1.0], 'bottom':[0.2, 0.2, 0.2, 1.0],\
                      'top':[0.2, 0.2, 0.2, 1.0], 'left2':[0.2, 0.2, 0.2, 1.0], 'right2':[0.2, 0.2, 0.2, 1.0]}
    self.axisFont = {'x': 'DejaVu Sans', 'y': 'DejaVu Sans', 'y2': 'DejaVu Sans', 'xs': 'DejaVu Sans'}
    self.axisPosition = {'left':'axes', 'right':'axes', 'bottom':'axes', 'top':'axes', 'left2':'axes', 'right2':'axes'}
    self.axisPositionValue = {'left':0.0, 'right':1.0, 'bottom':0.0, 'top':1.0, 'left2':0.0, 'right2':1.0}
    self.axisBoundary = {'left':[0.0, 1.0], 'right':[0.0, 1.0], 'bottom':[0.0, 1.0], 'top':[0.0, 1.0], 'left2':[0.0, 1.0], 'right2':[0.0, 1.0]}
    self.axisBoundaryCheck = {'left':False, 'right':False, 'bottom':False, 'top':False, 'left2':False, 'right2':False}
    self.tickFont = {'x': 'DejaVu Sans', 'y': 'DejaVu Sans', 'y2': 'DejaVu Sans', 'xs': 'DejaVu Sans'}
    self.ticksVisible = {'left':True, 'right':True, 'bottom':True, 'top':True, 'left2':False, 'right2':True, 'slaved': True}
    self.ticksWidth = {'left':0.5, 'right':0.5, 'bottom':0.5, 'top':0.5, 'left2':0.5, 'right2':0.5, 'slaved': .5}
    self.ticksLength = {'left':5.0, 'right':5.0, 'bottom':5.0, 'top':5.0, 'left2':5.0, 'right2':5.0, 'slaved': 5.}
    self.ticksColor = {'left':[0.2, 0.2, 0.2, 1.0], 'right':[0.2, 0.2, 0.2, 1.0], 'bottom':[0.2, 0.2, 0.2, 1.0],\
                       'top':[0.2, 0.2, 0.2, 1.0], 'left2':[0.2, 0.2, 0.2, 1.0], 'right2':[0.2, 0.2, 0.2, 1.0], 'slaved': [0.9, 0.2, 0.2, 1.0]}
    self.ticksDirection = {'left':'in', 'right':'in', 'bottom':'in', 'top':'in', 'left2':'in', 'right2':'in', 'slaved': 'in'}
    self.ticksLabelShow = {'left': True, 'right': False, 'bottom': True, 'top': False, 'left2': False, 'right2': True, 'slaved': True}
    self.canvasColor = [0.9, 0.9, 0.9, 1.0]
    self.canvasFill, self.figureFill, self.frameDraw = True, True, False
    self.figureColor = [1.0, 1.0, 1.0, 1.0]
    self.ticksX, self.ticksY, self.ticksResidY = [], [], []
    self.ticksXShow, self.ticksYShow, self.ticksResidYShow = True, True, True
    self.ticksXAuto, self.ticksYAuto, self.ticksResidYAuto = True, True, True
    self.ticksXMinor, self.ticksYMinor, self.ticksY2Minor, self.ticksResidYMinor = 2, 2, 2, 2
    self.ticksXMinorRelativeLength, self.ticksYMinorRelativeLength, self.ticksY2MinorRelativeLength, self.ticksResidYMinorRelativeLength = 0.5, 0.5, 0.5, 0.5
    self.ticksXLabel, self.ticksXDataSet = [], -1
    self.ticksYLabel, self.ticksYDataSet = [], -1
    self.ticksXColor = [0.2, 0.2, 0.2, 1.0]
    self.ticksYColor = [0.2, 0.2, 0.2, 1.0]
    self.ticksXSize, self.ticksYSize = 12.0, 12.0
    self.ticksXWeight, self.ticksYWeight = 'normal', 'normal'
    self.ticksXStyle, self.ticksYStyle = 'normal', 'normal'
    self.ticksXAngle, self.ticksYAngle = 0.0, 0.0
    self.ticksXAlignment, self.ticksYAlignment = 'center', 'right'
    self.ticksXAlignmentVertical, self.ticksYAlignmentVertical = 'top', 'center'
    self.ticksXLinespacing, self.ticksYLinespacing = 1., 1.
    self.ticksXPad, self.ticksYPad = 4.0, 4.0
    self.ticksXPad2, self.ticksYPad2 = 0.0, 0.0
    self.ticksXFormat, self.ticksYFormat, self.ticksY2Format, self.ticksResidYFormat = 'default', 'default', 'default', 'default'
    self.ticksXFormatPrecision, self.ticksYFormatPrecision, self.ticksY2FormatPrecision, self.ticksResidYFormatPrecision = 2, 2, 2, 2
    self.ticksXFormatTrailZero, self.ticksYFormatTrailZero, self.ticksY2FormatTrailZero, self.ticksResidYFormatTrailZero = False, False, False, False
    self.ticksXFormatSeparator, self.ticksYFormatSeparator, self.ticksY2FormatSeparator, self.ticksResidYFormatSeparator = False, False, False, False
    self.ticksXFormatComma, self.ticksYFormatComma, self.ticksY2FormatComma, self.ticksResidYFormatComma = False, False, False, False
    self.ticksXFormatPrefix, self.ticksYFormatPrefix, self.ticksY2FormatPrefix, self.ticksResidYFormatPrefix = '', '', '', ''
    self.ticksXFormatPostfix, self.ticksYFormatPostfix, self.ticksY2FormatPostfix, self.ticksResidYFormatPostfix = '', '', '', ''
    self.fallback_ticksXFormat, self.fallback_ticksYFormat, self.fallback_ticksY2Format, self.fallback_ticksResidYFormat = None, None, None, None
    self.gridVisible = {'x': False, 'x2': False, 'y': False, 'y2': False}
    self.gridLinesStore = {'x': [], 'x2': [], 'y': [], 'y2': []}
    self.gridLinesStore_resid = {'x': [], 'x2': [], 'y': [], 'y2': []}
    self.gridWidth = {'x': 0.5, 'x2': 0.5, 'y': 0.5, 'y2': 0.5}
    self.gridStyle = {'x': 'solid', 'x2': 'solid', 'y': 'solid', 'y2': 'solid'}
    self.gridDashStyle = {'x': 'butt', 'x2': 'butt', 'y': 'butt', 'y2': 'butt'}
    self.gridColor = {'x': [0.2, 0.2, 0.2, 1.0], 'x2': [0.2, 0.2, 0.2, 1.0], 'y': [0.2, 0.2, 0.2, 1.0], 'y2': [0.2, 0.2, 0.2, 1.0]}
    self.gridOrder = {'x': 'back', 'x2': 'back', 'y': 'back', 'y2': 'back'}
    self.gridMinorVisible = {'x': False, 'x2': False, 'y': False, 'y2': False}
    self.gridMinorLinesStore = {'x': [], 'x2': [], 'y': [], 'y2': []}
    self.gridMinorLinesStore_resid = {'x': [], 'x2': [], 'y': [], 'y2': []}
    self.gridMinorWidth = {'x': 0.5, 'x2': 0.5, 'y': 0.5, 'y2': 0.5}
    self.gridMinorStyle = {'x': 'dashed', 'x2': 'dashed', 'y': 'dashed', 'y2': 'dashed'}
    self.gridMinorDashStyle = {'x': 'butt', 'x2': 'butt', 'y': 'butt', 'y2': 'butt'}
    self.gridMinorColor = {'x': [0.2, 0.2, 0.2, 1.0], 'x2': [0.2, 0.2, 0.2, 1.0], 'y': [0.2, 0.2, 0.2, 1.0], 'y2': [0.2, 0.2, 0.2, 1.0]}
    self.gridMinorOrder = {'x': 'back', 'x2': 'back', 'y': 'back', 'y2': 'back'}
    self.gridRectStore = {'x': [], 'x2': [], 'y': [], 'y2': []}
    self.gridRectStore_resid = {'x': [], 'x2': [], 'y': [], 'y2': []}
    self.gridRectVisible = {'x': False, 'x2': False, 'y': False, 'y2': False}
    self.gridRectColor = {'x': [0.2, 0.2, 0.2, .2], 'x2': [0.2, 0.2, 0.2, .2], 'y': [0.2, 0.2, 0.2, .2], 'y2': [0.2, 0.2, 0.2, .2]}
    
    self.exportWidth, self.exportHeight = 8.0, 6.0
    self.padSize = {'left': 0.15, 'right': 0.95, 'bottom': 0.15, 'top': 0.95}
    self.frameWidth, self.frameColor, self.frameStyle, self.frameDashStyle = 0.5, [0.2, 0.2, 0.2, 1.0], 'solid', 'butt'
    self.visibilityResidLine = True
    self.zorderResidLine = 1
    self.legendVisible = True
    self.legendPlacement = 'best'
    self.placementStyles = 'best;upper right;upper left;lower left;lower right;right;center left;center right;lower center;upper center;center;custom'.split(';')
    self.legendX, self.legendY = 0.5, 0.5
    self.legendColor = {'face':[1.0, 1.0, 1.0, 0.5], 'edge':[0.2, 0.2, 0.2, 1.0]}
    self.legendEdgeWidth = 0.5
    self.legendShadow = False
    self.legendShadowDeltaX, self.legendShadowDeltaY, self.legendShadowFaceColor = 2, -2, [0.5, 0.5, 0.5, 1.0]
    self.legendLabelColor = [0.0, 0.0, 0.0, 1.0]
    self.legendLabelSize = 14
    self.legendLabelLinespacing = 1.
    self.legendLabelWeight, self.legendLabelStyle = 'normal', 'normal'
    self.legendLabelFont = 'DejaVu Sans'
    self.legendNumPoints = 1
    self.legendMarkerFirst = True
    self.legendNumCol = 1
    self.legendMarkerScale = 1.0
    self.legendBorderPad = 0.4
    self.legendLabelSpacing = 0.4
    self.legendLabelPad = 1.
    self.legendColumnSpacing = 2
    self.legendFill, self.legendEdge = True, True
    self.legendHatch, self.legendHatchMultiply, self.legendHatchLinewidth, self.legendRounding = '', 1, 1., 0.2
    self.xkcd = False
    self.xkcdScale, self.xkcdLength, self.xkcdRandomness = 1.0, 100.0, 2.0
    self.xkcdStoreFonts = ['DejaVu Sans']
    self.applyPathStroke = False
    self.pathStrokeWidth, self.pathStrokeColor = 2.0, [1.0, 1.0, 1.0, 1.0]
    self.applyPathShadow = False
    self.pathShadowX, self.pathShadowY, self.pathShadowColor, self.pathShadowAlpha = 2, -2, [0.4, 0.4, 0.4, 0.5], 0.5
    self.pathRhoCheck, self.pathRho = True, 0.5
    self.handleArrow = {'x': None, 'y': None, 'y2': None}
    self.handleArrowResid = {'x': None, 'y': None}
    self.handleArrow2 = {'x': None, 'y': None, 'y2': None}
    self.handleArrowResid2 = {'x': None, 'y': None}
    self.arrowVisible = {'x': False, 'y': False, 'y2': False}
    self.arrowOverhang = {'x': 0.1, 'y': 0.1, 'y2': 0.1}
    self.arrowColor = {'x': [0.2, 0.2, 0.2, 1.0], 'y': [0.2, 0.2, 0.2, 1.0], 'y2': [0.2, 0.2, 0.2, 1.0]}
    self.arrowFill = {'x': [0.2, 0.2, 0.2, 1.0], 'y': [0.2, 0.2, 0.2, 1.0], 'y2': [0.2, 0.2, 0.2, 1.0]}
    self.arrowHeadLength = {'x': 0.2, 'y': 0.2, 'y2': 0.2}
    self.arrowHeadWidth = {'x': 0.2, 'y': 0.2, 'y2': 0.2}
    self.arrowEdge = {'x': 0.5, 'y': 0.5, 'y2': 0.5}
    self.arrowOffset = {'x': 0.2, 'y': 0.2, 'y2': 0.2}
    self.arrowEdgeShow = {'x': True, 'y': True, 'y2': True}
    self.arrowFillShow = {'x': True, 'y': True, 'y2': True}
    self.arrowLocation = {'x': 'bottom', 'y': 'left', 'y2': 'right'}
    self.arrowZ = {'x': 'axis', 'y': 'axis', 'y2': 'axis'}
    self.tickLabelData = False
    self.cursorVisible = False
    self.cursor = None
    self.pickedExtra, self.pickedAxes, self.pickedBackground, self.pickedMode, self.inAxes = None, None, None, 0, None
    self.pickedLegend = False
    
    # second y axis
    self.minY2, self.maxY2 = 0.0, 1.0
    self.modeY2 = 'linear'
    self.autoScaleY2 = True
    self.ticksY2 = []
    self.ticksY2Show = True
    self.ticksY2Auto = True
    self.ticksY2Color = [0.2, 0.2, 0.2, 1.0]
    self.ticksY2Size = 12.0
    self.ticksY2Weight = 'normal'
    self.ticksY2Style = 'normal'
    self.ticksY2Angle = 0.0
    self.ticksY2Alignment = 'left'
    self.ticksY2AlignmentVertical = 'center'
    self.ticksY2Linespacing = 1.
    self.ticksY2Pad, self.ticksY2Pad2 = 4.0, 0.0
    self.ticksY2Label, self.ticksY2DataSet = [], -1
    self.labelY2 = 'y'
    self.labelY2Show = True
    self.labelY2Color = [0.2, 0.2, 0.2, 1.0]
    self.labelY2Size = 14.0
    self.labelY2Weight = 'normal'
    self.labelY2Style = 'italic'
    self.labelY2Variant = 'normal'
    self.labelY2Pad = 4.0
    self.labelY2Pos = 0.5
    self.labelY2Angle = 90.0
    self.labelY2Alignment = 'center'
    self.labelY2AlignmentVertical = 'top'
    self.labelY2Linespacing = 1.
    
    self.splitShow, self.splitFraction, self.splitPad = False, 1.0, 0.1
    self.splitDivider, self.splitDividerLength, self.splitDividerWidth, self.splitDividerLocation = True, 0.07, 0.5, 'both'
    self.splitDividerColor, self.splitDividerDashStyle, self.splitDividerAngle = [0.2, 0.2, 0.2, 1.0], 'round', 60
    self.handleDivider, self.handleDividerResid, self.handleDividerY = None, None, None
    self.minX_div, self.maxX_div, self.modeX_div = 0, 1.0, 'linear'
    self.innerAxes, self.innerTicks = False, False
    self.ticksX_div, self.ticksXLabel_div, self.ticksXAuto_div, self.ticksXShow_div, self.ticksXShow_resid_div = [], [], True, True, True
    self.ticksXFormat_div, self.ticksXFormatPrecision_div, self.fallback_ticksXFormat_div, self.ticksXFormatTrailZero_div, self.ticksXFormatSeparator_div, self.ticksXFormatComma_div = 'default', 2, None, False, False, False
    self.ticksXFormatPrefix_div, self.ticksXFormatPostfix_div = '', ''
    self.cursor_div, self.cursor2, self.cursor2_div = None, None, None
    self.autoScaleX_div = True
    self.ticksXMinor_div = 2
    
    self.canvasGradientVisible = False
    self.canvasGradient = []
    self.canvasGradientStyle = 'linear'
    self.canvasGradientColor1, self.canvasGradientColor2 = [0.8, 0.8, 0.8, 1.0], [0.2, 0.2, 0.2, 1.0]
    self.canvasGradientAngle, self.canvasGradientWidth, self.canvasGradientCenter = 0.0, 0.5, [0.5, 0.5]
    
    # controls for split y axis
    self.splitY = False
    self.splitYFraction, self.splitYPad = 1.0, 0.03
    self.innerAxesY, self.innerTicksY = False, False
    self.splitYDivider, self.splitYDividerLength, self.splitYDividerWidth, self.splitYDividerLocation = True, 0.07, 0.5, 'both'
    self.splitYDividerColor, self.splitYDividerDashStyle, self.splitYDividerAngle = [0.2, 0.2, 0.2, 1.0], 'round', 30
    self.splitYBackup = {}
    
    # parameters for second x axes labels
    self.secondXLines, self.secondXLabels = [], []
    self.secondXLines_div, self.secondXLabels_div = [], []
    self.secondXLinesResid, self.secondXLabelsResid = [], []
    self.secondXLinesResid_div, self.secondXLabelsResid_div = [], []
    self.slavedX = False
    self.handleSlavedXLabel, self.handleSlavedXLabelResid = None, None
    self.slavedXLabel, self.slavedXLabelShow = 'transformed x', True
    self.slavedXTransform = []
    self.slavedLabelXColor, self.slavedLabelXSize, self.slavedLabelXWeight, self.slavedLabelXStyle = [0.2, 0.2, 0.2, 1.0], 14., 'normal', 'italic'
    self.slavedLabelXVariant, self.slavedLabelXPad, self.slavedLabelXLinespacing, self.slavedLabelXPos = 'normal', 4., 1., .5
    self.slavedLabelXAngle, self.slavedLabelXAlignment, self.slavedLabelXAlignmentVertical = 0.0, 'center', 'bottom'
    self.slavedTicksXMinor, self.slavedTicksXMinorRelativeLength = 2, 0.5
    self.slavedTicksXColor, self.slavedTicksXSize, self.slavedTicksXWeight, self.slavedTicksXStyle = [0.2, 0.2, 0.2, 1.0], 12., 'normal', 'normal'
    self.slavedTicksXAngle, self.slavedTicksXAlignment, self.slavedTicksXAlignmentVertical = 0., 'center', 'bottom'
    self.slavedTicksXLinespacing, self.slavedTicksXPad, self.slavedTicksXPad2 = 1.0, 4.0, 0.
    self.slavedXFormat, self.slavedXFormatPrecision, self.slavedXFormatTrailZero, self.slavedXFormatSeparator = 'default', 2, False, False
    self.slavedXFormatComma, self.slavedXFormatPrefix, self.slavedXFormatPostfix = False, '', ''
    self.slavedXFormatter, self.fallback_slavedXFormat = None, None
    self.slavedTicksX, self.slavedTicksXAuto = [], True
    
    # initialize values for toggleCrossHairEncore()
    self.toggleShift, self.toggleCtrl = False, False
    
    # interactive drawing of freehand shape
    self.drawFreehand = False
    self.pick2Freehand = False
    
    # store UI style here (does not really belong but put here for convenience to allow style updating)
    self.themeUI = 'default'
    
    # temporarily store selector and cursors to resolve blit shenanigans under recent matplotlib versions
    self.selector2Restore, self.cursors2Restore = None, []

    # generate copies of certain settings for improved drawing control for resid window
    items = 'axisVisible,axisWidth,axisStyle,axisDashStyle,axisPosition,axisPositionValue,axisBoundaryCheck,axisBoundary,axisColor,labelX,labelY,labelXShow,labelYShow'.split(',')
    items.extend('labelXColor,labelYColor,labelXSize,labelYSize,labelXWeight,labelYWeight,labelXStyle,labelYStyle,labelXVariant,labelYVariant'.split(','))
    items.extend('labelXPad,labelYPad,labelXPos,labelYPos,labelXAlignment,labelYAlignment,labelXAlignmentVertical,labelYAlignmentVertical,labelXLinespacing,labelYLinespacing,labelXAngle,labelYAngle,axisFont'.split(','))
    items.extend('tickFont,ticksVisible,ticksWidth,ticksLength,ticksColor,ticksDirection,ticksLabelShow'.split(','))
    items.extend('ticksXShow,ticksYShow,ticksXColor,ticksYColor,ticksXSize,ticksYSize,ticksXWeight,ticksYWeight,ticksXStyle,ticksYStyle'.split(','))
    items.extend('ticksXAngle,ticksYAngle,ticksXMinorRelativeLength'.split(','))
    items.extend('ticksXAlignment,ticksYAlignment,ticksXAlignmentVertical,ticksYAlignmentVertical,ticksXLinespacing,ticksYLinespacing,ticksXPad,ticksYPad,ticksXPad2,ticksYPad2'.split(','))
    items.extend('gridVisible,gridWidth,gridStyle,gridDashStyle,gridColor,gridOrder,padSize,frameWidth,frameColor,frameStyle,frameDashStyle,frameDraw'.split(','))
    items.extend('gridMinorVisible,gridMinorWidth,gridMinorStyle,gridMinorDashStyle,gridMinorColor,gridMinorOrder,gridRectVisible,gridRectColor'.split(','))
    items.extend('canvasGradientStyle,canvasGradientColor1,canvasGradientColor2,canvasGradientAngle,canvasGradientWidth,canvasGradientCenter'.split(','))
    items.extend('arrowEdgeShow,arrowFillShow,arrowLocation,arrowZ'.split(','))
    items.extend('slavedX,slavedXLabel,slavedXLabelShow,slavedXTransform'.split(','))
    items.extend('slavedLabelXColor,slavedLabelXSize,slavedLabelXWeight,slavedLabelXStyle'.split(','))
    items.extend('slavedLabelXVariant,slavedLabelXPad,slavedLabelXLinespacing,slavedLabelXPos'.split(','))
    items.extend('slavedLabelXAngle,slavedLabelXAlignment,slavedLabelXAlignmentVertical'.split(','))
    items.extend('slavedTicksXMinor,slavedTicksXMinorRelativeLength,slavedTicksXColor,slavedTicksXSize,slavedTicksXWeight'.split(','))
    items.extend('slavedTicksXStyle,slavedTicksXAngle,slavedTicksXAlignment,slavedTicksXAlignmentVertical'.split(','))
    items.extend('slavedTicksXLinespacing,slavedTicksXPad,slavedTicksXPad2'.split(','))
    items.extend('slavedXFormat,slavedXFormatPrecision,slavedXFormatTrailZero,slavedXFormatSeparator'.split(','))
    items.extend('slavedXFormatComma,slavedXFormatPrefix,slavedXFormatPostfix'.split(','))
    for entry in items:
      self.__dict__[entry + '_resid'] = deepcopy(self.__dict__[entry])

    if('bmh' in matplotlib.style.available):
      self.stylemodel = 'bmh'
    else:
      self.stylemodel = matplotlib.style.available[0]
      
    # pick event handling
    self.lastPickEvent = None
            
  def buildRessource(self):
    # set up matplotlib and canvas
    self.vLayout_master = QtWidgets.QVBoxLayout(self)
    self.vLayout_master.setContentsMargins(0, 0, 0, 0)

    self.masterwidget = QtWidgets.QSplitter(QtCore.Qt.Orientation.Vertical)
    self.vLayout_master.addWidget(self.masterwidget)
    self.masterwidget.setSizePolicy(QtWidgets.QSizePolicy.Policy.Expanding, QtWidgets.QSizePolicy.Policy.Expanding)
    self.masterwidget.setChildrenCollapsible(False)
    self.masterwidget.setOpaqueResize(False)
    
    self.mainPlotBox = QWidgetMac()
    self.masterwidget.addWidget(self.mainPlotBox)
    self.hLayout = QtWidgets.QHBoxLayout(self.mainPlotBox)
    self.hLayout.setContentsMargins(0, 0, 0, 0)
    self.hLayout.setSpacing(1)
    
    # controls for y-axis
    self.yControlBox = QWidgetMac(self)
    self.yControlBox.setGeometry(QtCore.QRect(0, 0, int(50 * SCALEFONT), 500))
    self.yControlBox.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), 16777215))
    self.yControlBox.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(200 * SCALEFONT)))

    self.hLayout.addWidget(self.yControlBox)
    self.vLayout = QtWidgets.QVBoxLayout(self.yControlBox)
    self.vLayout.setContentsMargins(0, 0, 0, 0)

    self.SpacerBox0 = QWidgetMac(self)
    self.SpacerBox0.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.SpacerBox0.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.vLayout.addWidget(self.SpacerBox0)
    
    # controls for main plot
    self.yControlsPlotContainer = QWidgetMac(self)
    self.vLayout.addWidget(self.yControlsPlotContainer)
    self.LayoutYControlsPlotContainer = QtWidgets.QVBoxLayout(self.yControlsPlotContainer)
    self.LayoutYControlsPlotContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutYControlsPlotContainer.setSpacing(1)
    
    self.autoScaleBoxY = QWidgetMac(self)
    self.autoScaleBoxY.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.autoScaleBoxY.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.Layout_ScaleBoxY = QtWidgets.QHBoxLayout(self.autoScaleBoxY)
    self.Layout_ScaleBoxY.setContentsMargins(0, 0, 0, 0)
    self.Layout_ScaleBoxY.setSpacing(1)
    self.Layout_ScaleBoxY.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft | QtCore.Qt.AlignmentFlag.AlignTop)
    self.autoScaleButtonY = QPushButtonMac()
    self.autoScaleButtonY.setText('Auto')
    self.autoScaleButtonY.setToolTip('Scale y axis to current data set and/or curve')
    self.autoScaleButtonY.setMinimumHeight(int(BASE_SIZE))
    self.autoScaleButtonY.setMaximumHeight(int(BASE_SIZE))
    self.autoScaleButtonY.clicked.connect(partial(self.doAutoScale, 'y', True))
    self.Layout_ScaleBoxY.addWidget(self.autoScaleButtonY)
    self.autoScaleCheckY = QPushButtonCheckable()
    self.autoScaleCheckY.setToolTip('Automatically scale y axis')
    self.autoScaleCheckY.setCheckMe(True)
    self.autoScaleCheckY.setChecked(self.autoScaleY)
    self.autoScaleCheckY.setMaximumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.autoScaleCheckY.setMinimumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.autoScaleCheckY.clicked.connect(partial(self.setAutoScale, 'y'))
    self.Layout_ScaleBoxY.addWidget(self.autoScaleCheckY)
    self.LayoutYControlsPlotContainer.addStretch()
    self.LayoutYControlsPlotContainer.addWidget(self.autoScaleBoxY)
    
    self.upperLimity = QLineEditClick()
    self.upperLimity.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.upperLimity.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.upperLimity.setValidator(self.validFloat)
    self.upperLimity.setText(str(self.parent.formatNumber(self.maxY)))
    self.upperLimity.setToolTip('Upper limit of y axis')
    self.upperLimity.editingFinished.connect(partial(self.changeAxisLimits, 'y', 'plot', True))
    self.LayoutYControlsPlotContainer.addStretch()
    self.LayoutYControlsPlotContainer.addWidget(self.upperLimity)

    self.modeSelectory = QComboBoxMac()
    self.modeSelectory.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.modeSelectory.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.LayoutYControlsPlotContainer.addStretch()
    self.LayoutYControlsPlotContainer.addWidget(self.modeSelectory)
    self.modeSelectory.addItem('linear')
    self.modeSelectory.addItem('log')
    self.modeSelectory.currentIndexChanged.connect(partial(self.changeAxisMode, 'y', True))
    self.modeSelectory.setToolTip('Set scale of y axis')
    
    self.lowerLimity = QLineEditClick()
    self.lowerLimity.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.lowerLimity.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.lowerLimity.setValidator(self.validFloat)
    self.lowerLimity.setText(str(self.parent.formatNumber(self.minY)))
    self.lowerLimity.setToolTip('Lower limit of y axis')
    self.lowerLimity.editingFinished.connect(partial(self.changeAxisLimits, 'y', 'plot', True))
    self.LayoutYControlsPlotContainer.addStretch()
    self.LayoutYControlsPlotContainer.addWidget(self.lowerLimity)

    self.flipAxisButtonY = QPushButtonCheckable()
    self.flipAxisButtonY.setText('Flip')
    self.flipAxisButtonY.setToolTip('Swap upper and lower limits of y axis')
    self.flipAxisButtonY.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonY.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonY.clicked.connect(partial(self.flipAxis, 'y', redraw=True))
    self.LayoutYControlsPlotContainer.addStretch()
    self.LayoutYControlsPlotContainer.addWidget(self.flipAxisButtonY)

    self.LayoutYControlsPlotContainer.addStretch()
    self.splitYButton = QPushButtonCheckable()
    self.splitYButton.setText('Split')
    self.splitYButton.setToolTip('Toggle split y axis')
    self.splitYButton.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.splitYButton.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.splitYButton.setChecked(self.splitY)
    self.LayoutYControlsPlotContainer.addWidget(self.splitYButton)
    self.splitYButton.clicked.connect(self.toggleSplitYCallback)
    blah = HLine()
    self.LayoutYControlsPlotContainer.addWidget(blah)

    # middle box
    self.mainMiddleBox = QWidgetMac(self)
    self.hLayout.addWidget(self.mainMiddleBox)
    self.vLayout_mainMiddleBox = QtWidgets.QVBoxLayout(self.mainMiddleBox)
    self.vLayout_mainMiddleBox.setContentsMargins(0, 0, 0, 0)
    self.vLayout_mainMiddleBox.setSpacing(1)
    
    # horizontal ruler
    self.horizontalRulerBox = QWidgetMac(self)
    self.horizontalRulerBox.setMinimumHeight(int(BASE_SIZE))
    self.horizontalRulerBox.setMaximumHeight(int(BASE_SIZE))
    self.vLayout_mainMiddleBox.addWidget(self.horizontalRulerBox)
    self.LayoutHorizontalRulerBox = QtWidgets.QHBoxLayout(self.horizontalRulerBox)
    self.LayoutHorizontalRulerBox.setContentsMargins(0, 0, 0, 0)
    self.LayoutHorizontalRulerBox.setSpacing(1)
    
    self.horizontalRuler = Ruler(parent=self, mode=0)
    self.horizontalRuler.setMinimumHeight(int(BASE_SIZE))
    self.horizontalRuler.setMaximumHeight(int(BASE_SIZE))
    self.LayoutHorizontalRulerBox.addWidget(self.horizontalRuler)
    
    self.SpacerBox1 = QWidgetMac(self)
    self.SpacerBox1.setMaximumSize(QtCore.QSize(int(BASE_SIZE), int(BASE_SIZE)))
    self.SpacerBox1.setMinimumSize(QtCore.QSize(int(BASE_SIZE), int(BASE_SIZE)))
    self.LayoutHorizontalRulerBox.addWidget(self.SpacerBox1)

    # container for plot and residuals
    self.plotContainer = ShrinkoWidget(notify=self, parent=self)
    self.LayoutPlotContainer = QtWidgets.QHBoxLayout(self.plotContainer)
    self.LayoutPlotContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutPlotContainer.setSpacing(1)
    self.vLayout_mainMiddleBox.addWidget(self.plotContainer)
    
    # the actual matplotlib canvas
    self.matplot = plt.figure()
    self.originalPlotDPI = self.matplot.get_dpi()
    zoomDifference = [abs(int(self.originalPlotDPI) - i) for i in self.parent.zoomLevels]
    zoomIndex = zoomDifference.index(min(zoomDifference))
    self.matplot.set_dpi(self.parent.zoomLevels[zoomIndex])
    self.dataplotwidget = MyFigureCanvas(self.matplot, self, 'plot')
    self.LayoutPlotContainer.addWidget(self.dataplotwidget)

    # vertical ruler
    self.verticalRuler = Ruler(parent=self, mode=1)
    self.verticalRuler.setMinimumWidth(int(BASE_SIZE))
    self.verticalRuler.setMaximumWidth(int(BASE_SIZE))
    self.LayoutPlotContainer.addWidget(self.verticalRuler)
 
    # controls for resid plot
    self.residPlotBox = ShrinkoWidget(notify=self, notifyResid=True, parent=self)
    self.masterwidget.addWidget(self.residPlotBox)
    self.hLayout_resid = QtWidgets.QHBoxLayout(self.residPlotBox)
    self.hLayout_resid.setContentsMargins(0, 0, 0, 0)
    self.hLayout_resid.setSpacing(1)
    
    # on Windows and Linux add markers for splitter
    if(not (platform == 'darwin')):
      self.addSplitterHandleSymbols(parentHandle=self.masterwidget, direction='horizontal', number=1, repeat=5)

    self.yControlsResidContainer = QWidgetMac(self)
    self.yControlsResidContainer.setMaximumWidth(int(50 * SCALEFONT))
    self.hLayout_resid.addWidget(self.yControlsResidContainer)
    self.LayoutYControlsResidContainer = QtWidgets.QVBoxLayout(self.yControlsResidContainer)
    self.LayoutYControlsResidContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutYControlsResidContainer.setSpacing(1)
    
    self.upperLimitResidy = QLineEditClick()
    self.upperLimitResidy.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.upperLimitResidy.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.upperLimitResidy.setValidator(self.validFloat)
    self.upperLimitResidy.setText(str(self.parent.formatNumber(self.maxResidY)))
    self.upperLimitResidy.setToolTip('Upper limit of residuals y axis')
    self.upperLimitResidy.editingFinished.connect(partial(self.changeAxisLimits, 'y', 'resid', True))
    self.LayoutYControlsResidContainer.addWidget(self.upperLimitResidy)
    
    self.autoScaleButtonResidY = QPushButtonMac()
    self.autoScaleButtonResidY.setText('Auto')
    self.autoScaleButtonResidY.setToolTip('Scale y axis to current residuals object')
    self.autoScaleButtonResidY.setMinimumHeight(int(BASE_SIZE))
    self.autoScaleButtonResidY.setMaximumHeight(int(BASE_SIZE))
    self.autoScaleButtonResidY.clicked.connect(partial(self.doAutoScale, 'resid', True))
    self.LayoutYControlsResidContainer.addWidget(self.autoScaleButtonResidY)

    self.lowerLimitResidy = QLineEditClick()
    self.lowerLimitResidy.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.lowerLimitResidy.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.lowerLimitResidy.setValidator(self.validFloat)
    self.lowerLimitResidy.setText(str(self.parent.formatNumber(self.minResidY)))
    self.lowerLimitResidy.setToolTip('Lower limit of residuals y axis')
    self.lowerLimitResidy.editingFinished.connect(partial(self.changeAxisLimits, 'y', 'resid', True))
    self.LayoutYControlsResidContainer.addWidget(self.lowerLimitResidy)

    # define the plot for the residuals
    plt.rcParams.update({'svg.fonttype': 'none'})
    self.residplot = plt.figure()
    self.residplot.set_dpi(self.parent.zoomLevels[zoomIndex])

    # residuals middle box
    self.residMiddleBox = QWidgetMac(self)
    self.hLayout_resid.addWidget(self.residMiddleBox)
    self.vLayout_residMiddleBox = QtWidgets.QVBoxLayout(self.residMiddleBox)
    self.vLayout_residMiddleBox.setContentsMargins(0, 0, 0, 0)
    self.vLayout_residMiddleBox.setSpacing(1)
    
    # set up canvas for the residuals
    self.residContainer = QWidgetMac()
    self.vLayout_residMiddleBox.addWidget(self.residContainer)
    self.LayoutResidContainer = QtWidgets.QHBoxLayout(self.residContainer)
    self.LayoutResidContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutResidContainer.setSpacing(1)
    self.residplotwidget = MyFigureCanvas(self.residplot, self, 'resid')
    self.LayoutResidContainer.addWidget(self.residplotwidget)
    
    # vertical ruler resid
    self.verticalRulerResid = Ruler(parent=self, mode=1, resid=1)
    self.verticalRulerResid.setMinimumWidth(int(BASE_SIZE))
    self.verticalRulerResid.setMaximumWidth(int(BASE_SIZE))
    self.hLayout_resid.addWidget(self.verticalRulerResid)
    
    # controls for x-axis
    self.xControlBox = QWidgetMac(self)
    self.xControlBox.setGeometry(QtCore.QRect(0, 0, int(500 * SCALEFONT), int(BASE_SIZE + 1)))
    self.xControlBox.setMaximumSize(QtCore.QSize(16777215, int(BASE_SIZE + 1)))
    self.xControlBox.setMinimumSize(QtCore.QSize(int(200 * SCALEFONT), int(BASE_SIZE + 1)))
    self.xControlBox.setContentsMargins(0, 0, 0, 0)
    self.vLayout_master.addWidget(self.xControlBox)
    self.hLayout2 = QtWidgets.QHBoxLayout(self.xControlBox)
    self.hLayout2.setContentsMargins(0, 0, 0, 0)
    self.hLayout2.setSpacing(2)

    self.SpacerBox_xctrl = QWidgetMac(self)
    self.SpacerBox_xctrl.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.SpacerBox_xctrl.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout2.addWidget(self.SpacerBox_xctrl)

    self.clearButton = QPushButtonMac()
    self.clearButton.setText(' Clear Selection')
    self.clearButton.setToolTip('Clear data selection')
    self.clearButton.setMaximumSize(QtCore.QSize(int(100 * SCALEFONT), int(BASE_SIZE)))
    self.clearButton.setMinimumSize(QtCore.QSize(int(100 * SCALEFONT), int(BASE_SIZE)))
    self.clearButton.clicked.connect(self.clearSelection)
    self.clearButton.setIcon(FOM_ICON_DIALOG_CANCEL_BUTTON)
    self.hLayout2.addWidget(self.clearButton)
    self.hLayout2.addStretch()
    self.clearButton.setEnabled(False)

    self.autoScaleBoxX = QWidgetMac(self)
    self.autoScaleBoxX.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.autoScaleBoxX.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout2.addWidget(self.autoScaleBoxX)
    self.Layout_ScaleBoxX = QtWidgets.QHBoxLayout(self.autoScaleBoxX)
    self.Layout_ScaleBoxX.setContentsMargins(0, 0, 0, 0)
    self.Layout_ScaleBoxX.setSpacing(1)
    self.Layout_ScaleBoxX.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.autoScaleButtonX = QPushButtonMac()
    self.autoScaleButtonX.setText('Auto')
    self.autoScaleButtonX.setToolTip('Scale x axis to current data set')
    self.autoScaleButtonX.setMinimumHeight(int(BASE_SIZE))
    self.autoScaleButtonX.setMaximumHeight(int(BASE_SIZE))
    self.autoScaleButtonX.clicked.connect(partial(self.doAutoScale, 'x', True))
    self.Layout_ScaleBoxX.addWidget(self.autoScaleButtonX)
    self.autoScaleCheckX = QPushButtonCheckable()
    self.autoScaleCheckX.setToolTip('Automatically scale x axis')
    self.autoScaleCheckX.setCheckMe(True)
    self.autoScaleCheckX.setChecked(self.autoScaleX)
    self.autoScaleCheckX.setMaximumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.autoScaleCheckX.setMinimumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.autoScaleCheckX.clicked.connect(partial(self.setAutoScale, 'x'))
    self.Layout_ScaleBoxX.addWidget(self.autoScaleCheckX)

    self.lowerLimitx = QLineEditClick()
    self.lowerLimitx.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.lowerLimitx.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.lowerLimitx.setValidator(self.validFloat)
    self.lowerLimitx.setText(str(self.parent.formatNumber(self.minX)))
    self.lowerLimitx.setToolTip('Lower limit of x axis')
    self.lowerLimitx.editingFinished.connect(partial(self.changeAxisLimits, 'x', 'plot', True))
    self.hLayout2.addWidget(self.lowerLimitx)

    self.modeSelectorx = QComboBoxMac()
    self.modeSelectorx.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.modeSelectorx.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout2.addWidget(self.modeSelectorx)
    self.modeSelectorx.addItem('linear')
    self.modeSelectorx.addItem('log')
    self.modeSelectorx.currentIndexChanged.connect(partial(self.changeAxisMode, 'x', True))
    self.modeSelectorx.setToolTip('Set scale of x axis')
    
    self.upperLimitx = QLineEditClick()
    self.upperLimitx.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.upperLimitx.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.upperLimitx.setValidator(self.validFloat)
    self.upperLimitx.setText(str(self.parent.formatNumber(self.maxX)))
    self.upperLimitx.setToolTip('Upper limit of x axis')
    self.upperLimitx.editingFinished.connect(partial(self.changeAxisLimits, 'x', 'plot', True))
    self.hLayout2.addWidget(self.upperLimitx)
    
    self.flipAxisButtonX = QPushButtonCheckable()
    self.flipAxisButtonX.setText('Flip')
    self.flipAxisButtonX.setToolTip('Swap upper and lower limits of x axis')
    self.flipAxisButtonX.setMaximumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonX.setMinimumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonX.clicked.connect(partial(self.flipAxis, 'x', redraw=True))
    self.hLayout2.addWidget(self.flipAxisButtonX)

    self.firstSplitBox = QWidgetMac()
    self.firstSplitBox.setMaximumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.firstSplitBox.setMinimumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout2.addWidget(self.firstSplitBox)
    self.LayoutFirstSplitBox = QtWidgets.QHBoxLayout(self.firstSplitBox)
    self.LayoutFirstSplitBox.setContentsMargins(0, 0, 0, 0)

    self.splitAxisButton = QPushButtonCheckable()
    self.splitAxisButton.setText('Split')
    self.splitAxisButton.setToolTip('Toggle split x axis')
    self.splitAxisButton.setMaximumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.splitAxisButton.setMinimumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.splitAxisButton.setChecked(self.splitShow)
    self.LayoutFirstSplitBox.addWidget(self.splitAxisButton)
    self.splitAxisButton.clicked.connect(partial(self.toggleSplit, True, True))
    
    # controls for split x-axis
    self.xSplitControlBox = QWidgetMac(self)
    self.xSplitControlBox.setGeometry(QtCore.QRect(0, 0, int(500 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitControlBox.setMaximumSize(QtCore.QSize(16777215, int(BASE_SIZE)))
    self.xSplitControlBox.setMinimumSize(QtCore.QSize(int(200 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitControlBox.setContentsMargins(0, 0, 0, 0)
    self.vLayout_master.addWidget(self.xSplitControlBox)
    self.hLayout3 = QtWidgets.QHBoxLayout(self.xSplitControlBox)
    self.hLayout3.setContentsMargins(0, 0, 0, 0)
    self.hLayout3.setSpacing(2)
    self.hLayout3.addStretch()
    
    self.xSplitLabel = QtWidgets.QLabel('split x')
    self.hLayout3.addWidget(self.xSplitLabel)
    
    self.xSplitAutoScaleBoxX = QWidgetMac(self)
    self.xSplitAutoScaleBoxX.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitAutoScaleBoxX.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout3.addWidget(self.xSplitAutoScaleBoxX)
    self.Layout_xSplitScaleBoxX = QtWidgets.QHBoxLayout(self.xSplitAutoScaleBoxX)
    self.Layout_xSplitScaleBoxX.setContentsMargins(0, 0, 0, 0)
    self.Layout_xSplitScaleBoxX.setSpacing(1)
    self.Layout_xSplitScaleBoxX.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.xSplitAutoScaleButtonX = QPushButtonMac()
    self.xSplitAutoScaleButtonX.setText('Auto')
    self.xSplitAutoScaleButtonX.setToolTip('Scale split x axis to current data set')
    self.xSplitAutoScaleButtonX.setMinimumHeight(int(BASE_SIZE))
    self.xSplitAutoScaleButtonX.setMaximumHeight(int(BASE_SIZE))
    self.xSplitAutoScaleButtonX.clicked.connect(partial(self.doAutoScale, 'x2', True))
    self.Layout_xSplitScaleBoxX.addWidget(self.xSplitAutoScaleButtonX)
    self.xSplitAutoScaleCheckX = QPushButtonCheckable()
    self.xSplitAutoScaleCheckX.setToolTip('Automatically scale split x axis')
    self.xSplitAutoScaleCheckX.setCheckMe(True)
    self.xSplitAutoScaleCheckX.setChecked(self.autoScaleX_div)
    self.xSplitAutoScaleCheckX.setMaximumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.xSplitAutoScaleCheckX.setMinimumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.xSplitAutoScaleCheckX.clicked.connect(partial(self.setAutoScale, 'x2'))
    self.Layout_xSplitScaleBoxX.addWidget(self.xSplitAutoScaleCheckX)

    self.xSplitLowerLimitx = QLineEditClick()
    self.xSplitLowerLimitx.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitLowerLimitx.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitLowerLimitx.setValidator(self.validFloat)
    self.xSplitLowerLimitx.setText(str(self.parent.formatNumber(self.minX_div)))
    self.xSplitLowerLimitx.setToolTip('Lower limit of split x axis')
    self.xSplitLowerLimitx.editingFinished.connect(partial(self.changeAxisLimits, 'x2', 'plot', True))
    self.hLayout3.addWidget(self.xSplitLowerLimitx)

    self.xSplitModeSelectorx = QComboBoxMac()
    self.xSplitModeSelectorx.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitModeSelectorx.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout3.addWidget(self.xSplitModeSelectorx)
    self.xSplitModeSelectorx.addItem('linear')
    self.xSplitModeSelectorx.addItem('log')
    self.xSplitModeSelectorx.currentIndexChanged.connect(partial(self.changeAxisMode, 'x2', True))
    self.xSplitModeSelectorx.setToolTip('Set scale of split x axis')
    
    self.xSplitUpperLimitx = QLineEditClick()
    self.xSplitUpperLimitx.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitUpperLimitx.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.xSplitUpperLimitx.setValidator(self.validFloat)
    self.xSplitUpperLimitx.setText(str(self.parent.formatNumber(self.maxX_div)))
    self.xSplitUpperLimitx.setToolTip('Upper limit of split x axis')
    self.xSplitUpperLimitx.editingFinished.connect(partial(self.changeAxisLimits, 'x2', 'plot', True))
    self.hLayout3.addWidget(self.xSplitUpperLimitx)

    self.flipAxisButtonX2 = QPushButtonCheckable()
    self.flipAxisButtonX2.setText('Flip')
    self.flipAxisButtonX2.setToolTip('Swap upper and lower limits of split x axis')
    self.flipAxisButtonX2.setMaximumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonX2.setMinimumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonX2.clicked.connect(partial(self.flipAxis, 'x2', redraw=True))
    self.hLayout3.addWidget(self.flipAxisButtonX2)

    self.altSplitBox = QWidgetMac()
    self.altSplitBox.setMaximumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.altSplitBox.setMinimumSize(QtCore.QSize(int(40 * SCALEFONT), int(BASE_SIZE)))
    self.hLayout3.addWidget(self.altSplitBox)
    self.LayoutAltSplitBox = QtWidgets.QHBoxLayout(self.altSplitBox)
    self.LayoutAltSplitBox.setContentsMargins(0, 0, 0, 0)

    # controls for secondAxes
    self.secondAxesContainer = QWidgetMac(self)
    self.secondAxesContainer.setGeometry(QtCore.QRect(0, 0, int(50 * SCALEFONT + 1), 500))
    self.secondAxesContainer.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT + 1), 16777215))
    self.secondAxesContainer.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT + 1), 200))
    self.hLayout.addWidget(self.secondAxesContainer)
    self.LayoutSecondAxesContainer = QtWidgets.QVBoxLayout(self.secondAxesContainer)
    self.LayoutSecondAxesContainer.setContentsMargins(0, 1, 0, 0)
    self.LayoutSecondAxesContainer.setSpacing(1)

    self.SpacerBox3 = QWidgetMac(self)
    self.SpacerBox3.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.SpacerBox3.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.LayoutSecondAxesContainer.addWidget(self.SpacerBox3)
    
    self.secondAxesControlContainer = QWidgetMac(self)
    self.LayoutSecondAxesContainer.addWidget(self.secondAxesControlContainer)
    self.LayoutSecondAxesControlContainer = QtWidgets.QVBoxLayout(self.secondAxesControlContainer)
    self.LayoutSecondAxesControlContainer.setContentsMargins(0, 0, 0, 0)
    
    self.LayoutSecondAxesControlContainer.addStretch()
    self.secondAutoScaleBoxY = QWidgetMac(self)
    self.secondAutoScaleBoxY.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.secondAutoScaleBoxY.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.Layout_secondScaleBoxY = QtWidgets.QHBoxLayout(self.secondAutoScaleBoxY)
    self.Layout_secondScaleBoxY.setContentsMargins(0, 0, 0, 0)
    self.Layout_secondScaleBoxY.setSpacing(1)
    self.Layout_secondScaleBoxY.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft|QtCore.Qt.AlignmentFlag.AlignTop)
    self.secondAutoScaleButtonY = QPushButtonMac()
    self.secondAutoScaleButtonY.setText('Auto')
    self.secondAutoScaleButtonY.setToolTip('Scale second y axis to current data set and/or curve')
    self.secondAutoScaleButtonY.setMinimumHeight(int(BASE_SIZE))
    self.secondAutoScaleButtonY.setMaximumHeight(int(BASE_SIZE))
    self.secondAutoScaleButtonY.clicked.connect(partial(self.doAutoScale, 'y2', True))
    self.Layout_secondScaleBoxY.addWidget(self.secondAutoScaleButtonY)
    self.secondAutoScaleCheckY = QPushButtonCheckable()
    self.secondAutoScaleCheckY.setToolTip('Automatically scale second y axis')
    self.secondAutoScaleCheckY.setCheckMe(True)
    self.secondAutoScaleCheckY.setChecked(self.autoScaleY2)
    self.secondAutoScaleCheckY.setMaximumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.secondAutoScaleCheckY.setMinimumSize(QtCore.QSize(int(BASE_SIZE - 2), int(BASE_SIZE - 2)))
    self.secondAutoScaleCheckY.clicked.connect(partial(self.setAutoScale, 'y2'))
    self.Layout_secondScaleBoxY.addWidget(self.secondAutoScaleCheckY)
    self.LayoutSecondAxesControlContainer.addWidget(self.secondAutoScaleBoxY)

    self.secondUpperLimit = QLineEditClick()
    self.secondUpperLimit.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.secondUpperLimit.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.secondUpperLimit.setValidator(self.validFloat)
    self.secondUpperLimit.setText(str(self.parent.formatNumber(self.maxY2)))
    self.secondUpperLimit.setToolTip('Upper limit of second y axis')
    self.secondUpperLimit.editingFinished.connect(partial(self.changeAxisLimits, 'y2', 'plot', True))
    self.LayoutSecondAxesControlContainer.addStretch()
    self.LayoutSecondAxesControlContainer.addWidget(self.secondUpperLimit)

    self.secondModeSelector = QComboBoxMac()
    self.secondModeSelector.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.secondModeSelector.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.LayoutSecondAxesControlContainer.addStretch()
    self.LayoutSecondAxesControlContainer.addWidget(self.secondModeSelector)
    self.secondModeSelector.addItem('linear')
    self.secondModeSelector.addItem('log')
    self.secondModeSelector.currentIndexChanged.connect(partial(self.changeAxisMode, 'y2', True))
    self.secondModeSelector.setToolTip('Set scale of second y axis')
    
    self.secondLowerLimit = QLineEditClick()
    self.secondLowerLimit.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.secondLowerLimit.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.secondLowerLimit.setValidator(self.validFloat)
    self.secondLowerLimit.setText(str(self.parent.formatNumber(self.minY2)))
    self.secondLowerLimit.setToolTip('Lower limit of second y axis')
    self.secondLowerLimit.editingFinished.connect(partial(self.changeAxisLimits, 'y2', 'plot', True))
    self.LayoutSecondAxesControlContainer.addStretch()
    self.LayoutSecondAxesControlContainer.addWidget(self.secondLowerLimit)
    
    self.flipAxisButtonY2 = QPushButtonCheckable()
    self.flipAxisButtonY2.setText('Flip')
    self.flipAxisButtonY2.setToolTip('Swap upper and lower limits of second y axis')
    self.flipAxisButtonY2.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonY2.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.flipAxisButtonY2.clicked.connect(partial(self.flipAxis, 'y2', redraw=True))
    self.LayoutSecondAxesControlContainer.addStretch()
    self.LayoutSecondAxesControlContainer.addWidget(self.flipAxisButtonY2)
    
    self.LayoutSecondAxesControlContainer.addStretch()
    self.splitYSpacer = QWidgetMac(self)
    self.splitYSpacer.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.splitYSpacer.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), int(BASE_SIZE)))
    self.LayoutSecondAxesControlContainer.addWidget(self.splitYSpacer)
    blah = HLine()
    self.LayoutSecondAxesControlContainer.addWidget(blah)    

    # controls for display toggle buttons of data, curve, and extra
    self.toggleDataButtons, self.toggleFitButtons, self.toggleExtraButtons = [], [], []
    self.currentActiveData, self.currentActiveFit = 0, 0
    ###divider = VLine()
    ###self.hLayout.addWidget(divider)
    self.toggloContainer = QtWidgets.QScrollArea()
    self.toggloContainer.setWidgetResizable(True)
    self.hLayout.addWidget(self.toggloContainer)

    self.toggloContainerInner = QWidgetMac()
    self.LayoutToggloContainerInner = QtWidgets.QHBoxLayout(self.toggloContainerInner)
    self.LayoutToggloContainerInner.setContentsMargins(0, 1, 0, 0)
    self.LayoutToggloContainerInner.setSpacing(1)
    self.LayoutToggloContainerInner.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.toggloContainer.setWidget(self.toggloContainerInner)

    self.toggloDataContainer = QWidgetMac()
    self.toggloDataContainer.setMaximumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 16777215))
    self.toggloDataContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 200))
    self.LayoutToggloContainerInner.addWidget(self.toggloDataContainer)
    self.LayoutToggloDataContainer = QtWidgets.QVBoxLayout(self.toggloDataContainer)
    self.LayoutToggloDataContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutToggloDataContainer.setSpacing(1)
    self.LayoutToggloDataContainer.setAlignment(QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop)

    self.toggloFitContainer = QWidgetMac()
    self.toggloFitContainer.setMaximumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 16777215))
    self.toggloFitContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 200))
    self.LayoutToggloContainerInner.addWidget(self.toggloFitContainer)
    self.LayoutToggloFitContainer = QtWidgets.QVBoxLayout(self.toggloFitContainer)
    self.LayoutToggloFitContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutToggloFitContainer.setSpacing(1)
    self.LayoutToggloFitContainer.setAlignment(QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop)
    
    self.toggloExtraContainer = QWidgetMac()
    self.toggloExtraContainer.setMaximumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 16777215))
    self.toggloExtraContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 200))
    self.LayoutToggloContainerInner.addWidget(self.toggloExtraContainer)
    self.LayoutToggloExtraContainer = QtWidgets.QVBoxLayout(self.toggloExtraContainer)
    self.LayoutToggloExtraContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutToggloExtraContainer.setSpacing(1)
    self.LayoutToggloExtraContainer.setAlignment(QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop)

    # spacer and toggle controls for residuals
    self.pseudoContainer = QWidgetMac(self)
    self.pseudoContainer.setMaximumWidth(int(50 * SCALEFONT))
    self.hLayout_resid.addWidget(self.pseudoContainer)
    self.LayoutPseudoContainer = QtWidgets.QVBoxLayout(self.pseudoContainer)
    self.LayoutPseudoContainer.setContentsMargins(0, 1, 0, 0)

    spacerBox5 = QWidgetMac(self)
    spacerBox5.setMaximumSize(QtCore.QSize(int(50 * SCALEFONT), 25))
    spacerBox5.setMinimumSize(QtCore.QSize(int(50 * SCALEFONT), 25))
    self.LayoutPseudoContainer.addWidget(spacerBox5)

    # controls for display toggle buttons of data, curve, and extra
    self.toggleResidButtons = []
    self.toggloContainerResid = QtWidgets.QScrollArea()
    self.toggloContainerResid.setWidgetResizable(True)
    self.hLayout_resid.addWidget(self.toggloContainerResid)

    self.toggloContainerInnerResid = QWidgetMac()
    self.LayoutToggloContainerInnerResid = QtWidgets.QHBoxLayout(self.toggloContainerInnerResid)
    self.LayoutToggloContainerInnerResid.setContentsMargins(0, 0, 0, 0)
    self.LayoutToggloContainerInnerResid.setSpacing(1)
    self.LayoutToggloContainerInnerResid.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)
    self.toggloContainerResid.setWidget(self.toggloContainerInnerResid)

    self.toggloResidContainer = QWidgetMac(self)
    self.toggloResidContainer.setMaximumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 16777215))
    self.toggloResidContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), 200))
    self.LayoutToggloContainerInnerResid.addWidget(self.toggloResidContainer)
    self.LayoutToggloResidContainer = QtWidgets.QVBoxLayout(self.toggloResidContainer)
    self.LayoutToggloResidContainer.setContentsMargins(0, 0, 0, 0)
    self.LayoutToggloResidContainer.setSpacing(1)
    self.LayoutToggloResidContainer.setAlignment(QtCore.Qt.AlignmentFlag.AlignHCenter|QtCore.Qt.AlignmentFlag.AlignTop)

    self.toggloExtraContainerResid = QWidgetMac()
    self.toggloExtraContainerResid.setMaximumWidth(int(11 * SCALEFONT + 1))
    self.toggloExtraContainerResid.setMinimumWidth(int(11 * SCALEFONT + 1))
    self.LayoutToggloContainerInnerResid.addWidget(self.toggloExtraContainerResid)

    # introduce master buttons that control the display of all current objects
    self.toggleDataMaster = QPushButtonCheckableMini(parent=self, kind='master_data')
    self.toggleDataMaster.alignMeRight = True
    self.toggleDataMaster.setChecked(True)
    self.toggleDataMaster.setText('\u2115')
    self.toggleDataMaster.setToolTip('- Click left to set visibility of data objects\n- Click middle to vary data objects\n- Click right to hide all data objects')
    self.LayoutToggloDataContainer.addWidget(self.toggleDataMaster)
    self.toggleDataMaster.clicked.connect(partial(self.toggloCallbackMaster, 'data', self.toggleDataMaster))
    
    self.toggleFitMaster = QPushButtonCheckableMini(parent=self, kind='master_fit')
    self.toggleFitMaster.alignMeRight = True
    self.toggleFitMaster.setChecked(True)
    self.toggleFitMaster.setText('\u2115')
    self.toggleFitMaster.setToolTip('- Click left to set visibility of curve objects\n- Click middle to vary curve objects\n- Click right to hide all curve objects')
    self.LayoutToggloFitContainer.addWidget(self.toggleFitMaster)
    self.toggleFitMaster.clicked.connect(partial(self.toggloCallbackMaster, 'fit', self.toggleFitMaster))
    
    self.toggleExtraMaster = QPushButtonCheckableMini(parent=self, kind='master_extra')
    self.toggleExtraMaster.alignMeRight = True
    self.toggleExtraMaster.setChecked(True)
    self.toggleExtraMaster.setText('\u2115')
    self.toggleExtraMaster.setToolTip('- Click left to set visibility of extra objects\n- Click right to hide all extra objects')
    self.LayoutToggloExtraContainer.addWidget(self.toggleExtraMaster)
    self.toggleExtraMaster.clicked.connect(partial(self.toggloCallbackMaster, 'extra', self.toggleExtraMaster))
    
    self.toggleResidMaster = QPushButtonCheckableMini(parent=self, kind='master_resid')
    self.toggleResidMaster.alignMeRight = True
    self.toggleResidMaster.setChecked(True)
    self.toggleResidMaster.setText('\u2115')
    self.toggleResidMaster.setToolTip('- Click left to set visibility of residuals objects\n- Click middle to vary residuals objects\n- Click right to hide all residuals objects')
    self.LayoutToggloResidContainer.addWidget(self.toggleResidMaster)
    self.toggleResidMaster.clicked.connect(partial(self.toggloCallbackMaster, 'resid', self.toggleResidMaster))

    self.updateToggloContainer()

    # initially turn off the second axes controls
    self.toggleSecondAxes(state=False)

    # set initial size ratio of QSplitter()
    # this call of no consequence as widget will be resized after creation
    # => readjust in main ui, see below
    ###self.masterwidget.setSizes([4, 1])

  def updateToggloContainer(self):
    # updates push buttons in toggloContainer
    # keep track on whether we need to allow space for display of scrollbars
    needScrollbar = False
    
    # set visbility of data buttons
    for index, entry in enumerate(self.toggleDataButtons):
      if(index < len(self.parent.data)):
        self.toggleDataButtons[index].show()
      else:
        self.toggleDataButtons[index].hide()
    # expand list of data buttons as needed
    while(len(self.toggleDataButtons) < len(self.parent.data)):
      self.toggleDataButtons.append(QPushButtonCheckableMini(parent=self, kind='data'))
      # make button identifiable
      self.toggleDataButtons[-1].alignMeRight = True
      self.LayoutToggloDataContainer.addWidget(self.toggleDataButtons[-1])
      # connect new slot
      self.toggleDataButtons[-1].clicked.connect(partial(self.toggloCallback, 'data', self.toggleDataButtons[-1]))
    # cycle over all buttons
    for index, item in enumerate(self.toggleDataButtons[:len(self.parent.data)]):
      item.setText(str(index + 1))
      item.setChecked(self.parent.data[index].visibility)
      ttstring = '- Click left to set visibility of data object ' + str(index + 1)
      ttstring += '\n- Click middle to modify data object ' + str(index + 1)
      ttstring += '\n- Click right to make data object ' + str(index + 1) + ' active'
      item.setToolTip(ttstring)
    # need to manually adjust size of container as PyQt is not doing it ...
    size = self.toggloDataContainer.sizeHint()
    left, top, right, bottom = self.LayoutToggloDataContainer.getContentsMargins()
    spacing = self.LayoutToggloDataContainer.spacing()
    useHeight = size.height() + top + bottom + spacing * (len(self.parent.data) - 1)
    self.toggloDataContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), useHeight))
    if(useHeight > self.toggloContainer.viewport().height()):
      needScrollbar = True
      
    # set visbility of fit buttons
    for index, entry in enumerate(self.toggleFitButtons):
      if(index < len(self.parent.fit)):
        self.toggleFitButtons[index].show()
      else:
        self.toggleFitButtons[index].hide()
    # expand list of fit buttons as needed
    while(len(self.toggleFitButtons) < len(self.parent.fit)):
      self.toggleFitButtons.append(QPushButtonCheckableMini(parent=self, kind='fit'))
      # make button identifiable
      self.toggleFitButtons[-1].alignMeRight = True
      self.LayoutToggloFitContainer.addWidget(self.toggleFitButtons[-1])
      # connect new slot
      self.toggleFitButtons[-1].clicked.connect(partial(self.toggloCallback, 'fit', self.toggleFitButtons[-1]))
    # cycle over all buttons
    for index, item in enumerate(self.toggleFitButtons[:len(self.parent.fit)]):
      item.setText(str(index + 1))
      item.setChecked(self.parent.fit[index].visibility)
      ttstring = '- Click left to set visibility of curve object ' + str(index + 1)
      ttstring += '\n- Click middle to modify curve object ' + str(index + 1)
      ttstring += '\n- Click right to make curve object ' + str(index + 1) + ' active'
      item.setToolTip(ttstring)
    # need to manually adjust size of container as PyQt is not doing it ...
    size = self.toggloFitContainer.sizeHint()
    left, top, right, bottom = self.LayoutToggloFitContainer.getContentsMargins()
    spacing = self.LayoutToggloFitContainer.spacing()
    useHeight = size.height() + top + bottom + spacing * (len(self.parent.fit) - 1)
    self.toggloFitContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), useHeight))
    if(useHeight > self.toggloContainer.viewport().height()):
      needScrollbar = True

    # set visbility of extra buttons
    for index, entry in enumerate(self.toggleExtraButtons):
      if(index < len(self.parent.extras)):
        self.toggleExtraButtons[index].show()
      else:
        self.toggleExtraButtons[index].hide()
    # expand list of extra buttons as needed
    while(len(self.toggleExtraButtons) < len(self.parent.extras)):
      self.toggleExtraButtons.append(QPushButtonCheckableMini(parent=self, kind='extra'))
      # make button identifiable
      self.toggleExtraButtons[-1].alignMeRight = True
      self.LayoutToggloExtraContainer.addWidget(self.toggleExtraButtons[-1])
      # connect new slot
      self.toggleExtraButtons[-1].clicked.connect(partial(self.toggloCallback, 'extra', self.toggleExtraButtons[-1]))
    # cycle over all buttons
    for index, item in enumerate(self.toggleExtraButtons[:len(self.parent.extras)]):
      item.setText(str(index + 1))
      item.setChecked(self.parent.extras[index].visibility)
      ttstring = '- Click left to set visibility of extra object ' + str(index + 1)
      ttstring += '\n- Click middle to modify extra object ' + str(index + 1)
      item.setToolTip(ttstring)
    # need to manually adjust size of container as PyQt is not doing it ...
    size = self.toggloExtraContainer.sizeHint()
    left, top, right, bottom = self.LayoutToggloExtraContainer.getContentsMargins()
    spacing = self.LayoutToggloExtraContainer.spacing()
    # deal with situation when no extras exist
    useHeight = max(1, size.height() + top + bottom + spacing * (len(self.parent.extras) - 1))
    self.toggloExtraContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), useHeight))
    if(useHeight > self.toggloContainer.viewport().height()):
      needScrollbar = True
    
    # adjust of togglo sidebar as needed
    if(needScrollbar):
      # have to impose max size of scrollbar as geometry returns insanely large value before first display of scollbar
      addon = min(15, self.toggloContainer.verticalScrollBar().geometry().width())
    else:
      addon = 0
    if(len(self.parent.extras)):
      self.toggloExtraContainer.show()
      self.toggloExtraContainerResid.show()
      # have to hard-code the size as layout updating by Qt does not work
      self.toggloContainer.setMaximumWidth(addon + int(40 * SCALEFONT + 1))
      self.toggloContainerResid.setMaximumWidth(addon + int(40 * SCALEFONT + 1))
    else:
      self.toggloExtraContainer.hide()
      self.toggloExtraContainerResid.hide()
      self.toggloContainer.setMaximumWidth(addon + int(26 * SCALEFONT + 1))
      self.toggloContainerResid.setMaximumWidth(addon + int(26 * SCALEFONT + 1))

    # set visbility of resid buttons
    for index, entry in enumerate(self.toggleResidButtons):
      if(index < len(self.parent.data) + 1):
        self.toggleResidButtons[index].show()
      else:
        self.toggleResidButtons[index].hide()
    # expand list of resid buttons as needed
    while(len(self.toggleResidButtons) < len(self.parent.data) + 1):
      self.toggleResidButtons.append(QPushButtonCheckableMini(parent=self, kind='resid'))
      # make button identifiable
      self.toggleResidButtons[-1].alignMeRight = True
      self.LayoutToggloResidContainer.addWidget(self.toggleResidButtons[-1])
      # connect new slot
      self.toggleResidButtons[-1].clicked.connect(partial(self.toggloCallback, 'resid', self.toggleResidButtons[-1]))
    # cycle over all buttons
    for index, item in enumerate(self.toggleResidButtons[:len(self.parent.data) + 1]):
      item.setText(str(index))
      if(index):
        item.setChecked(self.parent.data[index - 1].visibilityResid)
      else:
        item.setChecked(self.visibilityResidLine)
      if(index):
        ttstring = '- Click left to set visibility of residuals object ' + str(index)
        ttstring += '\n- Click middle to modify residuals object ' + str(index)
      else:
        ttstring = '- Click left to set visibility of zero line object'
        ttstring += '\n- Click middle to modify zero line object'
      item.setToolTip(ttstring)

    # need to manually adjust size of container as PyQt is not doing it ...
    size = self.toggloResidContainer.sizeHint()
    left, top, right, bottom = self.LayoutToggloResidContainer.getContentsMargins()
    spacing = self.LayoutToggloResidContainer.spacing()
    useHeight = size.height() + top + bottom + spacing * len(self.parent.data)
    self.toggloResidContainer.setMinimumSize(QtCore.QSize(int(11 * SCALEFONT + 1), useHeight))
    if(useHeight > self.toggloContainerResid.viewport().height()):
      needScrollbar = True

    # update labels
    self.updateToggloActive(activeData=self.parent.activeData, activeFit=self.parent.activeFit)
    
    # update master buttons
    for kind in ['data', 'fit', 'resid']:
      self.updateToggloMaster(kind=kind)
    if(len(self.parent.extras)):
      self.updateToggloMaster(kind='extra')
    
  def updateToggloActive(self, activeData=None, activeFit=None):
    # highlights currently active data, fit and resid objects
    if((activeData != None) and (activeData < len(self.parent.data))):
      # cycle over all data sets and adjust highlight
      for index in range(len(self.parent.data)):
        self.toggleDataButtons[index].setHigh(index==activeData)
        self.toggleResidButtons[index + 1].setHigh(index==activeData)
      
      self.currentActiveData = activeData
    
    if((activeFit != None) and (activeFit < len(self.parent.fit))):
      # cycle over all curves and adjust highlight
      for index in range(len(self.parent.fit)):
        self.toggleFitButtons[index].setHigh(index==activeFit)
      
      self.currentActiveFit = activeFit
      
  def toggloCallback(self, kind='data', button=None):
    # call back function for visibility push buttons
    if(button != None):
      index = int(button.text()) - 1
      if(kind == 'data'):
        if(index < len(self.parent.data)):
          state = self.toggleDataButtons[index].isChecked()
          self.parent.objectsarea.dataSetTable.cellWidget(index, 0).setChecked(state)
          self.parent.objectsarea.toggleVisibilityData(index=index, redraw=True)
      elif(kind == 'fit'):
        if(index < len(self.parent.fit)):
          state = self.toggleFitButtons[index].isChecked()
          self.parent.objectsarea.curvesTable.cellWidget(index, 0).setChecked(state)
          self.parent.objectsarea.toggleVisibilityCurve(index=index, redraw=True)
      elif(kind == 'extra'):
        if(index < len(self.parent.extras)):
          state = self.toggleExtraButtons[index].isChecked()
          self.parent.objectsarea.extrasTable.cellWidget(index, 0).setChecked(state)
          self.parent.objectsarea.toggleVisibilityExtras(index=index, redraw=True)
      elif(kind == 'resid'):
        index += 1
        if(index < len(self.parent.data) + 1):
          state = self.toggleResidButtons[index].isChecked()
          self.parent.objectsarea.residTable.cellWidget(index, 0).setChecked(state)
          self.parent.objectsarea.toggleVisibilityResid(index=index, redraw=True)

  def toggloCallbackMaster(self, kind='data', button=None):
    # call back function for visibility push buttons
    if(button != None):
      flag = False
      state = button.isChecked()
      if(kind == 'data'):
        for index in range(len(self.parent.data)):
          state2 = self.toggleDataButtons[index].isChecked()
          if(state != state2):
            flag = True
            self.parent.objectsarea.dataSetTable.cellWidget(index, 0).setChecked(state)
            self.parent.objectsarea.toggleVisibilityData(index=index, redraw=False)
        if(flag):
          self.dataplotwidget.myRefresh()
      elif(kind == 'fit'):
        for index in range(len(self.parent.fit)):
          state2 = self.toggleFitButtons[index].isChecked()
          if(state != state2):
            flag = True
            self.parent.objectsarea.curvesTable.cellWidget(index, 0).setChecked(state)
            self.parent.objectsarea.toggleVisibilityCurve(index=index, redraw=False)
        if(flag):
          self.dataplotwidget.myRefresh()
      elif(kind == 'extra'):
        for index in range(len(self.parent.extras)):
          state2 = self.toggleExtraButtons[index].isChecked()
          if(state != state2):
            flag = True
            self.parent.objectsarea.extrasTable.cellWidget(index, 0).setChecked(state)
            self.parent.objectsarea.toggleVisibilityExtras(index=index, redraw=False)
        if(flag):
          self.dataplotwidget.myRefresh()
      elif(kind == 'resid'):
        for index in range(len(self.parent.data)):
          state2 = self.toggleResidButtons[index + 1].isChecked()
          if(state != state2):
            flag = True
            self.parent.objectsarea.residTable.cellWidget(index + 1, 0).setChecked(state)
            self.parent.objectsarea.toggleVisibilityResid(index=index + 1, redraw=False)
        if(flag):
          self.residplotwidget.myRefresh()

  def checkToggloItem(self, kind='data', index=0, state=True):
    # used to keep track of visibility changes
    if(kind == 'data'):
      if(index < len(self.parent.data)):
        self.toggleDataButtons[index].setChecked(state)
    elif(kind == 'fit'):
      if(index < len(self.parent.fit)):
        self.toggleFitButtons[index].setChecked(state)
    elif(kind == 'extra'):
      if(index < len(self.parent.extras)):
        self.toggleExtraButtons[index].setChecked(state)
    elif(kind == 'resid'):
      if(index < len(self.parent.data) + 1):
        self.toggleResidButtons[index].setChecked(state)
        
    # update master buttons
    self.updateToggloMaster(kind=kind)
    
  def updateToggloMaster(self, kind='data'):
    # helper function that adjusts master button checked status appropriately according to state of indivdual buttons
    if(kind in ['data', 'fit', 'extra', 'resid']):
      if(kind == 'data'):
        checko = True
        for i in range(len(self.parent.data)):
          if(i):
            checko = checko and (self.toggleDataButtons[0].isChecked() == self.toggleDataButtons[i].isChecked())
        # do all buttons have the same status?
        # if so, make sure that master button has the same status
        if(checko):
          self.toggleDataMaster.setChecked(self.toggleDataButtons[0].isChecked())
      elif(kind == 'fit'):
        checko = True
        for i in range(len(self.parent.fit)):
          if(i):
            checko = checko and (self.toggleFitButtons[0].isChecked() == self.toggleFitButtons[i].isChecked())
        # do all buttons have the same status?
        # if so, make sure that master button has the same status
        if(checko):
          self.toggleFitMaster.setChecked(self.toggleFitButtons[0].isChecked())
      elif(kind == 'extra'):
        checko = True
        for i in range(len(self.parent.extras)):
          if(i):
            checko = checko and (self.toggleExtraButtons[0].isChecked() == self.toggleExtraButtons[i].isChecked())
        # do all buttons have the same status?
        # if so, make sure that master button has the same status
        if(checko):
          self.toggleExtraMaster.setChecked(self.toggleExtraButtons[0].isChecked())
      elif(kind == 'resid'):
        checko = True
        for i in range(len(self.parent.data)):
          if(i):
            checko = checko and (self.toggleResidButtons[1].isChecked() == self.toggleResidButtons[i + 1].isChecked())
        # do all buttons have the same status?
        # if so, make sure that master button has the same status
        if(checko):
          self.toggleResidMaster.setChecked(self.toggleResidButtons[1].isChecked())

  def configureMeHere(self, kind='data', index=0, callButton=None):
    # call back to bring up config menu
    if(kind == 'data'):
      self.parent.objectsarea.changeStyle(self.parent.data[index], True, False, callButton=callButton)
    elif(kind == 'fit'):
      self.parent.objectsarea.changeStyle(self.parent.fit[index], False, False, callButton=callButton)
    elif(kind == 'extra'):
      self.parent.objectsarea.changeStyleExtra(index, callButton=callButton)
    elif(kind == 'resid'):
      index += 1
      if(index > -1):
        self.parent.objectsarea.changeStyle(self.parent.data[index], False, True, callButton=callButton)
      else:
        self.parent.objectsarea.changeResidZeroStyle(callButton=callButton)

  def configureMeHereMyMaster(self, kind='data', callButton=None):
    # call back to bring up config menu
    self.parent.objectsarea.changeStyleMyMaster(callButton=callButton, kind=kind)

  def makeMeActive(self, kind='data', index=0):
    # call back to make item active
    field = None
    if((kind == 'data') and (index != self.parent.activeData)):
      field = self.parent.objectsarea.dataSetTable.cellWidget(index, 1)
    elif((kind == 'fit') and (index != self.parent.activeFit)):
      field = self.parent.objectsarea.curvesTable.cellWidget(index, 1)
    
    if(field != None):
      field.setChecked(True)
      if(kind == 'data'):
        self.updateToggloActive(activeData=index)
      else:
        self.updateToggloActive(activeFit=index)

  def addSplitterHandleSymbols(self, parentHandle=None, direction='horizontal', number=1, repeat=5):
    # adds spacer symbols to splitter handle
    if(parentHandle != None):
      if(direction in ['horizontal', 'vertical']):
        parentHandle.setHandleWidth(5)
        handli = parentHandle.handle(number)
        if(handli != None):
          if(direction == 'horizontal'):
            layoutli = QtWidgets.QHBoxLayout(handli)
          else:
            layoutli = QtWidgets.QVBoxLayout(handli)
          layoutli.addStretch()
          for repeat in range(repeat):
            bogus = QtWidgets.QWidget()
            bogus.setStyleSheet('QWidget {background-color: #888888;border-radius: 2px;}')
            bogus.setMinimumSize(4, 4)
            bogus.setMaximumSize(4, 4)
            layoutli.addWidget(bogus)
          layoutli.addStretch()
          layoutli.setSpacing(3)
          layoutli.setContentsMargins(1, 0, 0, 0)

  def clearSelection(self):
    # callback for clear selection button
    if(hasattr(self.parent, 'resultsarea')):
      self.parent.resultsarea.clearSelection()

  def toggleSecondAxes(self, state=False):
    # toggles visibility of control elements for second axes
    ###if(state):
    if(True):
      self.secondAxesContainer.setVisible(state)
      self.pseudoContainer.setVisible(state)
      if((hasattr(self.parent, 'tabWidget')) and (hasattr(self.parent, 'tab_7'))):
        if(state):
          if(self.parent.tabWidget.indexOf(self.parent.tab_7) < 0):
            self.parent.tabWidget.addTab(self.parent.tab_7, "2nd Axes")
            # need to set again the tool tip upon recreation of tab
            toolText = 'This tab contains options for configuring the second axes graphics.\n(Ctrl-7)'
            if(myapp.toolTipsShow):
              self.parent.tabWidget.setTabToolTip(self.parent.tabWidget.indexOf(self.parent.tab_7), toolText)
            elif(len(self.parent.tabWidget.storedToolTips) <= self.parent.tabWidget.indexOf(self.parent.tab_7)):
              self.parent.tabWidget.storedToolTips.append(toolText)
        else:
          if(self.parent.tabWidget.indexOf(self.parent.tab_7) > -1):
            self.parent.tabWidget.removeTab(self.parent.tabWidget.indexOf(self.parent.tab_7))
      # toggle visibility of plot elements
      if(hasattr(self.parent, 'plotArea')):
        self.toggleSecondAxesItems(state=state)
    # need to solve inner situation -- but check whether plot object is already instantiated
    if(hasattr(self, 'ax')):
      self.updateInnerSituation()
      
  def toggleSecondAxesItems(self, state=False):
    # turns on/off all relevant plot elements of self.ax2
#    if(state):
    if(True):
      self.ax2.spines['right'].set_visible(state and self.axisVisible['right2'])
      self.ax2.yaxis.set_visible(state)
  
      self.ax2_div.spines['left'].set_visible(state and self.axisVisible['left2'])
      self.ax2_div.spines['right'].set_visible(state and self.axisVisible['right2'])
      self.ax2_div.yaxis.set_visible(state)
    
    # delete all grid lines on the second axes
    if(state):
      if(self.gridVisible['y2']):
        self.drawAxisGrid(axis='y2', redraw=False, target='plot')
    else:
      for line in self.gridLinesStore['y2']:
        line.remove()
      self.gridLinesStore['y2'] = []
      
    # deal with axis arrows
    if(state):
      if(self.arrowVisible['y2']):
        # need to redraw arrow as it will likely have been destroyed since last display of 2nd axes
        self.setAxisArrow(state=True, axis='y2', item='all', redraw=False, target='plot')
    else:
      if(self.handleArrow['y2'] != None):
        self.handleArrow['y2'].remove()
        self.handleArrow['y2'] = None
      if(self.handleArrow2['y2'] != None):
        self.handleArrow2['y2'].remove()
        self.handleArrow2['y2'] = None
    
  def isSecondAxesActive(self):
    # checks whether any items are on second axis
    onSecondItems = [1 if (i.onSecondAxes) else 0 for i in (self.parent.data + self.parent.fit + self.parent.extras)]
    if(sum(onSecondItems) or self.splitY):
      return True

    return False

  def toggleSplit(self, redraw=True, toggled=False):
    # toggles on/off split x axis
    state = self.splitAxisButton.isChecked()
    self.splitShow = state
    self.ax_div.set_visible(state)
    self.ax2_div.set_visible(state)
    self.ax_resid_div.set_visible(state)
    if(state):
      # implement check for too large split fraction
      maxSplit = 1 - self.padSize['left'] - (1 - self.padSize['right']) - 0.1
      if(self.splitPad > maxSplit):
        if(hasattr(self.parent, 'statusbar')):
          self.parent.statusbar.showMessage('Had to reduce split x axis padding to ' + self.parent.formatNumber(maxSplit) + ', rather than ' + self.parent.formatNumber(self.splitPad) + '!', self.parent.STATUS_TIME)
        self.splitPad = maxSplit

      if(hasattr(self.parent, 'graphicsarea')):
        self.parent.graphicsarea.configInnerBox.show()
        self.parent.graphicsarea.configInnerTickXBox.show()
        self.parent.graphicsarea.outerBox4b.show()
        self.parent.graphicsarea.configGridBox['x2'].show()
        # check whether advanced graphics enabled
        if(self.parent.graphicsarea.advancedExport['advancedGraphics']):
          if(hasattr(self.parent.graphicsarea, 'configInnerBox2')):
            self.parent.graphicsarea.configInnerBox2.show()
          if(hasattr(self.parent.graphicsarea, 'configInnerBox3')):
            self.parent.graphicsarea.configInnerBox3.show()
          if(hasattr(self.parent.graphicsarea, 'configInnerBox5')):
            self.parent.graphicsarea.configInnerBox5.show()
          if(hasattr(self.parent.graphicsarea, 'configInnerBox6')):
            self.parent.graphicsarea.configInnerBox6.show()
          if(hasattr(self.parent.graphicsarea, 'configInnerMinorTickBox')):
            self.parent.graphicsarea.configInnerMinorTickBox.show()
          self.parent.graphicsarea.configGridBox2['x2'].show()
          self.parent.graphicsarea.configGridBox3['x2'].show()
      
      xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
      # since we abandoned make_axes_locatable, we also need to implement check for split on y axis
      if(self.splitY):
        ybreak = (self.padSize['top'] - self.padSize['bottom'] - self.splitYPad) * self.splitYFraction / (1.0 + self.splitYFraction)
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['left'] + xbreak,  self.padSize['top']]]))
        self.ax_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['right'],  self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['bottom'] + ybreak]]))
        self.ax2_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['bottom'] + ybreak]]))
      else:
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak,  self.padSize['top']]]))
        self.ax_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'],  self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['top']]]))
        self.ax2_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
      self.ax_resid.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['top']]]))
      self.ax_resid_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))

      # if called further up, the following command produces a warning under Linux
      # b/c the underlying call to axis._get_pixel_distance_along_axis() fails
      # => work around: move it here
      self.xSplitControlBox.show()
      self.LayoutAltSplitBox.addWidget(self.splitAxisButton)
      
      # need to ensure correct positioning of second axis label
      if(self.isSecondAxesActive()):
        self.ax2.yaxis.label.set_visible(False)
        self.ax2_div.yaxis.label.set_visible(self.labelY2Show)
      else:
        # hopefully, this kills the mysterious black line
        self.ax2.spines['right'].set_visible(False)
        self.ax2_div.spines['left'].set_visible(False)
      
      # set visibility of inner items correctly
      self.updateInnerSituation()
      
      # set axis mode and limits to match those of first axes
      if(toggled):
        self.xSplitModeSelectorx.blockSignals(True)
        index = self.xSplitModeSelectorx.findText(self.modeX)
        self.xSplitModeSelectorx.setCurrentIndex(index)
        self.xSplitModeSelectorx.blockSignals(False)
        self.changeAxisMode('x2', redraw=False)
      # set axes limits for split effect
      if(toggled):
        # only do this upon first display of split x axes
        if(self.modeX == 'linear'):
          splitXAt = self.minX + 1.0 / (self.splitFraction + 1.0) * (self.maxX - self.minX)
        else:
          splitXAt = np.exp(np.log(self.minX) + 1.0 / (self.splitFraction + 1.0) * (np.log(self.maxX) - np.log(self.minX)))
        self.setAxisLimits(lower=splitXAt, upper=self.maxX, axis='x2', updateLabel=False, target='plot', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.setAxisLimits(lower=splitXAt, upper=self.maxX, axis='x2', updateLabel=False, target='resid', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.xSplitLowerLimitx.setText(self.parent.formatNumber(splitXAt))
        self.xSplitUpperLimitx.setText(self.upperLimitx.text())
        self.setAxisLimits(lower=self.minX, upper=splitXAt, axis='x', updateLabel=False, target='plot', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.setAxisLimits(lower=self.minX, upper=splitXAt, axis='x', updateLabel=False, target='resid', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.upperLimitx.setText(self.parent.formatNumber(splitXAt))

      # need to update tick formatting if split axis is turned on
      for target in ['plot', 'resid']:
        self.setTickOne4All(axis='x', redraw=False, target=target)
    else:
      self.xSplitControlBox.hide()
      self.LayoutFirstSplitBox.addWidget(self.splitAxisButton)
      if(hasattr(self.parent, 'graphicsarea')):
        self.parent.graphicsarea.configInnerBox.hide()
        self.parent.graphicsarea.configInnerTickXBox.hide()
        self.parent.graphicsarea.outerBox4b.hide()
        self.parent.graphicsarea.configGridBox['x2'].hide()
        if(hasattr(self.parent.graphicsarea, 'configInnerBox2')):
          self.parent.graphicsarea.configInnerBox2.hide()
        if(hasattr(self.parent.graphicsarea, 'configInnerBox3')):
          self.parent.graphicsarea.configInnerBox3.hide()
        if(hasattr(self.parent.graphicsarea, 'configInnerBox5')):
          self.parent.graphicsarea.configInnerBox5.hide()
          if(hasattr(self.parent.graphicsarea, 'configInnerBox6')):
            self.parent.graphicsarea.configInnerBox6.hide()
        if(hasattr(self.parent.graphicsarea, 'configInnerMinorTickBox')):
          self.parent.graphicsarea.configInnerMinorTickBox.hide()
        self.parent.graphicsarea.configGridBox2['x2'].hide()
        self.parent.graphicsarea.configGridBox3['x2'].hide()
      
      # since we abandoned make_axes_locatable, we also need to implement check for split on y axis
      if(self.splitY):
        ybreak = (self.padSize['top'] - self.padSize['bottom'] - self.splitYPad) * self.splitYFraction / (1.0 + self.splitYFraction)
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['right'], self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['bottom'] + ybreak]]))
        # this should take care of the obnoxious line randomly popping up in the the middle of the screen
        # not sure why, suspect Matplotlib bug
        # but we can workaround by moving second axes out of screen (in addition to not displaying them)
        # awful fix, very ugly, but cannot be helped
        self.ax_div.set_position(matplotlib.transforms.Bbox([[1.5, self.padSize['bottom'] + ybreak + self.splitYPad], [2.0,  self.padSize['top']]]))
        self.ax2_div.set_position(matplotlib.transforms.Bbox([[1.5, self.padSize['bottom']], [2.0, self.padSize['bottom'] + ybreak]]))
      else:
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
        # and here goes the ugly fix again
        self.ax_div.set_position(matplotlib.transforms.Bbox([[1.5, self.padSize['bottom']], [2.0,  self.padSize['top']]]))
        self.ax2_div.set_position(matplotlib.transforms.Bbox([[1.5, self.padSize['bottom']], [2.0, self.padSize['top']]]))
      self.ax_resid.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))

      # clean up the mess that may have been caused by toggling visibility of inner items
      self.setAxisVisibility(value=self.axisVisible['right'], axis='right', redraw=False, target='plot')
      self.ax2.spines['right'].set_visible(self.axisVisible['right2'] and self.isSecondAxesActive())
      self.setAxisVisibility(value=self.axisVisible_resid['right'], axis='right', redraw=False, target='resid')

      self.setTickMarkVisibility(value=self.ticksVisible['right'], axis='right', redraw=False, target='plot', initMode=True)
      self.setTickMarkVisibility(value=self.ticksVisible['left2'], axis='left2', redraw=False, target='plot', initMode=True)
      self.setTickMarkVisibility(value=self.ticksVisible_resid['right'], axis='right', redraw=False, target='resid', initMode=True)
      
      # need to ensure correct positioning of second axis label
      if(self.isSecondAxesActive()):
        self.ax2.yaxis.label.set_visible(self.labelY2Show)
        self.ax2_div.yaxis.label.set_visible(False)
      else:
        # hopefully, this kills the mysterious black line
        self.ax2.spines['right'].set_visible(False)
        self.ax2_div.spines['left'].set_visible(False)
        
      # reset axis limits to nonsplit
      if(self.modeX == self.modeX_div):
        self.setAxisLimits(lower=self.minX, upper=self.maxX_div, axis='x', updateLabel=False, target='plot', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.setAxisLimits(lower=self.minX, upper=self.maxX_div, axis='x', updateLabel=False, target='resid', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.upperLimitx.setText(self.parent.formatNumber(self.maxX_div))

      # also need to update tick formatting if split axis is turned off
      for target in ['plot', 'resid']:
        self.setTickOne4All(axis='x', redraw=False, target=target)
      if(self.isSecondAxesActive()):
        self.setTickOne4All(axis='y2', redraw=False, target='plot')

    # draw plot and resid line (again) to ensure coverage of entire x range
    if(toggled):
      # only do this upon first display of split x axes
      if(hasattr(self.parent, 'plotArea')):
        self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.parent.plotArea.plotFunction(\
          fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot,\
          handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False, doAutoZoom=False)
        self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.parent.plotArea.plotFunction(\
          fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div,\
          handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True, doAutoZoom=False)
        # copy in case split axes are shown
        curve = self.parent.fit[self.parent.activeFit]
        if(self.parent.plotArea.splitY and curve.onBothAxes):
          curve.duplicateForSplit()
          if(not curve.visibility):
            curve.setVisibility(state=False, redraw=False)
        self.parent.plotArea.handleResidZero = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero, redraw=False)
        self.parent.plotArea.handleResidZero_div = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero_div, redraw=False, splitX=True)

    # refresh plots
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

      # finally we have to take care of the crosshair cursor b/c as of matplotlib 3.1.2 we can only initialize
      # it upon first display of div plot --- bummer!
      if(state and (self.cursor != None) and (self.cursor_div == None)):
        self.cursor_div = MyCursor(self.ax_div, useblit=True, color='black', linewidth=1)
        self.cursor_div.setParent(self)
        self.cursor_div.setAx2(self.ax2_div)
        
        # link cursors
        self.cursor.setTwin(self.cursor_div)
        self.cursor_div.setTwin(self.cursor)
        
        # toggle visibility
        self.cursor_div.toggleVisibility(self.cursorVisible, None)

      if(state and self.splitY and (self.cursor_div != None) and (self.cursor2_div == None)):
        self.cursor2_div = MyCursor(self.ax2_div, useblit=True, color='black', linewidth=0.5)
        self.cursor2_div.setParent(self)
        self.cursor2_div.setAx2(self.ax2_div)
        
        # link cursors
        if(self.cursor_div != None):
          self.cursor_div.setTwinVertical(self.cursor2_div)
          self.cursor2_div.setTwinVertical(self.cursor_div)
        if(self.cursor2 != None):
          self.cursor2.setTwin(self.cursor2_div)
          self.cursor2_div.setTwin(self.cursor2)

    # also need to update tick entry fields
    self.updateTickEntryField(axis='x')
    self.updateTickEntryField(axis='x2')

    # for some strange reason we should reset the axis ticks here b/c minor autor locator gets confused
    # reason unclear, could be matplotlib bug??
    ###if(self.splitShow):
    if(True):
      redrawAgain = False
      if(self.ticksXAuto):
        self.setAutoTicks(axis='x', redraw=False, target='plot')
      else:
        # need to preserve axis limits before calling setAxisTicks()
        minX, maxX, ticksXLabel = self.minX, self.maxX, self.ticksXLabel
        self.setAxisTicks(value=self.ticksX, axis='x', redraw=False, target='plot')
        # restore tick labels
        self.ticksXLabel = ticksXLabel
        if(len(self.ticksXLabel) and (len(self.ticksXLabel) == len(self.ticksX))):
          for axisobject in [self.ax2, self.ax_resid]:
            axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksX))
            axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksXLabel))

        # now restore the original limits and (alas!) issue another redraw
        self.setAxisLimits(lower=minX, upper=maxX, axis='x', updateLabel=False, target='plot', redraw=False, updateTicks=False, updateGrid=True, preserveInverted=True)
        self.setAxisLimits(lower=minX, upper=maxX, axis='x', updateLabel=False, target='resid', redraw=False, updateTicks=False, updateGrid=True, preserveInverted=True)
        redrawAgain = True
      if(self.ticksXAuto_div):
        self.setAutoTicks(axis='x2', redraw=False, target='plot')
      else:
        # need to preserve axis limits before calling setAxisTicks()
        minX, maxX, ticksXLabel = self.minX_div, self.maxX_div, self.ticksXLabel_div
        self.setAxisTicks(value=self.ticksX_div, axis='x2', redraw=False, target='plot')
        # restore tick labels
        self.ticksXLabel_div = ticksXLabel
        if(len(self.ticksXLabel_div) and (len(self.ticksXLabel_div) == len(self.ticksX_div))):
          for axisobject in [self.ax2_div, self.ax_resid_div]:
            axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksX_div))
            axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksXLabel_div))
        # if split displayed, restore the original limits and (alas!) issue another redraw
        if(state):
          self.setAxisLimits(lower=minX, upper=maxX, axis='x2', updateLabel=False, target='plot', redraw=False, updateTicks=False, updateGrid=True, preserveInverted=True)
          self.setAxisLimits(lower=minX, upper=maxX, axis='x2', updateLabel=False, target='resid', redraw=False, updateTicks=False, updateGrid=True, preserveInverted=True)
          redrawAgain = True
      # do we need to redraw?
      if(redrawAgain):
        self.dataplotwidget.myRefresh()
        self.residplotwidget.myRefresh()
      
  def updateSplitDivider(self, param='color', value=[0.2, 0.2, 0.2, 1.0], redraw=True):
    # update parameter for split x axis inner divider line
    items = {'color': 'splitDividerColor', 'length': 'splitDividerLength', 'width': 'splitDividerWidth',\
             'dash': 'splitDividerDashStyle', 'angle': 'splitDividerAngle', 'location': 'splitDividerLocation'}
    if(param in items):
      if(value == self.__dict__[items[param]]):
        redraw = False
      self.__dict__[items[param]] = value

    # refresh plots
    if(redraw and self.splitShow and self.splitDivider):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

  def setSplitDivider(self, value=True, redraw=True):
    # toggles display of split divider lines
    if(value == self.splitDivider):
      redraw = False
    self.splitDivider = value

    # refresh plots
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

  def changeSplitFraction(self, value=1.0, redraw=True):
    # update parameters
    if (value == self.splitFraction):
      redraw = False
    self.splitFraction = value
    # call toggleSplit fxn
    self.toggleSplit(redraw=redraw, toggled=False)

  def changeSplitPad(self, value=0.02, redraw=True):
    # update parameters
    if (value == self.splitPad):
      redraw = False
    # implement check for too large split fraction
    maxSplit = 1 - self.padSize['left'] - (1 - self.padSize['right']) - 0.1
    if(value > maxSplit):
      if(hasattr(self.parent, 'statusbar')):
        self.parent.statusbar.showMessage('Cannot set split padding to ' + self.parent.formatNumber(value) + ', using ' + self.parent.formatNumber(maxSplit) + ' instead!', self.parent.STATUS_TIME)
      value, redraw = maxSplit, True
    self.splitPad = value
    # call toggleSplit fxn
    self.toggleSplit(redraw=redraw, toggled=False)

  def toggleSplitYCallback(self):
    # callback function splitY button
    state = self.splitYButton.isChecked()
    # toggle button on 2nd axes tab
    if(hasattr(self.parent, 'graphicsarea2')):
      self.parent.graphicsarea2.configSplitYLabel.setChecked(state)
    # do the actual toggling
    self.setSplitY(value=state, redraw=True)

  def updateSplitYDivider(self, param='color', value=[0.2, 0.2, 0.2, 1.0], redraw=True):
    # update parameter for split x axis inner divider line
    items = {'color': 'splitYDividerColor', 'length': 'splitYDividerLength', 'width': 'splitYDividerWidth',\
             'dash': 'splitYDividerDashStyle', 'angle': 'splitYDividerAngle', 'location': 'splitYDividerLocation'}
    if(param in items):
      if(value == self.__dict__[items[param]]):
        redraw = False
      self.__dict__[items[param]] = value

    # refresh plots
    if(redraw and self.splitY and self.splitYDivider):
      self.dataplotwidget.myRefresh()

  def setSplitYDivider(self, value=True, redraw=True):
    # toggles display of split y divider lines
    if(value == self.splitYDivider):
      redraw = False
    self.splitYDivider = value

    # refresh plots
    if(redraw):
      self.dataplotwidget.myRefresh()

  def setSplitY(self, value=True, redraw=True):
    # toggles splitting of y axes
    if(value == self.splitY):
      redraw = False
    self.splitY = value
    # update button in y control menu
    self.splitYButton.setChecked(self.splitY)
    # call toggleSplitY fxn
    self.toggleSplitY(redraw=redraw)

  def changeSplitYFraction(self, value=1.0, redraw=True):
    # update parameters
    if (value == self.splitYFraction):
      redraw = False
    self.splitYFraction = value
    # call toggleSplitY fxn
    self.toggleSplitY(redraw=redraw, toggled=False)

  def changeSplitYPad(self, value=0.02, redraw=True):
    # update parameters
    if (value == self.splitYPad):
      redraw = False
    # implement check for too large split fraction
    maxSplit = 1 - self.padSize['bottom'] - (1 - self.padSize['top']) - 0.1
    if(value > maxSplit):
      if(hasattr(self.parent, 'statusbar')):
        self.parent.statusbar.showMessage('Cannot set split padding to ' + self.parent.formatNumber(value) + ', using ' + self.parent.formatNumber(maxSplit) + ' instead!', self.parent.STATUS_TIME)
      value, redraw = maxSplit, True
    self.splitYPad = value
    # call toggleSplit fxn
    self.toggleSplitY(redraw=redraw, toggled=False)

  def setInnerYParameter(self, param='axes', state=True, redraw=True):
    # changes parameters for display of items at split y axis interface
    if(param in ['axes', 'ticks']):
      if(param == 'axes'):
        self.innerAxesY = state
      else:
        self.innerTicksY = state
        
      # apply these settings
      self.updateInnerYSituation()

      # issue plot refreshs
      if(redraw):
        self.dataplotwidget.myRefresh()

  def updateInnerYSituation(self):
    # deals with axis and tick settings at inner boundary of split y axis
    if(self.splitY):
      # deal with axis spines at interface
      for axis in [self.ax, self.ax_div]:
        axis.spines['bottom'].set_visible(self.innerAxesY and self.axisVisible['bottom'])
        axis.spines['bottom'].set_linewidth(self.axisWidth['bottom'])
        axis.spines['bottom'].set_linestyle(self.axisStyle['bottom'])
        axis.spines['bottom'].set_capstyle(self.axisDashStyle['bottom'])
        axis.spines['bottom'].set_color(self.axisColor['bottom'])
        if(self.axisBoundaryCheck['bottom']):
          lower, upper = self.axisBoundary['bottom']
          axis.spines['bottom'].set_bounds(lower, upper)
        else:
          axis.spines['bottom']._bounds = None

      # deal with axis on secondary plot
      for axis in [self.ax, self.ax_div, self.ax2, self.ax2_div]:
        if(axis in [self.ax, self.ax_div]):
          axis.spines['top'].set_visible(self.axisVisible['top'])
        else:
          axis.spines['top'].set_visible(self.innerAxesY and self.axisVisible['top'])
        axis.spines['top'].set_linewidth(self.axisWidth['top'])
        axis.spines['top'].set_linestyle(self.axisStyle['top'])
        axis.spines['top'].set_capstyle(self.axisDashStyle['top'])
        axis.spines['top'].set_color(self.axisColor['top'])
        if(self.axisBoundaryCheck['top']):
          lower, upper = self.axisBoundary['top']
          axis.spines['top'].set_bounds(lower, upper)
        else:
          axis.spines['top']._bounds = None
      
      # now deal with ticks
      for axis in [self.ax, self.ax_div]:
        # need to enable axis visibility to show ticks
        axis.xaxis.set_visible(True)
        axis.xaxis.set_tick_params(which='both', top=self.ticksVisible['top'], bottom=self.ticksVisible['bottom'] and self.innerTicksY,\
                                   width=self.ticksWidth['top'], color=self.ticksColor['top'], direction=self.ticksDirection['top'])
        axis.xaxis.set_tick_params(which='major', length=self.ticksLength['top'])
        axis.xaxis.set_tick_params(which='minor', length=self.ticksLength['top'] * self.ticksXMinorRelativeLength)

      for axis in [self.ax2, self.ax2_div]:
        axis.xaxis.set_tick_params(which='both', top=self.ticksVisible['top'] and self.innerTicksY, bottom=self.ticksVisible['bottom'])

      # ensure that no tick labels displayed at split
      self.ax.xaxis.set_tick_params(which='both', labeltop=self.ticksLabelShow['top'] and self.ticksXShow, labelbottom=False)
      self.ax_div.xaxis.set_tick_params(which='both', labeltop=self.ticksLabelShow['top'] and self.ticksXShow, labelbottom=False)
      self.ax2.xaxis.set_tick_params(which='both', labeltop=False, labelbottom=self.ticksLabelShow['bottom'] and self.ticksXShow)
      self.ax2_div.xaxis.set_tick_params(which='both', labeltop=False, labelbottom=self.ticksLabelShow['bottom'] and self.ticksXShow)

  def toggleSplitYHelper(self, state=True):
    # helper function called by toggleSplitY and loadState to deal with split cursors
    if(state):
      # update coords for rectangle selector
      self.storeCoord = []
  
      if((self.cursor != None) and (self.cursor2 == None)):
        self.cursor2 = MyCursor(self.ax2, useblit=True, color='black', linewidth=0.5)
        self.cursor2.setParent(self)
        self.cursor2.setAx2(self.ax2)
        
        # link cursors
        self.cursor.setTwinVertical(self.cursor2)
        self.cursor2.setTwinVertical(self.cursor)
        
        # toggle visibility
        self.cursor2.toggleVisibility(self.cursorVisible, None)
        
      if(self.splitShow and (self.cursor_div != None) and (self.cursor2_div == None)):
        self.cursor2_div = MyCursor(self.ax2_div, useblit=True, color='black', linewidth=0.5)
        self.cursor2_div.setParent(self)
        self.cursor2_div.setAx2(self.ax2_div)
        
        # link cursors
        if(self.cursor_div != None):
          self.cursor_div.setTwinVertical(self.cursor2_div)
          self.cursor2_div.setTwinVertical(self.cursor_div)
        if(self.cursor2 != None):
          self.cursor2.setTwin(self.cursor2_div)
          self.cursor2_div.setTwin(self.cursor2)
        
        # toggle visibility
        self.cursor2_div.toggleVisibility(self.cursorVisible, None)
    else:
      # update coords for rectangle selector
      self.storeCoord = []
      
      if(self.cursor != None):
        self.cursor.setTwinVertical(None)
      if(self.cursor_div != None):
        self.cursor_div.setTwinVertical(None)
      if(self.cursor2 != None):
        del self.cursor2
        self.cursor2 = None
      if(self.cursor2_div != None):
        del self.cursor2_div
        self.cursor2_div = None

  def toggleSplitY(self, redraw=True, toggled=True):
    # toggle second axes
    self.toggleSecondAxes(self.isSecondAxesActive())
    
    # helper function for split y axes
    if(self.splitY):
      # implement check for too large split fraction
      maxSplit = 1 - self.padSize['bottom'] - (1 - self.padSize['top']) - 0.1
      if(self.splitYPad > maxSplit):
        if(hasattr(self.parent, 'statusbar')):
          self.parent.statusbar.showMessage('Had to reduce split y axis padding to ' + self.parent.formatNumber(maxSplit) + ', rather than ' + self.parent.formatNumber(self.splitYPad) + '!', self.parent.STATUS_TIME)
        self.splitYPad = maxSplit

      # since we abandoned make_axes_locatable, we also need to implement check for split on x axis
      ybreak = (self.padSize['top'] - self.padSize['bottom'] - self.splitYPad) * self.splitYFraction / (1.0 + self.splitYFraction)
      if(self.splitShow):
        xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['left'] + xbreak,  self.padSize['top']]]))
        self.ax_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['right'],  self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['bottom'] + ybreak]]))
        self.ax2_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['bottom'] + ybreak]]))
      else:
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['right'], self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['bottom'] + ybreak]]))
    else:
      if(self.splitShow):
        xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak,  self.padSize['top']]]))
        self.ax_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'],  self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['top']]]))
        self.ax2_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
      else:
        self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
        self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))

    # adjust some graphic settings if y axes are split/unsplit
    if(self.splitY):
      if(toggled):
        if(self.canvasFill):
          # deal with patch background
          self.ax.patch.set_visible(True)
          self.ax_div.patch.set_visible(True)
          if(self.canvasGradientVisible):
            self.ax.patch.set_facecolor('none')
            self.ax_div.patch.set_facecolor('none')
          else:
            self.ax.patch.set_facecolor(self.canvasColor)
            self.ax_div.patch.set_facecolor(self.canvasColor)
  
        # store some settings
        items = 'axisVisible,axisWidth,axisStyle,axisDashStyle,axisColor,axisPosition,axisPositionValue'.split(',')
        items.extend('ticksVisible,ticksWidth,ticksLength,ticksColor,ticksDirection,ticksLabelShow'.split(','))
        for item in items:
          self.splitYBackup[item] = {}
          for item2 in ['left', 'right']:
            self.splitYBackup[item][item2] = self.__dict__[item][item2 + '2']
            
        items = 'ticksY2Show,ticksY2Color,ticksY2Size,ticksY2Weight,ticksY2Style,ticksY2Angle'.split(',')
        items.extend('ticksY2Alignment,ticksY2AlignmentVertical,ticksY2Pad,ticksY2Pad2,labelY2Show'.split(','))
        items.extend('ticksY2Minor,ticksY2MinorRelativeLength,ticksY2Format,ticksY2FormatPrecision,ticksY2FormatTrailZero'.split(','))
        items.extend('ticksY2FormatSeparator,ticksY2FormatComma,ticksY2FormatPrefix,ticksY2FormatPostfix'.split(','))
        for item in items:
          self.splitYBackup[item] = self.__dict__[item]
  
        items = 'tickFont,gridVisible,gridWidth,gridStyle,gridDashStyle,gridColor,gridOrder'.split(',')
        for item in items:
          self.splitYBackup[item] = self.__dict__[item]['y2']
          
        # adjust display of axis
        for item in ['left', 'right']:
          self.setAxisVisibility(value=self.axisVisible[item], axis=item + '2', redraw=False, target='plot')
          self.setAxisWidth(value=self.axisWidth[item], axis=item + '2', redraw=False, target='plot')
          self.setAxisStyle(value=self.axisStyle[item], axis=item + '2', redraw=False, target='plot')
          self.setAxisDashStyle(value=self.axisDashStyle[item], axis=item + '2', redraw=False, target='plot')
          self.setAxisColor(value=self.axisColor[item], axis=item + '2', redraw=False, target='plot')
          # also need to copy axis position settings
          self.setAxisPosition(value=self.axisPosition[item], axis=item + '2', redraw=False, target='plot')
          self.setAxisPositionValue(value=self.axisPositionValue[item], axis=item + '2', redraw=False, target='plot')
          
        # also need to turn on shifted top and bottom axis
        for item in ['top', 'bottom']:
          # phew, we will need to update setAxisPositionHelper() to accommodate this option
          self.setAxisPositionHelper(axis=item, plotobject=self.dataplotwidget, axisobject=self.ax, target='plot', secondAxes=False, splitX=False, splitY=True)
          self.setAxisPositionHelper(axis=item, plotobject=self.dataplotwidget, axisobject=self.ax_div, target='plot', secondAxes=False, splitX=False, splitY=True)
  
        # adjust display of ticks
        for item in ['left', 'right']:
          self.setTickMarkVisibility(value=self.ticksVisible[item], axis=item + '2', redraw=False, target='plot')
          self.setTickMarkWidth(value=self.ticksWidth[item], axis=item + '2', redraw=False, target='plot')
          self.setTickMarkLength(value=self.ticksLength[item], axis=item + '2', redraw=False, target='plot')
          self.setTickMarkColor(value=self.ticksColor[item], axis=item + '2', redraw=False, target='plot')
          self.setTickMarkDirection(value=self.ticksDirection[item], axis=item + '2', redraw=False, target='plot')
          self.setTickMarkLabelShow(value=self.ticksLabelShow[item], axis=item + '2', redraw=False, target='plot')
  
        # tick marks
        self.toggleTicksLabel(value=self.ticksYShow, axis='y2', redraw=False, target='plot')
        self.setTickLabelColor(value=self.ticksYColor, axis='y2', redraw=False, target='plot')
        self.setTickLabelSize(value=self.ticksYSize, axis='y2', redraw=False, target='plot')
        self.setTickLabelBold(value=self.ticksYWeight, axis='y2', redraw=False, target='plot')
        self.setTickLabelItalic(value=self.ticksYStyle, axis='y2', redraw=False, target='plot')
        self.setTickLabelAngle(value=self.ticksYAngle, axis='y2', redraw=False, target='plot')
        self.setTickLabelAlignment(value=self.ticksYAlignment, axis='y2', redraw=False, target='plot')
        self.setTickLabelAlignmentVertical(value=self.ticksYAlignmentVertical, axis='y2', redraw=False, target='plot')
        self.setTickLabelPad(value=self.ticksYPad, axis='y2', redraw=False, target='plot')
        self.setTickLabelPad2(value=self.ticksYPad2, axis='y2', redraw=False, target='plot')
        self.setTickFont(value=self.tickFont['y'], axis='y2', redraw=False, target='plot')
  
        # grid and axis labels
        self.toggleAxisLabel(value=False, axis='y2', redraw=False, target='plot')
        self.setGridVisibility(value=self.gridVisible['y'], axis='y2', redraw=False, target='plot')
        self.setGridWidth(value=self.gridWidth['y'], axis='y2', redraw=False, target='plot')
        self.setGridStyle(value=self.gridStyle['y'], axis='y2', redraw=False, target='plot')
        self.setGridDashStyle(value=self.gridDashStyle['y'], axis='y2', redraw=False, target='plot')
        self.setGridColor(value=self.gridColor['y'], axis='y2', redraw=False, target='plot')
        self.setGridOrder(value=self.gridOrder['y'], axis='y2', redraw=False, target='plot')
        ###consider also updating minor grid (although should be slaved to major ticks)
  
        # deal with the innner situation
        self.updateInnerYSituation()
  
        # update graphiscarea 2
        if(hasattr(self.parent, 'graphicsarea2')):
          self.parent.graphicsarea2.updateFields()
  
          self.parent.graphicsarea2.outerBox8.show()
          ###self.parent.graphicsarea2.configSplitYBox.show()
          ###self.parent.graphicsarea2.splitYSeparator.show()
          # check whether advanced graphics enabled
          if(self.parent.graphicsarea.advancedExport['advancedGraphics']):
            if(hasattr(self.parent.graphicsarea2, 'configSplitYBox2')):
              self.parent.graphicsarea2.configSplitYBox2.show()
            if(hasattr(self.parent.graphicsarea2, 'configSplitYBox3')):
              self.parent.graphicsarea2.configSplitYBox3.show()
  
        # draw all items on split axes and check them
        # change of plan -- duplicate all items but hide the ones that are not visible
        for data in self.parent.data:
          data.onBothAxes = True
          data.duplicateForSplit()
          if(not data.visibility):
            data.setVisibility(state=False, redraw=False)
  
        for curve in self.parent.fit:
          curve.onBothAxes = True
          curve.duplicateForSplit()
          if(not curve.visibility):
            curve.setVisibility(state=False, redraw=False)
          
        for extra in self.parent.extras:
          extra.onBothAxes = True
          extra.duplicateForSplit()
          if(not extra.visibility):
            extra.setVisibility(state=False, redraw=False)
            
        # also draw selected data on both axes
        if(self.parent.selectedData.isLive):
          self.parent.selectedData.duplicateForSplit()
          self.parent.selectedData.style.update(self.parent.selectedData.theseStyles)
        
        # set axis mode and limits to match those of first axes
        self.secondModeSelector.blockSignals(True)
        index = self.secondModeSelector.findText(self.modeY)
        self.secondModeSelector.setCurrentIndex(index)
        self.secondModeSelector.blockSignals(False)
        self.changeAxisMode('y2', redraw=False)
        # set axes limits for split effect
        if(self.modeY == 'linear'):
          splitYAt = self.minY + self.splitYFraction / (self.splitYFraction + 1) * (self.maxY - self.minY)
        else:
          splitYAt = np.exp(np.log(self.minY) + self.splitYFraction / (self.splitYFraction + 1) * (np.log(self.maxY) - np.log(self.minY)))
        # discard flipped state of second axis
        # need to check whether main axis is inverted or not
        if(self.minY < self.maxY):
          lower, upper = sorted([self.minY, splitYAt])
        else:
          upper, lower = sorted([self.minY, splitYAt])
        self.setAxisLimits(lower=lower, upper=upper, axis='y2', updateLabel=False, target='plot', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=False)
        self.secondLowerLimit.setText(self.lowerLimity.text())
        self.secondUpperLimit.setText(self.parent.formatNumber(splitYAt))
        self.setAxisLimits(lower=splitYAt, upper=self.maxY, axis='y', updateLabel=False, target='plot', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.lowerLimity.setText(self.parent.formatNumber(splitYAt))

        # also need to copy custom tick formatting (if axis modes on y axes are the same)
        # => but we just copied the mode, so this is always the case
        self.setMinorTick(value=self.ticksYMinor, axis='y2', redraw=False, target='plot')
        self.setMinorTickRelativeLength(value=self.ticksYMinorRelativeLength, axis='y2', redraw=False, target='plot')
        self.setTickFormat(axis='y2', value=self.ticksYFormat, redraw=False)
        self.setTickFormatPrecision(axis='y2', value=self.ticksYFormatPrecision, redraw=False)
        self.setFormatTrailZero(axis='y2', value=self.ticksYFormatTrailZero, redraw=False)
        self.setFormatSeparator(axis='y2', value=self.ticksYFormatSeparator, redraw=False)
        self.setFormatComma(axis='y2', value=self.ticksYFormatComma, redraw=False)
        self.setFormatFix(axis='y2', value=self.ticksYFormatPrefix, redraw=False, prefix=True)
        self.setFormatFix(axis='y2', value=self.ticksYFormatPostfix, redraw=False, prefix=False)
        
        # activate rectangle selectors and generate new cursor if needed
        self.toggleSplitYHelper(state=True)
    else:
      # deal with patch background
      self.ax.patch.set_visible(False)
      self.ax_div.patch.set_visible(False)
      
      # adjust display of axis
      for item in ['left', 'right']:
        if(('axisVisible' in self.splitYBackup) and (item in self.splitYBackup['axisVisible'])):
          if(self.splitShow and (item == 'right')):
            self.setAxisVisibility(value=self.innerAxes and self.axisVisible['right2'], axis='right2', redraw=False, target='plot')
          else:
            if(self.isSecondAxesActive()):
              self.setAxisVisibility(value=self.splitYBackup['axisVisible'][item], axis=item + '2', redraw=False, target='plot')
            else:
              # hide axes of second plot in case no items exist on second axes
              self.setAxisVisibility(value=False, axis=item + '2', redraw=False, target='plot')
        if(('axisWidth' in self.splitYBackup) and (item in self.splitYBackup['axisWidth'])):
          self.setAxisWidth(value=self.splitYBackup['axisWidth'][item], axis=item + '2', redraw=False, target='plot')
        if(('axisStyle' in self.splitYBackup) and (item in self.splitYBackup['axisStyle'])):
          self.setAxisStyle(value=self.splitYBackup['axisStyle'][item], axis=item + '2', redraw=False, target='plot')
        if(('axisDashStyle' in self.splitYBackup) and (item in self.splitYBackup['axisDashStyle'])):
          self.setAxisDashStyle(value=self.splitYBackup['axisDashStyle'][item], axis=item + '2', redraw=False, target='plot')
        if(('axisColor' in self.splitYBackup) and (item in self.splitYBackup['axisColor'])):
          self.setAxisColor(value=self.splitYBackup['axisColor'][item], axis=item + '2', redraw=False, target='plot')
        if(('axisPosition' in self.splitYBackup) and (item in self.splitYBackup['axisPosition'])):
          self.setAxisPosition(value=self.splitYBackup['axisPosition'][item], axis=item + '2', redraw=False, target='plot')
        if(('axisPositionValue' in self.splitYBackup) and (item in self.splitYBackup['axisPositionValue'])):
          self.setAxisPositionValue(value=self.splitYBackup['axisPositionValue'][item], axis=item + '2', redraw=False, target='plot')

      # adjust display of ticks
      for item in ['left', 'right']:
        if(('ticksVisible' in self.splitYBackup) and (item in self.splitYBackup['ticksVisible'])):
          self.setTickMarkVisibility(value=self.splitYBackup['ticksVisible'][item], axis=item + '2', redraw=False, target='plot')
        if(('ticksWidth' in self.splitYBackup) and (item in self.splitYBackup['ticksWidth'])):
          self.setTickMarkWidth(value=self.splitYBackup['ticksWidth'][item], axis=item + '2', redraw=False, target='plot')
        if(('ticksLength' in self.splitYBackup) and (item in self.splitYBackup['ticksLength'])):
          self.setTickMarkLength(value=self.splitYBackup['ticksLength'][item], axis=item + '2', redraw=False, target='plot')
        if(('ticksColor' in self.splitYBackup) and (item in self.splitYBackup['ticksColor'])):
          self.setTickMarkColor(value=self.splitYBackup['ticksColor'][item], axis=item + '2', redraw=False, target='plot')
        if(('ticksDirection' in self.splitYBackup) and (item in self.splitYBackup['ticksDirection'])):
          self.setTickMarkDirection(value=self.splitYBackup['ticksDirection'][item], axis=item + '2', redraw=False, target='plot')
        if(('ticksLabelShow' in self.splitYBackup) and (item in self.splitYBackup['ticksLabelShow'])):
          self.setTickMarkLabelShow(value=self.splitYBackup['ticksLabelShow'][item], axis=item + '2', redraw=False, target='plot')

      # tick marks
      if('ticksY2Show' in self.splitYBackup):
        self.toggleTicksLabel(value=self.splitYBackup['ticksY2Show'], axis='y2', redraw=False, target='plot')
      if('ticksY2Color' in self.splitYBackup):
        self.setTickLabelColor(value=self.splitYBackup['ticksY2Color'], axis='y2', redraw=False, target='plot')
      if('ticksY2Size' in self.splitYBackup):
        self.setTickLabelSize(value=self.splitYBackup['ticksY2Size'], axis='y2', redraw=False, target='plot')
      if('ticksY2Weight' in self.splitYBackup):
        self.setTickLabelBold(value=self.splitYBackup['ticksY2Weight'], axis='y2', redraw=False, target='plot')
      if('ticksY2Style' in self.splitYBackup):
        self.setTickLabelItalic(value=self.splitYBackup['ticksY2Style'], axis='y2', redraw=False, target='plot')
      if('ticksY2Angle' in self.splitYBackup):
        self.setTickLabelAngle(value=self.splitYBackup['ticksY2Angle'], axis='y2', redraw=False, target='plot')
      if('ticksY2Alignment' in self.splitYBackup):
        self.setTickLabelAlignment(value=self.splitYBackup['ticksY2Alignment'], axis='y2', redraw=False, target='plot')
      if('ticksY2AlignmentVertical' in self.splitYBackup):
        self.setTickLabelAlignmentVertical(value=self.splitYBackup['ticksY2AlignmentVertical'], axis='y2', redraw=False, target='plot')
      if('ticksY2Pad' in self.splitYBackup):
        self.setTickLabelPad(value=self.splitYBackup['ticksY2Pad'], axis='y2', redraw=False, target='plot')
      if('ticksY2Pad2' in self.splitYBackup):
        self.setTickLabelPad2(value=self.splitYBackup['ticksY2Pad2'], axis='y2', redraw=False, target='plot')
      if(('tickFont' in self.splitYBackup) and ('y2' in self.splitYBackup['tickFont'])):
        self.setTickFont(value=self.splitYBackup['tickFont']['y2'], axis='y2', redraw=False, target='plot')
        
      # custom tick formatting
      if('ticksY2Minor' in self.splitYBackup):
        self.setMinorTick(value=self.splitYBackup['ticksY2Minor'], axis='y2', redraw=False, target='plot')
      if('ticksY2MinorRelativeLength' in self.splitYBackup):
        self.setMinorTickRelativeLength(value=self.splitYBackup['ticksY2MinorRelativeLength'], axis='y2', redraw=False, target='plot')
      if('ticksY2Format' in self.splitYBackup):
        self.setTickFormat(axis='y2', value=self.splitYBackup['ticksY2Format'], redraw=False)
      if('ticksY2FormatPrecision' in self.splitYBackup):
        self.setTickFormatPrecision(axis='y2', value=self.splitYBackup['ticksY2FormatPrecision'], redraw=False)
      if('ticksY2FormatTrailZero' in self.splitYBackup):
        self.setFormatTrailZero(axis='y2', value=self.splitYBackup['ticksY2FormatTrailZero'], redraw=False)
      if('ticksY2FormatSeparator' in self.splitYBackup):
        self.setFormatSeparator(axis='y2', value=self.splitYBackup['ticksY2FormatSeparator'], redraw=False)
      if('ticksY2FormatComma' in self.splitYBackup):
        self.setFormatComma(axis='y2', value=self.splitYBackup['ticksY2FormatComma'], redraw=False)
      if('ticksY2FormatPrefix' in self.splitYBackup):
        self.setFormatFix(axis='y2', value=self.splitYBackup['ticksY2FormatPrefix'], redraw=False, prefix=True)
      if('ticksYFormatPostfix' in self.splitYBackup):
        self.setFormatFix(axis='y2', value=self.splitYBackup['ticksYFormatPostfix'], redraw=False, prefix=False)

      # grid and axis labels
      if('labelY2Show' in self.splitYBackup):
        self.toggleAxisLabel(value=self.splitYBackup['labelY2Show'], axis='y2', redraw=False, target='plot')
      if('gridVisible' in self.splitYBackup):
        self.setGridVisibility(value=self.splitYBackup['gridVisible'], axis='y2', redraw=False, target='plot')
      if('gridWidth' in self.splitYBackup):
        self.setGridWidth(value=self.splitYBackup['gridWidth'], axis='y2', redraw=False, target='plot')
      if('gridStyle' in self.splitYBackup):
        self.setGridStyle(value=self.splitYBackup['gridStyle'], axis='y2', redraw=False, target='plot')
      if('gridDashStyle' in self.splitYBackup):
        self.setGridDashStyle(value=self.splitYBackup['gridDashStyle'], axis='y2', redraw=False, target='plot')
      if('gridColor' in self.splitYBackup):
        self.setGridColor(value=self.splitYBackup['gridColor'], axis='y2', redraw=False, target='plot')
      if('gridOrder' in self.splitYBackup):
        self.setGridOrder(value=self.splitYBackup['gridOrder'], axis='y2', redraw=False, target='plot')
      ###consider also updating minor grid (although should be slaved to major ticks)

      # deal with axis on secondary plot
      for axis in [self.ax, self.ax_div]:
        axis.spines['top'].set_visible(False)
      for axis in [self.ax2, self.ax2_div]:
        axis.spines['top'].set_visible(self.axisVisible['top'])

      # update graphiscarea 2
      if(hasattr(self.parent, 'graphicsarea2')):
        self.parent.graphicsarea2.updateFields()

        ###self.parent.graphicsarea2.configSplitYBox.hide()
        self.parent.graphicsarea2.outerBox8.hide()
        self.parent.graphicsarea2.configSplitYBox2.hide()
        self.parent.graphicsarea2.configSplitYBox3.hide()
        ###self.parent.graphicsarea2.splitYSeparator.hide()

      # clear all duplicated graphics
      for data in self.parent.data:
        data.removeFromSplit()

      for curve in self.parent.fit:
        curve.removeFromSplit()

      for extra in self.parent.extras:
        extra.removeFromSplit()

      # remove selected data from both axes
      if(self.parent.selectedData.isLive):
        self.parent.selectedData.removeFromSplit()
          
      # clear stored info
      self.splitYBackup = {}
      
      # reset axis limits to nonsplit
      if((self.modeY == 'linear') or (self.minY2 > 0)):
        self.setAxisLimits(lower=self.minY2, upper=self.maxY, axis='y', updateLabel=False, target='plot', redraw=False, updateTicks=True, updateGrid=True, preserveInverted=True)
        self.lowerLimity.setText(self.parent.formatNumber(self.minY2))
      
      # deactivate cursors
      self.toggleSplitYHelper(state=False)

    # refresh tables in objects tab
    self.parent.objectsarea.refreshDataTable()
    self.parent.objectsarea.refreshCurvesTable()
    self.parent.objectsarea.refreshExtrasTable()

    # refresh plots
    if(redraw):
      self.dataplotwidget.myRefresh()

    # for some strange reason we should reset the axis ticks here b/c minor autor locator gets confused
    # reason unclear, could be matplotlib bug??
    ###if(self.splitY):
    if(True):
      redrawAgain = False
      if(self.ticksYAuto):
        self.setAutoTicks(axis='y', redraw=False, target='plot')
      else:
        # need to preserve axis limits before calling setAxisTicks()
        minY, maxY = self.minY, self.maxY
        self.setAxisTicks(value=self.ticksY, axis='y', redraw=False, target='plot')

        # now restore the original limits and (alas!) issue another redraw
        self.setAxisLimits(lower=minY, upper=maxY, axis='y', updateLabel=True, target='plot', redraw=False, updateTicks=False, updateGrid=True, preserveInverted=True)
        redrawAgain = True
      if(self.ticksY2Auto):
        self.setAutoTicks(axis='y2', redraw=False, target='plot')
      else:
        # need to preserve axis limits before calling setAxisTicks()
        minY, maxY = self.minY2, self.maxY2
        self.setAxisTicks(value=self.ticksY, axis='y2', redraw=False, target='plot')

        # now restore the original limits and (alas!) issue another redraw
        self.setAxisLimits(lower=minY, upper=maxY, axis='y2', updateLabel=True, target='plot', redraw=False, updateTicks=False, updateGrid=True, preserveInverted=True)
        redrawAgain = True

      # do we need to redraw?
      if(redrawAgain):
        self.dataplotwidget.myRefresh()

  def reportState(self):
    # returns current settings for save state function
    reportItems = ['minX', 'maxX', 'minY', 'maxY', 'minResidY', 'maxResidY', 'modeX', 'modeY', 'autoScaleX', 'autoScaleY']
    reportItems.extend(['minY2', 'maxY2', 'modeY2', 'autoScaleY2'])
    reportItems.extend(['splitShow', 'splitY', 'minX_div', 'maxX_div', 'modeX_div', 'autoScaleX_div'])
    retv = {}

    for entry in reportItems:
      if(hasattr(self, entry)):
        value = getattr(self, entry)
        retv[entry] = value
        
    # also add current zoom level (which, however, belongs to parent object)
    currZoomIndex = self.parent.zoomSelector.currentIndex()
    retv['zoomLevel'] = self.parent.zoomLevels[::-1][currZoomIndex]
    
    return retv
  
  def restoreState(self, settings, updateTicks=True):
    # restores settings from load state function
    for entry in settings:
      if(hasattr(self, entry)):
        self.__dict__[entry] = settings[entry]

    # remember desired axes limits
    orig_minX, orig_maxX = self.minX, self.maxX
    orig_minY, orig_maxY = self.minY, self.maxY
    orig_minResidY, orig_maxResidY = self.minResidY, self.maxResidY
    orig_minY2, orig_maxY2 = self.minY2, self.maxY2
    orig_minX_div, orig_maxX_div = self.minX_div, self.maxX_div
    
    # apply these settings
    # autoscale
    self.autoScaleCheckX.setChecked(self.autoScaleX)
    self.autoScaleCheckY.setChecked(self.autoScaleY)
    self.xSplitAutoScaleCheckX.setChecked(self.autoScaleX_div)

    # axes modes
    index = self.modeSelectorx.findText(self.modeX)
    if((index + 1) and (index != self.modeSelectorx.currentIndex())):
      self.modeSelectorx.blockSignals(True)
      self.modeSelectorx.setCurrentIndex(index)
      self.modeSelectorx.blockSignals(False)
      self.changeAxisMode('x', redraw=False)
    index = self.modeSelectory.findText(self.modeY)
    if((index + 1) and (index != self.modeSelectory.currentIndex())):
      self.modeSelectory.blockSignals(True)
      self.modeSelectory.setCurrentIndex(index)
      self.modeSelectory.blockSignals(False)
      self.changeAxisMode('y', redraw=False)
    index = self.secondModeSelector.findText(self.modeY2)
    if((index + 1) and (index != self.secondModeSelector.currentIndex())):
      self.secondModeSelector.blockSignals(True)
      self.secondModeSelector.setCurrentIndex(index)
      self.secondModeSelector.blockSignals(False)
      self.changeAxisMode('y2', redraw=False)
    index = self.xSplitModeSelectorx.findText(self.modeX_div)
    if((index + 1) and (index != self.xSplitModeSelectorx.currentIndex())):
      self.xSplitModeSelectorx.blockSignals(True)
      self.xSplitModeSelectorx.setCurrentIndex(index)
      self.xSplitModeSelectorx.blockSignals(False)
      self.changeAxisMode('x2', redraw=False)
    
    # need to counteract resetting of axis ticks when adjusting axis mode
    minX, maxX = self.minX, self.maxX
    minY, maxY = self.minY, self.maxY
    minResidY, maxResidY = self.minResidY, self.maxResidY
    minY2, maxY2 = self.minY2, self.maxY2
    minX_div, maxX_div = self.minX_div, self.maxX_div
    
    # apply tick formatting
    self.ticksXDataSet = -1
    if(self.ticksXAuto):
      self.setAutoTicks(axis='x', redraw=False, target='plot')
      self.setAutoTicks(axis='x', redraw=False, target='resid')
    else:
      ticksXLabel = self.ticksXLabel
      self.setAxisTicks(value=self.ticksX, axis='x', redraw=False, target='plot')
      self.setAxisTicks(value=self.ticksX, axis='x', redraw=False, target='resid')
      # apply axis labels?
      self.ticksXLabel = ticksXLabel
      if(len(self.ticksXLabel) and (len(self.ticksXLabel) == len(self.ticksX))):
        for axisobject in [self.ax, self.ax_resid]:
          axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksX))
          axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksXLabel))
    if(self.ticksXAuto_div):
      self.setAutoTicks(axis='x2', redraw=False, target='plot')
      self.setAutoTicks(axis='x2', redraw=False, target='resid')
    else:
      ticksXLabel = self.ticksXLabel_div
      self.setAxisTicks(value=self.ticksX_div, axis='x2', redraw=False, target='plot')
      self.setAxisTicks(value=self.ticksX_div, axis='x2', redraw=False, target='resid')
      # apply axis labels?
      self.ticksXLabel_div = ticksXLabel
      if(len(self.ticksXLabel_div) and (len(self.ticksXLabel_div) == len(self.ticksX_div))):
        for axisobject in [self.ax2_div, self.ax_resid_div]:
          axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksX_div))
          axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksXLabel_div))
      
    if(self.ticksYAuto):
      self.setAutoTicks(axis='y', redraw=False, target='plot')
    else:
      self.setAxisTicks(value=self.ticksY, axis='y', redraw=False, target='plot')
      
    if(self.ticksYAuto):
      self.setAutoTicks(axis='resid', redraw=False, target='resid')
    else:
      self.setAxisTicks(value=self.ticksResidY, axis='resid', redraw=False, target='resid')

    if(self.ticksY2Auto):
      self.setAutoTicks(axis='y2', redraw=False, target='plot')
    else:
      self.setAxisTicks(value=self.ticksY2, axis='y2', redraw=False, target='plot')

    # retrieve settings
    self.minX, self.maxX = minX, maxX
    self.minY, self.maxY = minY, maxY
    self.minResidY, self.maxResidY = minResidY, maxResidY
    self.minY2, self.maxY2 = minY2, maxY2
    self.minX_div, self.maxX_div = minX_div, maxX_div
    
    # check wether we can go to original values specified in state
    if(self.modeX == 'linear'):
      self.minX, self.maxX = orig_minX, orig_maxX
    else:
      self.minX, self.maxX = [i if (i > 0) else j for i, j in zip([orig_minX, orig_maxX], [self.minX, self.maxX])]
    if(self.modeY == 'linear'):
      self.minY, self.maxY = orig_minY, orig_maxY
    else:
      self.minY, self.maxY = [i if (i > 0) else j for i, j in zip([orig_minY, orig_maxY], [self.minY, self.maxY])]
    if(self.modeY2 == 'linear'):
      self.minY2, self.maxY2 = orig_minY2, orig_maxY2
    else:
      self.minY2, self.maxY2 = [i if (i > 0) else j for i, j in zip([orig_minY2, orig_maxY2], [self.minY2, self.maxY2])]
    self.minResidY, self.maxResidY = orig_minResidY, orig_maxResidY
    if(self.modeX_div == 'linear'):
      self.minX_div, self.maxX_div = orig_minX_div, orig_maxX_div
    else:
      self.minX_div, self.maxX_div = [i if (i > 0) else j for i, j in zip([orig_minX_div, orig_maxX_div], [self.minX_div, self.maxX_div])]
    
    # adjust zoom level if specified
    if('zoomLevel' in settings):
      # set actual zoom level
      self.setZoomLevel(settings['zoomLevel'], redraw=False)
      # adjust selection dialog
      zoomDifference = [abs(settings['zoomLevel'] - i) for i in self.parent.zoomLevels[::-1]]
      zoomIndex = zoomDifference.index(min(zoomDifference))
      self.parent.zoomSelector.blockSignals(True)
      self.parent.zoomSelector.setCurrentIndex(zoomIndex)
      self.parent.zoomSelector.blockSignals(False)
      
    # toggle split x axis
    if(self.splitAxisButton.isChecked() != self.splitShow):
      self.splitAxisButton.setChecked(self.splitShow)
      self.toggleSplit(redraw=False, toggled=False)
      
    # toggle split y axis
    if(self.splitYButton.isChecked() != self.splitY):
      self.splitYButton.setChecked(self.splitY)
      self.toggleSplitY(redraw=False, toggled=False)

    # axes limits
    self.setAxisLimits(lower=self.minX, upper=self.maxX, axis='x', updateLabel=True, target='plot', redraw=False, updateTicks=updateTicks, updateGrid=True)
    self.setAxisLimits(lower=self.minY2, upper=self.maxY2, axis='y2', updateLabel=True, target='plot', redraw=False, updateTicks=updateTicks, updateGrid=True)
    self.setAxisLimits(lower=self.minY, upper=self.maxY, axis='y', updateLabel=True, target='plot', redraw=False, updateTicks=updateTicks, updateGrid=True)
    self.setAxisLimits(lower=self.minX, upper=self.maxX, axis='x', updateLabel=True, target='resid', redraw=False, updateTicks=updateTicks, updateGrid=True)
    self.setAxisLimits(lower=self.minResidY, upper=self.maxResidY, axis='y', updateLabel=True, target='resid', redraw=False, updateTicks=updateTicks, updateGrid=True)

    self.setAxisLimits(lower=self.minX_div, upper=self.maxX_div, axis='x2', updateLabel=True, target='plot', redraw=False, updateTicks=updateTicks, updateGrid=True)
    self.setAxisLimits(lower=self.minX_div, upper=self.maxX_div, axis='x2', updateLabel=True, target='resid', redraw=False, updateTicks=updateTicks, updateGrid=True)
    
    # draw zero resid line over (updated) x range
    self.handleResidZero = self.plotResidZero(self.handleResidZero, redraw=False)
    self.handleResidZero_div = self.plotResidZero(self.handleResidZero_div, redraw=False, splitX=True)

    # update tick entry fields
    for axis in ['x', 'x2', 'y', 'y2', 'resid']:
      self.updateTickEntryField(axis=axis)

  def incZoomLevel(self, increment=1):
    # used by mouse wheel to adjust zoom level
    currZoomIndex = self.parent.zoomSelector.currentIndex()
    # alter zoom level but restrain in bounds
    nuZoomIndex = currZoomIndex + increment
    nuZoomIndex = max(0, nuZoomIndex)
    nuZoomIndex = min(len(self.parent.zoomLevels) -1, nuZoomIndex)
    # do we need to change anything?
    if(currZoomIndex != nuZoomIndex):
      self.parent.zoomSelector.setCurrentIndex(nuZoomIndex)

  def setZoomLevel(self, zoomLevel=100, redraw=True):
    # sets zoom level of plots
    if(self.matplot.get_dpi() != zoomLevel):
      # get current size and scale factor
      scaleFactor = self.matplot.get_dpi() / zoomLevel
      currSize = self.matplot.get_size_inches()
      # change canvas DPI
      self.matplot.set_dpi(zoomLevel)
      # adjust size accordingly
      nuSize = [i * scaleFactor for i in currSize]
      self.matplot.set_size_inches(nuSize)
      self.horizontalRuler.updateRuler()
      self.verticalRuler.updateRuler()
      # check for presence of data selection (which would need to be adjusted)
      if(self.parent.selectedData.isLive):
        self.parent.selectedData.adjustMarkerSize(resid=False)
      # issue plot refresh
      if(redraw):
        self.dataplotwidget.myRefresh()
    
    if(self.residplot.get_dpi() != zoomLevel):
      # get current size and scale factor
      scaleFactor = self.residplot.get_dpi() / zoomLevel
      currSize = self.residplot.get_size_inches()
      # change canvas DPI
      self.residplot.set_dpi(zoomLevel)
      # adjust size accordingly
      nuSize = [i * scaleFactor for i in currSize]
      self.residplot.set_size_inches(nuSize)
      self.verticalRulerResid.updateRuler()
      # check for presence of data selection (which would need to be adjusted)
      if(self.parent.selectedData.isLive):
        self.parent.selectedData.adjustMarkerSize(resid=True)
      # issue plot refresh
      if(redraw):
        self.residplotwidget.myRefresh()

  def setTickFormatHelper(self, axis='x', redraw=True):
    # accessory function to format tick labels
    if(axis in ['x', 'residx']):
      useAxisobject, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, useFallback =\
        self.ax2, self.ticksXFormat, self.ticksXFormatPrecision, self.ticksXFormatTrailZero, self.ticksXFormatSeparator, self.ticksXFormatComma, self.fallback_ticksXFormat
      usePrefix, usePostfix = self.ticksXFormatPrefix, self.ticksXFormatPostfix
      if(axis == 'residx'):
        useAxisobject, item = self.ax_resid, 'resid'
      else:
        item = 'plot'
    elif(axis in ['x2', 'residx2']):
      useAxisobject, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, useFallback =\
        self.ax2_div, self.ticksXFormat_div, self.ticksXFormatPrecision_div, self.ticksXFormatTrailZero_div, self.ticksXFormatSeparator_div, self.ticksXFormatComma_div, self.fallback_ticksXFormat_div
      usePrefix, usePostfix = self.ticksXFormatPrefix_div, self.ticksXFormatPostfix_div
      if(axis == 'residx2'):
        useAxisobject, item = self.ax_resid_div, 'resid'
      else:
        item = 'plot'
    elif(axis == 'y'):
      useAxisobject, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, useFallback =\
        self.ax, self.ticksYFormat, self.ticksYFormatPrecision, self.ticksYFormatTrailZero, self.ticksYFormatSeparator, self.ticksYFormatComma, self.fallback_ticksYFormat
      usePrefix, usePostfix = self.ticksYFormatPrefix, self.ticksYFormatPostfix
      item = 'plot'
    elif(axis == 'y2'):
      useAxisobject, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, useFallback =\
        self.ax2, self.ticksY2Format, self.ticksY2FormatPrecision, self.ticksY2FormatTrailZero, self.ticksY2FormatSeparator, self.ticksY2FormatComma, self.fallback_ticksY2Format
      usePrefix, usePostfix = self.ticksY2FormatPrefix, self.ticksY2FormatPostfix
      item = 'plot'
    elif(axis == 'resid'):
      useAxisobject, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, useFallback =\
        self.ax_resid, self.ticksResidYFormat, self.ticksResidYFormatPrecision, self.ticksResidYFormatTrailZero, self.ticksResidYFormatSeparator, self.ticksResidYFormatComma, self.fallback_ticksResidYFormat
      usePrefix, usePostfix = self.ticksResidYFormatPrefix, self.ticksResidYFormatPostfix
      item = 'resid'
    elif(axis in ['xs', 'residxs']):
      useAxisobject, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, useFallback =\
        None, self.slavedXFormat, self.slavedXFormatPrecision, self.slavedXFormatTrailZero, self.slavedXFormatSeparator, self.slavedXFormatComma, self.fallback_slavedXFormat
      usePrefix, usePostfix = self.slavedXFormatPrefix, self.slavedXFormatPostfix
      if(axis == 'residxs'):
        useAxisobject, item = None, 'resid'
      else:
        item = 'plot'
    else:
      useTicksFormat = 'tilt'
      item = 'none'
    
    if(item in ['plot', 'resid']):
      # now apply the altered tick formatting
      # this should be easy as long as we turn on custom formatting
      if(useTicksFormat != 'default'):
        autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, useTicksFormat, useTicksFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, usePrefix, usePostfix))
      elif(useFallback != None):
        autolabels = useFallback
      else:
        # scalar formatter should be safe (except when using data ticks but then fallback should exist)
        autolabels = matplotlib.ticker.ScalarFormatter()
      if(axis in ['x', 'x2', 'residx', 'residx2']):
        useAxisobject.xaxis.set_major_formatter(autolabels)
      elif(axis in ['xs', 'residxs']):
        self.slavedXFormatter = autolabels
      else:
        useAxisobject.yaxis.set_major_formatter(autolabels)
      
      # issue plot redraw
      if(redraw):
        if(item == 'plot'):
          self.dataplotwidget.myRefresh()
        else:
          self.residplotwidget.myRefresh()    

  def setTickFormatPrecision(self, axis='x', value=2, redraw=True):
    # changes precision in custom tick label formatting
    if(axis in ['x', 'y', 'y2', 'resid', 'x2', 'xs', 'residx', 'residx2', 'residxs']):
      useTicksFormat = 'default'
      if(axis in ['x', 'residx']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.ticksXFormatPrecision) or (axis == 'residx')):
          self.ticksXFormatPrecision, useTicksFormat = value, self.ticksXFormat
        else:
          redraw = False
      elif(axis in ['xs', 'residxs']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.slavedXFormatPrecision) or (axis == 'residxs')):
          self.slavedXFormatPrecision, useTicksFormat = value, self.slavedXFormat
        else:
          redraw = False
      elif(axis in ['x2', 'residx2']):
        if((value != self.ticksXFormatPrecision_div) or (axis == 'residx2')):
          self.ticksXFormatPrecision_div, useTicksFormat = value, self.ticksXFormat_div
        else:
          redraw = False
      elif(axis == 'y'):
        if(value != self.ticksYFormatPrecision):
          self.ticksYFormatPrecision, useTicksFormat = value, self.ticksYFormat
        else:
          redraw = False
      elif(axis == 'y2'):
        if(value != self.ticksY2FormatPrecision):
          self.ticksY2FormatPrecision, useTicksFormat = value, self.ticksY2Format
        else:
          redraw = False
      elif(axis == 'resid'):
        if(value != self.ticksResidYFormatPrecision):
          self.ticksResidYFormatPrecision, useTicksFormat = value, self.ticksResidYFormat
        else:
          redraw = False
      else:
        redraw = False

      # don't redraw when in default mode as changes won't be visible anyhow
      if(useTicksFormat == 'default'):
        redraw = False
      self.setTickFormatHelper(axis=axis, redraw=redraw)
          
  def setTickFormat(self, axis='x', value='float', redraw=True):
    # changes type of custom tick label formatting
    if(axis in ['x', 'y', 'y2', 'resid', 'x2', 'residx', 'residx2', 'xs', 'residxs']):
      if(axis in ['x', 'residx']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.ticksXFormat) or (axis == 'residx')):
          self.ticksXFormat = value
        else:
          redraw = False
      elif(axis in ['x2', 'residx2']):
        if((value != self.ticksXFormat_div) or (axis == 'residx2')):
          self.ticksXFormat_div = value
        else:
          redraw = False
      elif(axis == 'y'):
        if(value != self.ticksYFormat):
          self.ticksYFormat = value
        else:
          redraw = False
      elif(axis == 'y2'):
        if(value != self.ticksY2Format):
          self.ticksY2Format = value
        else:
          redraw = False
      elif(axis == 'resid'):
        if(value != self.ticksResidYFormat):
          self.ticksResidYFormat = value
        else:
          redraw = False
      elif(axis in ['xs', 'residxs']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.slavedXFormat) or (axis == 'residxs')):
          self.slavedXFormat = value
        else:
          redraw = False
      else:
        redraw = False

      self.setTickFormatHelper(axis=axis, redraw=redraw)

  def setFormatTrailZero(self, axis='x', value=True, redraw=True):
    # set trailing zeros of tick labels in custom tick label formatting
    if(axis in ['x', 'y', 'y2', 'resid', 'x2', 'residx', 'residx2', 'xs', 'residxs']):
      useTicksFormat = 'default'
      if(axis in ['x', 'residx']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.ticksXFormatTrailZero) or (axis == 'residx')):
          self.ticksXFormatTrailZero, useTicksFormat = value, self.ticksXFormat
        else:
          redraw = False
      elif(axis in ['xs', 'residxs']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.slavedXFormatTrailZero) or (axis == 'residxs')):
          self.slavedXFormatTrailZero, useTicksFormat = value, self.slavedXFormat
        else:
          redraw = False
      elif(axis in ['x2', 'residx2']):
        if((value != self.ticksXFormatTrailZero_div) or (axis == 'residx2')):
          self.ticksXFormatTrailZero_div, useTicksFormat = value, self.ticksXFormat_div
        else:
          redraw = False
      elif(axis == 'y'):
        if(value != self.ticksYFormatTrailZero):
          self.ticksYFormatTrailZero, useTicksFormat = value, self.ticksYFormat
        else:
          redraw = False
      elif(axis == 'y2'):
        if(value != self.ticksY2FormatTrailZero):
          self.ticksY2FormatTrailZero, useTicksFormat = value, self.ticksY2Format
        else:
          redraw = False
      elif(axis == 'resid'):
        if(value != self.ticksResidYFormatTrailZero):
          self.ticksResidYFormatTrailZero, useTicksFormat = value, self.ticksResidYFormat
        else:
          redraw = False
      else:
        redraw = False

      # don't redraw when in default mode as changes won't be visible anyhow
      if(useTicksFormat == 'default'):
        redraw = False
      self.setTickFormatHelper(axis=axis, redraw=redraw)

  def setFormatComma(self, axis='x', value=True, redraw=True):
    # set comma as decimal sign in tick labels in custom tick label formatting
    if(axis in ['x', 'y', 'y2', 'resid', 'x2', 'residx', 'residx2', 'xs', 'residxs']):
      useTicksFormat = 'default'
      if(axis in ['x', 'residx']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.ticksXFormatComma) or (axis == 'residx')):
          self.ticksXFormatComma, useTicksFormat = value, self.ticksXFormat
        else:
          redraw = False
      elif(axis in ['xs', 'residxs']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.slavedXFormatComma) or (axis == 'residxa')):
          self.slavedXFormatComma, useTicksFormat = value, self.slavedXFormat
        else:
          redraw = False
      elif(axis in ['x2', 'residx2']):
        if((value != self.ticksXFormatComma_div) or (axis == 'residx2')):
          self.ticksXFormatComma_div, useTicksFormat = value, self.ticksXFormat_div
        else:
          redraw = False
      elif(axis == 'y'):
        if(value != self.ticksYFormatComma):
          self.ticksYFormatComma, useTicksFormat = value, self.ticksYFormat
        else:
          redraw = False
      elif(axis == 'y2'):
        if(value != self.ticksY2FormatComma):
          self.ticksY2FormatComma, useTicksFormat = value, self.ticksY2Format
        else:
          redraw = False
      elif(axis == 'resid'):
        if(value != self.ticksResidYFormatComma):
          self.ticksResidYFormatComma, useTicksFormat = value, self.ticksResidYFormat
        else:
          redraw = False
      else:
        redraw = False

      # don't redraw when in default mode as changes won't be visible anyhow
      if(useTicksFormat == 'default'):
        redraw = False
      self.setTickFormatHelper(axis=axis, redraw=redraw)

  def setFormatSeparator(self, axis='x', value=True, redraw=True):
    # set separate 1000s of tick labels in custom tick label formatting
    if(axis in ['x', 'y', 'y2', 'resid', 'x2', 'residx', 'residx2', 'xs', 'residxs']):
      useTicksFormat = 'default'
      if(axis in ['x', 'residx']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.ticksXFormatSeparator) or (axis == 'residx')):
          self.ticksXFormatSeparator, useTicksFormat = value, self.ticksXFormat
        else:
          redraw = False
      elif(axis in ['xs', 'residxs']):
        # always update and redraw for resid plot unless you implement a better check
        if((value != self.slavedXFormatSeparator) or (axis == 'residxs')):
          self.slavedXFormatSeparator, useTicksFormat = value, self.slavedXFormat
        else:
          redraw = False
      elif(axis in ['x2', 'residx2']):
        if((value != self.ticksXFormatSeparator_div) or (axis == 'residx2')):
          self.ticksXFormatSeparator_div, useTicksFormat = value, self.ticksXFormat_div
        else:
          redraw = False
      elif(axis == 'y'):
        if(value != self.ticksYFormatSeparator):
          self.ticksYFormatSeparator, useTicksFormat = value, self.ticksYFormat
        else:
          redraw = False
      elif(axis == 'y2'):
        if(value != self.ticksY2FormatSeparator):
          self.ticksY2FormatSeparator, useTicksFormat = value, self.ticksY2Format
        else:
          redraw = False
      elif(axis == 'resid'):
        if(value != self.ticksResidYFormatSeparator):
          self.ticksResidYFormatSeparator, useTicksFormat = value, self.ticksResidYFormat
        else:
          redraw = False
      else:
        redraw = False

      # don't redraw when in default mode as changes won't be visible anyhow
      if(useTicksFormat != 'float'):
        redraw = False
      self.setTickFormatHelper(axis=axis, redraw=redraw)

  def setFormatFix(self, axis='x', value=True, redraw=True, prefix=True):
    # sets pre/postfix of tick labels in custom tick label formatting
    if(axis in ['x', 'y', 'y2', 'resid', 'x2', 'xs', 'residx', 'residx2', 'residxs']):
      useTicksFormat = 'default'
      if(axis in ['x', 'residx']):
        # always update and redraw for resid plot unless you implement a better check
        if(prefix):
          if((value != self.ticksXFormatPrefix) or (axis == 'residx')):
            self.ticksXFormatPrefix, useTicksFormat = value, self.ticksXFormat
          else:
            redraw = False
        else:
          if((value != self.ticksXFormatPostfix) or (axis == 'residx')):
            self.ticksXFormatPostfix, useTicksFormat = value, self.ticksXFormat
          else:
            redraw = False
      elif(axis in ['xs', 'residxs']):
        if(prefix):
          if((value != self.slavedXFormatPrefix) or (axis == 'residxs')):
            self.slavedXFormatPrefix, useTicksFormat = value, self.slavedXFormat
          else:
            redraw = False
        else:
          if((value != self.slavedXFormatPostfix) or (axis == 'residx')):
            self.slavedXFormatPostfix, useTicksFormat = value, self.slavedXFormat
          else:
            redraw = False
      elif(axis in ['x2', 'residx2']):
        if(prefix):
          if((value != self.ticksXFormatPrefix_div) or (axis == 'residx2')):
            self.ticksXFormatPrefix_div, useTicksFormat = value, self.ticksXFormat_div
          else:
            redraw = False
        else:
          if((value != self.ticksXFormatPostfix_div) or (axis == 'residx2')):
            self.ticksXFormatPostfix_div, useTicksFormat = value, self.ticksXFormat_div
          else:
            redraw = False
      elif(axis == 'y'):
        if(prefix):
          if(value != self.ticksYFormatPrefix):
            self.ticksYFormatPrefix, useTicksFormat = value, self.ticksYFormat
          else:
            redraw = False
        else:
          if(value != self.ticksYFormatPostfix):
            self.ticksYFormatPostfix, useTicksFormat = value, self.ticksYFormat
          else:
            redraw = False
      elif(axis == 'y2'):
        if(prefix):
          if(value != self.ticksY2FormatPrefix):
            self.ticksY2FormatPrefix, useTicksFormat = value, self.ticksY2Format
          else:
            redraw = False
        else:
          if(value != self.ticksY2FormatPostfix):
            self.ticksY2FormatPostfix, useTicksFormat = value, self.ticksY2Format
          else:
            redraw = False
      elif(axis == 'resid'):
        if(prefix):
          if(value != self.ticksResidYFormatPrefix):
            self.ticksResidYFormatPrefix, useTicksFormat = value, self.ticksResidYFormat
          else:
            redraw = False
        else:
          if(value != self.ticksResidYFormatPostfix):
            self.ticksResidYFormatPostfix, useTicksFormat = value, self.ticksResidYFormat
          else:
            redraw = False
      else:
        redraw = False

      # don't redraw when in default mode as changes won't be visible anyhow
      if(useTicksFormat == 'default'):
        redraw = False
      self.setTickFormatHelper(axis=axis, redraw=redraw)

  def setInnerParameter(self, param='axes', state=True, redraw=True):
    # changes parameters for display of items at split axis interface
    if(param in ['axes', 'ticks']):
      if(param == 'axes'):
        self.innerAxes = state
      else:
        self.innerTicks = state
        
      # apply these settings
      self.updateInnerSituation()

      # issue plot refreshs
      if(redraw):
        self.dataplotwidget.myRefresh()
        self.residplotwidget.myRefresh()

  def updateInnerSituation(self):
    # updates graphic items at interface between split axes
    # do this only when split is activated as otherwise right axis and tick marks will be messed up
    if(self.splitShow):
      # deal with axis spines
      secondAxes = self.isSecondAxesActive()
      self.ax.spines['right'].set_visible(self.innerAxes and self.axisVisible['right'])
      self.ax_div.spines['left'].set_visible(self.innerAxes and self.axisVisible['left'])
      self.ax_resid.spines['right'].set_visible(self.innerAxes and self.axisVisible['right'])
      self.ax_resid_div.spines['left'].set_visible(self.innerAxes and self.axisVisible['left'])
      if(secondAxes):
        self.ax2.spines['right'].set_visible(self.innerAxes and self.axisVisible['right2'])
        self.ax2_div.spines['left'].set_visible(self.innerAxes and self.axisVisible['left2'])
      else:
        # hopefully, this kills the mysterious black line
        self.ax2.spines['right'].set_visible(False)
        self.ax2_div.spines['left'].set_visible(False)
      
      # now deal with ticks
      self.ax.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'], right=self.ticksVisible['right'] and self.innerTicks)
      self.ax_div.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'] and self.innerTicks, right=self.ticksVisible['right'])
      self.ax_resid.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'], right=self.ticksVisible['right'] and self.innerTicks)
      self.ax_resid_div.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'] and self.innerTicks, right=self.ticksVisible['right'])
  
      # ensure that no tick labels displayed on self.ax_div nor on self.ax_resid_div
      self.ax.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left'] and self.ticksYShow, labelright=False)
      self.ax_div.yaxis.set_tick_params(which='both', labelleft=False, labelright=self.ticksLabelShow['right'] and self.ticksYShow)
      self.ax_resid.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left'] and self.ticksResidYShow, labelright=False)
      self.ax_resid_div.yaxis.set_tick_params(which='both', labelleft=False, labelright=self.ticksLabelShow['right'] and self.ticksResidYShow)
  
      # now do the same for second axes
      if(secondAxes):
        # now deal with ticks
        self.ax2.yaxis.set_tick_params(which='both', left=self.ticksVisible['left2'], right=self.ticksVisible['right2'] and self.innerTicks)
        self.ax2_div.yaxis.set_tick_params(which='both', left=self.ticksVisible['left2'] and self.innerTicks, right=self.ticksVisible['right2'])
  
        # ensure that no tick labels displayed on self.ax_div nor on self.ax_resid_div
        self.ax2.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left2'] and self.ticksY2Show, labelright=False)
        self.ax2_div.yaxis.set_tick_params(which='both', labelleft=False, labelright=self.ticksLabelShow['right2'] and self.ticksY2Show)

  def setZOrderResidLine(self, zorder=0, redraw=True):
    # updates z order of residuals
    if(self.zorderResidLine != zorder):
      self.zorderResidLine = zorder
      # update plot if necessary
      plotUpdate = False
      if(self.handleResidZero != None):
        self.handleResidZero.set_zorder(self.zorderResidLine + self.parent.zOffset)
        plotUpdate = True
      if(self.handleResidZero_div != None):
        self.handleResidZero_div.set_zorder(self.zorderResidLine + self.parent.zOffset)
        plotUpdate = True

      # update plot
      if(redraw and plotUpdate):
        self.residplotwidget.myRefresh()

  def setVisibilityResidLine(self, state=True, redraw=True):
    # toggles visibility of residual zero line
    if(self.visibilityResidLine != state):
      self.visibilityResidLine = state
      # update plot if necessary
      plotUpdate = False
      if(self.handleResidZero != None):
        self.handleResidZero.set_visible(state)
        plotUpdate = True
      if(self.handleResidZero_div != None):
        self.handleResidZero_div.set_visible(state)
        plotUpdate = True

      # update plot
      if(redraw and plotUpdate):
        self.residplotwidget.myRefresh()

  def legendHelper(self, axisobject=None):
    # helper function called by legend formatters
    if(axisobject == None):
      axisobject = self.ax
    
    # build axis legend objects
    items = []
    for entry in self.parent.data:
      if((entry.handleData != None) and (entry.visibility)):
        # manually process escape characters
        name = entry.name.replace('\n', '\\n')
        name = name.replace('\t', '\\t')
        name = '\n'.join([i for i in name.split('\\n') if (len(i.strip()))])
        name = '\t'.join([i for i in name.split('\\t') if (len(i.strip()))])
        # now also eliminate double $$ sign b/c mathtext freaks out
        name = name.replace('$$', '')
        # test for potential Mathttext errors by creating a dummy text label
        tempText = axisobject.text(1, 1, name)
        try:
          tempText._get_layout(self.matplot.canvas.renderer)
        except:
          # some kind of problem with item label
          self.parent.statusbar.showMessage('Problems with data set label ' + name, self.parent.STATUS_TIME)
          name = name.replace('$', '')
        tempText.remove()
        if(not name.startswith('_')):
          if((entry.handleData != None) and ((not entry.style['marker'] in ['None', None, '', ' ']) or (entry.style['linestyle'] != 'None'))):
            useHandle = entry.handleData
          elif((entry.handleScatter != None) and entry.Scatterstyle['showScatter']):
            useHandle = entry.handleScatter
          elif((entry.handleBar != None) and entry.Barstyle['showBar']):
            useHandle = entry.handleBar
          elif((entry.handleStack != None) and entry.Stackstyle['showStack']):
            useHandle = entry.handleStack
          else:
            useHandle = entry.handleData
          items.append([useHandle, name, entry.zorder])
    for entry in self.parent.fit:
      if((entry.handlePlot != None) and (entry.visibility)):
        # manually process escape characters
        name = entry.name.replace('\n', '\\n')
        name = name.replace('\t', '\\t')
        name = '\n'.join([i for i in name.split('\\n') if (len(i.strip()))])
        name = '\t'.join([i for i in name.split('\\t') if (len(i.strip()))])
        # now also eliminate double $$ sign b/c mathtext freaks out
        name = name.replace('$$', '')
        # test for potential Mathttext errors by creating a dummy text label
        tempText = axisobject.text(1, 1, name)
        try:
          tempText._get_layout(self.matplot.canvas.renderer)
        except:
          # some kind of problem with item label
          self.parent.statusbar.showMessage('Problems with curve label ' + name, self.parent.STATUS_TIME)
          name = name.replace('$', '')
        tempText.remove()
        if(not name.startswith('_')):
          items.append([entry.handlePlot, name, entry.zorder])
    # order according to zorder
    items = sorted(items, key=lambda k: k[2])
    handles = [i[0] for i in items]
    labels = [i[1] for i in items]
    bbox_to_anchor = (self.padSize['left'], self.padSize['bottom'],\
                      self.padSize['right'] - self.padSize['left'], self.padSize['top'] - self.padSize['bottom'])
    
    # have to discriminate for Linux versions b/c markerfirst may be unknown
    if(self.legendPlacement == 'custom'):
      location = (self.legendX, self.legendY)
    else:
      location = self.legendPlacement
    try:
      self.legendHandle = self.createCustomLegend(axisobject, handles, labels, loc=location, shadow=self.legendShadow,\
        numpoints=self.legendNumPoints, markerfirst=self.legendMarkerFirst, ncol=self.legendNumCol, markerscale=self.legendMarkerScale,\
        borderpad=self.legendBorderPad, labelspacing=self.legendLabelSpacing, columnspacing=self.legendColumnSpacing,\
        handlelength=2.0 * self.legendMarkerScale, handletextpad=self.legendLabelPad)#, bbox_to_anchor=bbox_to_anchor)
    except:
      self.legendHandle = self.createCustomLegend(axisobject, handles, labels, loc=location, shadow=self.legendShadow,\
        numpoints=self.legendNumPoints, ncol=self.legendNumCol, markerscale=self.legendMarkerScale,\
        borderpad=self.legendBorderPad, labelspacing=self.legendLabelSpacing, columnspacing=self.legendColumnSpacing,\
        handlelength=2.0 * self.legendMarkerScale, handletextpad=self.legendLabelPad, bbox_to_anchor=bbox_to_anchor)
    
    if(self.legendHandle != None):
      # go via frame properties for enhanced controls
      frame = self.legendHandle.get_frame()
      if(frame != None):
        frame.set_linewidth(self.legendEdgeWidth)
        if(self.legendEdge):
          frame.set_edgecolor(self.legendColor['edge'])
        else:
          frame.set_edgecolor('none')
        if(self.legendFill):
          frame.set_facecolor(self.legendColor['face'])
        else:
          frame.set_facecolor('none')
        frame.set_alpha(self.legendColor['face'][-1])
      
      # set text properties
      texts = self.legendHandle.texts
      for entry in texts:
        entry.set_color(self.legendLabelColor)
        entry.set_fontsize(self.legendLabelSize)
        entry.set_fontweight(self.legendLabelWeight)
        entry.set_fontstyle(self.legendLabelStyle)
        entry.set_fontname(self.legendLabelFont)
        entry.set_linespacing(self.legendLabelLinespacing)
        
      # set additional config settings (brought to you by AM)
      self.legendHandle.additionalConfig(shadowDeltaX=self.legendShadowDeltaX, shadowDeltaY=self.legendShadowDeltaY,\
                                         shadowFaceColor=self.legendShadowFaceColor, legendHatch=self.legendHatch,\
                                         legendHatchMultiply=self.legendHatchMultiply, legendHatchLinewidth=self.legendHatchLinewidth,\
                                         legendEdgeFallbackColor=self.legendColor['edge'],\
                                         legendRounding=self.legendRounding)
      
      # set z-order to display in front (note that aboutLogo is at 1000)
      self.legendHandle.set_zorder(999)
      
  def createCustomLegend(self, axisobject, handles, labels, *args, **kwargs):
    # adapter function to allow generation of custom legend
    # the proper way throws an error under Linux due to matplotlib versions
    # => better avoid this call, which we can as we have a fairly good idea which arguments we pass
    ###handles, labels, extra_args, kwargs = Legend._parse_legend_args([axisobject], *args, **kwargs)
    ###self.legend_ = MyLegend(axisobject, handles, labels, **kwargs)
    # put the legend on the figure, not on the axes => better for split plots and the likes
    if(('loc' in kwargs) and (not (type(kwargs['loc']) is tuple)) and (not (kwargs['loc'] in self.placementStyles))):
      kwargs['loc'] = self.placementStyles[0]
    if(('loc' in kwargs) and (kwargs['loc'] == 'best')):
      # in case of 'best' location, have to use axis object not figure (makes kinda sense)
      self.legend_ = MyLegend(axisobject, handles, labels, **kwargs)
    else:
      self.legend_ = MyLegend(self.matplot, handles, labels, **kwargs)
    # trick axisobject into accepting MyLegend
    try:
      self.legend_._remove_method = axisobject._remove_legend
    except:
      self.legend_._remove_method = lambda h: setattr(self, 'legend_', None)

    axisobject.legend_ = self.legend_
    return self.legend_

  def toggleLegendEdge(self, value=True, redraw=True, target='plot'):
    # toggles whether legend is boxed or not
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      # sets canvas color
      if(self.legendEdge == value):
        redraw = False

      self.legendEdge = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def toggleLegendFill(self, value=True, redraw=True, target='plot'):
    # toggles whether legend is filled or not
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      # sets canvas color
      if(self.legendFill == value):
        redraw = False

      self.legendFill = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendMarkerFirst(self, value=True, redraw=True, target='plot'):
    # sets marker location in legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      if(self.legendMarkerFirst == value):
        redraw = False
        
      self.legendMarkerFirst= value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendMarkerScale(self, value=1.0, redraw=True, target='plot'):
    # sets marker scale in legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      if(self.legendMarkerScale == value):
        redraw = False
        
      self.legendMarkerScale = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendNCol(self, value=1, redraw=True, target='plot'):
    # sets number of columns in legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      if(self.legendNumCol == value):
        redraw = False
        
      self.legendNumCol = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendMarkerNumPoints(self, value=1, redraw=True, target='plot'):
    # sets number of marker points in legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      if(self.legendNumPoints == value):
        redraw = False
        
      self.legendNumPoints = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()
          
  def setLegendPlacement(self, value='upper right', redraw=True, target='plot'):
    # sets placement of legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      if(self.legendPlacement == value):
        redraw = False
        
      self.legendPlacement = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendShadow(self, value=False, redraw=True, target='plot'):
    # sets placement of legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      if(self.legendShadow == value):
        redraw = False
        
      self.legendShadow = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegend(self, value=True, redraw=True, target='plot'):
    # sets legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
      
      self.legendVisible = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
      else:
        legend = self.legendHandle
        if(legend != None):
          try:
            legend.remove()
          except:
            pass
          self.ax.legend_ = None
          self.legendHandle = None
          
      if(redraw):
        plotobject.myRefresh()

  def setLegendColor(self, value=[0.5, 0.5, 0.5, 0.5], prop='face', redraw=True, target='plot'):
    # sets color of legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
    else:
      prop = 'abort'

    # update color
    if(prop in ['face', 'edge']):
      if(self.legendColor[prop] == value):
        redraw=False
      if((prop == 'face') and (not self.legendFill)):
        redraw = False
      if((prop == 'edge') and (not self.legendEdge) and (self.legendHatch == '')):
        redraw = False
        
      self.legendColor[prop] = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendRounding(self, value=0.2, redraw=True, target='plot'):
    # alters rounding for legend box
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      if(self.legendRounding == value):
        redraw = False
      self.legendRounding = value

      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendPosition(self, axis='x', value=0.2, redraw=True, target='plot'):
    # alters position for legend box
    if(axis in ['x', 'y']):
      if(target in ['plot', 'resid']):
        if(target == 'plot'):
          plotobject, axisobject = self.dataplotwidget, self.ax
        else:
          plotobject, axisobject = self.residplotwidget, self.ax_resid
  
        if(axis == 'x'):
          if(self.legendX == value):
            redraw = False
          else:
            # switch to custom placement when this value changes
            self.legendPlacement = 'custom'
          self.legendX = value
        else:
          if(self.legendY == value):
            redraw = False
          else:
            # switch to custom placement when this value changes
            self.legendPlacement = 'custom'
          self.legendY = value
  
        if(self.legendVisible):
          self.legendHelper(axisobject)
          if(redraw):
            plotobject.myRefresh()

  def setLegendHatchParameter(self, item='hatch', value='', redraw=True, target='plot'):
    # alters parameter for legend hatch
    if(item in ['hatch', 'multiply', 'linewidth']):
      if(target in ['plot', 'resid']):
        if(target == 'plot'):
          plotobject, axisobject = self.dataplotwidget, self.ax
        else:
          plotobject, axisobject = self.residplotwidget, self.ax_resid

        if(item == 'linewidth'):
          if(self.legendHatchLinewidth == value):
            redraw = False
          self.legendHatchLinewidth = value
        elif(item == 'hatch'):
          if(self.legendHatch == value):
            redraw = False
          self.legendHatch = value
        else:
          if(self.legendHatchMultiply == value):
            redraw = False
          self.legendHatchMultiply = value
          
        if(self.legendVisible):
          self.legendHelper(axisobject)
          if(redraw):
            plotobject.myRefresh()

  def setLegendShadowParameter(self, item='color', value=[0.5, 0.5, 0.5, 1.0], redraw=True, target='plot'):
    # alters parameter for legend shadow
    if(item in ['color', 'x', 'y']):
      if(target in ['plot', 'resid']):
        if(target == 'plot'):
          plotobject, axisobject = self.dataplotwidget, self.ax
        else:
          plotobject, axisobject = self.residplotwidget, self.ax_resid

        if(item == 'color'):
          if(self.legendShadowFaceColor == value):
            redraw = False
          self.legendShadowFaceColor = value
        elif(item == 'x'):
          if(self.legendShadowDeltaX == value):
            redraw = False
          self.legendShadowDeltaX = value
        else:
          if(self.legendShadowDeltaY == value):
            redraw = False
          self.legendShadowDeltaY = value
          
        if(self.legendVisible and self.legendShadow):
          self.legendHelper(axisobject)
          if(redraw):
            plotobject.myRefresh()

  def setLegendLabelColor(self, value=[0.5, 0.5, 0.5, 0.5], redraw=True, target='plot'):
    # sets color of legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # update color
      if(self.legendLabelColor == value):
        redraw = False
        
      self.legendLabelColor = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendLabelBold(self, value='normal', redraw=True, target='plot'):
    # sets formatting of legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # update color
      if(self.legendLabelWeight == value):
        redraw = False
        
      self.legendLabelWeight = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendLabelLinespacing(self, value=1., redraw=True, target='plot'):
    # sets linescpacing of legend entries
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # update color
      if(self.legendLabelLinespacing == value):
        redraw = False
        
      self.legendLabelLinespacing = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendLabelItalic(self, value='normal', redraw=True, target='plot'):
    # sets formatting of legend
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # update color
      if(self.legendLabelStyle == value):
        redraw = False
        
      self.legendLabelStyle = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendEdgeWidth(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      if((self.legendEdgeWidth == value) or (not self.legendEdge)):
        redraw = False
        
      self.legendEdgeWidth = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendPadBorder(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      if(self.legendBorderPad == value):
        redraw = False
        
      self.legendBorderPad = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendPadRow(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      if(self.legendLabelSpacing == value):
        redraw = False
        
      self.legendLabelSpacing = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendPadCol(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      if(self.legendColumnSpacing == value):
        redraw = False
        
      self.legendColumnSpacing = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendPadLabel(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      if(self.legendLabelPad == value):
        redraw = False
        
      self.legendLabelPad = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendLabelSize(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      if(self.legendLabelSize == value):
        redraw = False
        
      self.legendLabelSize = value
      if(self.legendVisible):
        self.legendHelper(axisobject)
        if(redraw):
          plotobject.myRefresh()

  def setLegendLabelFont(self, value='DejaVu Sans', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject = self.dataplotwidget, self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid

      # sets legend edge width
      prevFont = self.legendLabelFont
      if(self.legendLabelFont == value):
        redraw = False
        
      self.legendLabelFont = value
      self.legendHelper(axisobject)
      
      # have to capture errors in case a strange font is set
      try:
        if(redraw):
          plotobject.myRefresh()
      except:
        self.parent.statusbar.showMessage('Experiencing problems setting font ' + self.legendLabelFont + ' -- reverting to ' + prevFont, self.parent.STATUS_TIME)

        self.legendLabelFont = prevFont
        self.legendHelper(axisobject)
        # also capture errors with previous font (can happen if selecting two bad fonts in a row)
        try:
          if(redraw):
            plotobject.myRefresh()
        except:
          ###safeFont = 'DejaVu Sans'
          self.parent.statusbar.showMessage('Also experiencing problems setting font ' + self.legendLabelFont + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
          self.legendLabelFont = SAFE_FONT
          self.legendHelper(axisobject)

  def setDataAxisYTicks(self, dataSet=0, redraw=True, target='plot', axis='y'):
    # set y ticks to label values
    useData, roles = self.parent.data[dataSet].getData_n_Fit()
    if(('y' in roles) and (target == 'plot')):
      ycol = roles.index('y')
      yval = list(useData[:, ycol])
      labels2 = list(self.parent.data[self.parent.activeData].getLabels(labels2=True))
      minLength = np.min((len(yval), len(labels2)))
      if(minLength):
        # we have some labels to place
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          axisobject, useModeY, useMinY, useMaxY, useTicksYMinor = self.ax2, self.modeY, self.minY, self.maxY, self.ticksYMinor
          if(axis == 'y'):
            axisobject = self.ax
        else:
          # y labels on resid plot not implemented yet
          pass

        flag = False
        labels2 = labels2[:minLength]
        yval = yval[:minLength]

        # manually process labels for escape characters
        for index, entry in enumerate(labels2):
          # ensure that labels are string
          if(type(entry) != type('tse')):
            entry = str(entry)
          name = entry.replace('\n', '\\n')
          name = name.replace('\t', '\\t')
          name = '\n'.join([i for i in name.split('\\n') if (len(i.strip()))])
          name = '\t'.join([i for i in name.split('\\t') if (len(i.strip()))])
          # now also eliminate double $$ sign b/c mathtext freaks out
          name = name.replace('$$', '')
          labels2[index] = name
        
        # first set new ticks
        axisobject.yaxis.set_major_locator(matplotlib.ticker.FixedLocator(yval))
        if(1):
          minorAutoticks = MyAutoMinorLocator(useTicksYMinor)
          axisobject.yaxis.set_minor_locator(minorAutoticks)
          axisobject.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        lower, upper = useMinY, useMaxY
        if(len(yval)):
          # check for empty list
          if(np.min(yval) < np.min((useMinY, useMaxY))):
            flag = True
          if(np.max(yval) > np.max((useMinY, useMaxY))):
            flag = True
            
        # now set new tick labels
        axisobject.yaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(labels2))
        self.ticksYDataSet = self.parent.activeData
        
        # store settings
        if(axis == 'y'):
          self.ticksY, self.ticksYLabel, self.ticksYAuto = yval, labels2, False
          self.parent.graphicsarea.configTickYAuto.setChecked(False)
        else:
          self.ticksY2, self.ticksY2Label, self.ticksY2Auto = yval, labels2, False
          self.parent.graphicsarea2.configTickYAuto.setChecked(False)

        # check whether the new ticks necessitate axis rescaling
        if(flag):
          self.setAxisLimits(lower=lower, upper=upper, axis=axis, updateLabel=True, target=target, redraw=False, updateGrid=True)
        else:
          # update axis grid lines
          if(self.gridVisible[axis]):
            self.drawAxisGrid(axis=axis, redraw=False, target=target)
        
          if(redraw):
            plotobject.myRefresh()
      else:
        self.parent.statusbar.showMessage('Data set ' + str(dataSet) + ' contains no labels (use labels2 for y axis)!', self.parent.STATUS_TIME)
    else:
      self.parent.statusbar.showMessage('Current data set ' + str(dataSet) + ' is empty!', self.parent.STATUS_TIME)

  def setDataAxisTicks(self, dataSet=0, redraw=True, target='plot', splitX=False):
    # set x ticks to label values
    useData, roles = self.parent.data[dataSet].getData_n_Fit()
    if('x' in roles):
      xcol = roles.index('x')
      xval = list(useData[:, xcol])
      labels = list(self.parent.data[self.parent.activeData].getLabels())
      minLength = np.min((len(xval), len(labels)))
      if(minLength):
        # we have some labels to place
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(splitX):
            axisobject, useModeX, useMinX, useMaxX, useTicksXMinor = self.ax2_div, self.modeX_div, self.minX_div, self.maxX_div, self.ticksXMinor_div
            axisItems = [self.ax_div, self.ax2_div]
          else:
            axisobject, useModeX, useMinX, useMaxX, useTicksXMinor = self.ax2, self.modeX, self.minX, self.maxX, self.ticksXMinor
            axisItems = [self.ax, self.ax2]
        else:
          plotobject = self.residplotwidget
          if(splitX):
            axisobject, useModeX, useMinX, useMaxX, useTicksXMinor = self.ax_resid_div, self.modeX_div, self.minX_div, self.maxX_div, self.ticksXMinor_div
            axisItems = [self.ax_resid_div]
          else:
            axisobject, useModeX, useMinX, useMaxX, useTicksXMinor = self.ax_resid, self.modeX, self.minX, self.maxX, self.ticksXMinor
            axisItems = [self.ax_resid]

        flag = False
        labels = labels[:minLength]
        xval = xval[:minLength]

        # manually process labels for escape characters
        for index, entry in enumerate(labels):
          # ensure that labels are string
          if(type(entry) != type('tse')):
            entry = str(entry)
          name = entry.replace('\n', '\\n')
          name = name.replace('\t', '\\t')
          name = '\n'.join([i for i in name.split('\\n') if (len(i.strip()))])
          name = '\t'.join([i for i in name.split('\\t') if (len(i.strip()))])
          # now also eliminate double $$ sign b/c mathtext freaks out
          name = name.replace('$$', '')
          labels[index] = name
        
        # first set new ticks
        for useAxis in axisItems:
          useAxis.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(xval))
        ###if(useModeX == 'linear'):
        if(1):
          minorAutoticks = MyAutoMinorLocator(useTicksXMinor)
          for useAxis in axisItems:
            useAxis.xaxis.set_minor_locator(minorAutoticks)
            useAxis.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        lower, upper = useMinX, useMaxX
        if(len(xval)):
          # check for empty list
          if(np.min(xval) < np.min((useMinX, useMaxX))):
            flag = True
            # disable rescaling on assignment of axis labels
            ###lower = np.min(xval)
          if(np.max(xval) > np.max((useMinX, useMaxX))):
            flag = True
            # disable rescaling on assignment of axis labels
            ###upper = np.max(xval)
          # special treatment for resid plot
          if(target == 'resid'):
            flag = True
            
        # now set new tick labels
        axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(labels))
        self.ticksXDataSet = self.parent.activeData
        
        # store settings
        if(splitX):
          self.ticksX_div, self.ticksXLabel_div, self.ticksXAuto_div = xval, labels, False
          self.parent.graphicsarea.configInnerTickXAuto.setChecked(False)
        else:
          self.ticksX, self.ticksXLabel, self.ticksXAuto = xval, labels, False
          self.parent.graphicsarea.configTickXAuto.setChecked(False)

        # check whether the new ticks necessitate axis rescaling
        if(flag):
          if(splitX):
            self.setAxisLimits(lower=lower, upper=upper, axis='x2', updateLabel=True, target=target, redraw=False, updateGrid=True)
            if(target == 'plot'):
              # and we should redraw the fit function to cover new x-range
              self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.parent.plotArea.plotFunction(\
                fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div,\
                handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=splitX)
              # copy in case split axes are shown
              curve = self.parent.fit[self.parent.activeFit]
              if(self.parent.plotArea.splitY and curve.onBothAxes):
                curve.duplicateForSplit()
              if(redraw):
                self.parent.plotArea.dataplotwidget.myRefresh()
            else:
              # and we should update the resid plot (as x-axis will most likely have rescaled)
              self.parent.plotArea.handleResidZero_div = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero_div, redraw=redraw, splitX=splitX)
          else:
            self.setAxisLimits(lower=lower, upper=upper, axis='x', updateLabel=True, target=target, redraw=False, updateGrid=True)
            if(target == 'plot'):
              # and we should redraw the fit function to cover new x-range
              self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.parent.plotArea.plotFunction(\
                fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot,\
                handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False)
              # copy in case split axes are shown
              curve = self.parent.fit[self.parent.activeFit]
              if(self.parent.plotArea.splitY and curve.onBothAxes):
                curve.duplicateForSplit()
              if(redraw):
                self.parent.plotArea.dataplotwidget.myRefresh()
            else:
              # and we should update the resid plot (as x-axis will most likely have rescaled)
              self.parent.plotArea.handleResidZero = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero, redraw=redraw)
        else:
          # update axis grid lines
          if(splitX):
            if(self.gridVisible['x2']):
              self.drawAxisGrid(axis='x2', redraw=False, target=target)
          else:
            if(self.gridVisible['x']):
              self.drawAxisGrid(axis='x', redraw=False, target=target)
        
          if(redraw):
            plotobject.myRefresh()
      else:
        self.parent.statusbar.showMessage('Data set ' + str(dataSet) + ' contains no labels!', self.parent.STATUS_TIME)
    else:
      self.parent.statusbar.showMessage('Current data set ' + str(dataSet) + ' is empty!', self.parent.STATUS_TIME)

  def myFormatterFunction(self, mode='float', precision=2, trailZero=True, separator=True, comma=True, prefix='', postfix='', x=0, pos=None):
    # a custom formatter function
    prefix, postfix = prefix.replace('\\', '\\\\'), postfix.replace('\\', '\\\\')
    prefix, postfix = prefix.replace('$', '\\$'), postfix.replace('$', '\\$')
    if(mode == 'float'):
      # for float formatting consider separator for 1000s (by contrast, not relevant for scientific/mathdefault)
      if(separator):
        formatstr = '{:,.' + str(precision) + 'f}'
      else:
        formatstr = '{:.' + str(precision) + 'f}'
      formatstr = formatstr.format(x)
      # check whether commata should be used rather than decimal points
      if(comma):
        checkItem = ','
        formatstr = formatstr.replace('.', '|')
        formatstr = formatstr.replace(',', '.')
        formatstr = formatstr.replace('|', ',')
      else:
        checkItem = '.'
      # test at level of output string, as is_integer() runs into problems with precision of floats
      if((not trailZero) and (checkItem in formatstr)):
        while(formatstr.endswith('0')):
          formatstr = formatstr[:-1]
        if(formatstr.endswith(checkItem)):
          formatstr = formatstr[:-1]
      # need to enclose commata in curly brackets to prevent trailing space
      formatstr = formatstr.replace(',', '{,}')
      formatstr = '$\\mathdefault{' + formatstr + '}$'
      return prefix + formatstr + postfix
    elif(mode == 'scientific'):
      formatstr = '{:.' + str(precision) + 'e}'
      formatstr = formatstr.format(x)
      if(not trailZero):
        if('e' in formatstr.lower()):
          eString = formatstr[formatstr.lower().find('e')]
          preString, exponentString = formatstr.split(eString)
          while(preString.endswith('0')):
            preString = preString[:-1]
          if(preString.endswith('.')):
            preString = preString[:-1]
          formatstr = preString + eString + exponentString
      if(comma):
        formatstr = formatstr.replace('.', '{,}')
      formatstr = '$\\mathdefault{' + formatstr + '}$'
      return prefix + formatstr + postfix
    elif(mode == 'mathtext'):
      # calculate exponent
      if(x != 0.0):
        try:
          exponent = int(np.floor(np.log10(np.abs(x))))
        except:
          exponent = 0
        # calculate preexponent
        try:
          pre = x / (10 ** exponent)
        except:
          pre, exponent = x, 0
      # assemble string
      if(x == 0.0):
        retstr = '{:.' + str(precision) + 'f}'
        retstr = retstr.format(0.0)
        if(not trailZero):
          retstr = '0'
        if(comma):
          retstr = retstr.replace('.', '{,}')
        retstr = '$\\mathdefault{' + retstr + '}$' 
      elif(np.isclose(pre, 1.0)):
        retstr = '$\\mathdefault{10^{' + str(exponent) + '}}$'
      elif(np.isclose(pre, -1.0)):
        retstr = '$\\mathdefault{-10^{' + str(exponent) + '}}$'
      else:
        retstr = '{:.' + str(precision) + 'f}'
        retstr = retstr.format(pre)
        # testing with is_integer() struggles with float precision - test directly at level of string
        #if((not trailZero) and pre.is_integer()):
        if((not trailZero) and precision):
          front, trail = retstr.split('.')
          while(trail.endswith('0')):
            trail = trail[:-1]
          if(trail == ''):
            retstr = front
          else:
            retstr = front + '.' + trail
        if(comma):
          retstr = retstr.replace('.', '{,}')
        # should not use \times here as some fonts don't possess this symbol
        retstr = '$\\mathdefault{' + retstr + '\\ x\\ 10^{' + str(exponent) + '}}$' 
      # return value
      return prefix + retstr + postfix
    elif(mode == 'engineer'):
      # the following adapted from EngFormatter
      SI_PREFIXES = {-30: "q", -27: "r", -24: "y", -21: "z", -18: "a", -15: "f", -12: "p", -9: "n",
                     -6: "\N{MICRO SIGN}", -3: "m", 0: "", 3: "k", 6: "M", 9: "G", 12: "T", 15: "P",
                     18: "E", 21: "Z", 24: "Y", 27: "R", 30: "Q"}

      # calculate exponent
      if(x != 0.0):
        try:
          exponent = int(np.floor(np.log10(np.abs(x))))
          exponent = int(np.floor(exponent / 3.) * 3.)
        except:
          exponent = 0

        pow10 = np.clip(exponent, min(SI_PREFIXES), max(SI_PREFIXES))
        sistring = SI_PREFIXES[int(pow10)]

        # calculate preexponent
        try:
          # need to explicitly convert to int to avoid numpy type incompatibility
          pre = x / (10 ** int(pow10))
        except:
          pre, exponent = x, 0

      # assemble string
      if(x == 0.0):
        retstr = '{:.' + str(precision) + 'f}'
        retstr = retstr.format(0.0)
        if(not trailZero):
          retstr = '0'
      else:
        retstr = '{:.' + str(precision) + 'f}'
        if(np.isclose(pre, 1.0)):
          retstr = retstr.format(1)
        elif(np.isclose(pre, -1.0)):
          retstr = retstr.format(-1)
        else:
          retstr = retstr.format(pre)

        if((not trailZero) and precision):
          front, trail = retstr.split('.')
          while(trail.endswith('0')):
            trail = trail[:-1]
          if(trail == ''):
            retstr = front
          else:
            retstr = front + '.' + trail

        # deal with delimiter
        if(comma):
          retstr = retstr.replace('.', '{,}')
        # now append the si prefix
        retstr = '$\\mathdefault{' + retstr + '\\,' + SI_PREFIXES[pow10] + '}$'
      # return value
      return prefix + retstr + postfix
    else:
      return str(x)

  def setAutoTicks(self, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject, axisobject2 = self.dataplotwidget, self.ax2, self.ax2_div
        axisItems = [self.ax, self.ax2]
        if(axis in ['x2']):
          axisItems = [self.ax_div, self.ax2_div]
      else:
        plotobject, axisobject, axisobject2 = self.residplotwidget, self.ax_resid, self.ax_resid_div
        axisItems = [self.ax_resid]
        if(axis in ['x2']):
          axisItems = [self.ax_resid_div]
    else:
      axis = 'abort'
    # automatically sets axis ticks
    if(axis in ['x', 'y', 'y2', 'resid', 'x2']):
      if(axis in ['x', 'x2']):
        if(axis == 'x'):
          useModeX, useAxisobject, useTicksXFormat, useTicksXFormatPrecision, useTicksXMinor, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma =\
            self.modeX, axisobject, self.ticksXFormat, self.ticksXFormatPrecision, self.ticksXMinor, self.ticksXFormatTrailZero, self.ticksXFormatSeparator, self.ticksXFormatComma
          usePrefix, usePostfix = self.ticksXFormatPrefix, self.ticksXFormatPostfix
        else:
          useModeX, useAxisobject, useTicksXFormat, useTicksXFormatPrecision, useTicksXMinor, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma =\
            self.modeX_div, axisobject2, self.ticksXFormat_div, self.ticksXFormatPrecision_div, self.ticksXMinor_div, self.ticksXFormatTrailZero_div, self.ticksXFormatSeparator_div, self.ticksXFormatComma_div
          usePrefix, usePostfix = self.ticksXFormatPrefix_div, self.ticksXFormatPostfix_div

        if(useModeX == 'linear'):
          autoticks = matplotlib.ticker.AutoLocator()
          autolabels = matplotlib.ticker.ScalarFormatter()
          minorAutoticks = MyAutoMinorLocator(useTicksXMinor)
          for useAxis in axisItems:
            useAxis.xaxis.set_minor_locator(minorAutoticks)
            useAxis.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        else:
          autoticks = matplotlib.ticker.LogLocator()
          ###if(platform in ['linux', 'darwin']):
          try:
            autolabels = matplotlib.ticker.LogFormatterSciNotation()
          except:
            autolabels = matplotlib.ticker.ScalarFormatter()
          minorAutoticks = MyAutoMinorLocator(useTicksXMinor)
          for useAxis in axisItems:
            useAxis.xaxis.set_minor_locator(minorAutoticks)
            useAxis.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # apply custom formatting?
        if(useTicksXFormat != 'default'):
          # store fallback
          if(axis == 'x'):
            self.fallback_ticksXFormat = autolabels
          else:
            self.fallback_ticksXFormat_div = autolabels
          autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, useTicksXFormat, useTicksXFormatPrecision, useTicksFormatTrailZero, useTicksFormatSeparator, useTicksFormatComma, usePrefix, usePostfix))
        for useAxis in axisItems:
          useAxis.xaxis.set_major_locator(autoticks)
        useAxisobject.xaxis.set_major_formatter(autolabels)

        # store information
        if(axis == 'x'):
          self.ticksX = self.getAxisTicks(axis)
          nuticks = self.ticksX
          # clear labels
          self.ticksXLabel, self.ticksXAuto, self.ticksXDataSet = [], True, -1
        else:
          self.ticksX_div = self.getAxisTicks(axis)
          nuticks = self.ticksX_div
          # clear labels
          self.ticksXLabel_div, self.ticksXAuto_div = [], True
      elif(axis == 'y'):
        axisobject = self.ax; axisobject2 = self.ax_div
        for useAxisobject in [axisobject, axisobject2]:
          if(self.modeY == 'linear'):
            autoticks = matplotlib.ticker.AutoLocator()
            autolabels = matplotlib.ticker.ScalarFormatter()
            minorAutoticks = MyAutoMinorLocator(self.ticksYMinor)
            useAxisobject.yaxis.set_minor_locator(minorAutoticks)
            useAxisobject.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
          else:
            autoticks = matplotlib.ticker.LogLocator()
            ###if(platform in ['linux', 'darwin']):
            try:
              autolabels = matplotlib.ticker.LogFormatterSciNotation()
            except:
              autolabels = matplotlib.ticker.ScalarFormatter()
            minorAutoticks = MyAutoMinorLocator(self.ticksYMinor)
            useAxisobject.yaxis.set_minor_locator(minorAutoticks)
            useAxisobject.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
          # apply custom formatting?
          if(self.ticksYFormat != 'default'):
            # store fallback
            self.fallback_ticksYFormat = autolabels
            autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksYFormat, self.ticksYFormatPrecision, self.ticksYFormatTrailZero, self.ticksYFormatSeparator, self.ticksYFormatComma, self.ticksYFormatPrefix, self.ticksYFormatPostfix))
          useAxisobject.yaxis.set_major_locator(autoticks)
          useAxisobject.yaxis.set_major_formatter(autolabels)
        # store information
        self.ticksY = self.getAxisTicks(axis)
        nuticks = self.ticksY
        self.ticksYAuto = True
      elif(axis == 'y2'):
        for useAxisobject in [axisobject, axisobject2]:
          if(self.modeY2 == 'linear'):
            autoticks = matplotlib.ticker.AutoLocator()
            autolabels = matplotlib.ticker.ScalarFormatter()
            minorAutoticks = MyAutoMinorLocator(self.ticksY2Minor)
            useAxisobject.yaxis.set_minor_locator(minorAutoticks)
            useAxisobject.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
          else:
            autoticks = matplotlib.ticker.LogLocator()
            ###if(platform in ['linux', 'darwin']):
            try:
              autolabels = matplotlib.ticker.LogFormatterSciNotation()
            except:
              autolabels = matplotlib.ticker.ScalarFormatter()
            minorAutoticks = MyAutoMinorLocator(self.ticksY2Minor)
            useAxisobject.yaxis.set_minor_locator(minorAutoticks)
            useAxisobject.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
          # apply custom formatting?
          if(self.ticksY2Format != 'default'):
            # store fallback
            self.fallback_ticksY2Format = autolabels
            autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksY2Format, self.ticksY2FormatPrecision, self.ticksY2FormatTrailZero, self.ticksY2FormatSeparator, self.ticksY2FormatComma, self.ticksY2FormatPrefix, self.ticksY2FormatPostfix))
          useAxisobject.yaxis.set_major_locator(autoticks)
          useAxisobject.yaxis.set_major_formatter(autolabels)
        # store information
        self.ticksY2 = self.getAxisTicks(axis)
        nuticks = self.ticksY2
        self.ticksY2Auto = True
      else:
        for useAxisobject in [axisobject, axisobject2]:
          autoticks = matplotlib.ticker.AutoLocator()
          autolabels = matplotlib.ticker.ScalarFormatter()
          # apply custom formatting?
          if(self.ticksResidYFormat != 'default'):
            # store fallback
            self.fallback_ticksResidYFormat = autolabels
            autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksResidYFormat, self.ticksResidYFormatPrecision, self.ticksResidYFormatTrailZero, self.ticksResidYFormatSeparator, self.ticksResidYFormatComma, self.ticksResidYFormatPrefix, self.ticksResidYFormatPostfix))
          minorAutoticks = MyAutoMinorLocator(self.ticksResidYMinor)
          useAxisobject.yaxis.set_major_locator(autoticks)
          useAxisobject.yaxis.set_minor_locator(minorAutoticks)
          useAxisobject.yaxis.set_major_formatter(autolabels)
          useAxisobject.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # store information
        self.ticksResidY = self.getAxisTicks(axis)
        nuticks = self.ticksResidY
        self.ticksResidYAuto = True

      # update axis grid lines
      if(axis == 'resid'):
        item = 'y'
      else:
        item = axis
      if(self.gridVisible[item]):
        if((item != 'y2') or self.isSecondAxesActive()):
          self.drawAxisGrid(axis=item, redraw=False, target=target)
      
      if(redraw):
        plotobject.myRefresh()
      return nuticks

  def setAxisTicks(self, value=np.array([]), axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject, axisname, useTicksXMinor = self.dataplotwidget, self.ax2, 'ax2', self.ticksXMinor
        if(axis == 'x2'):
          axisobject, useTicksXMinor = self.ax2_div, self.ticksXMinor_div
          axisItems = [self.ax_div, self.ax2_div]
        elif(axis == 'y'):
          axisItems = [self.ax, self.ax_div]
        elif(axis == 'y2'):
          axisItems = [self.ax2, self.ax2_div]
        else:
          axisItems = [self.ax, self.ax2]
      else:
        plotobject, axisobject, axisname, useTicksXMinor = self.residplotwidget, self.ax_resid, 'ax_resid', self.ticksXMinor
        if(axis == 'x2'):
          axisobject, useTicksXMinor = self.ax_resid_div, self.ticksXMinor_div
          axisItems = [self.ax_resid_div]
        elif(axis == 'resid'):
          axisItems = [self.ax_resid, self.ax_resid_div]
        else:
          axisItems = [self.ax_resid]
    else:
      axis='abort'
    # sets axis ticks
    if(axis in ['x', 'y', 'y2', 'resid', 'x2']):
      flag = False
      if(axis == 'x'):
        if(self.modeX == 'linear'):
          autolabels = matplotlib.ticker.ScalarFormatter()
          formatterName = 'matplotlib.ticker.ScalarFormatter()'
        else:
          autolabels = matplotlib.ticker.LogFormatterSciNotation()
          ###if(platform in ['linux', 'darwin']):
          try:
            blah = matplotlib.ticker.LogFormatterSciNotation()
            formatterName = 'matplotlib.ticker.LogFormatterSciNotation()'
          except:
            formatterName = 'matplotlib.ticker.ScalarFormatter()'
        # apply custom formatting?
        if(self.ticksXFormat != 'default'):
          # store fallback
          self.fallback_ticksXFormat = autolabels
          autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksXFormat, self.ticksXFormatPrecision, self.ticksXFormatTrailZero, self.ticksXFormatSeparator, self.ticksXFormatComma, self.ticksXFormatPrefix, self.ticksXFormatPostfix))
        for useAxis in axisItems:
          useAxis.xaxis.set_major_formatter(autolabels)
          useAxis.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(value))
        ###if(self.modeX == 'linear'):
        if(1):
          minorAutoticks = MyAutoMinorLocator(useTicksXMinor)
          for useAxis in axisItems:
            useAxis.xaxis.set_minor_locator(minorAutoticks)
            useAxis.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # store settings
        self.ticksX = value
        lower, upper = self.minX, self.maxX
        if(len(value)):
          # check for empty list
          if(np.min(value) < np.min((self.minX, self.maxX))):
            flag = True
            lower = np.min(value)
          if(np.max(value) > np.max((self.minX, self.maxX))):
            flag = True
            upper = np.max(value)
          # special treatment for resid plot
          if(target == 'resid'):
            flag = True
        # clear labels
        self.ticksXLabel = []
        self.ticksXDataSet = -1
        self.ticksXAuto = False
        self.parent.graphicsarea.configTickXAuto.setChecked(False)
      elif(axis == 'x2'):
        if(self.modeX_div == 'linear'):
          autolabels = matplotlib.ticker.ScalarFormatter()
        else:
          autolabels = matplotlib.ticker.LogFormatterSciNotation()
        # apply custom formatting?
        if(self.ticksXFormat_div != 'default'):
          # store fallback
          self.fallback_ticksXFormat_div = autolabels
          autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksXFormat_div, self.ticksXFormatPrecision_div, self.ticksXFormatTrailZero_div, self.ticksXFormatSeparator_div, self.ticksXFormatComma_div, self.ticksXFormatPrefix_div, self.ticksXFormatPostfix_div))
        axisobject.xaxis.set_major_formatter(autolabels)
        axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(value))
        ###if(self.modeX_div == 'linear'):
        if(1):
          minorAutoticks = MyAutoMinorLocator(useTicksXMinor)
          axisobject.xaxis.set_minor_locator(minorAutoticks)
          axisobject.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # store settings
        self.ticksX_div = value
        lower, upper = self.minX_div, self.maxX_div
        if(len(value)):
          # check for empty list
          if(np.min(value) < np.min((self.minX_div, self.maxX_div))):
            flag = True
            lower = np.min(value)
          if(np.max(value) > np.max((self.minX_div, self.maxX_div))):
            flag = True
            upper = np.max(value)
          # special treatment for resid plot
          if(target == 'resid'):
            flag = True
        # clear labels
        self.ticksXLabel_div = []
        self.ticksXAuto_div = False
        self.parent.graphicsarea.configInnerTickXAuto.setChecked(False)
      elif(axis == 'y'):
        axisobject, axisname = self.ax, 'ax'
        if(self.modeY == 'linear'):
          autolabels = matplotlib.ticker.ScalarFormatter()
          formatterName = 'matplotlib.ticker.ScalarFormatter()'
        else:
          autolabels = matplotlib.ticker.LogFormatterSciNotation()
          ###if(platform in ['linux', 'darwin']):
          try:
            blah = matplotlib.ticker.LogFormatterSciNotation()
            formatterName = 'matplotlib.ticker.LogFormatterSciNotation()'
          except:
            formatterName = 'matplotlib.ticker.ScalarFormatter()'
        # apply custom formatting?
        if(self.ticksYFormat != 'default'):
          # store fallback
          self.fallback_ticksYFormat = autolabels
          autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksYFormat, self.ticksYFormatPrecision, self.ticksYFormatTrailZero, self.ticksYFormatSeparator, self.ticksYFormatComma, self.ticksYFormatPrefix, self.ticksYFormatPostfix))
        for useAxis in axisItems:
          useAxis.yaxis.set_major_formatter(autolabels)
          useAxis.yaxis.set_major_locator(matplotlib.ticker.FixedLocator(value))
        ###if(self.modeY == 'linear'):
        if(1):
          minorAutoticks = MyAutoMinorLocator(self.ticksYMinor)
          for useAxis in axisItems:
            useAxis.yaxis.set_minor_locator(minorAutoticks)
            useAxis.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # store settings
        self.ticksY = value
        lower, upper = self.minY, self.maxY
        if(len(value)):
          # check for empty list
          if(np.min(value) < np.min((self.minY, self.maxY))):
            flag = True
            lower = np.min(value)
          if(np.max(value) > np.max((self.minY, self.maxY))):
            flag = True
            upper = np.max(value)
        self.ticksYAuto = False
        self.parent.graphicsarea.configTickYAuto.setChecked(False)
      elif(axis == 'y2'):
        if(self.modeY2 == 'linear'):
          autolabels = matplotlib.ticker.ScalarFormatter()
          formatterName = 'matplotlib.ticker.ScalarFormatter()'
        else:
          autolabels = matplotlib.ticker.LogFormatterSciNotation()
          ###if(platform in ['linux', 'darwin']):
          try:
            blah = matplotlib.ticker.LogFormatterSciNotation()
            formatterName = 'matplotlib.ticker.LogFormatterSciNotation()'
          except:
            formatterName = 'matplotlib.ticker.ScalarFormatter()'
        # apply custom formatting?
        if(self.ticksY2Format != 'default'):
          # store fallback
          self.fallback_ticksY2Format = autolabels
          autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksY2Format, self.ticksY2FormatPrecision, self.ticksY2FormatTrailZero, self.ticksY2FormatSeparator, self.ticksY2FormatComma, self.ticksY2FormatPrefix, self.ticksY2FormatPostfix))
        for useAxis in axisItems:
          useAxis.yaxis.set_major_formatter(autolabels)
          useAxis.yaxis.set_major_locator(matplotlib.ticker.FixedLocator(value))
        ###if(self.modeY2 == 'linear'):
        if(1):
          minorAutoticks = MyAutoMinorLocator(self.ticksY2Minor)
          for useAxis in axisItems:
            useAxis.yaxis.set_minor_locator(minorAutoticks)
            useAxis.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # store settings
        self.ticksY2 = value
        lower, upper = self.minY2, self.maxY2
        if(len(value)):
          # check for empty list
          if(np.min(value) < np.min((self.minY2, self.maxY2))):
            flag = True
            lower = np.min(value)
          if(np.max(value) > np.max((self.minY2, self.maxY2))):
            flag = True
            upper = np.max(value)
        self.ticksY2Auto = False
        self.parent.graphicsarea2.configTickYAuto.setChecked(False)
      else:
        for useAxis in axisItems:
          useAxis.yaxis.set_major_locator(matplotlib.ticker.FixedLocator(value))
        autolabels = matplotlib.ticker.ScalarFormatter()
        # apply custom formatting?
        if(self.ticksResidYFormat != 'default'):
          # store fallback
          self.fallback_ticksResidYFormat = autolabels
          autolabels = matplotlib.ticker.FuncFormatter(partial(self.myFormatterFunction, self.ticksResidYFormat, self.ticksResidYFormatPrecision, self.ticksResidYFormatTrailZero, self.ticksResidYFormatSeparator, self.ticksResidYFormatComma, self.ticksResidYFormatPrefix, self.ticksResidYFormatPostfix))
        if(1):
          minorAutoticks = MyAutoMinorLocator(self.ticksResidYMinor)
          for useAxis in axisItems:
            useAxis.yaxis.set_minor_locator(minorAutoticks)
            useAxis.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        formatterName = 'matplotlib.ticker.ScalarFormatter()'
        for useAxis in axisItems:
          useAxis.yaxis.set_major_formatter(autolabels)
        lower, upper = self.minResidY, self.maxResidY
        if(len(value)):
          # check for empty list
          if(np.min(value) < np.min((self.minResidY, self.maxResidY))):
            flag = True
            lower = np.min(value)
          if(np.max(value) > np.max((self.minResidY, self.maxResidY))):
            flag = True
            upper = np.max(value)
        self.ticksResidYAuto = False
        self.parent.graphicsarea.configTickResidYAuto.setChecked(False)
        
      # check whether the new ticks necessitate axis rescaling
      if(flag):
        if(axis in ['y', 'y2']):
          self.setAxisLimits(lower = lower, upper = upper, axis = axis, updateLabel = True, target=target, redraw=redraw, updateGrid=True)
        elif(axis == 'resid'):
          self.setAxisLimits(lower = lower, upper = upper, axis = 'y', updateLabel = True, target='resid', redraw=redraw, updateGrid=True)
        else:
          self.setAxisLimits(lower = lower, upper = upper, axis = axis, updateLabel = True, target=target, redraw=False, updateGrid=True)
          if(target == 'plot'):
            # and we should redraw the fit function to cover new x-range
            if(axis == 'x'):
              self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.parent.plotArea.plotFunction(\
                fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot,\
                handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False)
              # copy in case split axes are shown
              curve = self.parent.fit[self.parent.activeFit]
              if(self.parent.plotArea.splitY and curve.onBothAxes):
                curve.duplicateForSplit()
              if(redraw):
                self.parent.plotArea.dataplotwidget.myRefresh()
            else:
              self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.parent.plotArea.plotFunction(\
                fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div,\
                handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True)
              # copy in case split axes are shown
              curve = self.parent.fit[self.parent.activeFit]
              if(self.parent.plotArea.splitY and curve.onBothAxes):
                curve.duplicateForSplit()
              if(redraw):
                self.parent.plotArea.dataplotwidget.myRefresh()
          else:
            # and we should update the resid plot (as x-axis will most likely have rescaled)
            if(axis == 'x'):
              self.parent.plotArea.handleResidZero = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero, redraw=redraw)
            else:
              self.parent.plotArea.handleResidZero_div = self.parent.plotArea.plotResidZero(self.parent.plotArea.handleResidZero_div, redraw=redraw, splitX=True)
      else:
        # update axis grid lines
        if(axis == 'resid'):
          item = 'y'
        else:
          item = axis
        if(self.gridVisible[item]):
          if((item != 'y2') or self.isSecondAxesActive()):
            self.drawAxisGrid(axis=item, redraw=False, target=target)
      
        if(redraw):
          plotobject.myRefresh()
    elif(axis in ['xs']):
      # wow, this is rather simple as all the hard work is done by draw2ndAxisTicks()
      self.slavedTicksX = value
      if(redraw):
        plotobject.myRefresh()

  def getAxisTicks(self, axis='x'):
    # reports back axis ticks
    if(axis in ['x', 'y', 'y2', 'resid', 'x2']):
      if(axis == 'x'):
        ticks = self.ax.xaxis.get_ticklocs()
      elif(axis == 'x2'):
        ticks = self.ax_div.xaxis.get_ticklocs()
      elif(axis == 'y'):
        ticks = self.ax.yaxis.get_ticklocs()
      elif(axis == 'y2'):
        ticks = self.ax2.yaxis.get_ticklocs()
      else:
        ticks = self.ax_resid.yaxis.get_ticklocs()
      return ticks
    else:
      return []

  def setTickLabelPad2(self, value=0.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname, plotfig = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2', self.matplot
          if((self.ticksXPad2 == value) or (not self.ticksXShow)):
            redraw = False
          self.ticksXPad2 = value
        else:
          plotobject, axisobject, axisobject2, axisname, plotfig = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid', self.residplot
          if((self.ticksXPad2_resid == value) or (not self.ticksXShow_resid)):
            redraw = False
          self.ticksXPad2_resid = value
        #
        tickLabels = axisobject.get_xticklabels(which='both')
        tickLabels.extend(axisobject2.get_xticklabels(which='both'))
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname, plotfig = self.dataplotwidget, self.ax, self.ax_div, 'ax', self.matplot
          if((self.ticksYPad2 == value) or (not self.ticksYShow)):
            redraw = False
          self.ticksYPad2 = value
        else:
          plotobject, axisobject, axisobject2, axisname, plotfig = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid', self.residplot
          if((self.ticksYPad2_resid == value) or (not self.ticksResidYShow)):
            redraw = False
          self.ticksYPad2_resid = value
        #
        tickLabels = axisobject.get_yticklabels(which='both')
        tickLabels.extend(axisobject2.get_yticklabels(which='both'))
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname, plotfig = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2', self.matplot
        if((self.ticksY2Pad2 == value) or (not self.ticksY2Show)):
          redraw = False
        self.ticksY2Pad2 = value
        #
        tickLabels = axisobject.get_yticklabels(which='both')
        tickLabels.extend(axisobject2.get_yticklabels(which='both'))
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXPad2 == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXPad2 = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXPad2_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXPad2_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      # set orthogonal padding using transforms
      if(axis == 'x'):
        offset = matplotlib.transforms.ScaledTranslation(value / 72.0, 0, plotfig.dpi_scale_trans)
      elif(axis in ['y', 'y2']):
        offset = matplotlib.transforms.ScaledTranslation(0, value / 72.0, plotfig.dpi_scale_trans)
      for entry in tickLabels:
        # strip transform down to inner CompositeGenericTransform
        transformA, transformB = entry.get_transform(), matplotlib.transforms.IdentityTransform()
        while(isinstance(transformA, matplotlib.transforms.CompositeGenericTransform)):
          transformA, transformB = transformA._a, transformA._b
        # compile new transform
        transAll = transformA + transformB + offset
        entry.set_transform(transAll)

      # take care of moved axes (otherwise padding set to zero)
      if(axis == 'x'):
        # adjust spine lines if needed
        for spine in ['top', 'bottom']:
          if(self.axisPosition[spine] in ['data', 'axes']):
            self.setAxisPositionHelper(axis=spine, plotobject=plotobject, axisobject=axisobject, target=target, secondAxes=False)
            self.setAxisPositionHelper(axis=spine, plotobject=plotobject, axisobject=axisobject2, target=target, secondAxes=False, splitX=True)
      elif(axis == 'y'):
        # adjust spine lines if needed
        for spine in ['left', 'right']:
          if(self.axisPosition[spine] in ['data', 'axes']):
            self.setAxisPositionHelper(axis=spine, plotobject=plotobject, axisobject=axisobject, target=target, secondAxes=False)
      elif(axis == 'y2'):
        for spine in ['left2', 'right2']:
          if(self.axisPosition[spine] in ['data', 'axes']):
            self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.dataplotwidget, axisobject=self.ax2, target='plot', secondAxes=True)
            self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.dataplotwidget, axisobject=self.ax2_div, target='plot', secondAxes=True)

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickLabelPad(self, value=4.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.ticksXPad == value) or (not self.ticksXShow)):
            redraw = False
          self.ticksXPad = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.ticksXPad_resid == value) or (not self.ticksXShow_resid)):
            redraw = False
          self.ticksXPad_resid = value
        #
        ticks = axisobject.xaxis.get_major_ticks()
        ticks.extend(axisobject.xaxis.get_minor_ticks())
        ticks.extend(axisobject2.xaxis.get_major_ticks())
        ticks.extend(axisobject2.xaxis.get_minor_ticks())
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.ticksYPad == value) or (not self.ticksYShow)):
            redraw = False
          self.ticksYPad = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.ticksYPad_resid == value) or (not self.ticksResidYShow)):
            redraw = False
          self.ticksYPad_resid = value
        #
        ticks = axisobject.yaxis.get_major_ticks()
        ticks.extend(axisobject.yaxis.get_minor_ticks())
        ticks.extend(axisobject2.yaxis.get_major_ticks())
        ticks.extend(axisobject2.yaxis.get_minor_ticks())
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.ticksY2Pad == value) or (not self.ticksY2Show)):
          redraw = False
        self.ticksY2Pad = value
        #
        ticks = axisobject.yaxis.get_major_ticks()
        ticks.extend(axisobject.yaxis.get_minor_ticks())
        ticks.extend(axisobject2.yaxis.get_major_ticks())
        ticks.extend(axisobject2.yaxis.get_minor_ticks())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXPad == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXPad = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXPad_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXPad_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        ticks = []
        
      # set padding
      for entry in ticks:
        entry.set_pad(value)
        
      # now we should call setTickLabelPad2 to prevent undoing of perpendicular shift (kind of ugly thing to do)
      # maybe in the future use a single handler function
      if(axis == 'x'):
        self.setTickLabelPad2(value=self.ticksXPad2, axis=axis, redraw=False, target=target)
      elif(axis == 'y'):
        if(target == 'plot'):
          self.setTickLabelPad2(value=self.ticksYPad2, axis=axis, redraw=False, target=target)
        else:
          self.setTickLabelPad2(value=self.ticksYPad2_resid, axis=axis, redraw=False, target=target)
      elif(axis == 'y2'):
        self.setTickLabelPad2(value=self.ticksY2Pad2, axis=axis, redraw=False, target=target)

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickLabelLinespacing(self, value='normal', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXLinespacing == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXLinespacing = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if((self.ticksYLinespacing == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYLinespacing = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif((axis == 'y2') or (not self.ticksY2Show)):
            if(self.ticksY2Linespacing == value):
              redraw = False
            self.ticksY2Linespacing = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(axis == 'x'):
            if((self.ticksXLinespacing_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXLinespacing_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYLinespacing_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYLinespacing_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXLinespacing == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXLinespacing = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXLinespacing_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXLinespacing_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_linespacing(value)

      # redraw?
      if(redraw):
        plotobject.myRefresh()
      
  def setTickLabelBold(self, value='normal', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXWeight == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXWeight = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if((self.ticksYWeight == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYWeight = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif((axis == 'y2') or (not self.ticksY2Show)):
            if(self.ticksY2Weight == value):
              redraw = False
            self.ticksY2Weight = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(axis == 'x'):
            if((self.ticksXWeight_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXWeight_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYWeight_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYWeight_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXWeight == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXWeight = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXWeight_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXWeight_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_fontweight(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()
      
  def setTickLabelItalic(self, value='normal', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXStyle == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXStyle = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if((self.ticksYStyle == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYStyle = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if((self.ticksY2Style == value) or (not self.ticksY2Show)):
              redraw = False
            self.ticksY2Style = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(axis == 'x'):
            if((self.ticksXStyle_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXStyle_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYStyle_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYStyle_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXStyle == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXStyle = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXStyle_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXStyle_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_fontstyle(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickLabelAngle(self, value=12.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXAngle == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXAngle = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject, axisobject2, axisname = self.ax, self.ax_div, 'ax'
            if((self.ticksYAngle == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYAngle = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if((self.ticksY2Angle == value) or (not self.ticksY2Show)):
              redraw = False
            self.ticksY2Angle = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(axis == 'x'):
            if((self.ticksXAngle_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXAngle_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYAngle_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYAngle_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXAngle == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXAngle = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXAngle_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXAngle_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_rotation(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickLabelAlignment(self, value=None, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXAlignment == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXAlignment = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if((self.ticksYAlignment == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYAlignment = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if((self.ticksY2Alignment == value) or (not self.ticksY2Show)):
              redraw = False
            self.ticksY2Alignment = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(axis == 'x'):
            if((self.ticksXAlignment_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXAlignment_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYAlignment_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYAlignment_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXAlignment == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXAlignment = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXAlignment_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXAlignment_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_horizontalalignment(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickLabelAlignmentVertical(self, value=None, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXAlignmentVertical == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXAlignmentVertical = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject, axisobject2, axisname = self.ax, self.ax_div, 'ax'
            if((self.ticksYAlignmentVertical == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYAlignmentVertical = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if((self.ticksY2AlignmentVertical == value) or (not self.ticksY2Show)):
              redraw = False
            self.ticksY2AlignmentVertical = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(axis == 'x'):
            if((self.ticksXAlignmentVertical_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXAlignmentVertical_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYAlignmentVertical_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYAlignmentVertical_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        # fix matplotlib glitch/feature?
        if(value == 'center'):
          if(len(tickLabels)):
            try:
              tickLabels[0].set_verticalalignment('center_baseline')
              value = 'center_baseline'
            except:
              pass
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXAlignmentVertical == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXAlignmentVertical = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXAlignmentVertical_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXAlignmentVertical_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_verticalalignment(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickLabelSize(self, value=12.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXSize == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXSize = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject, axisobject2, axisname = self.ax, self.ax_div, 'ax'
            if((self.ticksYSize == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYSize = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if((self.ticksY2Size == value) or (not self.ticksY2Show)):
              redraw = False
            self.ticksY2Size = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          # sets tick label size
          if(axis == 'x'):
            if((self.ticksXSize_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXSize_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            if((self.ticksYSize_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYSize_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXSize == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXSize = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXSize_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXSize_resid = value

        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_fontsize(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def toggleTicksLabel(self, value=True, axis='x', redraw=True, target='plot'):
    # toggles visibility of tick labels
    # Matplotlib 3.1.2 only returns handles to visible labels which is counter-intuitive and overall bad
    # rather go via set_tick_params
    if((target in ['plot', 'resid']) and (axis in ['x', 'x2', 'y', 'y2'])):
      if(target == 'plot'):
        plotobject, axisobject, axisobject2 = self.dataplotwidget, self.ax2, self.ax2_div 
        if(axis == 'y'):
          axisobject, axisobject2 = self.ax, self.ax_div
      else:
        plotobject, axisobject, axisobject2 = self.residplotwidget, self.ax_resid, self.ax_resid_div
        
      if(axis == 'x'):
        if(target == 'plot'):
          if(self.ticksXShow == value):
            redraw = False
          self.ticksXShow = value
        else:
          if(self.ticksXShow_resid == value):
            redraw = False
          self.ticksXShow_resid = value
        offsetLabel = axisobject.xaxis.get_offset_text()
      elif(axis == 'x2'):
        if(target == 'plot'):
          if(self.ticksXShow_div == value):
            redraw = False
          self.ticksXShow_div = value
        else:
          if(self.ticksXShow_resid_div == value):
            redraw = False
          self.ticksXShow_resid_div = value
        offsetLabel = axisobject2.xaxis.get_offset_text()
      elif(axis == 'y'):
        if(target == 'plot'):
          if(self.ticksYShow == value):
            redraw = False
          self.ticksYShow = value
        else:
          if(self.ticksResidYShow == value):
            redraw = False
          self.ticksResidYShow = value
        offsetLabel = axisobject.yaxis.get_offset_text()
      elif(axis == 'y2'):
        if(self.ticksY2Show == value):
          redraw = False
        self.ticksY2Show = value
        offsetLabel = axisobject.yaxis.get_offset_text()
      else:
        redraw = False
        offsetLabel = None

      # set visibility of tick labels
      if(axis in ['x', 'x2']):
        if(axis == 'x'):
          useAxisobject = axisobject
        else:
          useAxisobject = axisobject2

        if(value):
          useAxisobject.xaxis.set_tick_params(which='both', labelbottom=self.ticksLabelShow['bottom'], labeltop=self.ticksLabelShow['top'])
          if((axis == 'x') and (hasattr(self.parent, 'graphicsarea'))):
            for useAxis in ('bottom', 'top'):
              self.parent.graphicsarea.configTickMarkLabelShow[useAxis].setChecked(self.ticksLabelShow[useAxis])
        else:
          useAxisobject.xaxis.set_tick_params(which='both', labelbottom=False, labeltop=False)
          if((axis == 'x') and (hasattr(self.parent, 'graphicsarea'))):
            for useAxis in ('bottom', 'top'):
              self.parent.graphicsarea.configTickMarkLabelShow[useAxis].setChecked(False)
      elif(axis == 'y'):
        if(value):
          # let updateInnerSituation() handle this
          if(self.splitShow):
            self.updateInnerSituation()
          else:
            axisobject.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left'], labelright=self.ticksLabelShow['right'])            
            axisobject2.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left'], labelright=self.ticksLabelShow['right'])            
          if(hasattr(self.parent, 'graphicsarea')):
            for useAxis in ('left', 'right'):
              self.parent.graphicsarea.configTickMarkLabelShow[useAxis].setChecked(self.ticksLabelShow[useAxis])
        else:
          axisobject.yaxis.set_tick_params(which='both', labelleft=False, labelright=False)
          axisobject2.yaxis.set_tick_params(which='both', labelleft=False, labelright=False)
          if(hasattr(self.parent, 'graphicsarea')):
            for useAxis in ('left', 'right'):
              self.parent.graphicsarea.configTickMarkLabelShow[useAxis].setChecked(False)
      elif(axis == 'y2'):
        if(value):
          # let updateInnerSituation() handle this
          if(self.splitShow):
            self.updateInnerSituation()
          else:
            axisobject.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left2'], labelright=self.ticksLabelShow['right2'])            
            axisobject2.yaxis.set_tick_params(which='both', labelleft=self.ticksLabelShow['left2'], labelright=self.ticksLabelShow['right2'])            
          if(hasattr(self.parent, 'graphicsarea2')):
            for useAxis in ('left2', 'right2'):
              self.parent.graphicsarea2.configTickMarkLabelShow[useAxis].setChecked(self.ticksLabelShow[useAxis])
        else:
          axisobject.yaxis.set_tick_params(which='both', labelleft=False, labelright=False)
          axisobject2.yaxis.set_tick_params(which='both', labelleft=False, labelright=False)
          if(hasattr(self.parent, 'graphicsarea2')):
            for useAxis in ('left2', 'right2'):
              self.parent.graphicsarea2.configTickMarkLabelShow[useAxis].setChecked(False)
        
      # need to update tick formatting if labels are displayed again
      if(value):
        self.setTickOne4All(axis=axis, redraw=False, target=target)
        
      # offset label
      if(offsetLabel != None):
        offsetLabel.set_visible(value)
      
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def toggleAxisLabel(self, value=True, axis='x', redraw=True, target='plot'):
    # toggles visibility of axis label
    if((target in ['plot', 'resid']) and (axis in ['x', 'y', 'y2'])):
      if(target == 'plot'):
        plotobject, axisobject, axisobject2 = self.dataplotwidget, self.ax2, self.ax2_div
        if(axis == 'y'):
          axisobject = self.ax
      else:
        plotobject, axisobject = self.residplotwidget, self.ax_resid
        
      if(axis == 'x'):
        if(target == 'plot'):
          if(self.labelXShow == value):
            redraw = False
          self.labelXShow = value
        else:
          if(self.labelXShow_resid == value):
            redraw = False
          self.labelXShow_resid = value
        axisobject.xaxis.label.set_visible(value) 
      elif(axis == 'y'):
        if(target == 'plot'):
          if(self.labelYShow == value):
            redraw = False
          self.labelYShow = value
        else:
          if(self.labelYShow_resid == value):
            redraw = False
          self.labelYShow_resid = value
        axisobject.yaxis.label.set_visible(value) 
      elif(axis == 'y2'):
        if(self.labelY2Show == value):
          redraw = False
        self.labelY2Show = value
        if(self.splitShow):
          axisobject2.yaxis.label.set_visible(value) 
          # this is needed when activating split y axis
          if(not self.labelY2Show):
            axisobject.yaxis.label.set_visible(value) 
        else:
          axisobject.yaxis.label.set_visible(value) 
      else:
        redraw = False
      
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelLinespacing(self, value=12.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXLinespacing == value) or (not self.labelXShow)):
            redraw = False
          self.labelXLinespacing = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXLinespacing_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXLinespacing_resid = value

        handleAxis = axisobject.xaxis.label
        handleAxis.set_linespacing(value)
        handleAxis = axisobject2.xaxis.label
        handleAxis.set_linespacing(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYLinespacing == value) or (not self.labelYShow)):
            redraw = False
          self.labelYLinespacing = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYLinespacing_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYLinespacing_resid = value

        handleAxis = axisobject.yaxis.label
        handleAxis.set_linespacing(value)
        handleAxis = axisobject2.yaxis.label
        handleAxis.set_linespacing(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Linespacing == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Linespacing = value
        handleAxis = axisobject.yaxis.label
        handleAxis.set_linespacing(value)
        handleAxis = axisobject2.yaxis.label
        handleAxis.set_linespacing(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXLinespacing == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXLinespacing = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXLinespacing_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXLinespacing_resid = value

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelSize(self, value=12.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXSize == value) or (not self.labelXShow)):
            redraw = False
          self.labelXSize = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXSize_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXSize_resid = value

        handleAxis = axisobject.xaxis.label
        handleAxis.set_size(value)
        handleAxis = axisobject2.xaxis.label
        handleAxis.set_size(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYSize == value) or (not self.labelYShow)):
            redraw = False
          self.labelYSize = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYSize_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYSize_resid = value

        handleAxis = axisobject.yaxis.label
        handleAxis.set_size(value)
        handleAxis = axisobject2.yaxis.label
        handleAxis.set_size(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Size == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Size = value
        handleAxis = axisobject.yaxis.label
        handleAxis.set_size(value)
        handleAxis = axisobject2.yaxis.label
        handleAxis.set_size(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXSize == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXSize = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXSize_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXSize_resid = value

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelPad(self, value=12.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXPad == value) or (not self.labelXShow)):
            redraw = False
          self.labelXPad = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid, 'ax_resid'
          if((self.labelXPad_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXPad_resid = value

        axisobject.xaxis.labelpad = value
        axisobject2.xaxis.labelpad = value
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYPad == value) or (not self.labelYShow)):
            redraw = False
          self.labelYPad = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYPad_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYPad_resid = value

        axisobject.yaxis.labelpad = value
        axisobject2.yaxis.labelpad = value
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Pad == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Pad = value
        axisobject.yaxis.labelpad = value
        axisobject2.yaxis.labelpad = value
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXPad == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXPad = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXPad_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXPad_resid = value

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelPos(self, value=0.5, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXPos == value) or (not self.labelXShow)):
            redraw = False
          self.labelXPos = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXPos_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXPos_resid = value

        axisobject.xaxis.label.set_x(value)
        axisobject2.xaxis.label.set_x(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYPos == value) or (not self.labelYShow)):
            redraw = False
          self.labelYPos = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYPos_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYPos_resid = value

        axisobject.yaxis.label.set_y(value)
        axisobject2.yaxis.label.set_y(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Pos == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Pos = value
        axisobject.yaxis.label.set_y(value)
        axisobject2.yaxis.label.set_y(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXPos == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXPos = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXPos_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXPos_resid = value

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelAngle(self, value=0.0, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXAngle == value) or (not self.labelXShow)):
            redraw = False
          self.labelXAngle = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXAngle_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXAngle_resid = value

        axisobject.xaxis.label.set_rotation(value)
        axisobject2.xaxis.label.set_rotation(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYAngle == value) or (not self.labelYShow)):
            redraw = False
          self.labelYAngle = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYAngle_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYAngle_resid = value

        axisobject.yaxis.label.set_rotation(value)
        axisobject2.yaxis.label.set_rotation(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Angle == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Angle = value
        axisobject.yaxis.label.set_rotation(value)
        axisobject2.yaxis.label.set_rotation(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXAngle == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXAngle = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXAngle_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXAngle_resid = value

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelBold(self, value='normal', axis='x', redraw=True, target='plot'):
    # formats axis label bold
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXWeight == value) or (not self.labelXShow)):
            redraw = False
          self.labelXWeight = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXWeight_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXWeight_resid = value

        axisobject.xaxis.label.set_fontweight(value)
        axisobject2.xaxis.label.set_fontweight(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYWeight == value) or (not self.labelYShow)):
            redraw = False
          self.labelYWeight = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYWeight_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYWeight_resid = value

        axisobject.yaxis.label.set_fontweight(value)
        axisobject2.yaxis.label.set_fontweight(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Weight == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Weight = value
        axisobject.yaxis.label.set_fontweight(value)
        axisobject2.yaxis.label.set_fontweight(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXWeight == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXWeight = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXWeight_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXWeight_resid = value
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelItalic(self, value='normal', axis='x', redraw=True, target='plot'):
    # formats axis label italic
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXStyle == value) or (not self.labelXShow)):
            redraw = False
          self.labelXStyle = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXStyle_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXStyle_resid = value

        axisobject.xaxis.label.set_fontstyle(value)
        axisobject2.xaxis.label.set_fontstyle(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYStyle == value) or (not self.labelYShow)):
            redraw = False
          self.labelYStyle = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYStyle_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYStyle_resid = value

        axisobject.yaxis.label.set_fontstyle(value)
        axisobject2.yaxis.label.set_fontstyle(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Style == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Style = value
        axisobject.yaxis.label.set_fontstyle(value)
        axisobject2.yaxis.label.set_fontstyle(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXStyle == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXStyle = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXStyle_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXStyle_resid = value
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisLabelVariant(self, value='normal', axis='x', redraw=True, target='plot'):
    # formats axis label small-caps
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if((self.labelXVariant == value) or (not self.labelXShow)):
            redraw = False
          self.labelXVariant = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelXVariant_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXVariant_resid = value

        axisobject.xaxis.label.set_variant(value)
        axisobject2.xaxis.label.set_variant(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax, self.ax_div, 'ax'
          if((self.labelYVariant == value) or (not self.labelYShow)):
            redraw = False
          self.labelYVariant = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if((self.labelYVariant_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYVariant_resid = value

        axisobject.yaxis.label.set_variant(value)
        axisobject2.yaxis.label.set_variant(value)
      elif(axis == 'y2'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
        if((self.labelY2Variant == value) or (not self.labelY2Show)):
          redraw = False
        self.labelY2Variant = value
        axisobject.yaxis.label.set_variant(value)
        axisobject2.yaxis.label.set_variant(value)
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXVariant == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXVariant = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXVariant_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXVariant_resid = value
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setAxisArrow(self, state=True, axis='x', item='all', redraw=True, target='plot'):
    # toggles drawing of arrow
    if(item in ['all', 'edge', 'fill']):
      if(((target == 'plot') and (axis in ['x', 'y', 'y2'])) or ((target == 'resid') and (axis in ['x', 'y']))):
        if(target == 'plot'):
          plotobject, arrowhandle, arrowhandle2  = self.dataplotwidget, self.handleArrow, self.handleArrow2
        else:
          plotobject, arrowhandle, arrowhandle2 = self.residplotwidget, self.handleArrowResid, self.handleArrowResid2
        
        if(item == 'all'):
          self.arrowVisible[axis] = state
          if(state):
            self.drawAxisArrow(axis=axis, redraw=redraw, target=target)
          elif((arrowhandle[axis] != None) or (arrowhandle2[axis] != None)):
            if(arrowhandle[axis] != None):
              arrowhandle[axis].remove()
              arrowhandle[axis] = None
            if(arrowhandle2[axis] != None):
              arrowhandle2[axis].remove()
              arrowhandle2[axis] = None
            if(redraw):
              plotobject.myRefresh()
        else:
          if(item == 'edge'):
            if(target == 'plot'):
              if(self.arrowEdgeShow[axis] == state):
                redraw = False
              self.arrowEdgeShow[axis] = state
            else:
              if(self.arrowEdgeShow_resid[axis] == state):
                redraw = False
              self.arrowEdgeShow_resid[axis] = state
          else:
            if(target == 'plot'):
              if(self.arrowFillShow[axis] == state):
                redraw = False
              self.arrowFillShow[axis] = state
            else:
              if(self.arrowFillShow_resid[axis] == state):
                redraw = False
              self.arrowFillShow_resid[axis] = state
  
          if(redraw and self.arrowVisible[axis]):
            self.drawAxisArrow(axis=axis, redraw=redraw, target=target)

  def setAxisArrowColor(self, value=[0.0, 0.0, 0.0, 1.0], axis='x', item='fill', redraw=True):
    # changes color axis arrow
    if((axis in ['x', 'y', 'y2']) and (item in ['line', 'fill'])):
      if(item == 'line'):
        checkItem = self.arrowEdgeShow[axis]
        if(self.arrowColor[axis] == value):
          redraw = False
        self.arrowColor[axis] = value
      else:
        checkItem = self.arrowFillShow[axis]
        if(self.arrowFill[axis] == value):
          redraw = False
        self.arrowFill[axis] = value
      
      if(redraw and self.arrowVisible[axis] and checkItem):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowLineWidth(self, value=0.1, axis='x', redraw=True):
    # changes edge width of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowEdge[axis] == value):
        redraw = False
      self.arrowEdge[axis] = value

      if(redraw and self.arrowVisible[axis] and self.arrowEdgeShow[axis]):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowHeadWidth(self, value=0.1, axis='x', redraw=True):
    # changes width of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowHeadWidth[axis] == value):
        redraw = False
      self.arrowHeadWidth[axis] = value

      if((redraw) and (self.arrowVisible[axis])):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowHeadLength(self, value=0.1, axis='x', redraw=True):
    # changes width of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowHeadLength[axis] == value):
        redraw = False
      self.arrowHeadLength[axis] = value

      if((redraw) and (self.arrowVisible[axis])):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowOverhang(self, value=0.1, axis='x', redraw=True):
    # changes overhang of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowOverhang[axis] == value):
        redraw = False
      self.arrowOverhang[axis] = value

      if((redraw) and (self.arrowVisible[axis])):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowOffset(self, value=0, axis='x', redraw=True):
    # changes offset of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowOffset[axis] == value):
        redraw = False
      self.arrowOffset[axis] = value

      if((redraw) and (self.arrowVisible[axis])):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowLocation(self, value='both', axis='x', redraw=True):
    # changes location of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowLocation[axis] == value):
        redraw = False
      self.arrowLocation[axis] = value

      if((redraw) and (self.arrowVisible[axis])):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def setAxisArrowZ(self, value='axis', axis='x', redraw=True):
    # changes location of axis arrow
    if(axis in ['x', 'y', 'y2']):
      if(self.arrowZ[axis] == value):
        redraw = False
      self.arrowZ[axis] = value

      if((redraw) and (self.arrowVisible[axis])):
        self.drawAxisArrow(axis=axis, redraw=redraw, target='plot')
        if(axis != 'y2'):
          self.drawAxisArrow(axis=axis, redraw=redraw, target='resid')

  def drawSplitYAxisDivider(self, redraw=True):
    # draws divider marks on split y axis
    plotobject, axisobject, axisobject2, handleDividerY = self.dataplotwidget, self.ax, self.ax_div, self.handleDividerY
    drawobject, drawobject2 = self.ax, self.ax2

    # delete old lines
    if(handleDividerY != None):
      for entry in handleDividerY:
        entry.remove()
    handleDividerY = None

    # draw new lines
    if(self.splitY and self.splitYDivider):
      # determine draw coordinates
      xcoord, ycoord, handleDividerY, axisList = [], [], [], []
      if(self.splitYDividerLocation == 'both'):
        useItems = ['left', 'right']
      else:
        useItems = [self.splitYDividerLocation]
      # account for shifted axis
      for item in useItems:
        # have to go via axisobject as drawobject throws an error
        useAxis, useAxisobject = drawobject, axisobject
        if(self.splitShow and (item == 'right')):
          useAxisobject, useAxis = axisobject2, self.ax_div
        if(item in useAxisobject.spines):
          try:
            axisPosition, axisPositionValue = useAxisobject.spines[item].get_position()
            if(axisPosition == 'data'):
              drawX = axisPositionValue
            elif(axisPosition == 'axes'):
              axis_to_data = useAxis.transAxes + useAxis.transData.inverted()
              transformedPoint = axis_to_data.transform((axisPositionValue, 0))
              drawX = transformedPoint[0]
            # passed => store coords for drawing
            xcoord.append(drawX)
            ycoord.append(self.minY)
            axisList.append(useAxis)
          except:
            pass
            
      # now copy coordinates to axisobject2
      if(self.splitYPad):
        for number, x in enumerate([i for i in xcoord]):
          xcoord.append(x)
          ycoord.append(self.maxY2)
          if(axisList[number] == drawobject):
            axisList.append(drawobject2)
          else:
            axisList.append(self.ax2_div)

      for x, y, useAxis in zip(xcoord, ycoord, axisList):
        # assign use-x values
        if(useAxis in [drawobject, drawobject2]):
          useMinX, useMaxX, useModeX = self.minX, self.maxX, self.modeX
        else:
          useMinX, useMaxX, useModeX = self.minX_div, self.maxX_div, self.modeX_div
          
        if(useAxis in [drawobject, self.ax_div]):
          useMinY, useMaxY, useModeY = self.minY, self.maxY, self.modeY
        else:
          useMinY, useMaxY, useModeY = self.minY2, self.maxY2, self.modeY2
          
        # draw some lines
        if(useModeY == 'linear'):
          dimY = (useMaxY - useMinY)
        else:
          dimY = np.log(np.abs(useMaxY / useMinY))
        if(useModeX == 'linear'):
          dimX = (useMaxX - useMinX)
        else:
          dimX = np.log(np.abs(useMaxX / useMinX))
        plotX, plotY = self.matplot.get_size_inches()
        # also take into account plot padding
        plotX *= (self.padSize['right'] - self.padSize['left'])
        plotY *= (self.padSize['top'] - self.padSize['bottom'])
        deltaX = (1 + self.splitFraction) * self.splitYDividerLength * np.cos(self.splitYDividerAngle * np.pi / 180.0) / plotX * np.max((plotX, plotY))
        if(not useAxis in [drawobject, drawobject2]):
          deltaX /= self.splitFraction
        if(not self.splitShow):
          deltaX /= 2
        deltaY = (1 + self.splitYFraction) * self.splitYDividerLength * np.sin(self.splitYDividerAngle * np.pi / 180.0) / plotY * np.max((plotX, plotY))
        if(not useAxis in [drawobject, self.ax_div]):
          deltaY /= self.splitYFraction

        if(useModeX == 'linear'):
          cx1, cx2 = x - 0.5 * deltaX * dimX, x + 0.5 * deltaX * dimX
        else:
          cx1, cx2 = x / np.exp(0.5 * deltaX * dimX), x * np.exp(0.5 * deltaX * dimX)
        if(useModeY == 'linear'):
          cy1, cy2 = y - 0.5 * deltaY * dimY, y + 0.5 * deltaY * dimY
        else:
          cy1, cy2 = y / np.exp(0.5 * deltaY * dimY), y * np.exp(0.5 * deltaY * dimY)

        # transform to axiscoordinates
        drawLine = matplotlib.lines.Line2D([cx1, cx2], [cy1, cy2])
        handleDividerY.append(useAxis.add_line(drawLine))

      # apply styles
      for entry in handleDividerY:
        entry.set_color(self.splitYDividerColor)
        entry.set_solid_capstyle(self.splitYDividerDashStyle)
        entry.set_linewidth(self.splitYDividerWidth)
        entry.set_clip_on(False)
        entry.set_zorder(999)
        
    # save handles
    self.handleDividerY = handleDividerY

    if(redraw):
      plotobject.myRefresh()

  def drawSplitAxisDivider(self, redraw=True, target='plot'):
    # draws divider marks on split axis
    if(target in ['plot', 'resid']):
      # assign handles
      if(target == 'plot'):
        plotobject, axisobject, axisobject2, handleDivider = self.dataplotwidget, self.ax2, self.ax2_div, self.handleDivider
        useMinY, useMaxY, useModeY = self.minY, self.maxY, self.modeY
        drawobject, drawobject2 = self.ax, self.ax_div
      else:
        plotobject, axisobject, axisobject2, handleDivider = self.residplotwidget, self.ax_resid, self.ax_resid_div, self.handleDividerResid
        plotobject, axisobject, axisobject2, handleDivider = self.residplotwidget, self.ax_resid, self.ax_resid_div, self.handleDividerResid
        useMinY, useMaxY, useModeY = self.minResidY, self.maxResidY, 'linear'
        drawobject, drawobject2 = axisobject, axisobject2

      # delete old lines
      if(handleDivider != None):
        for entry in handleDivider:
          entry.remove()
      handleDivider = None

      # draw new lines
      if(self.splitShow and self.splitDivider):
        # determine draw coordinates
        xcoord, ycoord, handleDivider, axisList = [], [], [], []
        if(self.splitDividerLocation == 'both'):
          useItems = ['bottom', 'top']
        else:
          useItems = [self.splitDividerLocation]
        # account for shifted axis
        for item in useItems:
          # have to go via axisobject as drawobject throws an error
          useAxis, useAxisobject = drawobject, axisobject
          if((target == 'plot') and self.splitY and (item == 'bottom')):
            useAxisobject, useAxis = axisobject2, self.ax2
          if(item in useAxisobject.spines):
            try:
              axisPosition, axisPositionValue = useAxisobject.spines[item].get_position()
              if(axisPosition == 'data'):
                drawY = axisPositionValue
              elif(axisPosition == 'axes'):
                axis_to_data = useAxis.transAxes + useAxis.transData.inverted()
                transformedPoint = axis_to_data.transform((0, axisPositionValue))
                drawY = transformedPoint[1]
              # passed => store coords for drawing
              xcoord.append(self.maxX)
              ycoord.append(drawY)
              axisList.append(useAxis)
            except:
              pass
            
        # now copy coordinates to axisobject2
        if(self.splitPad):
          for number, y in enumerate([i for i in ycoord]):
            xcoord.append(self.minX_div)
            ycoord.append(y)
            if(axisList[number] == drawobject):
              axisList.append(drawobject2)
            else:
              axisList.append(self.ax2_div)
          
        for x, y, useAxis in zip(xcoord, ycoord, axisList):
          # assign use-x values
          if(target == 'plot'):
            if(useAxis in [drawobject, self.ax_div]):
              useMinY, useMaxY, useModeY = self.minY, self.maxY, self.modeY
            else:
              useMinY, useMaxY, useModeY = self.minY2, self.maxY2, self.modeY2

          if(useAxis in [drawobject, self.ax2]):
            useMinX, useMaxX, useModeX = self.minX, self.maxX, self.modeX
          else:
            useMinX, useMaxX, useModeX = self.minX_div, self.maxX_div, self.modeX_div
            
          # draw some lines
          if(useModeX == 'linear'):
            dimX = (useMaxX - useMinX)
          else:
            dimX = np.log(np.abs(useMaxX / useMinX))
          if(useModeY == 'linear'):
            dimY = (useMaxY - useMinY)
          else:
            dimY = np.log(np.abs(useMaxY / useMinY))
          plotX, plotY = self.matplot.get_size_inches()
          # also take into account plot padding
          plotX *= (self.padSize['right'] - self.padSize['left'])
          plotY *= (self.padSize['top'] - self.padSize['bottom'])
          deltaX = (1 + self.splitFraction) * self.splitDividerLength * np.cos(self.splitDividerAngle * np.pi / 180.0) / plotX * np.max((plotX, plotY))
          if(not useAxis in [drawobject2, self.ax2_div]):
            deltaX /= self.splitFraction
          deltaY = (1 + self.splitYFraction) * self.splitDividerLength * np.sin(self.splitDividerAngle * np.pi / 180.0) / plotY * np.max((plotX, plotY))
          if(target == 'plot'):
            if(self.splitY):
              if(not useAxis in [drawobject, self.ax_div]):
                deltaY /= self.splitYFraction
            else:
              deltaY /= 2.
          
          if(target == 'resid'):
            # account for size ratio plot/resid
            relativeSizes = self.masterwidget.sizes()
            sizeRatio = 1. * relativeSizes[0] / relativeSizes[1]
            deltaY *= sizeRatio / 2.
  
          if(useModeX == 'linear'):
            cx1, cx2 = x - 0.5 * deltaX * dimX, x + 0.5 * deltaX * dimX
          else:
            cx1, cx2 = x / np.exp(0.5 * deltaX * dimX), x * np.exp(0.5 * deltaX * dimX)
          if(useModeY == 'linear'):
            cy1, cy2 = y - 0.5 * deltaY * dimY, y + 0.5 * deltaY * dimY
          else:
            cy1, cy2 = y / np.exp(0.5 * deltaY * dimY), y * np.exp(0.5 * deltaY * dimY)

          # transform to axiscoordinates
          drawLine = matplotlib.lines.Line2D([cx1, cx2], [cy1, cy2])
          handleDivider.append(useAxis.add_line(drawLine))

        # apply styles
        for entry in handleDivider:
          entry.set_color(self.splitDividerColor)
          entry.set_solid_capstyle(self.splitDividerDashStyle)
          entry.set_linewidth(self.splitDividerWidth)
          entry.set_clip_on(False)
          entry.set_zorder(999)
          
      # save handles
      if(target == 'plot'):
        self.handleDivider = handleDivider
      else:
        self.handleDividerResid = handleDivider

      if(redraw):
        plotobject.myRefresh()

  def transformSlavedX(self, operation='+', number='1.0', redraw=True):
    # adds a transformation step for slaved x axis
    if(operation in ['+', '-', '/', '*', '**', 'e', 'ln']):
      self.slavedXTransform.append([operation, number])
    else:
      redraw = False
      
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()
        
    return self.slavedXTransform

  def resetSlavedX(self, redraw=True):
    # resets transformation for slaved x axis
    if(self.slavedXTransform == []):
      redraw = False
    self.slavedXTransform = []
        
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

  def toggleSlavedX(self, state=True, redraw=True):
    # toggles slaved x axis display
    if(state == self.slavedX):
      redraw = False
    self.slavedX = state
    
    if(redraw):
      # hide regular ticks on top axis
      if(state):
        self.setTickMarkVisibility(value=False, axis='top', redraw=False, target='plot')
        self.setTickMarkVisibility(value=False, axis='top', redraw=False, target='resid')
        if(hasattr(self.parent, 'graphicsarea')):
          self.parent.graphicsarea.configTickMarkLabel['top'].setChecked(False)
        
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

  def toggleSlavedXLabel(self, state=True, redraw=True):
    # toggles whether to display label for slaved x axis
    if(state == self.slavedXLabelShow):
      redraw = False
    self.slavedXLabelShow = state

    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()
          
  def changeSlavedXLabel(self, value='transformed x', redraw=True):
    # alters label for slaved x axis
    if(value == self.slavedXLabel):
      redraw = False
    self.slavedXLabel = value

    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

  def transform2ndAxisTicks(self, values, reverse=False):
    # transforms tick values back and forth for slaved x axis
    if(len(values)):
      values = np.array(values)
      
      if(reverse):
        for transform in self.slavedXTransform[::-1]:
          if(transform[0] == '+'):
            values -= transform[1]
          elif(transform[0] == '-'):
            values += transform[1]
          elif(transform[0] == '*'):
            values /= transform[1]
          elif(transform[0] == '/'):
            values *= transform[1]
          elif(transform[0] == '**'):
            values = values ** (1 / transform[1])
          elif(transform[0] == 'e'):
            values = np.log(values)
          elif(transform[0] == 'ln'):
            values = np.exp(values)
      else:
        for transform in self.slavedXTransform:
          if(transform[0] == '+'):
            values += transform[1]
          elif(transform[0] == '-'):
            values -= transform[1]
          elif(transform[0] == '*'):
            values *= transform[1]
          elif(transform[0] == '/'):
            values /= transform[1]
          elif(transform[0] == '**'):
            values = values ** transform[1]
          elif(transform[0] == 'e'):
            values = np.exp(values)
          elif(transform[0] == 'ln'):
            values = np.log(values)
          
    return values

  def draw2ndXAxisTicks(self, axis='x', target='plot'):
    # helper function to draw custom labels for 2nd x axis
    # subroutine for calculating position of minor ticks
    def calcMinorTicks(majorlocs, ndivs, vmin, vmax):
      'Return the locations of the ticks for specified parameters'
  
      if ndivs is None:
        ndivs = 9
  
      # view interval
      if vmin > vmax:
        vmin, vmax = vmax, vmin
  
      # calculate minor ticks
      locs = np.array([])
      for index in range(len(majorlocs) - 1):
        # can treat linear and log axes the same
        nulocs = np.linspace(majorlocs[index], majorlocs[index + 1], ndivs + 1)
        locs = np.hstack((locs, nulocs[1: -1]))
      
      # run through conditions
      cond1 = locs >= vmin
      cond2 = locs <= vmax
      locs = locs.compress(cond1 & cond2)
  
      return np.array(locs)

    # main function code
    if((target in ['plot', 'resid']) and (axis in ['x', 'x2'])):
      if(axis == 'x'):
        suffix = ''
        minX, maxX = self.minX, self.maxX
        if(target == 'plot'):
          axisobject = self.ax2
        else:
          axisobject = self.ax_resid
      else:
        suffix = '_div'
        minX, maxX = self.minX_div, self.maxX_div
        if(target == 'plot'):
          axisobject = self.ax2_div
        else:
          axisobject = self.ax_resid_div
          
      if(target == 'plot'):
        plotobject = self.matplot
      else:
        plotobject = self.residplot
        
      # remove old objects
      if(target == 'plot'):
        for killItem in self.__dict__['secondXLines' + suffix] + self.__dict__['secondXLabels' + suffix]:
          killItem.remove()
        self.__dict__['secondXLines' + suffix], self.__dict__['secondXLabels' + suffix]  = [], []
        if((self.handleSlavedXLabel != None) and (axis == 'x')):
          self.handleSlavedXLabel.remove()
          self.handleSlavedXLabel = None
      else:
        for killItem in self.__dict__['secondXLinesResid' + suffix] + self.__dict__['secondXLabelsResid' + suffix]:
          killItem.remove()
        self.__dict__['secondXLinesResid' + suffix], self.__dict__['secondXLabelsResid' + suffix]  = [], []
        if((self.handleSlavedXLabelResid != None) and (axis == 'x')):
          self.handleSlavedXLabelResid.remove()
          self.handleSlavedXLabelResid = None
          
      # draw new slaved ticks?
      if(self.slavedX and ((axis == 'x') or self.splitShow)):
        # by default use self.maxY2
        if(target == 'plot'):
          minY, maxY = self.minY2, self.maxY2
        else:
          minY, maxY = self.minResidY, self.maxResidY
        
        # check whether axis is shifted from default position
        try:
          axisPosition, axisPositionValue = axisobject.spines['top'].get_position()
          if(axisPosition == 'data'):
            maxY = axisPositionValue
          elif(axisPosition == 'axes'):
            axis_to_data = axisobject.transAxes + axisobject.transData.inverted()
            transformedPoint = axis_to_data.transform((0, axisPositionValue))
            maxY = transformedPoint[1]
        except:
          pass
  
        # transform x values according to formula
        try:
          [tmin, tmax] = self.transform2ndAxisTicks([minX, maxX], reverse=False)
          if(self.slavedTicksXAuto):
            autoticks = axisobject.xaxis.get_major_locator()
            transticks = autoticks.tick_values(tmin, tmax)
            # update tick entry field if auto ticks is enabled
            tickstr = self.parent.graphicsarea.magicTickstring(transticks)
            self.parent.graphicsarea.slavedTickXEntry.setText(tickstr)
          else:
            transticks = self.slavedTicksX
          ###print(autoticks, transticks)
          tickpositions = self.transform2ndAxisTicks(transticks, reverse=True)
          
          # deal with minor ticks
          if(self.slavedTicksXMinor):
            ndivs = self.slavedTicksXMinor
            transticks2 = calcMinorTicks(majorlocs=transticks, ndivs=ndivs, vmin=tmin, vmax=tmax)
            tickpositions2 = self.transform2ndAxisTicks(transticks2, reverse=True)
          else:
            transticks2, tickpositions2 = [], []
        except:
          transticks, tickpositions = [], []
          transticks2, tickpositions2 = [], []
          
        # need to calculate positions of axis ticks
        # the length is specified in units of 1 / 72 in
        currWidth, currHeight = plotobject.get_size_inches()
        currHeight *= self.padSize['top'] - self.padSize['bottom']
        useLength = self.ticksLength['slaved'] / currHeight / 72. * abs(maxY - minY)
        # need to correct useLength for split (if visible)
        # maybe need additional correction for visible plot area (i.e. take into account padding around and between plots)
        if(self.splitY and (target == 'plot')):
          useLength *= (1 + self.splitYFraction) / self.splitYFraction
        
        # consider direction of ticks
        if(self.ticksDirection['slaved'] == 'in'):
          pos1, pos2 = maxY, maxY - useLength
          mpos1, mpos2 = maxY, maxY - useLength * self.slavedTicksXMinorRelativeLength
        elif(self.ticksDirection['slaved'] == 'out'):
          pos1, pos2 = maxY, maxY + useLength
          mpos1, mpos2 = maxY, maxY + useLength * self.slavedTicksXMinorRelativeLength
        else:
          pos1, pos2 = maxY - .5 * useLength, maxY + .5 * useLength
          mpos1, mpos2 = maxY - .5 * useLength * self.slavedTicksXMinorRelativeLength, maxY + .5 * useLength * self.slavedTicksXMinorRelativeLength
        
        # now draw these on the current top axis
        if(self.slavedXFormatter != None):
          labelFormatter = self.slavedXFormatter
        else:
          labelFormatter = axisobject.xaxis.get_major_formatter()
        for tick, tickpos in zip(transticks, tickpositions):
          if((tickpos >= min(minX, maxX)) and (tickpos <= max(minX, maxX))):
            handle, = axisobject.plot([tickpos, tickpos], [pos1, pos2], 'r-')
            handle.set_clip_on(False)
            if(target == 'plot'):
              self.__dict__['secondXLines' + suffix].append(handle)
              # implement check for div axis
              # for some reason formatter returns empty string upon initialization
              # in this case, go via main axis
              labelText = labelFormatter.__call__(tick)
              if(not len(labelText)):
                labelText = self.ax2.xaxis.get_major_formatter().__call__(tick)
              self.__dict__['secondXLabels' + suffix].append(axisobject.text(tickpos, max(pos1, pos2), labelText))
            else:
              self.__dict__['secondXLinesResid' + suffix].append(handle)
              labelText = labelFormatter.__call__(tick)
              if(not len(labelText)):
                labelText = self.ax_resid.xaxis.get_major_formatter().__call__(tick)
              self.__dict__['secondXLabelsResid' + suffix].append(axisobject.text(tickpos, max(pos1, pos2), labelText))
        # and the minor ticklis as well
        for tickpos in tickpositions2:
          if((tickpos >= min(minX, maxX)) and (tickpos <= max(minX, maxX))):
            handle, = axisobject.plot([tickpos, tickpos], [mpos1, mpos2], 'r-')
            handle.set_clip_on(False)
            if(target == 'plot'):
              self.__dict__['secondXLines' + suffix].append(handle)
            else:
              self.__dict__['secondXLinesResid' + suffix].append(handle)
        # and finally the axis label
        if(target == 'plot'):
          useLabel = 'handleSlavedXLabel'
        else:
          useLabel = 'handleSlavedXLabelResid'
        if(self.slavedXLabelShow and (axis == 'x')):
          if(self.modeX == 'linear'):
            midX = (self.minX + self.maxX) / 2
          else:
            midX = np.exp((np.log(self.minX) + np.log(self.maxX)) / 2.)
          # prepare axis label string
          text = self.slavedXLabel.replace('\n', '\\n')
          text = text.replace('\t', '\\t')
          text = '\n'.join([i for i in text.split('\\n') if (len(i.strip()))])
          text = '\t'.join([i for i in text.split('\\t') if (len(i.strip()))])
          self.__dict__[useLabel] = axisobject.text(midX, mpos2, text)

        # format labels
        if(target == 'plot'):
          useLabels = 'secondXLabels'
        else:
          useLabels = 'secondXLabelsResid'
        for label in self.__dict__[useLabels + suffix]:
          label.set_visible(self.ticksLabelShow['slaved'])
          label.set_color(self.slavedTicksXColor)
          label.set_fontsize(self.slavedTicksXSize)
          label.set_fontweight(self.slavedTicksXWeight)
          label.set_fontstyle(self.slavedTicksXStyle)
          label.set_rotation(self.slavedTicksXAngle)
          label.set_horizontalalignment(self.slavedTicksXAlignment)
          label.set_verticalalignment(self.slavedTicksXAlignmentVertical)

        # phew, font setting can be dangerous as we can deviated from font of main ticks
        # have to check for problematic fonts
        # there is still a residual danger here that we get a mortal crash ... :(
        for label in self.__dict__[useLabels + suffix]:
          label.set_fontname(self.tickFont['xs'])
          _, _, descent = label._get_layout(self.matplot.canvas.renderer)
          if(not (descent > 0)):
            self.parent.statusbar.showMessage('Experiencing problems setting font ' + self.tickFont['xs'] + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
            self.tickFont['xs'] = SAFE_FONT
            label.set_fontname(self.tickFont['xs'])

        # deal with transformation on tick labels
        pad, pad2 = self.slavedTicksXPad, self.slavedTicksXPad2
        offset = matplotlib.transforms.ScaledTranslation(pad2 / 72., pad / 72., plotobject.dpi_scale_trans)
        if(target == 'plot'):
          useLabels = 'secondXLabels'
        else:
          useLabels = 'secondXLabelsResid'
        for label in self.__dict__[useLabels + suffix]:
          # strip transform down to inner CompositeGenericTransform
          transformA, transformB = label.get_transform(), matplotlib.transforms.IdentityTransform()
          while(isinstance(transformA, matplotlib.transforms.CompositeGenericTransform)):
            transformA, transformB = transformA._a, transformA._b
          # compile new transform
          transAll = transformA + transformB + offset
          label.set_transform(transAll)
          
        # format lines
        if(target == 'plot'):
          useLines = 'secondXLines'
        else:
          useLines = 'secondXLinesResid'
        for line in self.__dict__[useLines + suffix]:
          line.set_visible(self.ticksVisible['slaved'])
          line.set_linewidth(self.ticksWidth['slaved'])
          line.set_color(self.ticksColor['slaved'])

        # format axis label
        if(axis == 'x'):
          if(target == 'plot'):
            useLabel = self.__dict__['handleSlavedXLabel']
          else:
            useLabel = self.__dict__['handleSlavedXLabelResid']
          if(useLabel != None):
            useLabel.set_color(self.slavedLabelXColor)
            useLabel.set_size(self.slavedLabelXSize)
            useLabel.set_fontweight(self.slavedLabelXWeight)
            useLabel.set_fontstyle(self.slavedLabelXStyle)
            ###useLabel.setAxisLabelVariant(value = self.labelXVariant, axis = 'x', redraw = False, target=target)
            useLabel.set_horizontalalignment(self.slavedLabelXAlignment)
            useLabel.set_verticalalignment(self.slavedLabelXAlignmentVertical)
            useLabel.set_linespacing(self.slavedLabelXLinespacing)
            useLabel.set_rotation(self.slavedLabelXAngle)

            # phew, font setting can be dangerous as we can deviate from font of main axis label
            # have to check for problematic fonts
            # there is still a residual danger here that we get a mortal crash ... :(
            useLabel.set_fontname(self.axisFont['xs'])
            _, _, descent = useLabel._get_layout(self.matplot.canvas.renderer)
            if(not (descent > 0)):
              self.parent.statusbar.showMessage('Experiencing problems setting font ' + self.axisFont['xs'] + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
              self.axisFont['xs'] = SAFE_FONT
              useLabel.set_fontname(self.axisFont['xs'])

            # need to convert xpos of label to coordinates on slaved x axis -- brrrr
            if(self.modeX == 'linear'):
              usePos = self.minX + self.slavedLabelXPos * (self.maxX - self.minX)
            else:
              usePos = self.minX * np.exp(self.slavedLabelXPos * (np.log(self.maxX) - np.log(self.minX)))
            useLabel.set_x(usePos)
  
            # deal with transformation on axis labels
            # account for size and shift of axis ticks
            if((target == 'plot') and (len(self.__dict__['secondXLabels' + suffix]))):
              size = self.__dict__['secondXLabels' + suffix][0].get_size()
              shiftos = self.slavedTicksXPad + size
            elif((target == 'resid') and (len(self.__dict__['secondXLabelsResid' + suffix]))):
              size = self.__dict__['secondXLabelsResid' + suffix][0].get_size()
              shiftos = self.slavedTicksXPad + size
            else:
              shiftos = self.slavedTicksXPad
              
            pad, pad2 = self.slavedLabelXPad + shiftos, 0
            offset = matplotlib.transforms.ScaledTranslation(pad2 / 72., pad / 72., plotobject.dpi_scale_trans)
            # strip transform down to inner CompositeGenericTransform
            transformA, transformB = useLabel.get_transform(), matplotlib.transforms.IdentityTransform()
            while(isinstance(transformA, matplotlib.transforms.CompositeGenericTransform)):
              transformA, transformB = transformA._a, transformA._b
            # compile new transform
            transAll = transformA + transformB + offset
            useLabel.set_transform(transAll)
          
  def drawAxisArrow(self, axis='x', redraw=True, target='plot'):
    # draws arrowhead along axis
    drawOnDiv = self.splitShow and (axis == 'x')
    if((target in ['plot', 'resid']) and (axis in ['x', 'y', 'y2'])):
      # assign handles
      if(target == 'plot'):
        plotobject, axisobject, axisobject2, arrowhandle, arrowhandle2 = self.dataplotwidget, self.ax, self.ax_div, self.handleArrow, self.handleArrow2
        if(drawOnDiv):
          axisobject = self.ax_div
        if(axis == 'y2'):
          axisobject, axisobject2 = self.ax2, self.ax2_div
      else:
        plotobject, axisobject, axisobject2, arrowhandle, arrowhandle2 = self.residplotwidget, self.ax_resid, self.ax_resid_div, self.handleArrowResid, self.handleArrowResid2
        if(drawOnDiv):
          axisobject = self.ax_resid_div

      # assign values
      if(self.arrowEdgeShow[axis]):
        drawCol = self.arrowColor[axis]
      else:
        drawCol = 'None'
      if(self.arrowFillShow[axis]):
        drawFill = self.arrowFill[axis]
      else:
        drawFill = 'None'
      drawOverhang = self.arrowOverhang[axis]
      drawOffset = self.arrowOffset[axis]
      effectiveWidth = self.exportWidth * (self.padSize['right'] - self.padSize['left'])
      effectiveHeight = self.exportHeight * (self.padSize['top'] - self.padSize['bottom'])
      if(axis == 'x'):
        drawHeadWidth = self.arrowHeadWidth[axis] / effectiveHeight
        drawHeadLength = self.arrowHeadLength[axis] / effectiveWidth
      else:
        drawHeadWidth = self.arrowHeadWidth[axis] / effectiveWidth
        drawHeadLength = self.arrowHeadLength[axis] / effectiveHeight
      if(self.arrowZ[axis] == 'axis'):
        drawZ = axisobject.spines['bottom'].get_zorder() + 0.1
      else:
        # set to front
        drawZ = 500
      drawLw = self.arrowEdge[axis]
      
      # which axis to operate on?
      if(axis == 'x'):
        # calculate drawWidth
        if(drawOnDiv):
          useModeX, useMinX, useMaxX = self.modeX_div, self.minX_div, self.maxX_div
        else:
          useModeX, useMinX, useMaxX = self.modeX, self.minX, self.maxX
          
        if(useModeX == 'linear'):
          drawWidth = np.abs(useMaxX - useMinX)
          logx = False
        else:
          drawWidth = np.abs(np.log(useMaxX) - np.log(useMinX))
          logx = True

        if(self.splitShow):
          drawHeadLength *= drawWidth * (1 + 1 / self.splitFraction) * self.splitFraction
        else:
          drawHeadLength *= drawWidth
  
        # calculate drawHeight
        if(target == 'plot'):
          if(self.modeY == 'linear'):
            drawHeight = np.abs(self.maxY - self.minY)
            logy = False
          else:
            drawHeight = np.abs(np.log(self.maxY) - np.log(self.minY))
            logy = True
          if(self.modeY2 == 'linear'):
            drawHeight_2 = np.abs(self.maxY2 - self.minY2)
            logy_2 = False
          else:
            drawHeight_2 = np.abs(np.log(self.maxY2) - np.log(self.minY2))
            logy_2 = True

          if(self.splitY):
            drawHeadWidth_2 = drawHeadWidth * drawHeight_2 * (1 + self.splitYFraction) / self.splitYFraction
            drawHeadWidth *= drawHeight * (1 + self.splitYFraction)
          else:
            drawHeadWidth_2 = drawHeadWidth * drawHeight_2
            drawHeadWidth *= drawHeight
        else:
          drawHeight = np.abs(self.maxResidY - self.minResidY)
          # resid plot is always linear in y; account for difference in window size
          relativeSizes = self.masterwidget.sizes()
          sizeRatio = 1. * relativeSizes[0] / relativeSizes[1]
          drawHeadWidth *= drawHeight * sizeRatio
          logy = False

        # assign line width and xy coordinates
        ###drawLw = self.axisWidth['bottom']
        drawCs = self.axisDashStyle['bottom']
        
        if(useModeX == 'linear'):
          drawX = np.max((useMinX, useMaxX))
        else:
          drawX = np.max((np.log(useMinX), np.log(useMaxX)))

        # check for axis boundaries
        if('bottom' in self.ax2.spines):
          bounds = self.ax2.spines['bottom'].get_bounds()
        elif('top' in self.ax2.spines):
          bounds = self.ax2.spines['top'].get_bounds()
        else:
          bounds = None
        
        if((bounds != None) and len(bounds)):
          boundedMax = np.max(bounds)
          if((boundedMax < drawX) and (boundedMax >= np.min((useMinX, useMaxX)))):
            if(useModeX == 'linear'):
              drawX = boundedMax
            else:
              drawX = np.log(boundedMax)

        # apply offset
        drawX -= drawOffset * drawHeadLength
        
        # determine drawY and drawY2
        if(target == 'plot'):
          if(self.modeY == 'linear'):
            drawY, drawY2 = np.sort((self.minY, self.maxY))
          else:
            drawY, drawY2 = np.sort((np.log(self.minY), np.log(self.maxY)))
          if(self.modeY2 == 'linear'):
            drawY_2, drawY2_2 = np.sort((self.minY2, self.maxY2))
          else:
            drawY_2, drawY2_2 = np.sort((np.log(self.minY2), np.log(self.maxY2)))
        else:
          drawY, drawY2 = np.sort((self.minResidY, self.maxResidY))
        
        # account for shifted axis
        if(target == 'plot'):
          for item in ['bottom', 'top']:
            if(item in self.ax2.spines):
              if(self.splitY and (item == 'bottom')):
                useAxisObject, useAxisObject2, useModeY, useMinY, useMaxY = self.ax, self.ax2, self.modeY2, self.minY2, self.maxY2
              else:
                useAxisObject, useAxisObject2, useModeY, useMinY, useMaxY = self.ax2, self.ax, self.modeY, self.minY, self.maxY
              try:
                axisPosition, axisPositionValue = useAxisObject.spines[item].get_position()
                if(axisPosition == 'data'):
                  itemY = axisPositionValue
                  if(self.modeY == 'log'):
                    # have to test for zero/negative entry
                    if(itemY > 0):
                      itemY = np.log(itemY)
                    else:
                      if(item == 'bottom'):
                        itemY = np.log(np.min((useMinY, useMaxY)))
                      else:
                        itemY = np.log(np.min((useMinY, useMaxY)))
                elif(axisPosition in ['axes', 'outward']):
                  axis_to_data = useAxisObject2.transAxes + useAxisObject2.transData.inverted()
                  transformedPoint = axis_to_data.transform((0, axisPositionValue))
                  itemY = transformedPoint[1]
                  if(useModeY == 'log'):
                    itemY = np.log(itemY)
                if(axisPosition in ['data', 'axes', 'outward']):
                  if(item == 'bottom'):
                    if(self.splitY):
                      drawY_2 = itemY
                    else:
                      drawY = itemY
                  else:
                    drawY2 = itemY
              except:
                pass
        else:
          for item in ['bottom', 'top']:
            if(item in axisobject.spines):
              try:
                axisPosition, axisPositionValue = axisobject.spines[item].get_position()
                if(axisPosition == 'data'):
                  if(item == 'bottom'):
                    drawY = axisPositionValue
                  else:
                    drawY2 = axisPositionValue
                elif(axisPosition == 'axes'):
                  # in resid 
                  axis_to_data = axisobject.transAxes + axisobject.transData.inverted()
                  transformedPoint = axis_to_data.transform((0, axisPositionValue))
                  if(item == 'bottom'):
                    drawY = transformedPoint[1]
                  else:
                    drawY2 = transformedPoint[1]
              except:
                pass
      else:
        # this is the y arrows
        # calculate drawWidth
        if(self.modeX == 'linear'):
          drawHeight = np.abs(self.maxX - self.minX)
          logx = False
        else:
          drawHeight = np.abs(np.log(self.maxX) - np.log(self.minX))
          logx = True
        if(self.modeX_div == 'linear'):
          drawHeight_div = np.abs(self.maxX_div - self.minX_div)
          logx_div = False
        else:
          drawHeight_div = np.abs(np.log(self.maxX_div) - np.log(self.minX_div))
          logx_div = True

        if(self.splitShow):
          drawHeadWidth_div = drawHeadWidth * drawHeight_div * (1 + self.splitFraction)# / self.splitFraction
          drawHeadWidth *= drawHeight * (1 + self.splitFraction) / self.splitFraction
        else:
          drawHeadWidth_div = drawHeadWidth * drawHeight_div
          drawHeadWidth *= drawHeight
  
        # calculate drawHeight
        if(target == 'plot'):
          if(axis == 'y'):
            if(self.modeY == 'linear'):
              drawWidth = np.abs(self.maxY - self.minY)
              logy = False
            else:
              drawWidth = np.abs(np.log(self.maxY) - np.log(self.minY))
              logy = True
          else:
            if(self.modeY2 == 'linear'):
              drawWidth = np.abs(self.maxY2 - self.minY2)
              logy = False
            else:
              drawWidth = np.abs(np.log(self.maxY2) - np.log(self.minY2))
              logy = True

          if(self.splitY):
            drawHeadLength *= drawWidth * (1 + self.splitYFraction)
          else:
            drawHeadLength *= drawWidth
        else:
          drawWidth = np.abs(self.maxResidY - self.minResidY)
          # resid plot is always linear in y; account for difference in window size
          relativeSizes = self.masterwidget.sizes()
          sizeRatio = 1. * relativeSizes[0] / relativeSizes[1]
          drawHeadLength *= drawWidth * sizeRatio
          logy = False

        # assign line width and xy coordinates
        ###drawLw = self.axisWidth['left']
        drawCs = self.axisDashStyle['left']

        if(target == 'resid'):
          useMinY, drawY = np.sort((self.minResidY, self.maxResidY))
        elif(axis == 'y'):
          if (self.modeY == 'linear'):
            useMinY, drawY = np.sort((self.minY, self.maxY))
          else:
            useMinY, drawY = np.sort((np.log(self.minY), np.log(self.maxY)))
        else:
          if (self.modeY2 == 'linear'):
            useMinY, drawY = np.sort((self.minY2, self.maxY2))
          else:
            useMinY, drawY = np.sort((np.log(self.minY2), np.log(self.maxY2)))

        # check for axis boundaries
        bounds = None
        if('left' in axisobject.spines):
          bounds = axisobject.spines['left'].get_bounds()
        if((bounds == None) or (axis == 'y2')):
          # make arrow on second axes preferentially use the right axis
          if('right' in axisobject.spines):
            bounds = axisobject.spines['right'].get_bounds()
        
        if((bounds != None) and len(bounds)):
          boundedMax = np.max(bounds)
          if((boundedMax < drawY) and (boundedMax >= useMinY)):
            if((target == 'resid') or (self.modeY == 'linear')):
              drawY = boundedMax
            else:
              drawY = np.log(boundedMax)

        # apply offset
        drawY -= drawOffset * drawHeadLength
        
        # determine drawX
        if(self.modeX == 'linear'):
          drawX, drawX2 = np.sort((self.minX, self.maxX))
        else:
          drawX, drawX2 = np.sort((np.log(self.minX), np.log(self.maxX)))
        if(self.modeX_div == 'linear'):
          drawX_div, drawX2_div = np.sort((self.minX_div, self.maxX_div))
        else:
          drawX_div, drawX2_div = np.sort((np.log(self.minX_div), np.log(self.maxX_div)))

        # account for shifted axis
        for item in ['left', 'right']:
          if(item in axisobject.spines):
            if(self.splitShow and (item == 'right')):
              useAxisObject, useModeX, useMinX, useMaxX = axisobject2, self.modeX_div, self.minX_div, self.maxX_div
            else:
              useAxisObject, useModeX, useMinX, useMaxX = axisobject, self.modeX, self.minX, self.maxX
            try:
              axisPosition, axisPositionValue = useAxisObject.spines[item].get_position()
              if(axisPosition == 'data'):
                itemX = axisPositionValue
                if(useModeX == 'log'):
                  # have to test for zero/negative entry
                  if(itemX > 0):
                    itemX = np.log(itemX)
                  else:
                    if(item == 'left'):
                      itemX = np.log(np.min((useMinX, useMaxX)))
                    else:
                      itemX = np.log(np.max((useMinX, useMaxX)))
              elif(axisPosition == 'axes'):
                axis_to_data = useAxisObject.transAxes + useAxisObject.transData.inverted()
                transformedPoint = axis_to_data.transform((axisPositionValue, self.minY))
                itemX = transformedPoint[0]
                if(useModeX == 'log'):
                  itemX = np.log(itemX)
                  
              if(item == 'left'):
                drawX = itemX
              else:
                if(self.splitShow):
                  drawX2_div = itemX
                else:
                  drawX2 = itemX
            except:
              pass

      # remove previous arrows
      if(arrowhandle[axis] != None):
        arrowhandle[axis].remove()
        arrowhandle[axis] = None
      if(arrowhandle2[axis] != None):
        arrowhandle2[axis].remove()
        arrowhandle2[axis] = None

      # draw customized arrow
      if((axis == 'x') and (self.arrowLocation['x'] in ['both', 'bottom'])):
        if(self.splitY and (target == 'plot')):
          if(drawOnDiv):
            thisAxisObject = self.ax2_div
          else:
            thisAxisObject = self.ax2
          arrowhandle[axis] = self.drawMyArrow(axisobject=thisAxisObject, x=drawX, y=drawY_2, axis=axis,\
            head_width=drawHeadWidth_2, head_length=drawHeadLength, overhang=drawOverhang, linewidth=drawLw,\
            linecolor=drawCol, fillcolor=drawFill, capstyle=drawCs, zorder=drawZ, logx=logx, logy=logy_2)
        else:
          arrowhandle[axis] = self.drawMyArrow(axisobject=axisobject, x=drawX, y=drawY, axis=axis,\
            head_width=drawHeadWidth, head_length=drawHeadLength, overhang=drawOverhang, linewidth=drawLw,\
            linecolor=drawCol, fillcolor=drawFill, capstyle=drawCs, zorder=drawZ, logx=logx, logy=logy)
        arrowhandle[axis].set_clip_on(False)
      if((axis == 'x') and (self.arrowLocation['x'] in ['both', 'top'])):
        arrowhandle2[axis] = self.drawMyArrow(axisobject=axisobject, x=drawX, y=drawY2, axis=axis,\
          head_width=drawHeadWidth, head_length=drawHeadLength, overhang=drawOverhang, linewidth=drawLw,\
          linecolor=drawCol, fillcolor=drawFill, capstyle=drawCs, zorder=drawZ, logx=logx, logy=logy)
        arrowhandle2[axis].set_clip_on(False)

      if(axis in ['y', 'y2']):
        if(self.arrowLocation[axis] in ['both', 'left']):
          arrowhandle[axis] = self.drawMyArrow(axisobject=axisobject, x=drawX, y=drawY, axis=axis,\
            head_width=drawHeadWidth, head_length=drawHeadLength, overhang=drawOverhang, linewidth=drawLw,\
            linecolor=drawCol, fillcolor=drawFill, capstyle=drawCs, zorder=drawZ, logx=logx, logy=logy)
          arrowhandle[axis].set_clip_on(False)
        if(self.arrowLocation[axis] in ['both', 'right']):
          if(self.splitShow):
            arrowhandle2[axis] = self.drawMyArrow(axisobject=axisobject2, x=drawX2_div, y=drawY, axis=axis,\
              head_width=drawHeadWidth_div, head_length=drawHeadLength, overhang=drawOverhang, linewidth=drawLw,\
              linecolor=drawCol, fillcolor=drawFill, capstyle=drawCs, zorder=drawZ, logx=logx_div, logy=logy)
          else:
            arrowhandle2[axis] = self.drawMyArrow(axisobject=axisobject, x=drawX2, y=drawY, axis=axis,\
              head_width=drawHeadWidth, head_length=drawHeadLength, overhang=drawOverhang, linewidth=drawLw,\
              linecolor=drawCol, fillcolor=drawFill, capstyle=drawCs, zorder=drawZ, logx=logx, logy=logy)
          arrowhandle2[axis].set_clip_on(False)

      if(redraw):
        plotobject.myRefresh()

  def drawMyArrow(self, axisobject, x, y, axis, head_width, head_length, overhang, linewidth, linecolor, fillcolor, capstyle, zorder, logx=False, logy=False):
    # draw customized arrow
    # calculate coordinates
    coords = np.array(4 * [[x, y]])
    if(axis=='x'):
      coords[0, 0] += head_length * overhang
      coords[2, 0] += head_length
      coords[1, 1] += head_width / 2.0
      coords[3, 1] -= head_width / 2.0
    else:
      coords[1, 0] += head_width / 2.0
      coords[3, 0] -= head_width / 2.0
      coords[0, 1] += head_length * overhang
      coords[2, 1] += head_length

    # transform to log scale?
    if(logx):
      coords[:, 0] = np.exp(coords[:,0])
    if(logy):
      coords[:, 1] = np.exp(coords[:,1])

    if(axisobject is self.ax):
      axisname = 'ax'
    else:
      axisname = 'ax_resid'
      
    if(overhang < 1):
      polyPatch = matplotlib.patches.Polygon(coords, closed=True, facecolor=fillcolor, fill=True,\
        edgecolor=linecolor, linestyle='solid', linewidth=linewidth, zorder=zorder, capstyle=capstyle)
      retv = axisobject.add_patch(polyPatch)
    else:
      polyPatch = matplotlib.patches.Polygon(coords[1:,:], closed=False, fill=False,\
        edgecolor=linecolor, linestyle='solid', linewidth=linewidth, zorder=zorder, capstyle=capstyle)
      retv = axisobject.add_patch(polyPatch)

    return retv

  def setCanvasGradientStyle(self, value=0.0, redraw=True, target='plot'):
    # sets angle of canvas gradient
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        if(self.canvasGradientStyle == value):
          redraw = False
        else:
          self.canvasGradientStyle = value
      else:
        if(self.canvasGradientStyle_resid == value):
          redraw = False
        else:
          self.canvasGradientStyle_resid = value
          
      # issue redraw if needed
      if(redraw and self.canvasGradientVisible):
        self.setCanvasGradient(state=True, redraw=redraw, target=target)

  def setCanvasGradientColor(self, value=0.0, redraw=True, target='plot', color=0):
    # sets angle of canvas gradient
    if((target in ['plot', 'resid']) and (color in [0, 1])):
      if(target == 'plot'):
        if(color):
          if(self.canvasGradientColor2 == value):
            redraw = False
          else:
            self.canvasGradientColor2 = value
        else:
          if(self.canvasGradientColor1 == value):
            redraw = False
          else:
            self.canvasGradientColor1 = value
      else:
        if(color):
          if(self.canvasGradientColor2_resid == value):
            redraw = False
          else:
            self.canvasGradientColor2_resid = value
        else:
          if(self.canvasGradientColor1_resid == value):
            redraw = False
          else:
            self.canvasGradientColor1_resid = value
          
      # issue redraw if needed
      if(redraw and self.canvasGradientVisible):
        self.setCanvasGradient(state=True, redraw=redraw, target=target)

  def setCanvasGradientCenter(self, value=0.0, redraw=True, target='plot', axis='x'):
    # sets angle of canvas gradient
    if((target in ['plot', 'resid']) and (axis in ['x', 'y'])):
      if(axis == 'x'):
        index = 0
      else:
        index = 1
      if(target == 'plot'):
        if(self.canvasGradientCenter[index] == value):
          redraw = False
        else:
          self.canvasGradientCenter[index] = value
      else:
        if(self.canvasGradientCenter_resid[index] == value):
          redraw = False
        else:
          self.canvasGradientCenter_resid[index] = value
          
      # issue redraw if needed
      if(redraw and self.canvasGradientVisible and (self.canvasGradientStyle == 'radial')):
        self.setCanvasGradient(state=True, redraw=redraw, target=target)

  def setCanvasGradientWidth(self, value=0.0, redraw=True, target='plot'):
    # sets angle of canvas gradient
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        if(self.canvasGradientWidth == value):
          redraw = False
        else:
          self.canvasGradientWidth = value
      else:
        if(self.canvasGradientWidth_resid == value):
          redraw = False
        else:
          self.canvasGradientWidth_resid = value
          
      # issue redraw if needed
      if(redraw and self.canvasGradientVisible and (self.canvasGradientStyle == 'radial')):
        self.setCanvasGradient(state=True, redraw=redraw, target=target)

  def setCanvasGradientAngle(self, value=0.0, redraw=True, target='plot'):
    # sets angle of canvas gradient
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        if(self.canvasGradientAngle == value):
          redraw = False
        else:
          self.canvasGradientAngle = value
      else:
        if(self.canvasGradientAngle_resid == value):
          redraw = False
        else:
          self.canvasGradientAngle_resid = value
          
      # issue redraw if needed
      if(redraw and self.canvasGradientVisible and (self.canvasGradientStyle == 'linear')):
        self.setCanvasGradient(state=True, redraw=redraw, target=target)
  
  def setCanvasGradient(self, state=True, redraw=True, target='plot', steps=50):
    # draws a color gradient on axis background
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, planeobject, handleContour = self.dataplotwidget, self.colorPlane, self.colorBackground
        axisobject = self.ax2; axisobject2 = self.ax2_div
      else:
        plotobject, planeobject, handleContour = self.residplotwidget, self.colorPlane_resid, self.colorBackground_resid
        axisobject, axisobject2 = self.ax_resid, self.ax_resid_div
      
      # remove previous gradient (if present)
      if(handleContour != None):
        # unexpectedly collections became deprecated, so have to escape this
        try:
          for entry in handleContour.collections:
            entry.remove()
        except:
          # this should take care of that
          handleContour.remove()
        handleContour = None
  
      self.canvasGradientVisible = state
      if(state):
        # define new gradient
        gradientDict = {}
        for index, component in enumerate(['red', 'green', 'blue', 'alpha']):
          gradientDictTemp = {component: ((0.0, self.canvasGradientColor1[index], self.canvasGradientColor1[index]),
                                          (1.0, self.canvasGradientColor2[index], self.canvasGradientColor2[index]))}
          gradientDict.update(gradientDictTemp)
        gradientMap = matplotlib.colors.LinearSegmentedColormap('MyMap', gradientDict)
        x, y = np.linspace(0, 1, steps), np.linspace(0, 1, steps)
        X, Y = np.meshgrid(x, y)
        if(self.canvasGradientStyle == 'linear'):
          angle = self.canvasGradientAngle / 360.0 * 2 * np.pi
          Z = X * np.cos(angle) + Y * np.sin(angle)
        else:
          width = self.canvasGradientWidth
          Z = np.exp(-(((X - self.canvasGradientCenter[0]) / width) ** 2) - (((Y - self.canvasGradientCenter[1]) / width) ** 2))
        handleContour = planeobject.contourf(X, Y, Z, 128, cmap=gradientMap, vmax=1.0, vmin=0.0)
        # hide canvas background of top plots
        axisobject.patch.set_facecolor([1.0] * 3 + [0.0])
        axisobject2.patch.set_facecolor([1.0] * 3 + [0.0])
        if(self.splitY and (target == 'plot')):
          self.ax.patch.set_facecolor([1.0] * 3 + [0.0])
          self.ax_div.patch.set_facecolor([1.0] * 3 + [0.0])
      else:
        # restore canvas background of top plots
        if(self.canvasFill):
          axisobject.patch.set_facecolor(self.canvasColor)
          axisobject2.patch.set_facecolor(self.canvasColor)
          if(self.splitY and (target == 'plot')):
            self.ax.patch.set_facecolor(self.canvasColor)
            self.ax_div.patch.set_facecolor(self.canvasColor)
        else:
          axisobject.patch.set_facecolor('none')
          axisobject2.patch.set_facecolor('none')
          if(self.splitY and (target == 'plot')):
            self.ax.patch.set_facecolor('none')
            self.ax_div.patch.set_facecolor('none')
  
      if(redraw):
        plotobject.myRefresh()
  
      # assign handles
      if(target == 'plot'):
        self.colorBackground = handleContour
      else:
        self.colorBackground_resid = handleContour

  def setFigureColor(self, value=[0, 0, 0, 1], redraw=True, target='plot', silent=True):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, matobject, objectname = self.dataplotwidget, self.matplot, 'matplot'
      else:
        plotobject, matobject, objectname = self.residplotwidget, self.residplot, 'residplot'
      # sets figure background color
      if((self.figureColor == value) and (target == 'plot')):
        redraw = False
        
      self.figureColor = value
      # always change figure color on screen as we cannot assume which label and axis color etc. people are using
      if((not self.figureFill) and (not silent)):
        self.parent.statusbar.showMessage('Note that figure background of exported graphics is currently set to transparent.', self.parent.STATUS_TIME, color='blue')
      matobject.set_facecolor(self.figureColor)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def toggleFigureFill(self, value=True, redraw=True, target='plot', silent=True):
    # toggles whether figure background is filled or not
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject = self.dataplotwidget
      else:
        plotobject = self.residplotwidget

      # disable redraw and never change figure color on screen as we cannot assume which label and axis color etc. people are using
      redraw = False
      if((not value) and (not silent)):
        self.parent.statusbar.showMessage('Figure background of exported graphics set to transparent.', self.parent.STATUS_TIME, color='blue')
  
      self.figureFill = value
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def toggleFrameDraw(self, value=True, redraw=True, target='plot'):
    # toggles whether figure frame is drawn or not
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, figobject = self.dataplotwidget, self.matplot
        if(self.frameDraw == value):
          redraw = False
        self.frameDraw = value
      else:
        plotobject, figobject = self.residplotwidget, self.residplot
        if(self.frameDraw_resid == value):
          redraw = False
        self.frameDraw_resid = value

      # adjust value
      if(value):
        if(target == 'plot'):
          value = self.frameColor
        else:
          value = self.frameColor_resid
      else:
        value = 'none'
      # sets frame color
      figobject.patch.set_edgecolor(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def toggleCanvasFill(self, value=True, redraw=True, target='plot'):
    # toggles whether canvas is filled or not
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject, axisobject2 = self.dataplotwidget, self.ax2, self.ax2_div
      else:
        plotobject, axisobject, axisobject2 = self.residplotwidget, self.ax_resid, self.ax_resid_div
      # sets canvas color
      if((self.canvasFill == value) and (target == 'plot')):
        redraw = False

      self.canvasFill = value
      # adjust value
      if(value):
        value = self.canvasColor
      else:
        value = 'none'
      if(not self.canvasGradientVisible):
        axisobject.patch.set_facecolor(value)
        axisobject2.patch.set_facecolor(value)
        if(self.splitY and (target == 'plot')):
          self.ax.patch.set_facecolor(value)
          self.ax_div.patch.set_facecolor(value)

      # redraw?
      if(redraw):
        plotobject.myRefresh()
        
  def setCanvasColor(self, value=[0, 0, 0, 1], redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
      else:
        plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
      # sets canvas color
      if((self.canvasColor == value) and (target == 'plot')):
        redraw = False

      self.canvasColor = value
      if(self.canvasFill):
        if(not self.canvasGradientVisible):
          axisobject.patch.set_facecolor(self.canvasColor)
          axisobject2.patch.set_facecolor(self.canvasColor)
          if(self.splitY and (target == 'plot')):
            self.ax.patch.set_facecolor(self.canvasColor)
            self.ax_div.patch.set_facecolor(self.canvasColor)
      else:
        redraw = False
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setFrameWidth(self, value=0.5, redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, figobject, axisname, plotname = self.dataplotwidget, self.matplot, 'ax', 'matplot'
        if(self.frameWidth == value):
          redraw = False
        self.frameWidth = value
      else:
        plotobject, figobject, axisname, plotname = self.residplotwidget, self.residplot, 'ax_resid', 'residplot'
        if(self.frameWidth_resid == value):
          redraw = False
        self.frameWidth_resid = value

      # check frame style to avoid crash at zero width
      if(value < .01):
        if(target == 'plot'):
          if(self.frameStyle != 'solid'):
            self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
            self.frameStyle = 'solid'
            figobject.patch.set_linestyle('solid')
        else:
          if(self.frameStyle_resid != 'solid'):
            self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
            self.frameStyle_resid = 'solid'
            figobject.patch.set_linestyle('solid')

      # sets frame width
      figobject.patch.set_linewidth(value)
      
      # check whether frame is visible at all
      if(not self.frameDraw):
        redraw = False
      
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setFrameStyle(self, value='solid', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, figobject, axisname, plotname = self.dataplotwidget, self.matplot, 'ax', 'matplot'
        if(self.frameStyle == value):
          redraw = False
        self.frameStyle = value
      else:
        plotobject, figobject, axisname, plotname = self.residplotwidget, self.residplot, 'ax_resid', 'residplot'
        if(self.frameStyle_resid == value):
          redraw = False
        self.frameStyle_resid = value

      # check frame style to avoid crash at zero width
      if(value != 'solid'):
        if(target == 'plot'):
          if(self.frameWidth < .01):
            self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
            value = 'solid'
            self.frameStyle = 'solid'
        else:
          if(self.frameWidth_resid < .01):
            self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
            value = 'solid'
            self.frameStyle_resid = 'solid'

      # sets frame line style
      figobject.patch.set_linestyle(value)

      # check whether frame is visible at all
      if(not self.frameDraw):
        redraw = False
      
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setFrameDashStyle(self, value='solid', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, figobject, axisname, plotname = self.dataplotwidget, self.matplot, 'ax', 'matplot'
        if(self.frameDashStyle == value):
          redraw = False
        self.frameDashStyle = value
      else:
        plotobject, figobject, axisname, plotname = self.residplotwidget, self.residplot, 'ax_resid', 'residplot'
        if(self.frameDashStyle_resid == value):
          redraw = False
        self.frameDashStyle_resid = value

      # sets grid line style
      figobject.patch.set_capstyle(value)

      # check whether frame is visible at all
      if(not self.frameDraw):
        redraw = False
      
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setFrameColor(self, value=[0, 0, 0, 1], redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, figobject, axisname, plotname = self.dataplotwidget, self.matplot, 'ax', 'matplot'
        if(self.frameColor == value):
          redraw = False
        self.frameColor = value
      else:
        plotobject, figobject, axisname, plotname = self.residplotwidget, self.residplot, 'ax_resid', 'residplot'
        if(self.frameColor_resid == value):
          redraw = False
        self.frameColor_resid = value

      # sets grid color
      if(self.frameDraw):
        figobject.patch.set_edgecolor(value)
      else:
        redraw = False

      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setGridOrder(self, value='back', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridOrder[axis] == value):
            redraw = False
          self.gridOrder[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridOrder_resid[axis] == value):
            redraw = False
          self.gridOrder_resid[axis] = value
          
        if(value == 'back'):
          useZ = 1
        else:
          useZ = 500
        # so here is the deal: set_axisbelow() toggles axis zorder between 0.5 and 2.5
        # however, due to z offset, all our objects are at higher z values than 2.5
        # hence draw custom lines
        if(target == 'plot'):
          for line in self.gridLinesStore[axis]:
            line.set_zorder(useZ)
        else:
          for line in self.gridLinesStore_resid[axis]:
            line.set_zorder(useZ)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridStyle(self, value='solid', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridStyle[axis] == value):
            redraw = False
          self.gridStyle[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridStyle_resid[axis] == value):
            redraw = False
          self.gridStyle_resid[axis] = value

        # check grid style to avoid crash at zero width
        if(value != 'solid'):
          if(self.gridWidth[axis] < .01):
            self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
            value = 'solid'
            if(target == 'plot'):
              self.gridStyle[axis] = 'solid'
            else:
              self.gridStyle_resid[axis] = 'solid'

        # sets grid line style
        if(target == 'plot'):
          for line in self.gridLinesStore[axis]:
            line.set_linestyle(value)
        else:
          for line in self.gridLinesStore_resid[axis]:
            line.set_linestyle(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridDashStyle(self, value='butt', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridDashStyle[axis] == value):
            redraw = False
          self.gridDashStyle[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridDashStyle_resid[axis] == value):
            redraw = False
          self.gridDashStyle_resid[axis] = value

        # sets grid line dash style
        if(target == 'plot'):
          for line in self.gridLinesStore[axis]:
            line.set_dash_capstyle(value)
            line.set_solid_capstyle(value)
        else:
          for line in self.gridLinesStore_resid[axis]:
            line.set_dash_capstyle(value)
            line.set_solid_capstyle(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridWidth(self, value=0.5, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridWidth[axis] == value):
            redraw = False
          self.gridWidth[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridWidth_resid[axis] == value):
            redraw = False
          self.gridWidth_resid[axis] = value

        # check grid style to avoid crash at zero width
        if(value < .01):
          if(target == 'plot'):
            if(self.gridStyle[axis] != 'solid'):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              self.gridStyle[axis] = 'solid'
              for line in self.gridLinesStore[axis]:
                line.set_linestyle('solid')
          else:
            if(self.gridStyle_resid[axis] != 'solid'):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              self.gridStyle_resid[axis] = 'solid'
              for line in self.gridLinesStore_resid[axis]:
                line.set_linestyle('solid')

        # sets grid line width
        if(target == 'plot'):
          for line in self.gridLinesStore[axis]:
            line.set_linewidth(value)
        else:
          for line in self.gridLinesStore_resid[axis]:
            line.set_linewidth(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridColor(self, value=[0, 0, 0, 1], axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridColor[axis] == value):
            redraw = False
          self.gridColor[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridColor_resid[axis] == value):
            redraw = False
          self.gridColor_resid[axis] = value

        # sets grid line color
        if(target == 'plot'):
          for line in self.gridLinesStore[axis]:
            line.set_color(value)
        else:
          for line in self.gridLinesStore_resid[axis]:
            line.set_color(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridRectColor(self, value=[0, 0, 0, 1], axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridRectColor[axis] == value):
            redraw = False
          self.gridRectColor[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridRectColor_resid[axis] == value):
            redraw = False
          self.gridRectColor_resid[axis] = value

        # sets grid line color
        if(target == 'plot'):
          for rect in self.gridRectStore[axis]:
            rect.set_facecolor(value)
        else:
          for rect in self.gridRectStore_resid[axis]:
            rect.set_facecolor(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridRectVisibility(self, value=True, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridRectVisible[axis] == value):
            redraw = False
          self.gridRectVisible[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridRectVisible_resid[axis] == value):
            redraw = False
          self.gridRectVisible_resid[axis] = value

        # sets visibility of grid lines
        self.drawAxisGrid(axis=axis, redraw=False, target=target)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridVisibility(self, value=True, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridVisible[axis] == value):
            redraw = False
          self.gridVisible[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridVisible_resid[axis] == value):
            redraw = False
          self.gridVisible_resid[axis] = value

        # sets visibility of grid lines
        self.drawAxisGrid(axis=axis, redraw=False, target=target)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridMinorOrder(self, value='back', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridMinorOrder[axis] == value):
            redraw = False
          self.gridMinorOrder[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridMinorOrder_resid[axis] == value):
            redraw = False
          self.gridMinorOrder_resid[axis] = value
          
        if(value == 'back'):
          useZ = 1
        else:
          useZ = 500
        # so here is the deal: set_axisbelow() toggles axis zorder between 0.5 and 2.5
        # however, due to z offset, all our objects are at higher z values than 2.5
        # hence draw custom lines
        if(target == 'plot'):
          for line in self.gridMinorLinesStore[axis]:
            line.set_zorder(useZ)
        else:
          for line in self.gridMinorLinesStore_resid[axis]:
            line.set_zorder(useZ)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridMinorStyle(self, value='solid', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridMinorStyle[axis] == value):
            redraw = False
          self.gridMinorStyle[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridMinorStyle_resid[axis] == value):
            redraw = False
          self.gridMinorStyle_resid[axis] = value

        # check grid style to avoid crash at zero width
        if(value != 'solid'):
          if(self.gridMinorWidth[axis] < .01):
            self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
            value = 'solid'
            if(target == 'plot'):
              self.gridMinorStyle[axis] = 'solid'
            else:
              self.gridMinorStyle_resid[axis] = 'solid'

        # sets grid line style
        if(target == 'plot'):
          for line in self.gridMinorLinesStore[axis]:
            line.set_linestyle(value)
        else:
          for line in self.gridMinorLinesStore_resid[axis]:
            line.set_linestyle(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridMinorDashStyle(self, value='butt', axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridMinorDashStyle[axis] == value):
            redraw = False
          self.gridMinorDashStyle[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridMinorDashStyle_resid[axis] == value):
            redraw = False
          self.gridMinorDashStyle_resid[axis] = value

        # sets grid line dash style
        if(target == 'plot'):
          for line in self.gridMinorLinesStore[axis]:
            line.set_dash_capstyle(value)
            line.set_solid_capstyle(value)
        else:
          for line in self.gridMinorLinesStore_resid[axis]:
            line.set_dash_capstyle(value)
            line.set_solid_capstyle(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridMinorWidth(self, value=0.5, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridMinorWidth[axis] == value):
            redraw = False
          self.gridMinorWidth[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridMinorWidth_resid[axis] == value):
            redraw = False
          self.gridMinorWidth_resid[axis] = value

        # check grid style to avoid crash at zero width
        if(value < .01):
          if(target == 'plot'):
            if(self.gridMinorStyle[axis] != 'solid'):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              self.gridMinorStyle[axis] = 'solid'
              for line in self.gridMinorLinesStore[axis]:
                line.set_linestyle('solid')
          else:
            if(self.gridMinorStyle_resid[axis] != 'solid'):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              self.gridMinorStyle_resid[axis] = 'solid'
              for line in self.gridMinorLinesStore_resid[axis]:
                line.set_linestyle('solid')

        # sets grid line width
        if(target == 'plot'):
          for line in self.gridMinorLinesStore[axis]:
            line.set_linewidth(value)
        else:
          for line in self.gridMinorLinesStore_resid[axis]:
            line.set_linewidth(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridMinorColor(self, value=[0, 0, 0, 1], axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridMinorColor[axis] == value):
            redraw = False
          self.gridMinorColor[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridMinorColor_resid[axis] == value):
            redraw = False
          self.gridMinorColor_resid[axis] = value

        # sets grid line color
        if(target == 'plot'):
          for line in self.gridMinorLinesStore[axis]:
            line.set_color(value)
        else:
          for line in self.gridMinorLinesStore_resid[axis]:
            line.set_color(value)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setGridMinorVisibility(self, value=True, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'x2', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.gridMinorVisible[axis] == value):
            redraw = False
          self.gridMinorVisible[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.gridMinorVisible_resid[axis] == value):
            redraw = False
          self.gridMinorVisible_resid[axis] = value

        # sets visibility of grid lines
        self.drawMinorAxisGrid(axis=axis, redraw=False, target=target)

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setPadding(self, value=0.5, axis='bottom', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['bottom', 'top', 'left', 'right']):
        if(target == 'plot'):
          plotobject, matobject, objectname = self.dataplotwidget, self.matplot, 'matplot'
          if(self.padSize[axis] == value):
            redraw = False

          # implement check for too large padding in case of split plots
          if((axis in ['left', 'right']) and self.splitShow):
            flag = False
            if(axis == 'left'):
              maxPad = .45 - self.splitPad / 2.
              if(value > maxPad):
                flag = True
            else:
              maxPad = .55 + self.splitPad / 2.
              if(value < maxPad):
                flag = True
            if(flag):
              if(hasattr(self.parent, 'statusbar')):
                self.parent.statusbar.showMessage('Cannot set ' + axis + ' padding to ' + self.parent.formatNumber(value) + ', using ' + self.parent.formatNumber(maxPad) + ' instead!', self.parent.STATUS_TIME)
              value, redraw = maxPad, True
          elif((axis in ['bottom', 'top']) and self.splitY):
            flag = False
            if(axis == 'bottom'):
              maxPad = .45 - self.splitYPad / 2.
              if(value > maxPad):
                flag = True
            else:
              maxPad = .55 + self.splitYPad / 2.
              if(value < maxPad):
                flag = True
            if(flag):
              if(hasattr(self.parent, 'statusbar')):
                self.parent.statusbar.showMessage('Cannot set ' + axis + ' padding to ' + self.parent.formatNumber(value) + ', using ' + self.parent.formatNumber(maxPad) + ' instead!', self.parent.STATUS_TIME)
              value, redraw = maxPad, True

          self.padSize[axis] = value
          
          # set padding
          if(self.splitY):
            ybreak = (self.padSize['top'] - self.padSize['bottom'] - self.splitYPad) * self.splitYFraction / (1.0 + self.splitYFraction)
            if(self.splitShow):
              xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
              self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['left'] + xbreak,  self.padSize['top']]]))
              self.ax_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['right'],  self.padSize['top']]]))
              self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['bottom'] + ybreak]]))
              self.ax2_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['bottom'] + ybreak]]))
            else:
              self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom'] + ybreak + self.splitYPad], [self.padSize['right'], self.padSize['top']]]))
              self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['bottom'] + ybreak]]))
          else:
            if(self.splitShow):
              xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
              self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak,  self.padSize['top']]]))
              self.ax_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'],  self.padSize['top']]]))
              self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['top']]]))
              self.ax2_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
            else:
              self.ax.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
              self.ax2.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
              
          # also need to adjust color plane
          self.colorPlane.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
        else:
          plotobject, matobject, objectname = self.residplotwidget, self.residplot, 'residplot'
          if(matobject.subplotpars.__dict__[axis] == value):
            redraw = False

          # implement check for too large padding in case of split plots
          if((axis in ['left', 'right']) and self.splitShow):
            flag = False
            if(axis == 'left'):
              maxPad = .45 - self.splitPad / 2.
              if(value > maxPad):
                flag = True
            else:
              maxPad = .55 + self.splitPad / 2.
              if(value < maxPad):
                flag = True
            if(flag):
              if(hasattr(self.parent, 'statusbar')):
                self.parent.statusbar.showMessage('Cannot set ' + axis + ' padding to ' + self.parent.formatNumber(value) + ', using ' + self.parent.formatNumber(maxPad) + ' instead!', self.parent.STATUS_TIME)
              value, redraw = maxPad, True

          self.padSize_resid[axis] = value

          # set padding
          if(self.splitShow):
            xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
            self.ax_resid.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['left'] + xbreak, self.padSize['top']]]))
            self.ax_resid_div.set_position(matplotlib.transforms.Bbox([[self.padSize['left'] + xbreak + self.splitPad, self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
          else:
            self.ax_resid.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))

          # also need to adjust color plane
          self.colorPlane_resid.set_position(matplotlib.transforms.Bbox([[self.padSize['left'], self.padSize['bottom']], [self.padSize['right'], self.padSize['top']]]))
        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setTickOne4All(self, axis='x', redraw=True, target='plot'):
    # calls all tick label formatters successively to fix glitch when no. of tick labels change upon rescaling
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject,  axisobject, axisobject2 = self.dataplotwidget, self.ax2, self.ax2_div
        else:
          plotobject,  axisobject, axisobject2 = self.residplotwidget, self.ax_resid, self.ax_resid_div
        # which axis?  
        if(axis == 'x'):
          tickLabelColor, tickLabelSize, tickLabelWeight, tickLabelStyle, tickLabelFont = self.ticksXColor, self.ticksXSize, self.ticksXWeight, self.ticksXStyle, self.tickFont[axis]
          tickLabelAngle, tickLabelAlignment, tickLabelAlignmentVertical, tickLabelPad, tickLabelPad2 = self.ticksXAngle, self.ticksXAlignment, self.ticksXAlignmentVertical, self.ticksXPad, self.ticksXPad2
        elif(axis == 'y'):
          if(target == 'plot'):
            axisobject, axisobject2 = self.ax, self.ax_div
          tickLabelColor, tickLabelSize, tickLabelWeight, tickLabelStyle, tickLabelFont = self.ticksYColor, self.ticksYSize, self.ticksYWeight, self.ticksYStyle, self.tickFont[axis]
          tickLabelAngle, tickLabelAlignment, tickLabelAlignmentVertical, tickLabelPad, tickLabelPad2 = self.ticksYAngle, self.ticksYAlignment, self.ticksYAlignmentVertical, self.ticksYPad, self.ticksYPad2
        else:
          tickLabelColor, tickLabelSize, tickLabelWeight, tickLabelStyle, tickLabelFont = self.ticksY2Color, self.ticksY2Size, self.ticksY2Weight, self.ticksY2Style, self.tickFont[axis]
          tickLabelAngle, tickLabelAlignment, tickLabelAlignmentVertical, tickLabelPad, tickLabelPad2 = self.ticksY2Angle, self.ticksY2Alignment, self.ticksY2AlignmentVertical, self.ticksY2Pad, self.ticksY2Pad2
        
        if(axis == 'x'):
          tickLabels = axisobject.get_xticklabels(which='both')
          tickLabels.append(axisobject.xaxis.get_offset_text())
          tickLabels.extend(axisobject2.get_xticklabels(which='both'))
          tickLabels.append(axisobject2.xaxis.get_offset_text())
          ticks = axisobject.xaxis.get_major_ticks()
          ticks.extend(axisobject.xaxis.get_minor_ticks())
          ticks.extend(axisobject2.xaxis.get_major_ticks())
          ticks.extend(axisobject2.xaxis.get_minor_ticks())
          if(self.splitY and (target == 'plot')):
            tickLabels.extend(self.ax.get_xticklabels(which='both'))
            tickLabels.append(self.ax.xaxis.get_offset_text())
            ticks.extend(self.ax.xaxis.get_major_ticks())
            ticks.extend(self.ax.xaxis.get_minor_ticks())
            tickLabels.extend(self.ax_div.get_xticklabels(which='both'))
            tickLabels.append(self.ax_div.xaxis.get_offset_text())
            ticks.extend(self.ax_div.xaxis.get_major_ticks())
            ticks.extend(self.ax_div.xaxis.get_minor_ticks())
        else:
          tickLabels = axisobject.get_yticklabels(which='both')
          tickLabels.append(axisobject.yaxis.get_offset_text())
          tickLabels.extend(axisobject2.get_yticklabels(which='both'))
          tickLabels.append(axisobject2.yaxis.get_offset_text())
          ticks = axisobject.yaxis.get_major_ticks()
          ticks.extend(axisobject.yaxis.get_minor_ticks())
          ticks.extend(axisobject2.yaxis.get_major_ticks())
          ticks.extend(axisobject2.yaxis.get_minor_ticks())

        if((target == 'plot') or (axis != 'y2')):
          # don't call formatter functions as this is too slow
          ###safeFont = 'DejaVu Sans'
          useFont = tickLabelFont
          for entry in tickLabels:
            entry.set_color(tickLabelColor)
            entry.set_fontsize(tickLabelSize)
            entry.set_fontweight(tickLabelWeight)
            entry.set_fontstyle(tickLabelStyle)
            try:
              # we should be in the clear but there is a certain danger here
              entry.set_fontname(useFont)
              _, _, descent = entry._get_layout(self.matplot.canvas.renderer)
              if(not (descent > 0)):
                useFont = SAFE_FONT
                entry.set_fontname(useFont)
            except:
              useFont = SAFE_FONT
              entry.set_fontname(useFont)
            entry.set_rotation(tickLabelAngle)
            entry.set_horizontalalignment(tickLabelAlignment)
            if(tickLabelAlignmentVertical != 'center'):
              entry.set_verticalalignment(tickLabelAlignmentVertical)
            else:
              try:
                entry.set_verticalalignment('center_baseline')
              except:
                entry.set_verticalalignment('center')

          for entry in ticks:
            entry.set_pad(tickLabelPad)
         
          # call formatter for pad2 as this is more complicated
          self.setTickLabelPad2(value=tickLabelPad2, axis=axis, redraw=False, target=target)
          
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setTickLabelColor(self, value=[0, 0, 0, 1], axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          # sets axis label color
          if(axis == 'x'):
            if((self.ticksXColor == value) or (not self.ticksXShow)):
              redraw = False
            self.ticksXColor = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
            if(self.splitY):
              tickLabels.extend(self.ax.get_xticklabels(which='both'))
              tickLabels.append(self.ax.xaxis.get_offset_text())
              tickLabels.extend(self.ax_div.get_xticklabels(which='both'))
              tickLabels.append(self.ax_div.xaxis.get_offset_text())
          elif(axis == 'y'):
            axisobject, axisobject2, axisname = self.ax, self.ax_div, 'ax'
            if((self.ticksYColor == value) or (not self.ticksYShow)):
              redraw = False
            self.ticksYColor = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if((self.ticksY2Color == value) or (not self.ticksY2Show)):
              redraw = False
            self.ticksY2Color = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          # sets axis label color
          if(axis == 'x'):
            if((self.ticksXColor_resid == value) or (not self.ticksXShow_resid)):
              redraw = False
            self.ticksXColor_resid = value
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          else:
            if((self.ticksYColor_resid == value) or (not self.ticksResidYShow)):
              redraw = False
            self.ticksYColor_resid = value
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedTicksXColor == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXColor = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedTicksXColor_resid == value) or (not self.ticksXShow)):
            redraw = False
          self.slavedTicksXColor_resid = value
          
        # set tickLabels to empty list as draw2ndAxisTicks() will take care of the rest
        tickLabels = []
        
      for entry in tickLabels:
        entry.set_color(value)
      # redraw?
      if(redraw):
        plotobject.myRefresh()

  def setTickMarkLabelShow(self, value=True, axis='left', redraw=True, target='plot'):
    # sets where to display the axis labels
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2 = self.dataplotwidget, self.ax2, self.ax2_div
          if (axis in ['left', 'right']):
            axisobject, axisobject2, axisname = self.ax, self.ax_div, 'ax'
        else:
          plotobject, axisobject, axisobject2 = self.residplotwidget, self.ax_resid, self.ax_resid_div

        self.ticksLabelShow[axis] = value
        if(axis == 'bottom'):
          axisobject.xaxis.set_tick_params(which='both', labelbottom=value)
          axisobject2.xaxis.set_tick_params(which='both', labelbottom=value)
        elif(axis == 'top'):
          axisobject.xaxis.set_tick_params(which='both', labeltop=value)
          axisobject2.xaxis.set_tick_params(which='both', labeltop=value)
        elif(axis in ['left', 'left2']):
          axisobject.yaxis.set_tick_params(which='both', labelleft=value)
          axisobject2.yaxis.set_tick_params(which='both', labelleft=value)
        else:
          axisobject.yaxis.set_tick_params(which='both', labelright=value)
          axisobject2.yaxis.set_tick_params(which='both', labelright=value)

        # check general tick show dialog
        if(value):
          if(axis in ['bottom', 'top']):
            self.ticksXShow, self.ticksXShow_resid = True, True
            if(hasattr(self.parent, 'graphicsarea')):
              self.parent.graphicsarea.configTickXLabel.setChecked(True)
            # and activate TickOne4All()
            self.setTickOne4All(axis='x', redraw=False, target=target)
          elif(axis in ['left', 'right']):
            self.ticksYShow, self.ticksResidYShow  = True, True
            if(hasattr(self.parent, 'graphicsarea')):
              self.parent.graphicsarea.configTickYLabel.setChecked(True)
              self.parent.graphicsarea.configTickResidYLabel.setChecked(True)
              self.parent.graphicsarea.configInnerLabel.setChecked(True)
            self.setTickOne4All(axis='y', redraw=False, target=target)
          else:
            self.ticksY2Show  = True
            if(hasattr(self.parent, 'graphicsarea2')):
              self.parent.graphicsarea2.configTickYLabel.setChecked(True)
            self.setTickOne4All(axis='y2', redraw=False, target=target)
        
        # set visibility of inner items correctly
        self.updateInnerSituation()

        # redraw?
        if(redraw):
          plotobject.myRefresh()
      elif(axis == 'slaved'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.ticksLabelShow[axis] == value):
            redraw = False
          self.ticksLabelShow[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.ticksLabelShow_resid[axis] == value):
            redraw = False
          self.ticksLabelShow_resid[axis] = value

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setTickMarkDirection(self, value='in', axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if(self.ticksDirection[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis, axisobject, axisobject2, axisname = 'y', self.ax, self.ax_div, 'ax'
            self.ticksDirection['left'] = value
            self.ticksDirection['right'] = value
          elif (axis in ['left2', 'right2']):
            axis = 'y'
            self.ticksDirection['left2'] = value
            self.ticksDirection['right2'] = value
          else:
            axis = 'x'
            self.ticksDirection['top'] = value
            self.ticksDirection['bottom'] = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(self.ticksDirection_resid[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis = 'y'
            self.ticksDirection_resid['left'] = value
            self.ticksDirection_resid['right'] = value
          elif (axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            axis = 'x'
            self.ticksDirection_resid['top'] = value
            self.ticksDirection_resid['bottom'] = value

        # sets tick mark position
        if(axis != 'abort'):
          axisobject.tick_params(axis=axis, direction=value, which='both')
          axisobject2.tick_params(axis=axis, direction=value, which='both')
          if(self.splitY and (axis == 'x') and (target == 'plot')):
            self.ax.tick_params(axis=axis, direction=value, which='both')
            self.ax_div.tick_params(axis=axis, direction=value, which='both')
          
          # redraw?
          if(redraw):
            plotobject.myRefresh()
      elif(axis == 'slaved'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.ticksDirection[axis] == value):
            redraw = False
          self.ticksDirection[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.ticksDirection_resid[axis] == value):
            redraw = False
          self.ticksDirection_resid[axis] = value

        # redraw?
        if(redraw):
          plotobject.myRefresh()
          
  def setTickMarkVisibility(self, value=True, axis='left', redraw=True, target='plot', initMode=False):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if(axis in ['left', 'right']):
            axisobject, axisobject2, axisname = self.ax, self.ax_div, 'ax'
          if(self.ticksVisible[axis] == value):
            redraw = False
          self.ticksVisible[axis] = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(self.ticksVisible_resid[axis] == value):
            redraw = False
          self.ticksVisible_resid[axis] = value

        if(axis in ['left', 'right']):
          # modify ticks along y axes
          # adjust axis labels
          if(not initMode):
            if(axis == 'left'):
              if(value or self.ticksLabelShow['right']):
                self.ticksLabelShow['left'] = value
                # ensure that graphicsarea has already been created (required during program start)
                if(hasattr(self.parent, 'graphicsarea')):
                  self.parent.graphicsarea.configTickMarkLabelShow['left'].setChecked(value)
            else:
              if(value and (not self.ticksVisible['left'])):
                self.ticksLabelShow['left'] = False
                if(hasattr(self.parent, 'graphicsarea')):
                  self.parent.graphicsarea.configTickMarkLabelShow['left'].setChecked(False)
              if(value == (not self.ticksLabelShow['left'])):
                self.ticksLabelShow['right'] = value
                if(hasattr(self.parent, 'graphicsarea')):
                  self.parent.graphicsarea.configTickMarkLabelShow['right'].setChecked(value)
          # then set ticks and tick labels
          if(target == 'plot'):
            useShow = self.ticksYShow
          else:
            useShow = self.ticksResidYShow
          axisobject.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'], labelleft=self.ticksLabelShow['left'] and useShow, right=self.ticksVisible['right'], labelright=self.ticksLabelShow['right'] and useShow)
          axisobject2.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'], labelleft=self.ticksLabelShow['left'] and useShow, right=self.ticksVisible['right'], labelright=self.ticksLabelShow['right'] and useShow)
          if(self.splitY and (target == 'plot')):
            self.ax.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'], labelleft=self.ticksLabelShow['left'] and useShow, right=self.ticksVisible['right'], labelright=self.ticksLabelShow['right'] and useShow)
            self.ax_div.yaxis.set_tick_params(which='both', left=self.ticksVisible['left'], labelleft=self.ticksLabelShow['left'] and useShow, right=self.ticksVisible['right'], labelright=self.ticksLabelShow['right'] and useShow)
        elif (axis in ['left2', 'right2']):
          # adjust axis labels
          if(not initMode):
            if(axis == 'right2'):
              if(value or self.ticksLabelShow['left2']):
                self.ticksLabelShow['right2'] = value
                # ensure that graphicsarea has already been created (required during program start)
                if(hasattr(self.parent, 'graphicsarea2')):
                  self.parent.graphicsarea2.configTickMarkLabelShow['right2'].setChecked(value)
            else:
              if(value and (not self.ticksVisible['right2'])):
                self.ticksLabelShow['right2'] = False
                if(hasattr(self.parent, 'graphicsarea2')):
                  self.parent.graphicsarea2.configTickMarkLabelShow['right2'].setChecked(False)
              if(value == (not self.ticksLabelShow['right2'])):
                self.ticksLabelShow['left2'] = value
                if(hasattr(self.parent, 'graphicsarea2')):
                  self.parent.graphicsarea2.configTickMarkLabelShow['left2'].setChecked(value)
          # then set ticks and tick labels
          useShow = self.ticksY2Show
          axisobject.yaxis.set_tick_params(which='both', left=self.ticksVisible['left2'], labelleft=self.ticksLabelShow['left2'] and useShow, right=self.ticksVisible['right2'], labelright=self.ticksLabelShow['right2'] and useShow)
          axisobject2.yaxis.set_tick_params(which='both', left=self.ticksVisible['left2'], labelleft=self.ticksLabelShow['left2'] and useShow, right=self.ticksVisible['right2'], labelright=self.ticksLabelShow['right2'] and useShow)
        else:
          # modify ticks along x axes
          # adjust axis labels
          if(not initMode):
            if(axis == 'bottom'):
              if(value or self.ticksLabelShow['top']):
                self.ticksLabelShow['bottom'] = value
                # ensure that graphicsarea has already been created (required during program start)
                if(hasattr(self.parent, 'graphicsarea')):
                  self.parent.graphicsarea.configTickMarkLabelShow['bottom'].setChecked(value)
            else:
              if(value and (not self.ticksVisible['bottom'])):
                self.ticksLabelShow['bottom'] = False
                if(hasattr(self.parent, 'graphicsarea')):
                  self.parent.graphicsarea.configTickMarkLabelShow['bottom'].setChecked(False)
              if(value == (not self.ticksLabelShow['bottom'])):
                self.ticksLabelShow['top'] = value
                if(hasattr(self.parent, 'graphicsarea')):
                  self.parent.graphicsarea.configTickMarkLabelShow['top'].setChecked(value)
          # then set ticks and tick labels
          useShow, useShow2 = self.ticksXShow, self.ticksXShow_div
          axisobject.xaxis.set_tick_params(which='both', bottom=self.ticksVisible['bottom'], labelbottom=self.ticksLabelShow['bottom'] and useShow, top=self.ticksVisible['top'], labeltop=self.ticksLabelShow['top'] and useShow)
          axisobject2.xaxis.set_tick_params(which='both', bottom=self.ticksVisible['bottom'], labelbottom=self.ticksLabelShow['bottom'] and useShow2, top=self.ticksVisible['top'], labeltop=self.ticksLabelShow['top'] and useShow2)
              
        # remember settings
        if(axis in ['left', 'right']):
          axis = 'y'
        elif(axis in ['left2', 'right2']):
          if(target == 'plot'):
            axis = 'y'
          else:
            axis = 'abort'
        else:
          axis = 'x'
        
        if(axis != 'abort'):
          # need to solve inner situation
          self.updateInnerSituation()
          # redraw?
          if(redraw):
            plotobject.myRefresh()
      elif(axis == 'slaved'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.ticksVisible[axis] == value):
            redraw = False
          self.ticksVisible[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.ticksVisible_resid[axis] == value):
            redraw = False
          self.ticksVisible_resid[axis] = value

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setTickMarkColor(self, value=[0, 0, 0, 1], axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject, axisobject, axisobject2, axisname = self.dataplotwidget, self.ax2, self.ax2_div, 'ax2'
          if(self.ticksColor[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis, axisobject, axisobject2, axisname = 'y', self.ax, self.ax_div, 'ax'
            self.ticksColor['left'] = value
            self.ticksColor['right'] = value
          elif (axis in ['left2', 'right2']):
            axis = 'y'
            self.ticksColor['left2'] = value
            self.ticksColor['right2'] = value
          else:
            axis = 'x'
            self.ticksColor['top'] = value
            self.ticksColor['bottom'] = value
        else:
          plotobject, axisobject, axisobject2, axisname = self.residplotwidget, self.ax_resid, self.ax_resid_div, 'ax_resid'
          if(self.ticksColor_resid[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis = 'y'
            self.ticksColor_resid['left'] = value
            self.ticksColor_resid['right'] = value
          elif (axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            axis = 'x'
            self.ticksColor_resid['top'] = value
            self.ticksColor_resid['bottom'] = value

        # sets axis label color
        if(axis != 'abort'):
          axisobject.tick_params(axis=axis, color=value, which='both')
          axisobject2.tick_params(axis=axis, color=value, which='both')
          if(self.splitY and (axis == 'x') and (target == 'plot')):
            self.ax.tick_params(axis=axis, color=value, which='both')
            self.ax_div.tick_params(axis=axis, color=value, which='both')
    
          # redraw?
          if(redraw):
            plotobject.myRefresh()
      elif(axis == 'slaved'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.ticksColor[axis] == value):
            redraw = False
          self.ticksColor[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.ticksColor_resid[axis] == value):
            redraw = False
          self.ticksColor_resid[axis] = value

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setTickMarkWidth(self, value=1.0, axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(self.ticksWidth[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis = 'y'; axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            self.ticksWidth['left'] = value
            self.ticksWidth['right'] = value
          elif (axis in ['left2', 'right2']):
            axis = 'y'
            self.ticksWidth['left2'] = value
            self.ticksWidth['right2'] = value
          else:
            axis = 'x'
            self.ticksWidth['top'] = value
            self.ticksWidth['bottom'] = value
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
          if(self.ticksWidth_resid[axis] == value):
            redraw = False
          self.ticksWidth_resid[axis] = value
          if (axis in ['left', 'right']):
            axis = 'y'
            self.ticksWidth_resid['left'] = value
            self.ticksWidth_resid['right'] = value
          elif (axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            axis = 'x'
            self.ticksWidth_resid['top'] = value
            self.ticksWidth_resid['bottom'] = value

        if(axis != 'abort'):
          axisobject.tick_params(axis=axis, width=value, which='both')
          axisobject2.tick_params(axis=axis, width=value, which='both')
          if(self.splitY and (axis == 'x') and (target == 'plot')):
            self.ax.tick_params(axis=axis, width=value, which='both')
            self.ax_div.tick_params(axis=axis, width=value, which='both')
  
          # redraw?
          if(redraw):
            plotobject.myRefresh()
      elif(axis == 'slaved'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(self.ticksWidth[axis] == value):
            redraw = False
          self.ticksWidth[axis] = value
        else:
          plotobject = self.residplotwidget
          if(self.ticksWidth_resid[axis] == value):
            redraw = False
          self.ticksWidth_resid[axis] = value

        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setMinorTickRelativeLength(self, value, axis, redraw=True, target='plot'):
    # changes relative length of ticks
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2', 'xs']):
        if(target == 'plot'):
          if(axis == 'y'):
            if(self.ticksYMinorRelativeLength == value):
              redraw = False
            self.ticksYMinorRelativeLength = value
            item = 'left'
          elif(axis == 'y2'):
            if(self.ticksY2MinorRelativeLength == value):
              redraw = False
            self.ticksY2MinorRelativeLength = value
            item = 'left2'
          elif(axis == 'x'):
            if(self.ticksXMinorRelativeLength == value):
              redraw = False
            self.ticksXMinorRelativeLength = value
            item = 'top'
          elif(axis == 'xs'):
            if(self.slavedTicksXMinorRelativeLength == value):
              redraw = False
            self.slavedTicksXMinorRelativeLength = value
            item = 'xs'
        else: 
          if(axis == 'y'):
            if(self.ticksResidYMinorRelativeLength == value):
              redraw = False
            self.ticksResidYMinorRelativeLength = value
            item = 'left'
          elif(axis == 'y'):
            axis = 'abort'
          elif(axis == 'x'):
            if(self.ticksXMinorRelativeLength_resid == value):
              redraw = False
            self.ticksXMinorRelativeLength_resid = value
            item = 'top'
          elif(axis == 'xs'):
            if(self.slavedTicksXMinorRelativeLength_resid == value):
              redraw = False
            self.slavedTicksXMinorRelativeLength_resid = value
            item = 'xs'
            
        # update tick marks
        if(axis != 'xs'):
          self.setTickMarkLength(value=self.ticksLength[item], axis=item, redraw=redraw, target=target, forceRedraw=redraw)
        else:
          self.setTickMarkLength(value=value, axis=item, redraw=redraw, target=target, forceRedraw=redraw)
      
  def setTickMarkLength(self, value=1.0, axis='left', redraw=True, target='plot', forceRedraw=False):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(self.ticksLength[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis = 'y'; axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            self.ticksLength['left'] = value
            self.ticksLength['right'] = value
            ticksMinorRelativeLength = self.ticksYMinorRelativeLength
          elif (axis in ['left2', 'right2']):
            axis = 'y'
            self.ticksLength['left2'] = value
            self.ticksLength['right2'] = value
            ticksMinorRelativeLength = self.ticksY2MinorRelativeLength
          else:
            axis = 'x'
            self.ticksLength['top'] = value
            self.ticksLength['bottom'] = value
            ticksMinorRelativeLength = self.ticksXMinorRelativeLength
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
          if(self.ticksLength_resid[axis] == value):
            redraw = False
          # for time being, left/right and top/bottom are linked
          if (axis in ['left', 'right']):
            axis = 'y'
            self.ticksLength_resid['left'] = value
            self.ticksLength_resid['right'] = value
            ticksMinorRelativeLength = self.ticksResidYMinorRelativeLength
          elif (axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            axis = 'x'
            self.ticksLength_resid['top'] = value
            self.ticksLength_resid['bottom'] = value
            ticksMinorRelativeLength = self.ticksXMinorRelativeLength

        if(axis != 'abort'):
          axisobject.tick_params(axis=axis, length=value, which='major')
          axisobject.tick_params(axis=axis, length=value * ticksMinorRelativeLength, which='minor')
          axisobject2.tick_params(axis=axis, length=value, which='major')
          axisobject2.tick_params(axis=axis, length=value * ticksMinorRelativeLength, which='minor')
          if(self.splitY and (axis == 'x') and (target == 'plot')):
            self.ax.tick_params(axis=axis, length=value, which='major')
            self.ax.tick_params(axis=axis, length=value * ticksMinorRelativeLength, which='minor')
            self.ax_div.tick_params(axis=axis, length=value, which='major')
            self.ax_div.tick_params(axis=axis, length=value * ticksMinorRelativeLength, which='minor')
          # redraw?
          if(redraw or forceRedraw):
            plotobject.myRefresh()
      elif(axis in ['xs', 'slaved']):
        # note that 'xs' redirects from minor tick realtive length
        # by contrast, 'slaved' results from the UI call back
        # draw2ndAxisTicks() takes care of the nitty-gritty stuff
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if(axis == 'slaved'):
            if(self.ticksLength[axis] == value):
              redraw = False
            self.ticksLength[axis] = value
        else:
          plotobject = self.residplotwidget
          if(axis == 'slaved'):
            if(self.ticksLength_resid[axis] == value):
              redraw = False
            self.ticksLength_resid[axis] = value
        # redraw?
        if(redraw or forceRedraw):
          plotobject.myRefresh()

  def setAxisLabelColor(self, value=[0, 0, 0, 1], axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis == 'x'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if((self.labelXColor == value) or (not self.labelXShow)):
            redraw = False
          self.labelXColor = value
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
          if((self.labelXColor_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.labelXColor_resid = value

        handleAxis = axisobject.xaxis.label
        handleAxis.set_color(value)
        handleAxis = axisobject2.xaxis.label
        handleAxis.set_color(value)
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
          if((self.labelYColor == value) or (not self.labelYShow)):
            redraw = False
          self.labelYColor = value
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
          if((self.labelYColor_resid == value) or (not self.labelYShow_resid)):
            redraw = False
          self.labelYColor_resid = value

        handleAxis = axisobject.yaxis.label
        handleAxis.set_color(value)
        handleAxis = axisobject2.yaxis.label
        handleAxis.set_color(value)
      elif(axis == 'y2'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if((self.labelY2Color == value) or (not self.labelY2Show)):
            redraw = False
          self.labelY2Color = value
          handleAxis = axisobject.yaxis.label
          handleAxis.set_color(value)
          handleAxis = axisobject2.yaxis.label
          handleAxis.set_color(value)
        else:
          axis = 'abort'
      elif(axis == 'xs'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXColor == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXColor = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXColor_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXColor_resid = value

      # remember settings
      if(axis != 'abort'):
        # redraw?
        if(redraw):
          plotobject.myRefresh()

  def setAxisLabel(self, labeltext=None, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      # updates axis label
      if(labeltext == None):
        labeltext = axis

      if(axis == 'x'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; figobject = self.matplot; axisname = 'ax2'
          if((self.labelX == labeltext) or (not self.labelXShow)):
            redraw = False
          self.labelX = labeltext
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; figobject = self.residplot; axisname = 'ax_resid'
          if((self.labelX_resid == labeltext) or (not self.labelXShow_resid)):
            redraw = False
          self.labelX_resid = labeltext
        # manually process escape characters
        labeltext = labeltext.replace('\n', '\\n')
        labeltext = labeltext.replace('\t', '\\t')
        labeltext = '\n'.join([i for i in labeltext.split('\\n') if (len(i.strip()))])
        labeltext = '\t'.join([i for i in labeltext.split('\\t') if (len(i.strip()))])
        # check for potential Mathtext errors
        prevLabel = axisobject.xaxis.get_label_text()
        axisobject.xaxis.set_label_text(labeltext)
        try:
          axisobject.xaxis.label._get_layout(figobject.canvas.renderer)
        except:
          # revert to previous label
          self.parent.statusbar.showMessage('Cannot set axis label to ' + labeltext, self.parent.STATUS_TIME)
          
          axisobject.xaxis.set_label_text(prevLabel)
          labeltext = prevLabel
          redraw = False
      elif(axis == 'y'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax; figobject = self.matplot; axisname = 'ax'
          if((self.labelY == labeltext) or (not self.labelYShow)):
            redraw = False
          self.labelY = labeltext
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; figobject = self.residplot; axisname = 'ax_resid'
          if((self.labelY_resid == labeltext) or (not self.labelYShow_resid)):
            redraw = False
          self.labelY_resid = labeltext
        # manually process escape characters
        labeltext = labeltext.replace('\n', '\\n')
        labeltext = labeltext.replace('\t', '\\t')
        labeltext = '\n'.join([i for i in labeltext.split('\\n') if (len(i.strip()))])
        labeltext = '\t'.join([i for i in labeltext.split('\\t') if (len(i.strip()))])
        if((target == 'resid') and (labeltext != '')):
          labeltext = u'\N{GREEK CAPITAL LETTER DELTA}' + labeltext
        # check for potential Mathtext errors
        prevLabel = axisobject.yaxis.get_label_text()
        axisobject.yaxis.set_label_text(labeltext)
        try:
          axisobject.yaxis.label._get_layout(figobject.canvas.renderer)
        except:
          # revert to previous label
          self.parent.statusbar.showMessage('Cannot set axis label to ' + labeltext, self.parent.STATUS_TIME)
          axisobject.yaxis.set_label_text(prevLabel)
          labeltext = prevLabel
          redraw = False
      elif(axis == 'y2'):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; figobject = self.matplot; axisname = 'ax2'
          if((self.labelY2 == labeltext) or (not self.labelY2Show)):
            redraw = False
          self.labelY2 = labeltext
          # manually process escape characters
          labeltext = labeltext.replace('\n', '\\n')
          labeltext = labeltext.replace('\t', '\\t')
          labeltext = '\n'.join([i for i in labeltext.split('\\n') if (len(i.strip()))])
          labeltext = '\t'.join([i for i in labeltext.split('\\t') if (len(i.strip()))])
          # check for potential Mathtext errors
          prevLabel = axisobject.yaxis.get_label_text()
          axisobject.yaxis.set_label_text(labeltext)
          try:
            axisobject.yaxis.label._get_layout(figobject.canvas.renderer)
            axisobject2.yaxis.set_label_text(labeltext)
          except:
            # revert to previous label
            self.parent.statusbar.showMessage('Cannot set axis label to' + labeltext, self.parent.STATUS_TIME)
            axisobject.yaxis.set_label_text(prevLabel)
            axisobject2.yaxis.set_label_text(prevLabel)
            labeltext = prevLabel
            redraw = False
        else:
          axis = 'abort'
                        
      # remember settings
      if(axis != 'abort'):
        # capture errors with bad fonts (can occur when switching back from all-Mathtext label)
        try:
          if(redraw):
            plotobject.myRefresh()
        except:
          ###safeFont = 'DejaVu Sans'
          self.parent.statusbar.showMessage('Experiencing problems with font ' + self.axisFont[axis] + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
          self.axisFont[axis] = SAFE_FONT
          if(axis == 'x'):
            axisobject.xaxis.label.set_fontname(SAFE_FONT)
          else:
            axisobject.yaxis.label.set_fontname(SAFE_FONT)
            if(axis == 'y2'):
              axisobject2.yaxis.label.set_fontname(SAFE_FONT)
  
          if(redraw):
            plotobject.myRefresh()

  def setAxisVisibility(self, value=True, axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div
          if(self.axisVisible[axis] == value):
            redraw = False
          self.axisVisible[axis] = value
          if(axis in ['left', 'right']):
            axisname = 'ax'
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisVisible_resid[axis] == value):
              redraw = False
            self.axisVisible_resid[axis] = value

        # sets axis visibility
        if(axis != 'abort'):
          if(axis in axisobject.spines):
            axisobject.spines[axis].set_visible(value)
          if(axis in axisobject2.spines):
            axisobject2.spines[axis].set_visible(value)
  
          # need to solve inner situation
          if(axis in ['bottom', 'top']):
            self.updateInnerYSituation()
          else:
            self.updateInnerSituation()

          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisColor(self, value=True, axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div
          if(self.axisColor[axis] == value):
            redraw = False
          self.axisColor[axis] = value
          if(axis in ['left', 'right']):
            axisname = 'ax'
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisColor_resid[axis] == value):
              redraw = False
            self.axisColor_resid[axis] = value

        # set axis color
        if(axis != 'abort'):
          if(axis in axisobject.spines):
            axisobject.spines[axis].set_color(value)
          if(axis in axisobject2.spines):
            axisobject2.spines[axis].set_color(value)
          
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisWidth(self, value=1.0, axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div
          if(self.axisWidth[axis] == value):
            redraw = False
          self.axisWidth[axis] = value
          if(axis in ['left', 'right']):
            axisname = 'ax'
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisWidth_resid[axis] == value):
              redraw = False
            self.axisWidth_resid[axis] = value

        # check grid style to avoid crash at zero width
        if(value < .01):
          if(target == 'plot'):
            if(self.axisStyle[axis] != 'solid'):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              self.axisStyle[axis] = 'solid'
          else:
            if(self.axisStyle_resid[axis] != 'solid'):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              self.axisStyle_resid[axis] = 'solid'

          if(axis in axisobject.spines):
            axisobject.spines[axis].set_linestyle('solid')
          if(axis in axisobject2.spines):
            axisobject2.spines[axis].set_linestyle('solid')

        # updates axis width
        if(axis != 'abort'):
          if(axis in axisobject.spines):
            axisobject.spines[axis].set_linewidth(value)
          if(axis in axisobject2.spines):
            axisobject2.spines[axis].set_linewidth(value)
  
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisFont(self, value=None, axis='x', redraw=True, target='plot'):
    # check whether font exists
    ###safeFont = 'DejaVu Sans'
    if(not (value in self.parent.fontNames)):
      value = SAFE_FONT

    if(value in self.parent.fontNames):
      # check whether to operate on data or resid plot
      if(target in ['plot', 'resid']):
        if(axis in ['x', 'y', 'y2']):
          if(target == 'plot'):
            prevFont = self.axisFont[axis]
            plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
            if(axis == 'y'):
              axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if(self.axisFont[axis] == value):
              redraw = False
            self.axisFont[axis] = value
          else:
            if(axis == 'y2'):
              axis = 'abort'
            else:
              prevFont = self.axisFont_resid[axis]
              plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
              if(self.axisFont_resid[axis] == value):
                redraw = False
              self.axisFont_resid[axis] = value

          if(axis != 'abort'):
            useFont = value
            # try setting font value
            if(axis == 'x'):
              for useObject in [axisobject, axisobject2]:
                useObject.xaxis.label.set_fontname(useFont)
                _, _, descent = useObject.xaxis.label._get_layout(self.matplot.canvas.renderer)
                if(not (descent > 0)):
                  self.parent.statusbar.showMessage('Experiencing problems setting font ' + useFont + ' -- reverting to ' + prevFont, self.parent.STATUS_TIME)
                  useFont = prevFont
                  useObject.xaxis.label.set_fontname(useFont)
                  _, _, descent =useObject.xaxis.label._get_layout(self.matplot.canvas.renderer)
                  if(not (descent > 0)):
                    self.parent.statusbar.showMessage('Also experiencing problems setting font ' + useFont + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
                    useFont = SAFE_FONT
                    useObject.xaxis.label.set_fontname(useFont)
            else:
              for useObject in [axisobject, axisobject2]:
                useObject.yaxis.label.set_fontname(useFont)
                _, _, descent = useObject.yaxis.label._get_layout(self.matplot.canvas.renderer)
                if(not (descent > 0)):
                  self.parent.statusbar.showMessage('Experiencing problems setting font ' + useFont + ' -- reverting to ' + prevFont, self.parent.STATUS_TIME)
                  useFont = prevFont
                  useObject.yaxis.label.set_fontname(useFont)
                  _, _, descent =useObject.yaxis.label._get_layout(self.matplot.canvas.renderer)
                  if(not (descent > 0)):
                    self.parent.statusbar.showMessage('Also experiencing problems setting font ' + useFont + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
                    useFont = SAFE_FONT
                    useObject.yaxis.label.set_fontname(useFont)
  
            # save actual font value
            if(target == 'plot'):
              self.axisFont[axis] = useFont
            else:
              self.axisFont_resid[axis] = useFont
            
            # have to capture errors in case a strange font is set
            # retain this as 2nd-layer contro
            try:
              if(redraw):
                plotobject.myRefresh()
            except:
              self.parent.statusbar.showMessage('Experiencing problems setting font ' + value + ' -- reverting to ' + prevFont, self.parent.STATUS_TIME)
              
              # revert to previous font
              value = prevFont
              if(target == 'plot'):
                self.axisFont[axis] = prevFont
              else:
                self.axisFont_resid[axis] = prevFont
  
              if(axis == 'x'):
                axisobject.xaxis.label.set_fontname(prevFont)
                axisobject2.xaxis.label.set_fontname(prevFont)
              else:
                axisobject.yaxis.label.set_fontname(prevFont)
                axisobject2.yaxis.label.set_fontname(prevFont)
  
              # also capture errors with previous font (can happen if selecting two bad fonts in a row)
              try:
                if(redraw):
                  plotobject.myRefresh()
              except:
                self.parent.statusbar.showMessage('Also experiencing problems setting font ' + value + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
  
                # revert to previous font
                value = SAFE_FONT
                if(target == 'plot'):
                  self.axisFont[axis] = SAFE_FONT
                else:
                  self.axisFont_resid[axis] = SAFE_FONT
    
                if(axis == 'x'):
                  axisobject.xaxis.label.set_fontname(SAFE_FONT)
                  axisobject2.xaxis.label.set_fontname(SAFE_FONT)
                else:
                  axisobject.yaxis.label.set_fontname(SAFE_FONT)
                  axisobject2.yaxis.label.set_fontname(SAFE_FONT)
        elif(axis == 'xs'):
          # ideally, we just rely on draw2ndXAxisTicks() to do the right thing here
          # but we should issue a redraw as we skip the above
          if(target == 'plot'):
            plotobject = self.dataplotwidget
            if(self.axisFont[axis] == value):
              redraw = False
            self.axisFont[axis] = value
          else:
            if(self.axisFont_resid[axis] == value):
              redraw = False
            self.axisFont_resid[axis] = value
            plotobject = self.residplotwidget

          # have to capture errors in case a strange font is set
          # retain this as 2nd-layer control
          try:
            if(redraw):
              plotobject.myRefresh()
          except:
            self.parent.statusbar.showMessage('Experiencing problems setting font ' + value + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
            self.axisFont['xs'] = SAFE_FONT
            self.axisFont_resid['xs'] = SAFE_FONT
            plotobject.myRefresh()
  
  def setAxisLabelAlignment(self, value=None, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'

        if(axis == 'x'):
          axisobject.xaxis.label.set_horizontalalignment(value)
          axisobject2.xaxis.label.set_horizontalalignment(value)
          direction = 'horizontal'
          if(target == 'plot'):
            if((self.labelXAlignment == value) or (not self.labelXShow)):
              redraw = False
            self.labelXAlignment = value
          else:
            if((self.labelXAlignment_resid == value) or (not self.labelXShow_resid)):
              redraw = False
            self.labelXAlignment_resid = value
        elif(axis == 'y'):
          if(target == 'plot'):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if((self.labelYAlignment == value) or (not self.labelYShow)):
              redraw = False
            self.labelYAlignment = value
          else:
            if((self.labelYAlignment_resid == value) or (not self.labelYShow_resid)):
              redraw = False
            self.labelYAlignment_resid = value
          axisobject.yaxis.label.set_horizontalalignment(value)
          axisobject2.yaxis.label.set_horizontalalignment(value)
          direction = 'horizontal'
        elif(axis == 'y2'):
          if(target == 'plot'):
            axisobject.yaxis.label.set_horizontalalignment(value)
            axisobject2.yaxis.label.set_horizontalalignment(value)
            direction = 'horizontal'
            if((self.labelY2Alignment == value) or (not self.labelYShow_resid)):
              redraw = False
            self.labelY2Alignment = value
          else:
            axis = 'abort'
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXAlignment == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXAlignment = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXAlignment_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXAlignment_resid = value

        # remember settings
        if(axis != 'abort'):
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisLabelAlignmentVertical(self, value=None, axis='x', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
        else:
          plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'

        if(axis == 'x'):
          if(value != 'center'):
            axisobject.xaxis.label.set_verticalalignment(value)
            axisobject2.xaxis.label.set_verticalalignment(value)
          else:
            try:
              axisobject.xaxis.label.set_verticalalignment('center_baseline')
              axisobject2.xaxis.label.set_verticalalignment('center_baseline')
            except:
              axisobject.xaxis.label.set_verticalalignment('center')
              axisobject2.xaxis.label.set_verticalalignment('center')
          direction = 'vertical'
          if(target == 'plot'):
            if((self.labelXAlignmentVertical == value) or (not self.labelXShow)):
              redraw = False
            self.labelXAlignmentVertical = value
          else:
            if((self.labelXAlignmentVertical_resid == value) or (not self.labelXShow_resid)):
              redraw = False
            self.labelXAlignmentVertical_resid = value
        elif(axis == 'y'):
          if(target == 'plot'):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
            if((self.labelYAlignmentVertical == value) or (not self.labelYShow)):
              redraw = False
            self.labelYAlignmentVertical = value
          else:
            if((self.labelYAlignmentVertical_resid == value) or (not self.labelYShow_resid)):
              redraw = False
            self.labelYAlignmentVertical_resid = value
          if(value != 'center'):
            axisobject.yaxis.label.set_verticalalignment(value)
            axisobject2.yaxis.label.set_verticalalignment(value)
          else:
            try:
              axisobject.yaxis.label.set_verticalalignment('center_baseline')
              axisobject2.yaxis.label.set_verticalalignment('center_baseline')
            except:
              axisobject.yaxis.label.set_verticalalignment('center')
              axisobject2.yaxis.label.set_verticalalignment('center')
          direction = 'vertical'
        elif(axis == 'y2'):
          if(target == 'plot'):
            if(value != 'center'):
              axisobject.yaxis.label.set_verticalalignment(value)
              axisobject2.yaxis.label.set_verticalalignment(value)
            else:
              try:
                axisobject.yaxis.label.set_verticalalignment('center_baseline')
                axisobject2.yaxis.label.set_verticalalignment('center_baseline')
              except:
                axisobject.yaxis.label.set_verticalalignment('center')
                axisobject2.yaxis.label.set_verticalalignment('center')
            direction = 'vertical'
            if((self.labelY2AlignmentVertical == value) or (not self.labelY2Show)):
              redraw = False
            self.labelY2AlignmentVertical = value
          else:
            axis = 'abort'
      elif(axis in ['xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget
          if((self.slavedLabelXAlignmentVertical == value) or (not self.labelXShow)):
            redraw = False
          self.slavedLabelXAlignmentVertical = value
        else:
          plotobject = self.residplotwidget
          if((self.slavedLabelXAlignmentVertical_resid == value) or (not self.labelXShow_resid)):
            redraw = False
          self.slavedLabelXAlignmentVertical_resid = value

        # remember settings
        if(axis != 'abort'):
          # redraw?
          if(redraw):
            plotobject.myRefresh()
  
  def setTickFont(self, value=None, axis='x', redraw=True, target='plot'):
    # check whether font exists
    ###safeFont = 'DejaVu Sans'
    if(not (value in self.parent.fontNames)):
      value = SAFE_FONT
    if(value in self.parent.fontNames):
      # check whether to operate on data or resid plot
      if(target in ['plot', 'resid']):
        if(axis in ['x', 'y', 'y2', 'xs']):
          if(target == 'plot'):
            prevFont = self.tickFont[axis]
            plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
            if(axis == 'y'):
              axisobject = self.ax; axisname = 'ax'
            if(self.tickFont[axis] == value):
              redraw = False
            self.tickFont[axis] = value
          else:
            prevFont = self.tickFont_resid[axis]
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.tickFont_resid[axis] == value):
              redraw = False
            self.tickFont_resid[axis] = value
        
          # updates tick font
          if(axis == 'x'):
            tickLabels = axisobject.get_xticklabels(which='both')
            tickLabels.append(axisobject.xaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_xticklabels(which='both'))
            tickLabels.append(axisobject2.xaxis.get_offset_text())
          elif(axis == 'y'):
            tickLabels = axisobject.get_yticklabels(which='both')
            tickLabels.append(axisobject.yaxis.get_offset_text())
            tickLabels.extend(axisobject2.get_yticklabels(which='both'))
            tickLabels.append(axisobject2.yaxis.get_offset_text())
          elif(axis == 'y2'):
            if(target == 'plot'):
              tickLabels = axisobject.get_yticklabels(which='both')
              tickLabels.append(axisobject.yaxis.get_offset_text())
              tickLabels.extend(axisobject2.get_yticklabels(which='both'))
              tickLabels.append(axisobject2.yaxis.get_offset_text())
            else:
              axis = 'abort'
          elif(axis == 'xs'):
            # ideally, we just rely on draw2ndXAxisTicks() to do the right thing here
            axis = 'abort'
            # but we should issue a redraw as we skip the below
            # have to capture errors in case a strange font is set
            # retain this as 2nd-layer control
            try:
              if(redraw):
                plotobject.myRefresh()
            except:
              self.parent.statusbar.showMessage('Experiencing problems setting font ' + value + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
              self.tickFont['xs'] = SAFE_FONT
              self.tickFont_resid['xs'] = SAFE_FONT
              plotobject.myRefresh()
          
          if(axis != 'abort'):
            useFont = value
            # try setting font value
            for entry in tickLabels:
              entry.set_fontname(useFont)
              _, _, descent = entry._get_layout(self.matplot.canvas.renderer)
              if(not (descent > 0)):
                self.parent.statusbar.showMessage('Experiencing problems setting font ' + useFont + ' -- reverting to ' + prevFont, self.parent.STATUS_TIME)
                useFont = prevFont
                entry.set_fontname(useFont)
                _, _, descent = entry._get_layout(self.matplot.canvas.renderer)
                if(not (descent > 0)):
                  self.parent.statusbar.showMessage('Also experiencing problems setting font ' + useFont + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
                  useFont = SAFE_FONT
                  entry.set_fontname(useFont)
  
            # save actual font value
            if(target == 'plot'):
              self.tickFont[axis] = useFont
            else:
              self.tickFont_resid[axis] = useFont
            
            # have to capture errors in case a strange font is set
            # retain this as 2nd-layer control
            try:
              if(redraw):
                plotobject.myRefresh()
            except:
              self.parent.statusbar.showMessage('Experiencing problems setting font ' + useFont + ' -- reverting to ' + prevFont, self.parent.STATUS_TIME)
              
              # revert to previous font
              useFont = prevFont
              if(target == 'plot'):
                self.tickFont[axis] = prevFont
              else:
                self.tickFont_resid[axis] = prevFont
                
              for entry in tickLabels:
                entry.set_fontname(prevFont)
  
              # also capture errors with previous font (can happen if selecting two bad fonts in a row)
              try:
                if(redraw):
                  plotobject.myRefresh()
              except:
                self.parent.statusbar.showMessage('Also experiencing problems setting font ' + useFont + ' -- reverting to ' + SAFE_FONT, self.parent.STATUS_TIME)
  
                # revert to previous font
                value = SAFE_FONT
                if(target == 'plot'):
                  self.tickFont[axis] = SAFE_FONT
                else:
                  self.tickFont_resid[axis] = SAFE_FONT
  
                for entry in tickLabels:
                  entry.set_fontname(SAFE_FONT)
  
  def setMinorTick(self, value=2, axis='x', redraw=True, target='plot'):
    # changes number of minor tick marks (on linear axes)
    if(target in ['plot', 'resid']):
      if(axis in ['x', 'y', 'y2', 'x2', 'xs']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = [self.ax, self.ax2]; axisname = 'ax2'
          if(axis == 'x'):
            if(self.ticksXMinor == value):
              redraw = False
            self.ticksXMinor = value
          elif(axis == 'x2'):
            axisobject = [self.ax_div, self.ax2_div]
            if(self.ticksXMinor_div == value):
              redraw = False
            self.ticksXMinor_div = value
          elif(axis == 'xs'):
            if(self.slavedTicksXMinor == value):
              redraw = False
            self.slavedTicksXMinor = value
          elif(axis == 'y'):
            axisname = 'ax'
            axisobject = [self.ax, self.ax_div]
            if(self.ticksYMinor == value):
              redraw = False
            self.ticksYMinor = value
          else:
            axisobject = [self.ax2, self.ax2_div]
            if(self.ticksY2Minor == value):
              redraw = False
            self.ticksY2Minor = value
        else:
          plotobject = self.residplotwidget; axisobject = [self.ax_resid]; axisname = 'ax_resid'
          if(axis == 'y'):
            if(self.ticksResidYMinor == value):
              redraw = False
            self.ticksResidYMinor = value
          elif(axis == 'x2'):
            axisobject = [self.ax_resid_div]
            
        # remember settings
        if(axis != 'xs'):
          minorAutoticks = MyAutoMinorLocator(value)
          if(axis in ['x', 'x2']):
            for useAxis in axisobject:
              useAxis.xaxis.set_minor_locator(minorAutoticks)
              useAxis.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
          else:
            for useAxis in axisobject:
              useAxis.yaxis.set_minor_locator(minorAutoticks)
              useAxis.yaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
          # redraw minor grid lines
          self.drawMinorAxisGrid(axis=axis, target=target, redraw=False)

        # redraw?
        if(axis != 'abort') :
          if(redraw):
            plotobject.myRefresh()

  def setAxisStyle(self, value='solid', axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
          if(self.axisStyle[axis] == value):
            redraw = False
          self.axisStyle[axis] = value
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisStyle_resid[axis] == value):
              redraw = False
            self.axisStyle_resid[axis] = value

        if(axis != 'abort'):
          # check axis style to avoid crash at zero width
          if(value != 'solid'):
            if(self.axisWidth[axis] < .01):
              self.parent.statusbar.showMessage('Zero line width encountered, set line style to solid.', self.parent.STATUS_TIME, color='blue')
              value = 'solid'
              if(target == 'plot'):
                self.axisStyle[axis] = 'solid'
              else:
                self.axisStyle_resid[axis] = 'solid'
  
          if(axis in axisobject.spines):
            axisobject.spines[axis].set_linestyle(value)
          if(axis in axisobject2.spines):
            axisobject2.spines[axis].set_linestyle(value)
  
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisDashStyle(self, value='solid', axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
          if(self.axisDashStyle[axis] == value):
            redraw = False
          self.axisDashStyle[axis] = value
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisDashStyle_resid[axis] == value):
              redraw = False
            self.axisDashStyle_resid[axis] = value

        if(axis != 'abort'):
          if(axis in axisobject.spines):
            axisobject.spines[axis].set_capstyle(value)
          if(axis in axisobject2.spines):
            axisobject2.spines[axis].set_capstyle(value)
  
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def resetAxisPosition(self, axis='left', redraw=True):
    # resets axis position to original values
    originalPosition = {'left':('axes', 0), 'right':('axes', 1.0), 'bottom':('axes', 0), 'top':('axes', 1.0), 'left2':('axes', 0), 'right2':('axes', 1.0)}
    if(axis in originalPosition):
      # check whether to do anything
      if((originalPosition[axis][0] != self.axisPosition[axis]) or (originalPosition[axis][1] != self.axisPositionValue[axis])):
        for target in ['plot', 'resid']:
          self.setAxisPosition(value=originalPosition[axis][0], axis=axis, redraw=False, target=target)
          self.setAxisPositionValue(value=originalPosition[axis][1], axis=axis, redraw=False, target=target)
        # refresh plots
        if(redraw):
          self.dataplotwidget.myRefresh()
          self.residplotwidget.myRefresh()

  def setAxisBoundaryValue(self, value=0.0, lower=True, axis='left', redraw=True, target='plot'):
    # changes limits on the axes
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(lower):
          index = 0
        else:
          index = 1
        if(target == 'plot'):
          if(self.axisBoundary[axis][index] == value):
            redraw = False
          self.axisBoundary[axis][index] = value
        else:
          if(self.axisBoundary_resid[axis][index] == value):
            redraw = False
          self.axisBoundary_resid[axis][index] = value
        
        self.setAxisBoundaryHelper(axis=axis, redraw=redraw, target=target)

  def setAxisBoundary(self, value=False, axis='left', redraw=True, target='plot'):
    # sets limits on the axes
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        if(target == 'plot'):
          if(self.axisBoundaryCheck[axis] == value):
            redraw = False
          self.axisBoundaryCheck[axis] = value
        else:
          if(self.axisBoundaryCheck_resid[axis] == value):
            redraw = False
          self.axisBoundaryCheck_resid[axis] = value
        
        self.setAxisBoundaryHelper(axis=axis, redraw=redraw, target=target)

  def setAxisBoundaryHelper(self, axis='left', redraw=True, target='plot'):
    # helper function for setting axis boundaries
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        secondAxes = False
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
            secondAxes = True
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div
        
        if(axis != 'abort'):
          if(secondAxes):
            useAxis = axis + '2'
          else:
            useAxis = axis
          if(self.axisBoundaryCheck[useAxis]):
            lower, upper = self.axisBoundary[useAxis]
            for useobject in [axisobject, axisobject2]:
              if(axis in useobject.spines):
                useobject.spines[axis].set_bounds(lower, upper)
          else:
            for useobject in [axisobject, axisobject2]:
              if(axis in useobject.spines):
                useobject.spines[axis]._bounds = None

          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisPosition(self, value='axes', axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        secondAxes = False
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div; axisname = 'ax'
          if(self.axisPosition[axis] == value):
            redraw = False
          self.axisPosition[axis] = value
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
            secondAxes = True
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisPosition_resid[axis] == value):
              redraw = False
            self.axisPosition_resid[axis] = value

        if(axis != 'abort'):
          if(axis in axisobject.spines):
            value = self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=axisobject, target=target, secondAxes=secondAxes)
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=axisobject2, target=target, secondAxes=secondAxes, splitX=True)
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisPositionValue(self, value=1.0, axis='left', redraw=True, target='plot'):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      if(axis in ['left', 'right', 'bottom', 'top', 'left2', 'right2']):
        secondAxes = False
        if(target == 'plot'):
          plotobject = self.dataplotwidget; axisobject = self.ax2; axisobject2 = self.ax2_div; axisname = 'ax2'
          if(axis in ['left', 'right']):
            axisobject = self.ax; axisobject2 = self.ax_div
          if(self.axisPositionValue[axis] == value):
            redraw = False
          self.axisPositionValue[axis] = value
          if(axis in ['left', 'right']):
            axisname = 'ax'
          if(axis in ['left2', 'right2']):
            axis = axis[:-1]
            secondAxes = True
        else:
          if(axis in ['left2', 'right2']):
            axis = 'abort'
          else:
            plotobject = self.residplotwidget; axisobject = self.ax_resid; axisobject2 = self.ax_resid_div; axisname = 'ax_resid'
            if(self.axisPositionValue_resid[axis] == value):
              redraw = False
            self.axisPositionValue_resid[axis] = value

        # updates axis width
        if(axis != 'abort'):
          if(axis in axisobject.spines):
            value = self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=axisobject, target=target, secondAxes=secondAxes)
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=axisobject2, target=target, secondAxes=secondAxes, splitX=True)
          # redraw?
          if(redraw):
            plotobject.myRefresh()

  def setAxisPositionHelper(self, axis, plotobject, axisobject, target, secondAxes=False, splitX=False, splitY=False):
    # called by setAxisPosition
    # duh, this is ugly!
    # the following call breaks the connection b/w spine and axis and thus disables detrimental updates to ticks and labels
    saveAxis = axisobject.spines[axis].axis
    axisobject.spines[axis].axis = None
    # assign useModeX and transAxisobject1/2
    if(splitX):
      useModeX, useMinX, transAxisobject1, transAxisobject2  = self.modeX_div, self.minX_div, self.ax_div, self.ax2_div
      ##useModeX, useMinX, transAxisobject1, transAxisobject2  = self.modeX_div, self.minX_div, self.ax, self.ax2
      transAxisobjectResid = self.ax_resid_div
    else:
      useModeX, useMinX, transAxisobject1, transAxisobject2 = self.modeX, self.minX, self.ax, self.ax2
      transAxisobjectResid = self.ax_resid
    #
    if(secondAxes):
      if((self.axisPosition[axis + '2'] == 'data') and (useModeX == 'log')):
        # matplotlib somehow screws up for log data (a bug I discovered!) => map to plot axes
        try:
          # have to do this b/c matplotlib.scale on Linux is buggy
          axis_to_data = transAxisobject1.transAxes + transAxisobject1.transData.inverted()
          point_on_data = (self.axisPositionValue[axis + '2'], self.minY)
          point_on_axes = axis_to_data.inverted().transform(point_on_data)
          assembledValue = ('axes', point_on_axes[0])
        except:
          assembledValue = ('outward', 0.0)
      else:
        assembledValue = (self.axisPosition[axis + '2'], self.axisPositionValue[axis + '2'])
    elif(target == 'plot'):
      if(axis in ['top', 'bottom']):
        # argh -- top and bottom are on self.ax2, meaning they are linked to the second axes
        # => we hence need to convert data values from one axis to the other
        if(self.axisPosition[axis] == 'data'):
          if(self.splitY):
            assembledValue = ('data', self.axisPositionValue[axis])
          else:
            try:
              # have to do this b/c matplotlib.scale on Linux is buggy
              axis_to_data1 = transAxisobject1.transAxes + transAxisobject1.transData.inverted()
              axis_to_data2 = transAxisobject2.transAxes + transAxisobject2.transData.inverted()
              point_on_data1 = (useMinX, self.axisPositionValue[axis])
              # map to axes
              point_on_axes = axis_to_data1.inverted().transform(point_on_data1)
              assembledValue = ('axes', point_on_axes[1])
            except:
              assembledValue = ('outward', 0.0)
        else:
          assembledValue = (self.axisPosition[axis], self.axisPositionValue[axis])
      else:
        if((self.axisPosition[axis] == 'data') and (useModeX == 'log')):
          # matplotlib somehow screws up for log data (a bug I discovered!) => map to plot axes
          try:
            # have to do this b/c matplotlib.scale on Linux is buggy
            axis_to_data2 = transAxisobject2.transAxes + transAxisobject2.transData.inverted()
            point_on_data2 = (self.axisPositionValue[axis], self.minY2)
            point_on_axes = axis_to_data2.inverted().transform(point_on_data2)
            assembledValue = ('axes', point_on_axes[0])
          except:
            assembledValue = ('outward', 0.0)
        else:
          assembledValue = (self.axisPosition[axis], self.axisPositionValue[axis])
    else:
      # the resid plot
      if(axis in ['left', 'right']):
        if((self.axisPosition[axis] == 'data') and (useModeX == 'log')):
          # deal with matplotlib error => map to plot axes
          try:
            # have to do this b/c matplotlib.scale on Linux is buggy
            axis_to_data = transAxisobjectResid.transAxes + transAxisobjectResid.transData.inverted()
            point_on_data = (self.axisPositionValue[axis], self.minResidY)
            point_on_axes = axis_to_data.inverted().transform(point_on_data)
            assembledValue = ('axes', point_on_axes[0])
          except:
            assembledValue = ('outward', 0.0)
        else:
          assembledValue = (self.axisPosition[axis], self.axisPositionValue[axis])
      else:
        assembledValue = (self.axisPosition[axis], self.axisPositionValue[axis])
      
    # set the new axis mode and value
    axisobject.spines[axis].set_position(assembledValue)

    # for good measure restore axis setting
    axisobject.spines[axis].axis = saveAxis
    
    # now we have to manually update the position of the tick labels and ticks
    if(axis in ['left', 'right']):
      ticks = axisobject.yaxis.get_major_ticks()
      ticks.extend(axisobject.yaxis.get_minor_ticks())
      if(axis == 'left'):
        useTransform = axisobject.get_yaxis_transform(which='tick1')
        tick1 = True
      else:
        useTransform = axisobject.get_yaxis_transform(which='tick2')
        tick1 = False
    else:
      ticks = axisobject.xaxis.get_major_ticks()
      ticks.extend(axisobject.xaxis.get_minor_ticks())
      if(axis == 'bottom'):
        useTransform = axisobject.get_xaxis_transform(which='tick1')
        tick1 = True
      else:
        useTransform = axisobject.get_xaxis_transform(which='tick2')
        tick1 = False

    horizontal = True        
    if(target == 'plot'):
      plotobject = self.matplot
      if(axis in ['left', 'right']):
        usePad = self.ticksYPad2
        horizontal = False
        if(secondAxes):
          usePad = self.ticksY2Pad2
      elif(axis in ['bottom', 'top']):
        usePad = self.ticksXPad2
    else:
      plotobject = self.residplot
      if(axis in ['left', 'right']):
        usePad = self.ticksYPad2_resid
        horizontal = False
      else:
        usePad = self.ticksXPad2_resid

    for entry in ticks:
      if(tick1):
        entry.tick1line.set_transform(useTransform)
        trans, vert, horiz = entry._get_text1_transform()
        # also have to apply pad2 padding - phew!
        if(horizontal):
          offset = matplotlib.transforms.ScaledTranslation(usePad / 72.0, 0.0, plotobject.dpi_scale_trans)
        else:
          offset = matplotlib.transforms.ScaledTranslation(0.0, usePad / 72.0, plotobject.dpi_scale_trans)
        trans += offset
        entry.label1.set_transform(trans)
      else:
        entry.tick2line.set_transform(useTransform)
        trans, vert, horiz = entry._get_text2_transform()
        # also have to apply pad2 padding - phew!
        if(horizontal):
          offset = matplotlib.transforms.ScaledTranslation(usePad / 72.0, 0, plotobject.dpi_scale_trans)
        else:
          offset = matplotlib.transforms.ScaledTranslation(0, usePad / 72.0, plotobject.dpi_scale_trans)
        trans += offset
        entry.label2.set_transform(trans)

    return repr(assembledValue)

  def setAutoScale(self, axis='x'):
    # updates autoscale options for axes
    if(axis in ['x', 'y', 'y2', 'x2']):
      if(axis == 'x'):
        state = self.autoScaleCheckX.isChecked()
        self.autoScaleX = state
      elif(axis == 'x2'):
        state = self.xSplitAutoScaleCheckX.isChecked()
        self.autoScaleX_div = state
      elif(axis == 'y'):
        state = self.autoScaleCheckY.isChecked()
        self.autoScaleY = state
      else:
        state = self.secondAutoScaleCheckY.isChecked()
        self.autoScaleY2 = state
        
      # rescale contents on setting auto to True
      if(state):
        self.doAutoScale(axis=axis, redraw=True)
        
  def doAutoScale(self, axis='x', redraw=True):
    # performs autoscale on axis
    if(axis in ['x', 'y', 'y2', 'x2', 'resid']):
      currData, currRoles = self.parent.data[self.parent.activeData].getData_n_Fit()
      # we have some data that we could zoom to
      if(axis in ['x', 'x2']):
        if(axis == 'x'):
          useModeX, splitx = self.modeX, False
        else:
          useModeX, splitx = self.modeX_div, True
          
        # consider all data sets
        xval = np.array([])
        for dataset in self.parent.data:
          if(((dataset == self.parent.data[self.parent.activeData]) or (dataset.visibility)) and (dataset.onBothAxes or (dataset.onSecondAxes == splitx))):
            values, roles = dataset.getData_n_Fit()
            if('x' in roles):
              datax = list(values[:, roles.index('x')])
              if('xerr' in roles):
                dataxerr = list(values[:, roles.index('xerr')])
                xval = np.hstack((xval, [i - j for i, j in zip(datax, dataxerr)]))
                xval = np.hstack((xval, [i + j for i, j in zip(datax, dataxerr)]))
              else:
                xval = np.hstack((xval, datax))
        xval = np.array([i for i in xval if ((not np.isnan(i)) and (not np.isinf(i)))])

        if(len(xval)):
          temp_xmin = np.min(xval)
          temp_xmax = np.max(xval)

          # check for split x axis
          if(self.splitShow):
            if(axis == 'x'):
              temp_xmax = np.min((temp_xmax, self.maxX_div, self.minX_div))
            else:
              temp_xmin = np.max((temp_xmin, self.maxX, self.minX))
            
          # ensure minimum limit
          if(temp_xmax - temp_xmin < self.EPSILON):
            temp_xmax += self.EPSILON; temp_xmin -= self.EPSILON
          elif(self.data_spacer > 0):
            if(useModeX == 'linear'):
              difference = temp_xmax - temp_xmin
              if((not self.splitShow) or (axis == 'x2')):
                temp_xmax += difference * self.data_spacer
              if((not self.splitShow) or (axis == 'x')):
                temp_xmin -= difference * self.data_spacer
            else:
              # log scale -- isolate positive data
              pos_x = np.array(xval)
              pos_x = pos_x[pos_x > 0]
              pos_x = np.array([i for i in pos_x if ((not np.isnan(i)) and (not np.isinf(i)))])
              if(len(pos_x > 1)):
                # recalc. xmin to address error when restoring state
                temp_xmin, temp_xmax = np.min(pos_x), np.max(pos_x)
                if(self.splitShow and (axis == 'x2') and (np.max((self.minX, self.maxX)) > 0)):
                  posLimits = [i for i  in [self.minX, self.maxX] if (i > 0)]
                  temp_xmin = np.max((temp_xmin, np.max(posLimits)))
                if(self.splitShow and (axis == 'x') and (np.max((self.minX_div, self.maxX_div)) > 0)):
                  posLimits = [i for i  in [self.minX_div, self.maxX_div] if (i > 0)]
                  temp_xmax = np.min((temp_xmax, np.min(posLimits)))
                difference = np.log(pos_x[-1] / pos_x[0])
                if((not self.splitShow) or (axis == 'x')):
                  temp_xmin = np.exp(np.log(temp_xmin) - self.data_spacer * difference)
                if((not self.splitShow) or (axis == 'x2')):
                  temp_xmax = np.exp(np.log(temp_xmax) + self.data_spacer * difference)
          
          if(axis == 'x'):
            self.setAxisLimits(lower=temp_xmin, upper=temp_xmax, axis='x', updateLabel=True, target='plot', redraw=False, updateGrid=True)
            self.setAxisLimits(lower=temp_xmin, upper=temp_xmax, axis='x', updateLabel=True, target='resid', redraw=False, updateGrid=True)
            # plot current function over new x-range
            self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.parent.plotArea.plotFunction(\
              fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot,\
              handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False, doAutoZoom=False)
            # copy in case split axes are shown
            curve = self.parent.fit[self.parent.activeFit]
            if(self.parent.plotArea.splitY and curve.onBothAxes):
              curve.duplicateForSplit()
            if(redraw):
              self.parent.plotArea.dataplotwidget.myRefresh()
            # plot zeroResid line over new x-range
            self.parent.plotArea.handleResidZero = self.parent.plotArea.plotResidZero(\
              handleResidZero=self.parent.plotArea.handleResidZero, redraw=redraw)
          else:
            self.setAxisLimits(lower=temp_xmin, upper=temp_xmax, axis='x2', updateLabel=True, target='plot', redraw=False, updateGrid=True)
            self.setAxisLimits(lower=temp_xmin, upper=temp_xmax, axis='x2', updateLabel=True, target='resid', redraw=False, updateGrid=True)
            # plot current function over new x-range
            self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.parent.plotArea.plotFunction(\
              fitobject=self.parent.fit[self.parent.activeFit], x=[], handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div,\
              handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True, doAutoZoom=False)
            # copy in case split axes are shown
            curve = self.parent.fit[self.parent.activeFit]
            if(self.parent.plotArea.splitY and curve.onBothAxes):
              curve.duplicateForSplit()
            if(redraw):
              self.parent.plotArea.dataplotwidget.myRefresh()
            # plot zeroResid line over new x-range
            self.parent.plotArea.handleResidZero_div = self.parent.plotArea.plotResidZero(\
              handleResidZero=self.parent.plotArea.handleResidZero_div, redraw=redraw, splitX=True)
          # update tick entry fields
          self.updateTickEntryField(axis=axis)
      elif(axis in ['y', 'y2']):
        # only consider active function and data if on current y axis
        if(axis == 'y'):
          if(self.splitShow):
            funcY = np.hstack((self.y, self.y_div))
          else:
            funcY = self.y
        else:
          if(self.splitShow):
            funcY = np.hstack((self.y2, self.y2_div))
          else:
            funcY = self.y2
        if(self.splitY and self.parent.fit[self.parent.activeFit].onBothAxes):
          if(self.splitShow):
            funcY = np.hstack((self.y, self.y_div, self.y2, self.y2_div))
          else:
            funcY = np.hstack((self.y, self.y2))
        # check whether curve is visible at all
        if(not self.parent.fit[self.parent.activeFit].visibility):
          funcY = []

        # consider all data sets
        temp_y = np.array([])
        for dataset in self.parent.data:
          if(dataset.visibility and (dataset.onBothAxes or (dataset.onSecondAxes == (axis == 'y2')))):
            values, roles = dataset.getData_n_Fit()
            if('y' in roles):
              datay = list(values[:, roles.index('y')])
              if('yerr' in roles):
                datayerr = list(values[:, roles.index('yerr')])
                temp_y = np.hstack((temp_y, [i - j for i, j in zip(datay, datayerr)]))
                temp_y = np.hstack((temp_y, [i + j for i, j in zip(datay, datayerr)]))
              else:
                temp_y = np.hstack((temp_y, datay))
        # check whether any data present/visible
        if(not len(temp_y)):
          temp_y = funcY
          
        temp_y = np.array([i for i in temp_y if ((not np.isnan(i)) and (not np.isinf(i)))])
        # ensure that temp_y is at least one element long
        if(temp_y.size):
          temp_ymin = np.min(temp_y)
          temp_ymax = np.max(temp_y)
          
          # check for split y axis
          if(self.splitY):
            if(axis == 'y'):
              temp_ymin = np.max((temp_ymin, self.maxY2, self.minY2))
            else:
              temp_ymax = np.min((temp_ymax, self.maxY, self.minY))
  
          # ensure minimum limit
          if (temp_ymax - temp_ymin < self.EPSILON):
            temp_ymax += self.EPSILON; temp_ymin -= self.EPSILON
          elif(self.data_spacer > 0):
            if(axis == 'y'):
              useMode = self.modeY
            else:
              useMode = self.modeY2
            if(useMode == 'linear'):
              difference = temp_ymax - temp_ymin
              if((not self.splitY) or (axis == 'y')):
                temp_ymax += difference * self.data_spacer
              if((not self.splitY) or (axis == 'y2')):
                temp_ymin -= difference * self.data_spacer
            else:
              # log scale -- isolate positive data
              pos_y = np.array(temp_y)
              pos_y = pos_y[pos_y > 0]
              if(len(pos_y > 1)):
                temp_ymin, temp_ymax = np.min(pos_y), np.max(pos_y)
                if(self.splitY and (axis == 'y') and (np.max((self.minY2, self.maxY2)) > 0)):
                  posLimits = [i for i  in [self.minY2, self.maxY2] if (i > 0)]
                  temp_ymin = np.max((temp_ymin, np.max(posLimits)))
                if(self.splitY and (axis == 'y2') and (np.max((self.minY, self.maxY)) > 0)):
                  posLimits = [i for i  in [self.minY, self.maxY] if (i > 0)]
                  temp_ymax = np.min((temp_ymax, np.min(posLimits)))
                difference = np.log(temp_ymax / temp_ymin)
                if((not self.splitY) or (axis == 'y2')):
                  temp_ymin = np.exp(np.log(temp_ymin) - self.data_spacer * difference)
                if((not self.splitY) or (axis == 'y')):
                  temp_ymax = np.exp(np.log(temp_ymax) + self.data_spacer * difference)
          
          self.setAxisLimits(lower = temp_ymin, upper = temp_ymax, axis = axis, updateLabel = True, target='plot', redraw=redraw, updateGrid=True)
          # update tick entry fields
          self.updateTickEntryField(axis=axis)
      else:
        # consider all data sets
        resid = np.array([])
        for dataset in self.parent.data:
          if(dataset.visibilityResid):
            values, roles = dataset.getData_n_Fit()
            if('resid' in roles):
              dataresid = list(values[:, roles.index('resid')])
              resid = np.hstack((resid, dataresid))

        # ensure that temp_y is at least one element long
        if(resid.size):
          temp_residmin = np.min(resid)
          temp_residmax = np.max(resid)
  
          # ensure minimum limit
          if (temp_residmax - temp_residmin < self.EPSILON):
            temp_residmax += self.EPSILON; temp_residmin -= self.EPSILON
          else:
            # this needs to correspond to the setting in doFit()
            usespacer = 1.2
            temp_residmax *= usespacer
            temp_residmin *= usespacer
          # ensure symmetric limits
          uselimit = np.max((np.abs(temp_residmax), np.abs(temp_residmin)))
          temp_residmin, temp_residmax = -1. * uselimit, 1. * uselimit
          
          self.setAxisLimits(lower=temp_residmin, upper=temp_residmax, axis='y', updateLabel=True, target='resid', redraw=redraw, updateGrid=True)
          # update tick entry fields
          self.updateTickEntryField(axis='resid')

  def flipAxis(self, axis='x', redraw=True):
    # swaps upper and lower limit of axis
    if(axis in ['x', 'x2', 'y', 'y2']):
      if(axis == 'x'):
        self.minX, self.maxX = self.maxX, self.minX
        # better to swap actual text and thus avoid reformatting of entered text (avoids rounding issues)
        lowerText, upperText = self.lowerLimitx.text(), self.upperLimitx.text()
        self.upperLimitx.setText(lowerText)
        self.lowerLimitx.setText(upperText)
        # update axis
        self.ax2.set_xlim([self.minX, self.maxX])
        self.ax.set_xlim([self.minX, self.maxX])
        self.ax_resid.set_xlim([self.minX, self.maxX])
        updateResid = True
      elif(axis == 'x2'):
        self.minX_div, self.maxX_div = self.maxX_div, self.minX_div
        # better to swap actual text and thus avoid reformatting of entered text (avoids rounding issues)
        lowerText, upperText = self.xSplitLowerLimitx.text(), self.xSplitUpperLimitx.text()
        self.xSplitUpperLimitx.setText(lowerText)
        self.xSplitLowerLimitx.setText(upperText)
        # update axis
        self.ax2_div.set_xlim([self.minX_div, self.maxX_div])
        self.ax_div.set_xlim([self.minX_div, self.maxX_div])
        self.ax_resid_div.set_xlim([self.minX_div, self.maxX_div])
        updateResid = True
      elif(axis == 'y'):
        self.minY, self.maxY = self.maxY, self.minY
        # better to swap actual text and thus avoid reformatting of entered text (avoids rounding issues)
        lowerText, upperText = self.lowerLimity.text(), self.upperLimity.text()
        self.upperLimity.setText(lowerText)
        self.lowerLimity.setText(upperText)
        # update axis
        self.ax.set_ylim([self.minY, self.maxY])
        self.ax_div.set_ylim([self.minY, self.maxY])
        updateResid = False
      else:
        self.minY2, self.maxY2 = self.maxY2, self.minY2
        # better to swap actual text and thus avoid reformatting of entered text (avoids rounding issues)
        lowerText, upperText = self.secondLowerLimit.text(), self.secondUpperLimit.text()
        self.secondUpperLimit.setText(lowerText)
        self.secondLowerLimit.setText(upperText)
        # update axis
        self.ax2.set_ylim([self.minY2, self.maxY2])
        self.ax2_div.set_ylim([self.minY2, self.maxY2])
        updateResid = False
        
      # refresh the plot(s)
      if(redraw):
        self.dataplotwidget.myRefresh()
        if(updateResid):
          self.residplotwidget.myRefresh()

  def lassoCallback(self, vertices):
    # call back for lasso selector function
    
    # as a precaution hide our lassoli
    self.lassoSelector.cleanUp()
        
    # then take care of the actual fallback
    if(self.pick2Freehand or (self.pickedMode == 3)):
      self.pick2Freehand = False
      # generate a new extras object
      self.parent.extras.append(ExtrasObject(self.parent))
      valueDict = {'labeltext': 'Freehand', 'extrasType': 'shape', 'shape__type': 'freehand', 'shape__vertices': vertices, 'shape__doFill': False}
      self.parent.extras[-1].setValues(valueDict, redraw=False)
      # deal with duplication and update of menus
      if(self.splitY):
        self.parent.extras[-1].onBothAxes = True
        self.parent.extras[-1].duplicateForSplit()
      self.parent.extras[-1].drawMe(redraw=True)
      # update extras table
      self.parent.objectsarea.refreshExtrasTable()
      self.parent.objectsarea.refreshCurvesTable()
      self.parent.objectsarea.refreshDataTable()
      self.parent.plotArea.updateToggloContainer()
    else:
      # need to check if shift key was pressed at time of initial click
      if(self.toggleShift):
        # check whether current object is on second axes
        if(self.parent.data[self.parent.activeData].onSecondAxes and (not self.splitY)):
          # do we need to check for regular vs. split x axes?
          # => tests show that not which comes as a surprise
          tempVertices = self.ax.transData.transform(vertices)
          transformedVertices = self.ax2.transData.inverted().transform(tempVertices)
          lassoPath = matplotlib.path.Path(transformedVertices)
        else:
          lassoPath = matplotlib.path.Path(vertices)
        values, roles = self.parent.data[self.parent.activeData].getData_n_Fit()
        if(('x' in roles) and ('y' in roles) and len(values[roles.index('x')])):
          xyData = [(x, y) for x, y in zip(values[:, roles.index('x')], values[:, roles.index('y')])]
          if(len(xyData)):
            selectData = lassoPath.contains_points(xyData)
            self.parent.selectedData.mouseSelectIndices(selectData)
    
    # destroy lasso
    del self.lassoSelector
    self.lassoSelector = None
    
  def rectangleCallback(self, vertices):
    # call back for rectangle selector function
    # determine which axis we are on
    inaxes = self.rectangleSelector.ax
    if(inaxes in [self.ax, self.ax2]):
      divAxis = False
    else:
      divAxis = True
      
    # get extents of rectangle selector
    start, end = self.rectangleSelector.getMyExtents()
    if(start != None):
      x1, y1 = start
      x2, y2 = end
      
      if((np.abs(x1 - x2) > 0) and (np.abs(y1 - y2) > 0)):
        xmin, xmax = np.min((x1, x2)), np.max((x1, x2))
        ymin, ymax = np.min((y1, y2)), np.max((y1, y2))
        
        # now assign these values correctly
        if(inaxes in [self.ax, self.ax2]):
          minX, maxX = xmin, xmax
          minX_div, maxX_div = self.minX_div, self.maxX_div
        else:
          minX, maxX = self.minX, self.maxX
          minX_div, maxX_div = xmin, xmax
          
        if(inaxes in [self.ax, self.ax_div]):
          minY, maxY = ymin, ymax
          # check for split y axes
          if(not self.splitY):
            # determine event x and y & transform to second axes
            event_x, event_y = self.ax.transData.transform((xmin, ymin))
            xmin2, ymin2 = self.ax2.transData.inverted().transform((event_x, event_y))
            # et encore une fois
            event_x, event_y = self.ax.transData.transform((xmax, ymax))
            xmax2, ymax2 = self.ax2.transData.inverted().transform((event_x, event_y))
            # and assign values
            minY2, maxY2 = ymin2, ymax2
          else:
            minY2, maxY2 = self.minY2, self.maxY2
        else:
          minY, maxY = self.minY, self.maxY
          minY2, maxY2 = ymin, ymax

        # store current axis limits
        self.storeCoord.extend((self.minX, self.maxX, self.minY, self.maxY, self.minX_div, self.maxX_div, self.minY2, self.maxY2))

        coords = [minX, maxX, minY, maxY, minX_div, maxX_div, minY2, maxY2]
        self.rectSelectorCallbackHelper(coords=coords, divAxis=divAxis, restore=False)
   
    # as a precaution hide our rectangli
    self.rectangleSelector.cleanUp()

    # destroy rectangle
    del self.rectangleSelector
    self.rectangleSelector = None

  def rectSelectorCallbackHelper(self, coords=[], divAxis=False, restore=False):
    # utility function to handle zoom and unzoom
    # this became necessary as later matplotlib versions trigger events differently
    # have to capture right click (for unzoom) already in toggleCrosshairEncore()
    if(len(coords) == 8):
      # only continue if axes limits can be assigned
      [xmin, xmax, ymin, ymax, xmin_div, xmax_div, secondYmin, secondYmax] = coords
        
      if(divAxis or restore):
        # only change the x axis when really required
        if((xmin_div != self.minX_div) or (xmax_div != self.maxX_div)):
          self.setAxisLimits(lower=xmin_div, upper=xmax_div, axis='x2', updateLabel=True, target='plot', redraw=False, updateGrid=True)
          self.setAxisLimits(lower=xmin_div, upper=xmax_div, axis='x2', updateLabel=False, target='resid', redraw=False, updateGrid=True)
          self.updateTickEntryField(axis='x2')
      if(not divAxis or restore):
        # only change the x axis when really required
        if((xmin != self.minX) or (xmax != self.maxX)):
          self.setAxisLimits(lower=xmin, upper=xmax, axis='x', updateLabel=True, target='plot', redraw=False, updateGrid=True)
          self.setAxisLimits(lower=xmin, upper=xmax, axis='x', updateLabel=False, target='resid', redraw=False, updateGrid=True)
          self.updateTickEntryField(axis='x')
      # trigger redrawing of fit function with new axis limits
      if(divAxis or restore):
        self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.plotFunction(fitobject=self.parent.fit[self.parent.activeFit], x=[],\
                       handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div, handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True, doAutoZoom=False)
        # copy in case split axes are shown
        curve = self.parent.fit[self.parent.activeFit]
        if(self.parent.plotArea.splitY and curve.onBothAxes):
          curve.duplicateForSplit()
        self.handleResidZero_div = self.plotResidZero(self.handleResidZero_div, redraw=not restore, splitX=True)
      if(not divAxis or restore):
        self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.plotFunction(fitobject=self.parent.fit[self.parent.activeFit], x=[],\
                       handlePlot=self.parent.fit[self.parent.activeFit].handlePlot, handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False, doAutoZoom=False)
        # copy in case split axes are shown
        curve = self.parent.fit[self.parent.activeFit]
        if(self.parent.plotArea.splitY and curve.onBothAxes):
          curve.duplicateForSplit()
        self.handleResidZero = self.plotResidZero(self.handleResidZero, redraw=True)
  
      # reset y axis
      self.setAxisLimits(lower=secondYmin, upper=secondYmax, axis='y2', updateLabel=True, target='plot', redraw=False, updateGrid=True)
      self.setAxisLimits(lower=ymin, upper=ymax, axis='y', updateLabel=True, target='plot', redraw=True, updateGrid=True)
      self.updateTickEntryField(axis='y')
      self.updateTickEntryField(axis='y2')

  def setPathStroke(self, state=True, redraw=True):
    # applies path stroke
    self.applyPathStroke = state
    self.setPathEffects(redraw=redraw)

  def setPathStrokeColor(self, value=4*[0.0], redraw=True):
    # changes color of path stroke
    self.pathStrokeColor = value
    # do the path setting draw?
    if((redraw) and (self.applyPathStroke)):
      self.setPathEffects(redraw=redraw)

  def setPathStrokeWidth(self, value=1.0, redraw=True):
    # changes witdth of path stroke
    self.pathStrokeWidth = value
    # do the path setting draw?
    if((redraw) and (self.applyPathStroke)):
      self.setPathEffects(redraw=redraw)

  def setPathShadow(self, state=True, redraw=True):
    # applies path shadow
    self.applyPathShadow = state
    self.setPathEffects(redraw=redraw)

  def setPathShadowColor(self, value=4*[0.0], redraw=True):
    # changes color of path shadow
    self.pathShadowColor = value
    self.pathShadowAlpha = value[-1]
    # do the path setting draw?
    if((redraw) and (self.applyPathShadow)):
      self.setPathEffects(redraw=redraw)

  def setPathShadowOffset(self, value=1.0, direction='x', redraw=True):
    # changes offset of path shadow
    if(direction in ['x', 'y']):
      if(direction == 'x'):
        self.pathShadowX = value
      else:
        self.pathShadowY = value
      # do the path setting draw?
      if((redraw) and (self.applyPathShadow)):
        self.setPathEffects(redraw=redraw)

  def setPathEffects(self, redraw=True):
    # applies path effects
    if(self.applyPathStroke):
      baseEffect = []
    else:
      baseEffect = [PathEffects.Normal()]

    if(self.applyPathShadow):
      pathShadowX = self.pathShadowX
      pathShadowY = self.pathShadowY
      pathShadowColor = self.pathShadowColor
      pathShadowAlpha = self.pathShadowAlpha
      baseEffect = [PathEffects.SimpleLineShadow(offset=(pathShadowX, pathShadowY), shadow_color=pathShadowColor,\
        alpha=pathShadowAlpha)] + baseEffect
      baseEffectText = [PathEffects.SimplePatchShadow(offset=(pathShadowX, pathShadowY), shadow_rgbFace=pathShadowColor,\
        alpha=pathShadowAlpha)] + baseEffect
    else:
      baseEffectText = baseEffect
    
    # deal with existing objects
    if(self.applyPathStroke):
      pathStrokeWidth = self.pathStrokeWidth
      pathStrokeColor = self.pathStrokeColor

      # modify existing extras
      for entry in self.parent.extras:
        relevantHandles = [entry.handle, entry.handle_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handle', 'handle_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleExtra in relevantHandles:
          # exclude area objects from path effects => throws tuple/list error
          if((handleExtra != None) and (entry.extrasType != 'area')):
            if(hasattr(handleExtra, 'get_lw')):
              curr_linewidth = 2 * pathStrokeWidth + handleExtra.get_lw()
            else:
              curr_linewidth = 2 * pathStrokeWidth + 1
            # check whether we have a label here
            if(hasattr(handleExtra, 'get_text')):
              handleExtra.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            else:
              handleExtra.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            # set bbox if present
            if(hasattr(handleExtra, 'get_bbox_patch')):
              handle = handleExtra.get_bbox_patch()
              if(handle != None):
                curr_linewidth = 2 * pathStrokeWidth + handle.get_lw()
                handle.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            # set arrow patch if present
            if(hasattr(handleExtra, 'arrow_patch')):
              handle = handleExtra.arrow_patch
              if(handle != None):
                curr_linewidth = 2 * pathStrokeWidth + handle.get_lw()
                handle.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      # modify existing curves
      for entry in self.parent.fit:
        relevantHandles = [entry.handlePlot, entry.handlePlot_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handlePlot', 'handlePlot_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handlePlot in relevantHandles:
          if(handlePlot != None):
            curr_linewidth = 2 * pathStrokeWidth + handlePlot.get_lw()
            handlePlot.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
  
      # modify existing data sets
      for entry in self.parent.data:
        relevantHandles = [entry.handleData, entry.handleData_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleData', 'handleData_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleData in relevantHandles:
          if(handleData != None):
            curr_linewidth = 2 * pathStrokeWidth + handleData.get_lw()
            handleData.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            
        for handleResid in [entry.handleResid, entry.handleResid_div]:
          if(handleResid != None):
            curr_linewidth = 2 * pathStrokeWidth + handleResid.get_lw()
            handleResid.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      ### leave out scatter plot for now as patheffects invariably throw an error here
#      for entry in self.parent.data:
#        relevantHandles = [entry.handleScatter, entry.handleScatter_div]
#        if(self.splitY):
#          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleScatter', 'handleScatter_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
#        for handleScatter in relevantHandles:
#          if(handleScatter != None):
#            curr_linewidth = 2 * pathStrokeWidth + handleScatter.get_lw()[0]
#            handleScatter.set_path_effects([PathEffects.withStroke(linewidth=5., foreground=pathStrokeColor)])
                    
        relevantHandles = [entry.handleBar, entry.handleBar_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleBar', 'handleBar_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleBar in relevantHandles:
          if(handleBar != None):
            children = handleBar.get_children()
            for entry2 in children:
              curr_linewidth = 2 * pathStrokeWidth + entry2.get_lw()
              entry2.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
        # don't apply to stack style, throws strange error
        #if(entry.handleStack != None):
          #entry.handleStack.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

        for handleResidBar in [entry.handleResidBar, entry.handleResidBar_div]:
          if(handleResidBar != None):
            children = handleResidBar.get_children()
            for entry2 in children:
              curr_linewidth = 2 * pathStrokeWidth + entry2.get_lw()
              entry2.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
        # don't apply to stack style, throws strange error
        #if(entry.handleStack != None):
          #entry.handleStack.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

        relevantHandles = [entry.handleErr, entry.handleErr_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleErr', 'handleErr_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleErr in relevantHandles:
          if(handleErr != None):
            curr_linewidth = 2 * pathStrokeWidth + handleErr[0].get_lw()
            handleErr[0].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            for entry2 in handleErr[1]:
              # remember caps are drawn as markers not lines
              if(entry2.get_markeredgewidth() > 0):
                curr_linewidth = 2 * pathStrokeWidth + entry2.get_markeredgewidth()
                entry2.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
              else:
                # don't draw effects for zero width markers
                #entry2.set_path_effects(baseEffect)
                # in this case don't do anything since setting path effects would cause drawing of caps (w/ currently set path effects)
                pass
            for entry2 in handleErr[2]:
              # have to use get_linewidth() here as get_lw() not implemented?!
              curr_linewidth = 2 * pathStrokeWidth + entry2.get_linewidth()
              entry2.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

        # now handle violin and box plots
        relevantHandles = [entry.handleViolin, entry.handleViolin_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleViolin', 'handleViolin_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleViolin in relevantHandles:
          if(handleViolin != None):
            # bodies throws a crash when applying path effects
            validItems = ['cmins', 'cmaxes', 'cmeans', 'cmedians', 'cbars', 'boxes', 'medians', 'whiskers', 'caps', 'fliers']
            for subtarget in [i for i in handleViolin if i in validItems]:
              if(type(handleViolin[subtarget]) == type([])):
                itemList = handleViolin[subtarget]
              else:
                itemList = [handleViolin[subtarget]]
              # cycle through all items
              for item in itemList:
                if(hasattr(item, 'get_linewidth')):
                  curr_linewidth = 2 * pathStrokeWidth + item.get_linewidth()
                else:
                  curr_linewidth = 2 * pathStrokeWidth + 1.0
                if(hasattr(item, 'set_path_effects')):
                  item.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

        # now handle the text labels
        relevantHandles = [entry.handleText, entry.handleText_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleText', 'handleText_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleText in relevantHandles:
          if(handleText != None):
            for item in handleText:
              if(hasattr(item, 'get_linewidth')):
                curr_linewidth = 2 * pathStrokeWidth + item.get_linewidth()
              else:
                curr_linewidth = 2 * pathStrokeWidth + 1.0
              if(hasattr(item, 'set_path_effects')):
                item.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
       
      # modify axes
      for entry in self.ax.spines:
        if(entry in ['left', 'right']):
          curr_linewidth = 2 * pathStrokeWidth + self.ax.spines[entry].get_lw()
          self.ax.spines[entry].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in self.ax2.spines:
        curr_linewidth = 2 * pathStrokeWidth + self.ax2.spines[entry].get_lw()
        self.ax2.spines[entry].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in self.ax_resid.spines:
        curr_linewidth = 2 * pathStrokeWidth + self.ax_resid.spines[entry].get_lw()
        self.ax_resid.spines[entry].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in self.ax_div.spines:
        if(entry in ['left', 'right']):
          curr_linewidth = 2 * pathStrokeWidth + self.ax_div.spines[entry].get_lw()
          self.ax_div.spines[entry].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in self.ax2_div.spines:
        curr_linewidth = 2 * pathStrokeWidth + self.ax2_div.spines[entry].get_lw()
        self.ax2_div.spines[entry].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in self.ax_resid_div.spines:
        curr_linewidth = 2 * pathStrokeWidth + self.ax_resid_div.spines[entry].get_lw()
        self.ax_resid_div.spines[entry].set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      # modify grid lines
      for key in self.gridLinesStore:
        if(self.gridVisible[key]):
          for entry in self.gridLinesStore[key]:
            curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
            entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
        if(key in self.gridLinesStore_resid):
          for entry in self.gridLinesStore_resid[key]:
            curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
            entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            
      # modify tick lines
      #for entry in self.ax.xaxis.get_ticklines():
      #  curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
      #  entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in (self.ax.yaxis.get_ticklines() + self.ax.yaxis.get_ticklines(minor=True) + self.ax_div.yaxis.get_ticklines() + self.ax_div.yaxis.get_ticklines(minor=True)):
        curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
        entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in (self.ax_resid.xaxis.get_ticklines() + self.ax_resid.xaxis.get_ticklines(minor=True) + self.ax_resid_div.xaxis.get_ticklines() + self.ax_resid_div.xaxis.get_ticklines(minor=True)):
        curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
        entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in (self.ax_resid.yaxis.get_ticklines() + self.ax_resid.yaxis.get_ticklines(minor=True) + self.ax_resid_div.yaxis.get_ticklines() + self.ax_resid_div.yaxis.get_ticklines(minor=True)):
        curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
        entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in (self.ax2.xaxis.get_ticklines() + self.ax2.xaxis.get_ticklines(minor=True) + self.ax2_div.xaxis.get_ticklines() + self.ax2_div.xaxis.get_ticklines(minor=True)):
        curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
        entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      for entry in (self.ax2.yaxis.get_ticklines() + self.ax2.yaxis.get_ticklines(minor=True) + self.ax2_div.yaxis.get_ticklines() + self.ax2_div.yaxis.get_ticklines(minor=True)):
        curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
        entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
  
      # zero line
      for handleResidZero in [self.handleResidZero, self.handleResidZero_div]:
        if(handleResidZero != None):
          curr_linewidth = 2 * pathStrokeWidth + handleResidZero.get_lw()
          handleResidZero.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      # and the cursor
      for cursor in [self.cursor, self.cursor_div, self.cursor2, self.cursor2_div]:
        if(cursor != None):
          handles = cursor.getHandles()
          for entry in handles:
            if(hasattr(entry, 'get_lw')):
              curr_linewidth = 2 * pathStrokeWidth + entry.get_lw()
            else:
              curr_linewidth = 2 * pathStrokeWidth + 1
            # check for text
            if(hasattr(entry, 'get_text')):
              entry.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
            else:
              entry.set_path_effects(baseEffect + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      # and the axes labels
      curr_linewidth = 2 * pathStrokeWidth + 1
      for entry in [self.ax2, self.ax_resid, self.ax2_div, self.ax_resid_div]:
        entry.xaxis.label.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
        entry.yaxis.label.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      self.ax.yaxis.label.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])
      self.ax_div.yaxis.label.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      # and the tick labels
      curr_linewidth = 2 * pathStrokeWidth + 1
      tickLabels = []
      for entry in [self.ax, self.ax_resid, self.ax2, self.ax_div, self.ax_resid_div, self.ax2_div]:
        tickLabels.extend(entry.get_xticklabels(which='both'))
        tickLabels.extend(entry.get_yticklabels(which='both'))
        tickLabels.append(entry.xaxis.get_offset_text())
        tickLabels.append(entry.yaxis.get_offset_text())
      for entry in tickLabels:
        entry.set_path_effects(baseEffectText + [PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor)])

      # update baseEffect for new plot items
      curr_linewidth = 2 * pathStrokeWidth + 1.0
      baseEffect.append(PathEffects.withStroke(linewidth=curr_linewidth, foreground=pathStrokeColor))
    else:
      # check for complete turnoff of pathEffects for text labels
      if((not self.applyPathStroke) and (not self.applyPathShadow)):
        baseEffectText = []

      # modify existing extras
      for entry in self.parent.extras:
        relevantHandles = [entry.handle, entry.handle_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handle', 'handle_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleExtra in relevantHandles:
          # exclude area objects from path effects => throws tuple/list error
          if((handleExtra != None) and (entry.extrasType != 'area')):
            # check whether we have a label here
            if(hasattr(handleExtra, 'get_text')):
              handleExtra.set_path_effects(baseEffectText)
            else:
              handleExtra.set_path_effects(baseEffect)
            # set bbox if present
            if(hasattr(handleExtra, 'get_bbox_patch')):
              handle = handleExtra.get_bbox_patch()
              if(handle != None):
                handle.set_path_effects(baseEffect)
            # set arrow patch if present
            if(hasattr(handleExtra, 'arrow_patch')):
              handle = handleExtra.arrow_patch
              if(handle != None):
                handle.set_path_effects(baseEffect)

      # modify existing curves
      for entry in self.parent.fit:
        relevantHandles = [entry.handlePlot, entry.handlePlot_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handlePlot', 'handlePlot_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handlePlot in relevantHandles:
          if(handlePlot != None):
            handlePlot.set_path_effects(baseEffect)
  
      # modify existing data sets
      for entry in self.parent.data:
        relevantHandles = [entry.handleData, entry.handleData_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleData', 'handleData_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleData in relevantHandles:
          if(handleData != None):
            handleData.set_path_effects(baseEffect)
            
        for handleResid in [entry.handleResid, entry.handleResid_div]:
          if(handleResid != None):
            handleResid.set_path_effects(baseEffect)
            
        ### leave out scatter plot for now as patheffects invariably throw an error here
            
        relevantHandles = [entry.handleBar, entry.handleBar_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleBar', 'handleBar_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleBar in relevantHandles:
          if(handleBar != None):
            children = handleBar.get_children()
            for entry2 in children:
              entry2.set_path_effects(baseEffect)

        # don't apply to stack style, throws strange error
        for handleResidBar in [entry.handleResidBar, entry.handleResidBar_div]:
          if(handleResidBar != None):
            children = handleResidBar.get_children()
            for entry2 in children:
              entry2.set_path_effects(baseEffect)

        # don't apply to stack style, throws strange error
        relevantHandles = [entry.handleErr, entry.handleErr_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleErr', 'handleErr_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleErr in relevantHandles:
          if(handleErr != None):
            handleErr[0].set_path_effects(baseEffect)
            for entry2 in handleErr[1]:
              entry2.set_path_effects(baseEffect)
            for entry2 in handleErr[2]:
              entry2.set_path_effects(baseEffect)

        # now handle violin and box plots
        relevantHandles = [entry.handleViolin, entry.handleViolin_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleViolin', 'handleViolin_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleViolin in relevantHandles:
          if(handleViolin != None):
            # bodies throws a crash when applying path effects
            validItems = ['cmins', 'cmaxes', 'cmeans', 'cmedians', 'cbars', 'boxes', 'medians', 'whiskers', 'caps', 'fliers']
            for subtarget in [i for i in handleViolin if i in validItems]:
              if(type(handleViolin[subtarget]) == type([])):
                itemList = handleViolin[subtarget]
              else:
                itemList = [handleViolin[subtarget]]
              # cycle through all items
              for item in itemList:
                if(hasattr(item, 'set_path_effects')):
                  item.set_path_effects(baseEffect)

        # now handle the text labels
        relevantHandles = [entry.handleText, entry.handleText_div]
        if(self.splitY):
          relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleText', 'handleText_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
        for handleText in relevantHandles:
          if(handleText != None):
            for item in handleText:
              if(hasattr(item, 'set_path_effects')):
                item.set_path_effects(baseEffectText)
       
      # modify axes
      for entry in self.ax.spines:
        if(entry in ['left', 'right']):
          self.ax.spines[entry].set_path_effects(baseEffect)
      for entry in self.ax2.spines:
        self.ax2.spines[entry].set_path_effects(baseEffect)
      for entry in self.ax_resid.spines:
        self.ax_resid.spines[entry].set_path_effects(baseEffect)
      for entry in self.ax_div.spines:
        if(entry in ['left', 'right']):
          self.ax_div.spines[entry].set_path_effects(baseEffect)
      for entry in self.ax2_div.spines:
        self.ax2_div.spines[entry].set_path_effects(baseEffect)
      for entry in self.ax_resid_div.spines:
        self.ax_resid_div.spines[entry].set_path_effects(baseEffect)

      # modify grid lines
      for key in self.gridLinesStore:
        if(self.gridVisible[key]):
          for entry in self.gridLinesStore[key]:
            entry.set_path_effects(baseEffect)
        if(key in self.gridLinesStore_resid):
          for entry in self.gridLinesStore_resid[key]:
            entry.set_path_effects(baseEffect)

      # modify tick lines
      for entry in (self.ax.yaxis.get_ticklines() + self.ax.yaxis.get_ticklines(minor=True) + self.ax_div.yaxis.get_ticklines() + self.ax_div.yaxis.get_ticklines(minor=True)):
        entry.set_path_effects(baseEffect)
      for entry in (self.ax_resid.xaxis.get_ticklines() + self.ax_resid.xaxis.get_ticklines(minor=True) + self.ax_resid_div.xaxis.get_ticklines() + self.ax_resid_div.xaxis.get_ticklines(minor=True)):
        entry.set_path_effects(baseEffect)
      for entry in (self.ax_resid.yaxis.get_ticklines() + self.ax_resid.yaxis.get_ticklines(minor=True) + self.ax_resid_div.yaxis.get_ticklines() + self.ax_resid_div.yaxis.get_ticklines(minor=True)):
        entry.set_path_effects(baseEffect)
      for entry in (self.ax2.xaxis.get_ticklines() + self.ax2.xaxis.get_ticklines(minor=True) + self.ax2_div.xaxis.get_ticklines() + self.ax2_div.xaxis.get_ticklines(minor=True)):
        entry.set_path_effects(baseEffect)
      for entry in (self.ax2.yaxis.get_ticklines() + self.ax2.yaxis.get_ticklines(minor=True) + self.ax2_div.yaxis.get_ticklines() + self.ax2_div.yaxis.get_ticklines(minor=True)):
        entry.set_path_effects(baseEffect)

      # zero line
      for handleResidZero in [self.handleResidZero, self.handleResidZero_div]:
        if(handleResidZero != None):
          handleResidZero.set_path_effects(baseEffect)
        
      # and the axes labels
      for entry in [self.ax2, self.ax_resid, self.ax2_div, self.ax_resid_div]:
        entry.xaxis.label.set_path_effects(baseEffectText)
        entry.yaxis.label.set_path_effects(baseEffectText)
      self.ax.yaxis.label.set_path_effects(baseEffectText)
      self.ax_div.yaxis.label.set_path_effects(baseEffectText)

      # and the tick labels
      tickLabels = []
      for entry in [self.ax, self.ax_resid, self.ax2, self.ax_div, self.ax_resid_div, self.ax2_div]:
        tickLabels.extend(entry.get_xticklabels(which='both'))
        tickLabels.extend(entry.get_yticklabels(which='both'))
        tickLabels.append(entry.xaxis.get_offset_text())
        tickLabels.append(entry.yaxis.get_offset_text())
      for entry in tickLabels:
        entry.set_path_effects(baseEffectText)

      # and the cursor
      for cursor in [self.cursor, self.cursor_div, self.cursor2, self.cursor2_div]:
        if(cursor != None):
          handles = cursor.getHandles()
          for entry in handles:
            # check for text
            if(hasattr(entry, 'get_text')):
              entry.set_path_effects(baseEffectText)
            else:
              entry.set_path_effects(baseEffect)

    # introduces path effects for new plot items
    tempDict = {}
    tempDict['path.effects'] = baseEffect
    matplotlib.rcParams.update(tempDict)

    # update plot
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()
    
  def setExportSize(self, width=None, height=None, redraw=True):
    # sets canvas size
    if(width != None):
      self.exportWidth = width
    if(height != None):
      self.exportHeight = height
    
    # update plot
    redraw = redraw and ((width != None) or (height != None))
    redraw = redraw and (self.arrowVisible['x'] or self.arrowVisible['y'])
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

  def setXkcdSetting(self, value=1.0, item='scale', redraw=True):
    # update xckd setttings
    if(item in ['scale', 'length', 'random']):
      if(item == 'scale'):
        if(self.xkcdScale == value):
          redraw = False
        else:
          self.xkcdScale = value
      elif(item == 'length'):
        if(self.xkcdLength == value):
          redraw = False
        else:
          self.xkcdLength = value
      elif(item == 'random'):
        if(self.xkcdRandomness == value):
          redraw = False
        else:
          self.xkcdRandomness = value
          
      # do the xkcdify?
      if((redraw) and (self.xkcd)):
        self.xkcdify(state=self.xkcd, redraw=redraw)

  def xkcdify(self, state=True, redraw=True):
    # set xkcd-like parameters
    # store previous parameters
    if((not self.xkcd) and (state)):
      if ('font.sans-serif' in matplotlib.rcParams):
        self.xkcdStoreFonts = matplotlib.rcParams['font.sans-serif']
      else:
        self.xkcdStoreFonts = ['DejaVu Sans']
    
    # set new parameters
    self.xkcd = state
    tempDict = {}
    if(self.xkcd):
      xkcdScale = self.xkcdScale
      xkcdLength = self.xkcdLength
      xkcdRandomness = self.xkcdRandomness

      # check for presence of funny fonts
      addFonts = []
      fontCandidates = 'Humor Sans,Comic Sans MS'.split(',')
      for entry in fontCandidates:
        if entry in self.parent.fontNames:
          addFonts.append(entry)
          
      if(len(addFonts)):
        tempDict['font.sans-serif'] = addFonts
        tempDict['font.sans-serif'].extend(self.xkcdStoreFonts)
    else:
      xkcdScale = 0
      xkcdLength = 0
      xkcdRandomness = 0
      
      # restore original fonts
      tempDict['font.sans-serif'] = self.xkcdStoreFonts

    # introduces xkcd-type style for new plot items
    tempDict['path.sketch'] = (xkcdScale, xkcdLength, xkcdRandomness)
    matplotlib.rcParams.update(tempDict)
    
    # modify existing extras
    for entry in self.parent.extras:
      relevantHandles = [entry.handle, entry.handle_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handle', 'handle_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleExtra in relevantHandles:
        if(handleExtra != None):
          # modify object directly (for line)
          if(hasattr(handleExtra, 'set_sketch_params')):
            handleExtra.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
          # set bbox if present
          if(hasattr(handleExtra, 'get_bbox_patch')):
            handle = handleExtra.get_bbox_patch()
            if(handle != None):
              handle.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
          # set arrow patch if present
          if(hasattr(handleExtra, 'arrow_patch')):
            handle = handleExtra.arrow_patch
            if(handle != None):
              handle.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    
    # modify existing curves
    for entry in self.parent.fit:
      relevantHandles = [entry.handlePlot, entry.handlePlot_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handlePlot', 'handlePlot_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handlePlot in relevantHandles:
        if(handlePlot != None):
          handlePlot.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    
    # modify existing data sets
    for entry in self.parent.data:
      relevantHandles = [entry.handleData, entry.handleData_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleData', 'handleData_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleData in relevantHandles:
        if(handleData != None):
          handleData.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      for handleResid in [entry.handleResid, entry.handleResid_div]:
        if(handleResid != None):
          handleResid.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      relevantHandles = [entry.handleBar, entry.handleBar_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleBar', 'handleBar_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleBar in relevantHandles:
        if(handleBar != None):
          children = handleBar.get_children()
          for entry2 in children:
            entry2.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      relevantHandles = [entry.handleStack, entry.handleStack_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleStack', 'handleStack_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleStack in relevantHandles:
        if(handleStack != None):
          handleStack.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      relevantHandles = [entry.handleStackNeg, entry.handleStackNeg_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleStackNeg', 'handleStackNeg_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleStackNeg in relevantHandles:
        if(handleStackNeg != None):
          handleStackNeg.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      for handleResidBar in [entry.handleResidBar, entry.handleResidBar_div]:
        if(handleResidBar != None):
          children = handleResidBar.get_children()
          for entry2 in children:
            entry2.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      for handleResidStack in [entry.handleResidStack, entry.handleResidStack_div]:
        if(handleResidStack != None):
          handleResidStack.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      for handleResidStackNeg in [entry.handleResidStackNeg, entry.handleResidStackNeg_div]:
        if(handleResidStackNeg != None):
          handleResidStackNeg.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      relevantHandles = [entry.handleErr, entry.handleErr_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleErr', 'handleErr_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleErr in relevantHandles:
        if(handleErr != None):
          handleErr[0].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
          for entry2 in handleErr[1]:
            entry2.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
          for entry2 in handleErr[2]:
            entry2.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

      relevantHandles = [entry.handleViolin, entry.handleViolin_div]
      if(self.splitY):
        relevantHandles += [entry.handlesBothAxes[handle] for handle in ['handleViolin', 'handleViolin_div'] if ((handle in entry.handlesBothAxes) and (entry.handlesBothAxes != None))]
      for handleViolin in relevantHandles:
        if(handleViolin != None):
          for subtarget in handleViolin:
            if(type(handleViolin[subtarget]) == type([])):
              itemList = handleViolin[subtarget]
            else:
              itemList = [handleViolin[subtarget]]
            # cycle through all items
            for item in itemList:
              if(hasattr(item, 'set_sketch_params')):
                item.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # modify axes and background
    for entry in self.ax.spines:
      if(entry in ['left', 'right']):
        self.ax.spines[entry].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in self.ax_resid.spines:
      self.ax_resid.spines[entry].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in self.ax2.spines:
      self.ax2.spines[entry].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    self.ax2.patch.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    self.ax_resid.patch.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in self.ax_div.spines:
      if(entry in ['left', 'right']):
        self.ax_div.spines[entry].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in self.ax_resid_div.spines:
      self.ax_resid_div.spines[entry].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in self.ax2_div.spines:
      self.ax2_div.spines[entry].set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    self.ax2_div.patch.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    self.ax_resid_div.patch.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # modify grid lines
    for key in self.gridLinesStore:
      if(self.gridVisible[key]):
        for entry in self.gridLinesStore[key]:
          entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
      if(key in self.gridLinesStore_resid):
        for entry in self.gridLinesStore_resid[key]:
          entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # modify tick lines (somehow not heeded by matplotlib?!)
    for entry in (self.ax.yaxis.get_ticklines() + self.ax.yaxis.get_ticklines(minor=True) + self.ax_div.yaxis.get_ticklines() + self.ax_div.yaxis.get_ticklines(minor=True)):
      entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in (self.ax_resid.xaxis.get_ticklines() + self.ax_resid.xaxis.get_ticklines(minor=True) + self.ax_resid_div.xaxis.get_ticklines() + self.ax_resid_div.xaxis.get_ticklines(minor=True)):
      entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in (self.ax_resid.yaxis.get_ticklines() + self.ax_resid.yaxis.get_ticklines(minor=True) + self.ax_resid_div.yaxis.get_ticklines() + self.ax_resid_div.yaxis.get_ticklines(minor=True)):
      entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in (self.ax2.xaxis.get_ticklines() + self.ax2.xaxis.get_ticklines(minor=True) + self.ax2_div.xaxis.get_ticklines() + self.ax2_div.xaxis.get_ticklines(minor=True)):
      entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    for entry in (self.ax2.yaxis.get_ticklines() + self.ax2.yaxis.get_ticklines(minor=True) + self.ax2_div.yaxis.get_ticklines() + self.ax2_div.yaxis.get_ticklines(minor=True)):
      entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # zero line
    for handleResidZero in [self.handleResidZero, self.handleResidZero_div]:
      if(handleResidZero != None):
        handleResidZero.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
      
    # and the cursor
    for cursor in [self.cursor, self.cursor_div, self.cursor2, self.cursor2_div]:
      if(cursor != None):
        handles = cursor.getHandles()
        for entry in handles:
          entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # and the axes labels
    for entry in [self.ax2, self.ax_resid, self.ax2_div, self.ax_resid_div]:
      entry.xaxis.label.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
      entry.yaxis.label.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    self.ax.yaxis.label.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)
    self.ax_div.yaxis.label.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # and the tick labels
    tickLabels = []
    for entry in [self.ax, self.ax2, self.ax_resid, self.ax_div, self.ax2_div, self.ax_resid_div]:
      tickLabels.extend(entry.get_xticklabels(which='both'))
      tickLabels.extend(entry.get_yticklabels(which='both'))
    for entry in tickLabels:
      entry.set_sketch_params(xkcdScale, xkcdLength, xkcdRandomness)

    # update plot
    if(redraw):
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()
    
  def initPlot(self, initialize=True):
    if(initialize):
      plt.ioff()
      # initialize data plot
      self.ax2 = self.matplot.add_subplot(221)
      self.ax2.autoscale(enable=False, axis='both')
      self.ax2.xaxis.set_label_text('x')
      self.ax2.yaxis.set_label_text('y')
      self.ax2.yaxis.set_label_position('right')
      self.ax2.yaxis.set_ticks_position('right')
      self.ax2.grid(False, which='both')
      self.ax2.patch.set_facecolor('none')
      
      # set up axes
      self.ax2_div = self.matplot.add_subplot(222)
      self.ax2_div.set_xscale(self.modeX_div)
      self.ax2_div.set_xlim((self.minX_div, self.maxX_div))
      # hide the extra axis
      self.ax2_div.grid(False, which='both')
      self.ax2_div.set_visible(False)
      self.ax2_div.yaxis.set_label_position('right')

      # we cannot set entire axis invisible as this also removes canvas etc.
      self.toggleSecondAxesItems(state=False)
      
      # ax object should be twinned to ax2 (not vice versa), such that it ends on top
      self.ax = self.matplot.add_subplot(223)

      self.ax.autoscale(enable=False, axis='both')
      self.ax.xaxis.set_visible(False)
      self.ax.yaxis.set_label_position('left')
      self.ax.yaxis.set_ticks_position('left')
      self.ax.grid(False, which='both')
      # for good measure, reset ticks position on ax2
      self.ax2.yaxis.set_ticks_position('right')
      for entry in ['top', 'bottom']:
        self.ax.spines[entry].set_visible(False)
      self.matplot.patch.set_facecolor(self.canvasColor)

      # set up extra axes
      self.ax_div = self.matplot.add_subplot(224)

      self.ax_div.set_xscale(self.modeX_div)
      self.ax_div.set_xlim((self.minX_div, self.maxX_div))
      # hide diverse elements
      for entry in ['top', 'bottom', 'left', 'right']:
        self.ax_div.spines[entry].set_visible(False)
      self.ax_div.patch.set_facecolor('none')
      self.ax_div.grid(False, which='both')
      self.ax_div.xaxis.set_ticks_position('none')
      self.ax_div.yaxis.set_ticks_position('none')
      tickLabels = self.ax_div.get_xticklabels(which='both')
      tickLabels.append(self.ax_div.xaxis.get_offset_text())
      tickLabels.extend(self.ax_div.get_yticklabels(which='both'))
      tickLabels.append(self.ax_div.yaxis.get_offset_text())
      for entry in tickLabels:
        entry.set_visible(False)
      # hide the extra axis
      self.ax_div.set_visible(False)

      # adjust z-order to make original axes appear in front (important for mouse cursor)
      self.ax.set_zorder(self.ax_div.get_zorder() + 1)
      self.ax2.set_zorder(self.ax2_div.get_zorder() + 1)
      
      # generate additional plot object for colored background
      self.colorPlane = self.matplot.add_subplot(111, label='tilt')
      self.colorPlane.autoscale(enable=False, axis='both')
      # hide essentially all elements of this new plot
      self.colorPlane.xaxis.set_visible(False)
      self.colorPlane.yaxis.set_ticks_position('none')
      for entry in ['top', 'bottom', 'left', 'right']:
        self.colorPlane.spines[entry].set_visible(False)
      self.colorPlane.xaxis.set_label_text('')
      self.colorPlane.yaxis.set_label_text('')
      self.colorPlane.xaxis.label.set_visible(False)
      self.colorPlane.yaxis.label.set_visible(False)
      self.colorPlane.set_xlim(0, 1)
      self.colorPlane.set_ylim(0, 1)
      self.colorPlane.set_xticks([])
      self.colorPlane.set_yticks([])
      self.colorPlane.patch.set_facecolor([0.0] * 4)
      self.colorPlane.grid(False, which='both')
      # initialize handle for colored background
      if(self.canvasGradientVisible):
        self.setCanvasGradient(redraw=False)
      else:
        self.colorBackground = None
      self.colorPlane.set_zorder(self.ax_div.get_zorder() - 1)
        
      # initalize some values
      self.handleData = None
      self.handlePlot = None
      self.handleErr = None
      self.handleErrShady = None
    
      # initialize resid plot
      self.ax_resid = self.residplot.add_subplot(211)
      self.ax_resid.autoscale(enable=False, axis='both')
      self.ax_resid.grid(False, which='both')
      self.residplot.patch.set_facecolor(self.canvasColor)
      
      # set up extra axes
      self.ax_resid_div = self.residplot.add_subplot(212)
      self.ax_resid_div.set_xscale(self.modeX_div)
      self.ax_resid_div.set_xlim((self.minX_div, self.maxX_div))
      self.ax_resid_div.grid(False, which='both')
      tickLabels = self.ax_resid_div.get_yticklabels(which='both')
      tickLabels.append(self.ax_resid_div.yaxis.get_offset_text())
      for entry in tickLabels:
        entry.set_visible(False)
      # hide the extra axis
      self.ax_resid_div.set_visible(False)

      # adjust z-order to make original axes appear in front
      self.ax_resid.set_zorder(self.ax_resid_div.get_zorder() + 1)

      # refresh plots
      self.dataplotwidget.myRefresh()
      self.residplotwidget.myRefresh()

      # generate additional plot object for colored background
      self.colorPlane_resid = self.residplot.add_subplot(111, label='tilt_resid')
      self.colorPlane_resid.autoscale(enable=False, axis='both')
      # hide essentially all elements of this new plot
      self.colorPlane_resid.xaxis.set_visible(False)
      self.colorPlane_resid.yaxis.set_ticks_position('none')
      for entry in ['top', 'bottom', 'left', 'right']:
        self.colorPlane_resid.spines[entry].set_visible(False)
      self.colorPlane_resid.xaxis.set_label_text('')
      self.colorPlane_resid.yaxis.set_label_text('')
      self.colorPlane_resid.xaxis.label.set_visible(False)
      self.colorPlane_resid.yaxis.label.set_visible(False)
      self.colorPlane_resid.set_xlim(0, 1)
      self.colorPlane_resid.set_ylim(0, 1)
      self.colorPlane_resid.set_xticks([])
      self.colorPlane_resid.set_yticks([])
      self.colorPlane_resid.patch.set_facecolor([0.0] * 4)
      self.colorPlane_resid.grid(False, which='both')
      # initialize handle for colored background
      if(self.canvasGradientVisible):
        self.setCanvasGradient(redraw=False, target='resid')
      else:
        self.colorBackground_resid = None
      self.colorPlane_resid.set_zorder(self.ax_resid_div.get_zorder() - 1)

      # initalize some values
      self.handleResid = None
      self.handleResidZero = None
      self.handleResidZero_div = None
      self.handlesAbout = []
      
      # initialize the plot rectangle and lasso selector
      # (this is the new format with ad hoc selectors rather than several persistent instances)
      self.lassoSelector, self.rectangleSelector = None, None

      # set up cursor
      self.matplot.canvas.mpl_connect('button_press_event', self.toggleCrossHairEncore)
      self.matplot.canvas.mpl_connect('scroll_event', self.mouseScrolled)
      self.matplot.canvas.mpl_connect('motion_notify_event', self.moveHandler)
      self.matplot.canvas.mpl_connect('button_release_event', self.releaseHandler)
      
      # set up click handler in residplot
      self.residplot.canvas.mpl_connect('button_press_event', self.pickHandlerResid)

    # beware that setTickMarkLabelShow() may also alter self.ticksXShow etc.
    # => need to store original values and restore further down
    restoreXShow, restoreYShow, restoreY2Show, restoreResidYShow = self.ticksXShow, self.ticksYShow, self.ticksY2Show, self.ticksResidYShow

    for target in ['plot', 'resid']:
      # labels etc.
      self.toggleAxisLabel(self.labelXShow, axis='x', redraw=False, target=target)
      self.setAxisLabel(self.labelX, axis='x', redraw=False, target=target)
      self.setAxisLabelColor(value = self.labelXColor, axis = 'x', redraw = False, target=target)
      self.setAxisLabelSize(value = self.labelXSize, axis = 'x', redraw = False, target=target)
      self.setAxisLabelBold(value = self.labelXWeight, axis = 'x', redraw = False, target=target)
      self.setAxisLabelItalic(value = self.labelXStyle, axis = 'x', redraw = False, target=target)
      self.setAxisLabelVariant(value = self.labelXVariant, axis = 'x', redraw = False, target=target)
      self.setAxisLabelAlignment(value = self.labelXAlignment, axis = 'x', redraw = False, target=target)
      self.setAxisLabelAlignmentVertical(value = self.labelXAlignmentVertical, axis = 'x', redraw = False, target=target)
      self.setAxisLabelPad(value = self.labelXPad, axis = 'x', redraw = False, target=target)
      self.setAxisLabelPos(value = self.labelXPos, axis = 'x', redraw = False, target=target)
      self.setAxisLabelAngle(value = self.labelXAngle, axis = 'x', redraw = False, target=target)
      self.setAxisFont(value = self.axisFont['x'], axis = 'x', redraw = False, target=target)
      self.toggleAxisLabel(self.labelYShow, axis='y', redraw=False, target=target)
      self.setAxisLabel(labeltext=self.labelY, axis='y', redraw=False, target=target)
      if(target == 'plot'):
        self.setAxisLabel(labeltext=self.labelY2, axis='y2', redraw=False, target=target)
        self.toggleAxisLabel(self.labelY2Show, axis='y2', redraw=False, target=target)
      self.setAxisLabelColor(value = self.labelYColor, axis = 'y', redraw = False, target=target)
      self.setAxisLabelSize(value = self.labelYSize, axis = 'y', redraw = False, target=target)
      self.setAxisLabelBold(value = self.labelYWeight, axis = 'y', redraw = False, target=target)
      self.setAxisLabelItalic(value = self.labelYStyle, axis = 'y', redraw = False, target=target)
      self.setAxisLabelVariant(value = self.labelYVariant, axis = 'y', redraw = False, target=target)
      self.setAxisLabelAlignment(value = self.labelYAlignment, axis = 'y', redraw = False, target=target)
      self.setAxisLabelAlignmentVertical(value = self.labelYAlignmentVertical, axis = 'y', redraw = False, target=target)
      self.setAxisLabelPad(value = self.labelYPad, axis = 'y', redraw = False, target=target)
      self.setAxisLabelPos(value = self.labelYPos, axis = 'y', redraw = False, target=target)
      self.setAxisLabelAngle(value = self.labelYAngle, axis = 'y', redraw = False, target=target)
      self.setAxisFont(value = self.axisFont['y'], axis = 'y', redraw = False, target=target)
      if(target == 'plot'):
        self.setAxisLabelColor(value = self.labelY2Color, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelSize(value = self.labelY2Size, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelBold(value = self.labelY2Weight, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelItalic(value = self.labelY2Style, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelVariant(value = self.labelY2Variant, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelAlignment(value = self.labelY2Alignment, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelAlignmentVertical(value = self.labelY2AlignmentVertical, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelPad(value = self.labelY2Pad, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelPos(value = self.labelY2Pos, axis = 'y2', redraw = False, target=target)
        self.setAxisLabelAngle(value = self.labelY2Angle, axis = 'y2', redraw = False, target=target)
        self.setAxisFont(value = self.axisFont['y2'], axis = 'y2', redraw = False, target=target)
      
      # set axes properties
      for key in self.axisVisible:
        self.setAxisVisibility(value = self.axisVisible[key], axis = key, redraw = False, target=target)
      for key in self.axisWidth:
        self.setAxisWidth(value = self.axisWidth[key], axis = key, redraw = False, target=target)
      for key in self.axisStyle:
        self.setAxisStyle(value = self.axisStyle[key], axis = key, redraw = False, target=target)
      for key in self.axisDashStyle:
        self.setAxisDashStyle(value = self.axisDashStyle[key], axis = key, redraw = False, target=target)
      for key in self.axisColor:
        self.setAxisColor(value = self.axisColor[key], axis = key, redraw = False, target=target)

      if(target == 'plot'):
        useKeys = ['left', 'right', 'bottom', 'top', 'left2', 'right2']
      else:
        useKeys = ['left', 'right', 'bottom', 'top']
      if(not initialize):
        for key in useKeys:
          self.setAxisPosition(value=self.axisPosition[key], axis=key, redraw=False, target=target)
          self.setAxisPositionValue(value=self.axisPositionValue[key], axis=key, redraw=False, target=target)
      for key in useKeys:
        self.setAxisBoundaryValue(value=self.axisBoundary[key][0], lower=True, axis=key, redraw=False, target=target)
        self.setAxisBoundaryValue(value=self.axisBoundary[key][1], lower=False, axis=key, redraw=False, target=target)
        self.setAxisBoundary(value=self.axisBoundaryCheck[key], axis=key, redraw=False, target=target)
        
      # set axes arrows
      for axis in ['x', 'y']:
        self.setAxisArrowColor(value=self.arrowColor[axis], axis=axis, item='line', redraw=False)
        self.setAxisArrowColor(value=self.arrowFill[axis], axis=axis, item='fill', redraw=False)
        self.setAxisArrowHeadWidth(value=self.arrowHeadWidth[axis], axis=axis, redraw=False)
        self.setAxisArrowHeadLength(value=self.arrowHeadLength[axis], axis=axis, redraw=False)
        self.setAxisArrowOverhang(value=self.arrowOverhang[axis], axis=axis, redraw=False)
        self.setAxisArrow(state=self.arrowVisible[axis], axis=axis, item='all', redraw=False, target=target)
        self.setAxisArrow(state=self.arrowEdgeShow[axis], axis=axis, item='edge', redraw=False, target=target)
        self.setAxisArrow(state=self.arrowFillShow[axis], axis=axis, item='fill', redraw=False, target=target)
  
      # set color and size of ticks
      self.toggleTicksLabel(value=self.ticksXShow, axis='x', redraw=False, target=target)
      self.setTickLabelColor(value = self.ticksXColor, axis = 'x', redraw = False, target=target)
      self.setTickLabelSize(value = self.ticksXSize, axis = 'x', redraw = False, target=target)
      self.setTickLabelBold(value = self.ticksXWeight, axis = 'x', redraw = False, target=target)
      self.setTickLabelItalic(value = self.ticksXStyle, axis = 'x', redraw = False, target=target)
      self.setTickLabelAngle(value = self.ticksXAngle, axis = 'x', redraw = False, target=target)
      self.setTickLabelAlignment(value = self.ticksXAlignment, axis = 'x', redraw = False, target=target)
      self.setTickLabelAlignmentVertical(value = self.ticksXAlignmentVertical, axis = 'x', redraw = False, target=target)
      self.setTickLabelPad(value = self.ticksXPad, axis = 'x', redraw = False, target=target)
      self.setTickLabelPad2(value = self.ticksXPad2, axis = 'x', redraw = False, target=target)
      self.setTickFont(value = self.tickFont['x'], axis = 'x', redraw = False, target=target)
      ###self.setTickFormat(axis='x', value=self.ticksXFormat, redraw=False, target=target)
      ### do we need to call tick formatting at all? (will be updated on draw time)

      if(target == 'plot'):
        self.toggleTicksLabel(value=self.ticksYShow, axis='y', redraw=False, target=target)
      else:
        self.toggleTicksLabel(value=self.ticksResidYShow, axis='y', redraw=False, target=target)
      self.setTickLabelColor(value = self.ticksYColor, axis = 'y', redraw = False, target=target)
      self.setTickLabelSize(value = self.ticksYSize, axis = 'y', redraw = False, target=target)
      self.setTickLabelBold(value = self.ticksYWeight, axis = 'y', redraw = False, target=target)
      self.setTickLabelItalic(value = self.ticksYStyle, axis = 'y', redraw = False, target=target)
      self.setTickLabelAngle(value = self.ticksYAngle, axis = 'y', redraw = False, target=target)
      self.setTickLabelAlignment(value = self.ticksYAlignment, axis = 'y', redraw = False, target=target)
      self.setTickLabelAlignmentVertical(value = self.ticksYAlignmentVertical, axis = 'y', redraw = False, target=target)
      self.setTickLabelPad(value = self.ticksYPad, axis = 'y', redraw = False, target=target)
      self.setTickLabelPad2(value = self.ticksYPad2, axis = 'y', redraw = False, target=target)
      self.setTickFont(value = self.tickFont['y'], axis = 'y', redraw = False, target=target)
      if(target == 'plot'):
        self.toggleTicksLabel(value=self.ticksY2Show, axis='y2', redraw=False, target=target)
        self.setTickLabelColor(value = self.ticksY2Color, axis = 'y2', redraw = False, target=target)
        self.setTickLabelSize(value = self.ticksY2Size, axis = 'y2', redraw = False, target=target)
        self.setTickLabelBold(value = self.ticksY2Weight, axis = 'y2', redraw = False, target=target)
        self.setTickLabelItalic(value = self.ticksY2Style, axis = 'y2', redraw = False, target=target)
        self.setTickLabelAngle(value = self.ticksY2Angle, axis = 'y2', redraw = False, target=target)
        self.setTickLabelAlignment(value = self.ticksY2Alignment, axis = 'y2', redraw = False, target=target)
        self.setTickLabelAlignmentVertical(value = self.ticksY2AlignmentVertical, axis = 'y2', redraw = False, target=target)
        self.setTickLabelPad(value = self.ticksY2Pad, axis = 'y2', redraw = False, target=target)
        self.setTickLabelPad2(value = self.ticksY2Pad2, axis = 'y2', redraw = False, target=target)
        self.setTickFont(value = self.tickFont['y2'], axis = 'y2', redraw = False, target=target)
  
      # set tick properties
      if(target == 'plot'):
        validItems = ['left', 'right', 'top', 'bottom', 'left2', 'right2']
      else:
        validItems = ['left', 'right', 'top', 'bottom']
      for key in self.ticksVisible:
        if(key in validItems):
         self.setTickMarkVisibility(value = self.ticksVisible[key], axis = key, redraw = False, target=target, initMode=True)
      for key in self.ticksWidth:
        if(key in validItems):
          self.setTickMarkWidth(value = self.ticksWidth[key], axis = key, redraw = False, target=target)
      for key in self.ticksLength:
        if(key in validItems):
          self.setTickMarkLength(value = self.ticksLength[key], axis = key, redraw = False, target=target, forceRedraw=False)
      for key in self.ticksColor:
        if(key in validItems):
          self.setTickMarkColor(value = self.ticksColor[key], axis = key, redraw = False, target=target)
      for key in self.ticksDirection:
        if(key in validItems):
          self.setTickMarkDirection(value = self.ticksDirection[key], axis = key, redraw = False, target=target)
      for key in self.ticksLabelShow:
        if(key in validItems):
          self.setTickMarkLabelShow(value = self.ticksLabelShow[key], axis = key, redraw = False, target=target)

      # set grid properties
      for key in self.gridVisible:
         self.setGridVisibility(value = self.gridVisible[key], axis = key, redraw = False, target=target)
      for key in self.gridWidth:
        self.setGridWidth(value = self.gridWidth[key], axis = key, redraw = False, target=target)
      for key in self.gridStyle:
        self.setGridStyle(value = self.gridStyle[key], axis = key, redraw = False, target=target)
      for key in self.gridDashStyle:
        self.setGridDashStyle(value = self.gridDashStyle[key], axis = key, redraw = False, target=target)
      for key in self.gridColor:
        self.setGridColor(value = self.gridColor[key], axis = key, redraw = False, target=target)
      for key in self.gridOrder:
        self.setGridOrder(value = self.gridOrder[key], axis = key, redraw = False, target=target)
      # set minor grid properties
      for key in self.gridMinorVisible:
         self.setGridMinorVisibility(value = self.gridMinorVisible[key], axis = key, redraw = False, target=target)
      for key in self.gridMinorWidth:
        self.setGridMinorWidth(value = self.gridMinorWidth[key], axis = key, redraw = False, target=target)
      for key in self.gridMinorStyle:
        self.setGridMinorStyle(value = self.gridMinorStyle[key], axis = key, redraw = False, target=target)
      for key in self.gridMinorDashStyle:
        self.setGridMinorDashStyle(value = self.gridMinorDashStyle[key], axis = key, redraw = False, target=target)
      for key in self.gridMinorColor:
        self.setGridMinorColor(value = self.gridMinorColor[key], axis = key, redraw = False, target=target)
      for key in self.gridMinorOrder:
        self.setGridMinorOrder(value = self.gridMinorOrder[key], axis = key, redraw = False, target=target)

      # canvas color etc.
      if(self.splitY):
        # activate patch background
        self.ax.patch.set_visible(True)
        self.ax_div.patch.set_visible(True)
      else:
        self.ax.patch.set_visible(False)
        self.ax_div.patch.set_visible(False)
        
      # deal with the innner situation
      self.updateInnerYSituation()
      self.setCanvasColor(value=self.canvasColor, redraw=False, target=target)
      self.toggleCanvasFill(value=self.canvasFill, redraw=False, target=target)
      self.toggleFigureFill(value=self.figureFill, redraw=False, target=target, silent=True)
      self.toggleFrameDraw(value=self.frameDraw, redraw=False, target=target)
      self.setFigureColor(value=self.figureColor, redraw=False, target=target, silent=True)
      self.setCanvasGradientStyle(value=self.canvasGradientStyle, redraw=False, target=target)
      self.setCanvasGradientColor(value=self.canvasGradientColor1, redraw=False, target=target, color=0)
      self.setCanvasGradientColor(value=self.canvasGradientColor2, redraw=False, target=target, color=1)
      self.setCanvasGradientCenter(value=self.canvasGradientCenter[0], redraw=False, target=target, axis='x')
      self.setCanvasGradientCenter(value=self.canvasGradientCenter[1], redraw=False, target=target, axis='y')
      self.setCanvasGradientWidth(value=self.canvasGradientWidth, redraw=False, target=target)
      self.setCanvasGradientAngle(value=self.canvasGradientAngle, redraw=False, target=target)
      self.setCanvasGradient(state=self.canvasGradientVisible, redraw=False, target=target)
      
      # set padding
      self.setPadding(value=self.padSize['bottom'], axis='bottom', redraw=False, target=target)
      
      # frame
      self.setFrameColor(value=self.frameColor, redraw=False, target=target)
      self.setFrameWidth(value=self.frameWidth, redraw=False, target=target)
      self.setFrameStyle(value=self.frameStyle, redraw=False, target=target)
      self.setFrameDashStyle(value=self.frameDashStyle, redraw=False, target=target)

    # slaved x axis controls
    self.toggleSlavedX(state=self.slavedX, redraw=False)
    self.toggleSlavedXLabel(state=self.slavedXLabelShow, redraw=False)
    self.changeSlavedXLabel(value=self.slavedXLabel, redraw=False)

    # now restore visibility of tick labels
    self.toggleTicksLabel(value=restoreXShow, axis='x', redraw=False, target='plot')
    self.toggleTicksLabel(value=restoreYShow, axis='y', redraw=False, target='plot')
    self.toggleTicksLabel(value=restoreY2Show, axis='y2', redraw=False, target='plot')
    self.toggleTicksLabel(value=restoreXShow, axis='x', redraw=False, target='resid')
    self.toggleTicksLabel(value=restoreResidYShow, axis='y', redraw=False, target='resid')

    # the inner situation
    self.changeSplitFraction(value=self.splitFraction, redraw=False)
    self.changeSplitPad(value=self.splitPad, redraw=False)
    self.setInnerParameter(param='axes', state=self.innerAxes, redraw=False)
    self.setInnerParameter(param='ticks', state=self.innerTicks, redraw=False)
    for target in ['plot', 'resid']:
      self.toggleTicksLabel(value=self.ticksXShow_div, axis='x2', redraw=False, target=target)

    # xkcd etc.
    self.xkcdify(state=self.xkcd, redraw=False)
    self.setPathEffects(redraw=False)
    
    # deal with axis ticks
    if(self.ticksXAuto):
      self.setAutoTicks(axis='x', redraw=False, target='plot')
      self.setAutoTicks(axis='x', redraw=False, target='resid')
    else:
      ticksXLabel = self.ticksXLabel
      self.setAxisTicks(value=self.ticksX, axis='x', redraw=False, target='plot')
      self.setAxisTicks(value=self.ticksX, axis='x', redraw=False, target='resid')
      # apply axis labels?
      self.ticksXLabel = ticksXLabel
      if(len(self.ticksXLabel) and (len(self.ticksXLabel) == len(self.ticksX))):
        for axisobject in [self.ax, self.ax_resid]:
          axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksX))
          axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksXLabel))
    if(self.ticksXAuto_div):
      self.setAutoTicks(axis='x2', redraw=False, target='plot')
      self.setAutoTicks(axis='x2', redraw=False, target='resid')
    else:
      ticksXLabel = self.ticksXLabel_div
      self.setAxisTicks(value=self.ticksX_div, axis='x2', redraw=False, target='plot')
      self.setAxisTicks(value=self.ticksX_div, axis='x2', redraw=False, target='resid')
      # apply axis labels?
      self.ticksXLabel_div = ticksXLabel
      if(len(self.ticksXLabel_div) and (len(self.ticksXLabel_div) == len(self.ticksX_div))):
        for axisobject in [self.ax2_div, self.ax_resid_div]:
          axisobject.xaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksX_div))
          axisobject.xaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksXLabel_div))
      
    if(self.ticksYAuto):
      self.setAutoTicks(axis='y', redraw=False, target='plot')
    else:
      ticksYLabel = self.ticksYLabel
      self.setAxisTicks(value=self.ticksY, axis='y', redraw=False, target='plot')
      # apply axis labels?
      self.ticksYLabel = ticksYLabel
      if(len(self.ticksYLabel) and (len(self.ticksYLabel) == len(self.ticksY))):
        for axisobject in [self.ax]:
          axisobject.yaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksY))
          axisobject.yaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksYLabel))
      
    if(self.ticksY2Auto):
      self.setAutoTicks(axis='y2', redraw=False, target='plot')
    else:
      ticksY2Label = self.ticksY2Label
      self.setAxisTicks(value=self.ticksY2, axis='y2', redraw=False, target='plot')
      # apply axis labels?
      self.ticksY2Label = ticksY2Label
      if(len(self.ticksY2Label) and (len(self.ticksY2Label) == len(self.ticksY2))):
        for axisobject in [self.ax2]:
          axisobject.yaxis.set_major_locator(matplotlib.ticker.FixedLocator(self.ticksY2))
          axisobject.yaxis.set_major_formatter(matplotlib.ticker.FixedFormatter(self.ticksY2Label))
      
    if(self.ticksResidYAuto):
      self.setAutoTicks(axis='resid', redraw=False, target='resid')
    else:
      self.setAxisTicks(value=self.ticksResidY, axis='resid', redraw=False, target='resid')

    # retrieve axis ticks
    self.ticksX = self.getAxisTicks(axis='x')
    self.ticksY = self.getAxisTicks(axis='y')
    self.ticksY2 = self.getAxisTicks(axis='y2')
    self.ticksResidY = self.getAxisTicks(axis='resid')
    
    # issue call to setTickFormatHelper for slaved x axis to get its formatting right when processing style sheet
    self.setTickFormatHelper(axis='xs', redraw=False)
    self.setTickFormatHelper(axis='residxs', redraw=False)

    # issue plot redraw
    # the follwing call not needed as we will draw a curve afterwards
    self.handleResidZero = self.plotResidZero(self.handleResidZero, redraw=False)
    self.handleResidZero_div = self.plotResidZero(self.handleResidZero_div, redraw=False, splitX=True)
    self.setAxisLimits(lower=self.minResidY, upper=self.maxResidY, axis='y', updateLabel=False, target='resid', redraw=initialize, updateGrid=True)
    self.updateTickEntryField(axis='y')

  def pickHandlerResid(self, event):
    # handles click events in resid plot
    if((event.button == 1) and event.dblclick):
      # first check for double click on certain plot items
      searchItems = {}
      searchItems['xlabel'] = self.ax_resid.xaxis.label
      searchItems['ylabel'] = self.ax_resid.yaxis.label
      # need to expand for slaved second x axis
      if(self.handleSlavedXLabelResid != None):
        searchItems['slavedxlabel'] = self.handleSlavedXLabelResid

      # need to expand for slaved second x axis
      # these should come first b/c otherwise the regular upper labels win out
      for number, label in enumerate(self.secondXLabelsResid):
        searchItems['labliSlavedX_' + str(number)] = label
      for number, label in enumerate(self.secondXLabelsResid_div):
        searchItems['labliSlavedX2_' + str(number)] = label

      labli = self.ax_resid.get_xticklabels(which='both')
      for number, label in enumerate(labli):
        searchItems['labliX_' + str(number)] = label
      labli = self.ax_resid.get_yticklabels(which='both')
      for number, label in enumerate(labli):
        searchItems['labliResid_' + str(number)] = label
      if(self.splitShow):
        labli = self.ax_resid_div.get_xticklabels(which='both')
        for number, label in enumerate(labli):
          searchItems['labliX2_' + str(number)] = label

      # are split axes active?
      if(self.splitShow):
        if(self.handleDividerResid != None):
          for number, item in enumerate(self.handleDividerResid):
            searchItems['dividerX_' + str(number)] = item

      # need to expand for slaved second x axis
      # these should come first b/c otherwise the regular upper ticks win out
      for number, tick in enumerate(self.secondXLinesResid):
        searchItems['tickliSlavedX_' + str(number)] = tick
      for number, tick in enumerate(self.secondXLinesResid_div):
        searchItems['tickliSlavedX2_' + str(number)] = tick

      tickli = self.ax_resid.xaxis.get_ticklines() + self.ax_resid.xaxis.get_minorticklines()
      for number, tick in enumerate(tickli):
        searchItems['tickliX_' + str(number)] = tick
      if(self.splitShow):
        tickli = self.ax_resid_div.xaxis.get_ticklines() + self.ax_resid_div.xaxis.get_minorticklines()
        for number, tick in enumerate(tickli):
          searchItems['tickliX2_' + str(number)] = tick
      tickli = self.ax_resid.yaxis.get_ticklines() + self.ax_resid.yaxis.get_minorticklines()
      for number, tick in enumerate(tickli):
        searchItems['tickliY_' + str(number)] = tick

      axes = ['x', 'y']
      if(self.splitShow):
        axes += ['x2']
      for axis in axes:
        for number, line in enumerate(self.gridLinesStore_resid[axis]):
          if(self.gridOrder_resid[axis] == 'front'):
            searchItems['gridli' + axis.upper() + '_' + str(number)] = line
      for axis in axes:
        for number, line in enumerate(self.gridMinorLinesStore_resid[axis]):
          if(self.gridMinorOrder_resid[axis] == 'front'):
            searchItems['minorGridli' + axis.upper() + '_' + str(number)] = line

      # somehow axis spines don't respond to .contains() => work around via canvas click
      ###searchItems['spine2'] = self.ax2.spines['bottom']
      ###searchItems['spine'] = self.ax.spines['top']
      ###searchItems['canvas'] = self.ax2.patch
      
      fallThrough = True
      for itemType, item in searchItems.items():
        if(item.contains(event)[0]):
          fallThrough = False
          self.menu = ConfigPlotMenu(self, item=item, itemType=itemType)
          
      # now need an extra check to decide between axis spines (which don't react to .contains) and the canvas
      if(fallThrough):
        figX, figY = self.residplot.get_figwidth() * self.residplot.dpi, self.residplot.get_figheight() * self.residplot.dpi
        relX, relY = event.x / figX, (figY - event.y) / figY
        tolerance = 0.02
        # check closeness to axis spines
        if(abs(relX - self.padSize['left']) <= tolerance):
          self.menu = ConfigPlotMenu(self, item=self.ax_resid.spines['left'], itemType='spine_left')
          fallThrough = False
        elif(abs(relX - self.padSize['right']) <= tolerance):
          self.menu = ConfigPlotMenu(self, item=self.ax_resid.spines['right'], itemType='spine_right')
          fallThrough = False
        elif(abs(relY - 1 + self.padSize['bottom']) <= tolerance):
          self.menu = ConfigPlotMenu(self, item=self.ax_resid.spines['bottom'], itemType='spine_bottom')
          fallThrough = False
        elif(abs(relY - 1 + self.padSize['top']) <= tolerance):
          self.menu = ConfigPlotMenu(self, item=self.ax_resid.spines['top'], itemType='spine_top')
          fallThrough = False
        # implement checks for split x axis
        if(self.splitShow and fallThrough):
          xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
          if((abs(relX - xbreak - self.padSize['left']) <= tolerance) and (relY <= 1 - self.padSize['bottom']) and (relY >= 1 - self.padSize['top'])):
            self.menu = ConfigPlotMenu(self, item=None, itemType='splitX')
            fallThrough = False
          elif((abs(relX - xbreak - self.padSize['left'] - self.splitPad) <= tolerance) and (relY <= 1 - self.padSize['bottom']) and (relY >= 1 - self.padSize['top'])):
            self.menu = ConfigPlotMenu(self, item=None, itemType='splitX')
            fallThrough = False

      # hit on a graphics item, so should format config menu
      if(not fallThrough):
        needStyling = True
      else:
        needStyling = False
        # compile list of z-ordered items
        zOrderedItems = {}
        for index, entry in enumerate(self.parent.data):
          if((hasattr(entry, 'visibilityResid')) and entry.visibilityResid):
            zOrderedItems[entry.zorderResid] = ['resid', index]
        if(self.visibilityResidLine):
          zOrderedItems[self.zorderResidLine] = ['residLine', 0]
        # now check for click in descending z order
        zKeys = sorted(list(zOrderedItems.keys()))
        currIndex = len(zKeys) - 1
        while(currIndex >= 0):
          currKey = zKeys[currIndex]
          currItem = zOrderedItems[currKey]
          if(currItem[0] == 'resid'):
            relevantHandles = ['handleResid', 'handleResidStack', 'handleResidStackNeg', 'handleResid_div', 'handleResidStack_div', 'handleResidStackNeg_div']
            for handle in relevantHandles:
              if((self.parent.data[currItem[1]].__dict__[handle] != None) and (self.parent.data[currItem[1]].__dict__[handle].contains(event)[0])):
                # found an item that has been clicked on
                currIndex = -1
            # handle bars extra
            for handle in ['handleResidBar', 'handleResidBar_div']:
              if(self.parent.data[currItem[1]].__dict__[handle] != None):
                for patch in self.parent.data[currItem[1]].__dict__[handle].patches:
                  if(patch.contains(event)[0]):
                    # found an item that has been clicked on
                    currIndex = -1
            # did we hit the item?
            if(currIndex == -1):
              self.parent.objectsarea.changeStyle(self.parent.data[currItem[1]], False, True, callButton=None)
          elif(currItem[0] == 'residLine'):
            for handle in ['handleResidZero', 'handleResidZero_div']:
              if((self.__dict__[handle] != None) and (self.__dict__[handle].contains(event)[0])):
                # found an item that has been clicked on
                currIndex = -1
                
            if(currIndex == -1):
              self.parent.objectsarea.changeResidZeroStyle(callButton=None)

          if(currIndex == -1):
            # we hit an item, so don't fall through
            fallThrough = False
    
          # advance item
          currIndex -= 1

      # if still no item found, check for click on plot items
      if(fallThrough):
        searchItems = {}
        axes = ['x', 'y']
        if(self.splitShow):
          axes += ['x2']
        for axis in axes:
          for number, line in enumerate(self.gridLinesStore_resid[axis]):
            if(self.gridOrder_resid[axis] == 'back'):
              searchItems['gridli' + axis.upper() + '_' + str(number)] = line
        for axis in axes:
          for number, line in enumerate(self.gridMinorLinesStore_resid[axis]):
            if(self.gridMinorOrder_resid[axis] == 'back'):
              searchItems['minorGridli' + axis.upper() + '_' + str(number)] = line
  
        for itemType, item in searchItems.items():
          if(item.contains(event)[0]):
            fallThrough = False
            self.menu = ConfigPlotMenu(self, item=item, itemType=itemType)
        
        if(fallThrough):
          self.menu = ConfigPlotMenu(self, item=self.ax_resid.patch, itemType='canvas')
        needStyling = True
        
      # is there a context menu that would require styling?
      if(needStyling):
        # apply styles to popup window
        if(QSTYLE != None):
          self.menu.setStyle(QSTYLE)
        if(QSTYLESHEET != None):
          self.menu.setStyleSheet(QSTYLESHEET)
        # first need to display QMenu to get reliable size (even sizeHint fails)
        menuPos = QtGui.QCursor.pos()
        self.menu.popup(menuPos)
        # now move window to new position
        menuX = menuPos.x() - self.menu.width() / 2
        menuX = max(menuX, 0)
        menuPos.setX(int(menuX))
        self.menu.move(menuPos)

  def dynamicLegend(self, deltaX, deltaY, pickedAxes, pickedBackground, transient=True):
    # function called by interactive drag and resize
    if((pickedAxes != None) and (pickedBackground != None)):
      renderer = self.parent.plotArea.matplot.canvas.renderer
      
      # restore background
      pickedAxes.figure.canvas.restore_region(pickedBackground)

      # update object
      if(hasattr(self.legendHandle, 'draw')):
        self.legendHandle.set_loc((deltaX, deltaY))
        self.legendHandle.draw(renderer)
    
      # and blit just the redrawn area
      pickedAxes.figure.canvas.blit(pickedAxes.bbox)
      
      # accept these changes?
      if(not transient):
        self.legendX, self.legendY = deltaX, deltaY
        # update dialog
        self.parent.graphicsarea.configLegendX.setText(self.parent.formatNumber(self.legendX))
        self.parent.graphicsarea.configLegendY.setText(self.parent.formatNumber(self.legendY))
        
        # issue a refresh to accurately deal with cursor
        self.dataplotwidget.myRefresh()

  def releaseHandler(self, event):
    # called to turn off extra picking
    if(self.pickedLegend and (self.startX != None) and (self.startY != None)):
      width, height = self.matplot.get_size_inches() * self.matplot.dpi
      xdata, ydata = (event.x - self.startX + self.legendOrigX) / width, (event.y - self.startY + self.legendOrigY) / height
      self.dynamicLegend(xdata, ydata, pickedAxes=self.pickedAxes, pickedBackground=self.pickedBackground, transient=False)
    elif(self.pickedExtra != None):
      if(event.inaxes == self.inAxes):
        # map to correct axis
        xdata, ydata = self.pickedAxes.transData.inverted().transform((event.x, event.y))
        xdata, ydata = xdata - self.startX, ydata - self.startY
      else:
        xdata, ydata = 0.0, 0.0
      self.parent.extras[self.pickedExtra].updateMe(xdata, ydata, pickedAxes=self.pickedAxes,\
                        pickedBackground=self.pickedBackground, pickedMode=self.pickedMode,\
                        quadrantX=self.quadrantX, quadrantY=self.quadrantY, transient=False)
      if(self.splitY and self.parent.extras[self.pickedExtra].onBothAxes):
        self.parent.extras[self.pickedExtra].duplicateForSplit()
      self.parent.extras[self.pickedExtra].drawMe(redraw=True)
    self.pickedExtra, self.pickedAxes, self.pickedBackground, self.pickedMode, self.inAxes = None, None, None, 0, None
    self.pickedLegend = False
    # do we need to restore a rectangle selector?
    if(self.selector2Restore != None):
      self.selector2Restore.set_active(True)
      self.selector2Restore = None
    # do we need to restore cursors?
    for cursor in self.cursors2Restore:
      cursor.visible = True
      cursor.linev.set_visible(True)
      cursor.lineh.set_visible(True)
      cursor.label.set_visible(True)
    self.cursors2Restore = []

  def moveHandler(self, event):
    # called to allow moving and resizing of extras
    if(self.pickedLegend and (self.startX != None) and (self.startY != None)):
      width, height = self.matplot.get_size_inches() * self.matplot.dpi
      xdata, ydata = (event.x - self.startX + self.legendOrigX) / width, (event.y - self.startY + self.legendOrigY) / height
      self.dynamicLegend(xdata, ydata, pickedAxes=self.pickedAxes, pickedBackground=self.pickedBackground, transient=True)
    elif(self.pickedExtra != None):
      if(event.inaxes == self.inAxes):
        # map to correct axis
        xdata, ydata = self.pickedAxes.transData.inverted().transform((event.x, event.y))
        xdata, ydata = xdata - self.startX, ydata - self.startY
        self.parent.extras[self.pickedExtra].updateMe(xdata, ydata, pickedAxes=self.pickedAxes,\
                          pickedBackground=self.pickedBackground, pickedMode=self.pickedMode,\
                          quadrantX=self.quadrantX, quadrantY=self.quadrantY, transient=True)

  def toggleCrossHairEncore(self, event):
    # combined handler for all click events
    modifiers = QtWidgets.QApplication.queryKeyboardModifiers()
    # need to save event modifiers at time of click to get handlers right
    self.toggleShift, self.toggleCtrl = modifiers & QtCore.Qt.KeyboardModifier.ShiftModifier, modifiers & QtCore.Qt.KeyboardModifier.ControlModifier
    # determine which axis we are on
    inaxes = event.inaxes
    # and then on with all the rest
    fallThrough = False
    if(((modifiers & QtCore.Qt.KeyboardModifier.ControlModifier) or (modifiers & QtCore.Qt.KeyboardModifier.ShiftModifier) or self.drawFreehand) and (event.button == 1)):
      # determine mode
      if(self.drawFreehand):
        # implement check that a valid axes has been selected
        if(inaxes in [self.ax, self.ax2, self.ax_div, self.ax2_div]):
          # for myterious reasons, older Python version under Linux does not correctly assign pickedMode (it reverts to zero!)
          # => go via extra flag pick2Freehand
          self.pickedMode, self.pick2Freehand = 3, True
          # switch to ad hoc lasso to avoid shenanigans with the rectangle selector
          self.lassoSelector = MyLassoli(ax=inaxes, xy=(event.xdata, event.ydata), callback=self.lassoCallback, useblit=True, parent=self)
          self.lassoSelector.line.set_color([0.2, 0.9, 0.15, 1.0])
      else:
        if((modifiers & QtCore.Qt.KeyboardModifier.ControlModifier) and (modifiers & QtCore.Qt.KeyboardModifier.ShiftModifier)):
          # Ctrl/shift for rotation
          self.pickedMode = 2
        elif(modifiers & QtCore.Qt.KeyboardModifier.ControlModifier):
          # Ctrl for resizing
          self.pickedMode = 1
        else:
          # shift for moving
          self.pickedMode = 0
          # check whether click was on legend
          if(hasattr(self.legendHandle, 'contains')):
            if(self.legendHandle.contains(event)[0]):
              self.pickedLegend = True
              
              # save canvas for blitting
              if(inaxes == None):
                # default to self.ax to enable moving legend outside of canvas
                self.pickedAxes = self.ax
              else:
                self.pickedAxes = inaxes
                
              # save canvas for blitting
              self.pickedBackground = self.pickedAxes.figure.canvas.copy_from_bbox(self.pickedAxes.bbox)
              
              # find where legend is currently displayed
              # still somewhat buggy for 'best' legend placement, reason unclear
              bbox = self.legendHandle.get_tightbbox()
              self.legendOrigX, self.legendOrigY = bbox.x0, bbox.y0
              self.startX, self.startY = event.x, event.y
              
              # finally, toggle legend and menu entry to custom location
              if(self.legendPlacement != 'custom'):
                self.legendPlacement = 'custom'
                self.parent.graphicsarea.configLegendPlacement.setCurrentText('custom')
                self.parent.statusbar.showMessage('Switched to custom placement of legend.', self.parent.STATUS_TIME, color='blue')

              # if cursor active, temporarily disable it, to avoid blitting catastrophe
              if(self.cursorVisible):
                # toggle visibility (in correct order to get display of twins right)
                order = [self.cursor2_div, self.cursor2, self.cursor_div, self.cursor]
                
                for cursor in order:
                  if((cursor != None) and cursor.visible):
                    self.cursors2Restore.append(cursor)
                    cursor.visible = False
                    cursor.linev.set_visible(False)
                    cursor.lineh.set_visible(False)
                    cursor.label.set_visible(False)
          else:
            # for good measure, reinitialize a bunch of values that would be used in move and release handlers
            self.legendOrigX, self.legendOrigY, self.startX, self.startY = None, None, None, None
        if(not self.pickedLegend):
          # compile list of z-ordered extras
          zOrderedExtras = {}
          for index, entry in enumerate(self.parent.extras):
            if((hasattr(entry, 'visibility')) and entry.visibility):
              zOrderedExtras[entry.zorder] = index
          # now check for click in descending z order
          zKeys = sorted(list(zOrderedExtras.keys()))
          currIndex = len(zKeys) - 1
          while(currIndex >= 0):
            currKey = zKeys[currIndex]
            currItem = zOrderedExtras[currKey]          
    
            relevantHandles = ['handle', 'handle_div']
            # check for items on splitY
            if(self.splitY):
              # determine whether the event originated from the same axis as the object is on
              flag = False
              if((inaxes in [self.ax, self.ax_div]) and (not self.parent.extras[currItem].onSecondAxes)):
                flag = True
              if((inaxes in [self.ax2, self.ax2_div]) and self.parent.extras[currItem].onSecondAxes):
                flag = True
            else:
              flag = True
    
            check = False
            for handle in relevantHandles:
              if(flag):
                if((self.parent.extras[currItem].__dict__[handle] != None) and (self.parent.extras[currItem].__dict__[handle].contains(event)[0]) and (currIndex != -1)):
                  # found an item that has been clicked on - break out of loop
                  currIndex = -1
                  self.pickedExtra = currItem
              elif((handle in self.parent.extras[currItem].handlesBothAxes) and (self.parent.extras[currItem].handlesBothAxes[handle] != None) and (self.parent.extras[currItem].handlesBothAxes[handle].contains(event)[0]) and (currIndex != -1)):
                # found an item that has been clicked on - break out of loop
                currIndex = -1
                self.pickedExtra = currItem
                  
              if((currIndex == -1) and (not check)):
                check = True
                # determine which axes to draw on
                self.pickedAxes = inaxes
                self.inAxes = inaxes
                  
                # save canvas for blitting
                self.pickedBackground = self.pickedAxes.figure.canvas.copy_from_bbox(self.pickedAxes.bbox)
    
                # map to correct axis
                self.startX, self.startY = self.pickedAxes.transData.inverted().transform((event.x, event.y))
                # determine quadrant on which extra was clicked
                if(self.startX < self.parent.extras[self.pickedExtra].x):# + 0.5 * self.parent.extras[self.pickedExtra].x2)):
                  self.quadrantX = 'left'
                else:
                  self.quadrantX = 'right'
                if(self.startY < self.parent.extras[self.pickedExtra].y):# + 0.5 * self.parent.extras[self.pickedExtra].y2)):
                  self.quadrantY = 'bottom'
                else:
                  self.quadrantY = 'top'
                
                # temporarily make extra semi-transparent
                self.parent.extras[self.pickedExtra].tempTransparency()
                #print(self.parent.extras[self.pickedExtra].x, self.parent.extras[self.pickedExtra].y, self.parent.extras[self.pickedExtra].x2, self.parent.extras[self.pickedExtra].y2)
                
                # if cursor active, temporarily disable it, to avoid blitting catastrophe
                if(self.cursorVisible):
                  # toggle visibility (in correct order to get display of twins right)
                  order = [self.cursor2_div, self.cursor2, self.cursor_div, self.cursor]
                  
                  for cursor in order:
                    if((cursor != None) and cursor.visible):
                      self.cursors2Restore.append(cursor)
                      cursor.visible = False
                      cursor.linev.set_visible(False)
                      cursor.lineh.set_visible(False)
                      cursor.label.set_visible(False)
    
            # advance item
            currIndex -= 1
            
          # now check whether an extra has been picked
          # if no, we can launch our lassoli
          if((currIndex == -1) and (modifiers & QtCore.Qt.KeyboardModifier.ShiftModifier)):
            # implement check that a valid axes has been selected
            if(inaxes in [self.ax, self.ax2, self.ax_div, self.ax2_div]):
              # switch to ad hoc lasso to avoid shenanigans with the rectangle selector
              self.lassoSelector = MyLassoli(ax=inaxes, xy=(event.xdata, event.ydata), callback=self.lassoCallback, useblit=True, parent=self)
          
    elif((event.button == 1) and not event.dblclick):
      # implement check that a valid axes has been selected
      if(inaxes in [self.ax, self.ax2, self.ax_div, self.ax2_div]):
        self.rectangleSelector = MyRectangli(ax=inaxes, xy=(event.xdata, event.ydata), callback=self.rectangleCallback, useblit=True, parent=self)

    elif(event.button == 3):
      # right mouse button => unzoom
      if(len(self.storeCoord) > 0):
        coords = self.storeCoord[-8:]
        self.storeCoord = self.storeCoord[:-8]

        # if cursor active, temporarily disable it, lest new matplotlib slows down immensely
        if(self.cursorVisible):
          # toggle visibility (in correct order to get display of twins right)
          order = [self.cursor2_div, self.cursor2, self.cursor_div, self.cursor]
          
          for cursor in order:
            if((cursor != None) and cursor.visible):
              self.cursors2Restore.append(cursor)
              if(True):
                cursor.visible = False
                cursor.linev.set_visible(False)
                cursor.lineh.set_visible(False)
                cursor.label.set_visible(False)
        
        # set axes limits if required
        self.rectSelectorCallbackHelper(coords=coords, divAxis=False, restore=True)
        
    elif((event.button == 1) and event.dblclick):
      fallThrough = True
      # should first check for certain display items (because curves somehow hog click events even when not under cursor)
      searchItems = {}
      searchItems['xlabel'] = self.ax2.xaxis.label
      searchItems['ylabel'] = self.ax.yaxis.label
      if(self.isSecondAxesActive()):
        searchItems['ylabel2'] = self.ax2.yaxis.label
      # need to expand for slaved second x axis
      searchItems['slavedxlabel'] = self.handleSlavedXLabel
      searchItems['legend'] = self.legendHandle

      # need to expand for slaved second x axis
      # these should come first b/c otherwise the regular upper labels win out
      for number, label in enumerate(self.secondXLabels):
        searchItems['labliSlavedX_' + str(number)] = label
      for number, label in enumerate(self.secondXLabels_div):
        searchItems['labliSlavedX2_' + str(number)] = label

      labli = self.ax2.get_xticklabels(which='both')
      for number, label in enumerate(labli):
        searchItems['labliX_' + str(number)] = label
      if(self.splitShow):
        labli = self.ax2_div.get_xticklabels(which='both')
        for number, label in enumerate(labli):
          searchItems['labliX2_' + str(number)] = label
      labli = self.ax.get_yticklabels(which='both')
      for number, label in enumerate(labli):
        searchItems['labliY_' + str(number)] = label
      if(self.isSecondAxesActive()):
        labli = self.ax2.get_yticklabels(which='both')
        for number, label in enumerate(labli):
          searchItems['labliY2_' + str(number)] = label

      # are split axes active?
      if(self.splitShow):
        if(self.handleDivider != None):
          for number, item in enumerate(self.handleDivider):
            searchItems['dividerX_' + str(number)] = item
      if(self.splitY):
        if(self.handleDividerY != None):
          for number, item in enumerate(self.handleDividerY):
            searchItems['dividerY_' + str(number)] = item

      # need to expand for slaved second x axis
      # these should come first b/c otherwise the regular upper ticks win out
      for number, tick in enumerate(self.secondXLines):
        searchItems['tickliSlavedX_' + str(number)] = tick
      for number, tick in enumerate(self.secondXLines_div):
        searchItems['tickliSlavedX2_' + str(number)] = tick

      tickli = self.ax2.xaxis.get_ticklines() + self.ax2.xaxis.get_minorticklines()
      for number, tick in enumerate(tickli):
        searchItems['tickliX_' + str(number)] = tick
      if(self.splitShow):
        tickli = self.ax2_div.xaxis.get_ticklines() + self.ax2_div.xaxis.get_minorticklines()
        for number, tick in enumerate(tickli):
          searchItems['tickliX2_' + str(number)] = tick
      tickli = self.ax.yaxis.get_ticklines() + self.ax.yaxis.get_minorticklines()
      for number, tick in enumerate(tickli):
        searchItems['tickliY_' + str(number)] = tick
      if(self.isSecondAxesActive()):
        tickli = self.ax2.yaxis.get_ticklines() + self.ax2.yaxis.get_minorticklines()
        for number, tick in enumerate(tickli):
          searchItems['tickliY2_' + str(number)] = tick
          
      # need to check for gridOrder and gridMinorOrder => only consider grid lines here that are displayed in front
      axes = ['x', 'y']
      if(self.splitShow):
        axes += ['x2']
      if(self.isSecondAxesActive()):
        axes += ['y2']
      for axis in axes:
        for number, line in enumerate(self.gridLinesStore[axis]):
          if(self.gridOrder[axis] == 'front'):
            searchItems['gridli' + axis.upper() + '_' + str(number)] = line
      for axis in axes:
        for number, line in enumerate(self.gridMinorLinesStore[axis]):
          if(self.gridMinorOrder[axis] == 'front'):
            searchItems['minorGridli' + axis.upper() + '_' + str(number)] = line

      # somehow axis spines don't respond to .contains() => work around via canvas click
      ###searchItems['spine2'] = self.ax2.spines['bottom']
      ###searchItems['spine'] = self.ax.spines['top']
      ###searchItems['canvas'] = self.ax2.patch
      
      for itemType, item in searchItems.items():
        if(fallThrough and hasattr(item, 'contains') and item.contains(event)[0]):
          fallThrough = False
          self.menu = ConfigPlotMenu(self, item=item, itemType=itemType)
          
      # now need an extra check to decide between axis spines (which don't react to .contains) and the canvas
      if(fallThrough):
        figX, figY = self.matplot.get_figwidth() * self.matplot.dpi, self.matplot.get_figheight() * self.matplot.dpi
        relX, relY = event.x / figX, (figY - event.y) / figY
        tolerance = 0.02
        # check closeness to axis spines
        if(self.axisVisible['left'] and (abs(relX - self.padSize['left']) <= tolerance) and (relY <= 1 - self.padSize['bottom']) and (relY >= 1 - self.padSize['top'])):
          self.menu = ConfigPlotMenu(self, item=self.ax.spines['left'], itemType='spine_left')
          fallThrough = False
        elif(self.axisVisible['right'] and (abs(relX - self.padSize['right']) <= tolerance) and (relY <= 1 - self.padSize['bottom']) and (relY >= 1 - self.padSize['top'])):
          self.menu = ConfigPlotMenu(self, item=self.ax.spines['right'], itemType='spine_right')
          fallThrough = False
        elif(self.axisVisible['bottom'] and (abs(relY - 1 + self.padSize['bottom']) <= tolerance) and (relX >= self.padSize['left']) and (relX <= self.padSize['right'])):
          self.menu = ConfigPlotMenu(self, item=self.ax2.spines['bottom'], itemType='spine_bottom')
          fallThrough = False
        elif(self.axisVisible['top'] and (abs(relY - 1 + self.padSize['top']) <= tolerance) and (relX >= self.padSize['left']) and (relX <= self.padSize['right'])):
          self.menu = ConfigPlotMenu(self, item=self.ax2.spines['top'], itemType='spine_top')
          fallThrough = False
        # implement checks for split x and y axes
        if(self.splitShow and fallThrough):
          xbreak = (self.padSize['right'] - self.padSize['left'] - self.splitPad) * self.splitFraction / (1.0 + self.splitFraction)
          if((abs(relX - xbreak - self.padSize['left']) <= tolerance) and (relY <= 1 - self.padSize['bottom']) and (relY >= 1 - self.padSize['top'])):
            self.menu = ConfigPlotMenu(self, item=None, itemType='splitX')
            fallThrough = False
          elif((abs(relX - xbreak - self.padSize['left'] - self.splitPad) <= tolerance) and (relY <= 1 - self.padSize['bottom']) and (relY >= 1 - self.padSize['top'])):
            self.menu = ConfigPlotMenu(self, item=None, itemType='splitX')
            fallThrough = False
        if(self.splitY and fallThrough):
          ybreak = (self.padSize['top'] - self.padSize['bottom'] - self.splitYPad) * self.splitYFraction / (1.0 + self.splitYFraction)
          if((abs(1 - self.padSize['bottom'] - ybreak - relY) <= tolerance) and (relX >= self.padSize['left']) and (relX <= self.padSize['right'])):
            self.menu = ConfigPlotMenu(self, item=None, itemType='splitY')
            fallThrough = False
          elif((abs(1 - self.padSize['bottom'] - ybreak - self.splitYPad - relY) <= tolerance) and (relX >= self.padSize['left']) and (relX <= self.padSize['right'])):
            self.menu = ConfigPlotMenu(self, item=None, itemType='splitY')
            fallThrough = False

      # hit on a graphics item, so should format config menu
      if(not fallThrough):
        needStyling = True
      else:
        needStyling = False
        # check for click on plot items
        if(fallThrough):
          # compile list of z-ordered items
          zOrderedItems = {}
          for index, entry in enumerate(self.parent.fit):
            if((hasattr(entry, 'visibility')) and entry.visibility):
              zOrderedItems[entry.zorder] = ['fit', index]
          for index, entry in enumerate(self.parent.data):
            if((hasattr(entry, 'visibility')) and entry.visibility):
              zOrderedItems[entry.zorder] = ['data', index]
          for index, entry in enumerate(self.parent.extras):
            if((hasattr(entry, 'visibility')) and entry.visibility):
              zOrderedItems[entry.zorder] = ['extra', index]
          # now check for click in descending z order
          zKeys = sorted(list(zOrderedItems.keys()))
          currIndex = len(zKeys) - 1
          while(currIndex >= 0):
            currKey = zKeys[currIndex]
            currItem = zOrderedItems[currKey]
            if(currItem[0] == 'fit'):
              relevantHandles = ['handlePlot', 'handlePlot_div', 'handleBoot', 'handleBoot_div']
              # check for items on splitY
              if(self.splitY):
                # determine whether the event originated from the same axis as the object is on
                flag = False
                if((inaxes in [self.ax, self.ax_div]) and (not self.parent.fit[currItem[1]].onSecondAxes)):
                  flag = True
                if((inaxes in [self.ax2, self.ax2_div]) and self.parent.fit[currItem[1]].onSecondAxes):
                  flag = True
                # check whether we have item on both axes
                if(self.parent.fit[currItem[1]].onBothAxes and (not flag)):
                  for handle in relevantHandles:
                    if((handle in self.parent.fit[currItem[1]].handlesBothAxes) and (self.parent.fit[currItem[1]].handlesBothAxes[handle] != None) and (self.parent.fit[currItem[1]].handlesBothAxes[handle].contains(event)[0])):
                      # found an item that has been clicked on - break out of loop
                      currIndex = -1
              else:
                flag = True
              # normal click handling
              if(flag):
                for handle in relevantHandles:
                  if((self.parent.fit[currItem[1]].__dict__[handle] != None) and (self.parent.fit[currItem[1]].__dict__[handle].contains(event)[0])):
                    # found an item that has been clicked on - break out of loop
                    currIndex = -1
              if(currIndex == -1):
                self.parent.objectsarea.changeStyle(self.parent.fit[currItem[1]], False, False, callButton=None)
            elif(currItem[0] == 'data'):
              relevantHandles = ['handleData', 'handleStack', 'handleStackNeg', 'handleData_div', 'handleStack_div', 'handleStackNeg_div']
              # check for items on splitY
              if(self.splitY):
                # determine whether the event originated from the same axis as the object is on
                flag = False
                if((inaxes in [self.ax, self.ax_div]) and (not self.parent.data[currItem[1]].onSecondAxes)):
                  flag = True
                if((inaxes in [self.ax2, self.ax2_div]) and self.parent.data[currItem[1]].onSecondAxes):
                  flag = True
                flag2 = (not flag) and self.parent.data[currItem[1]].onBothAxes
              else:
                flag, flag2 = True, False
              # cycle through items
              for handle in relevantHandles:
                if(flag):
                  if((self.parent.data[currItem[1]].__dict__[handle] != None) and (self.parent.data[currItem[1]].__dict__[handle].contains(event)[0])):
                    # found an item that has been clicked on
                    currIndex = -1
                if(flag2):
                  if((handle in self.parent.data[currItem[1]].handlesBothAxes) and (self.parent.data[currItem[1]].handlesBothAxes[handle] != None) and (self.parent.data[currItem[1]].handlesBothAxes[handle].contains(event)[0])):
                    # found an item that has been clicked on - break out of loop
                    currIndex = -1
              # handle bars extra
              for handle in ['handleBar', 'handleBar_div']:
                if(flag):
                  if(self.parent.data[currItem[1]].__dict__[handle] != None):
                    for patch in self.parent.data[currItem[1]].__dict__[handle].patches:
                      if(patch.contains(event)[0]):
                        # found an item that has been clicked on
                        currIndex = -1
                if(flag2):
                  if((handle in self.parent.data[currItem[1]].handlesBothAxes) and (self.parent.data[currItem[1]].handlesBothAxes[handle] != None)):
                    for patch in self.parent.data[currItem[1]].handlesBothAxes[handle].patches:
                      if(patch.contains(event)[0]):
                        # found an item that has been clicked on
                        currIndex = -1
              # handle violins extra
              for handle in ['handleViolin', 'handleViolin_div']:
                if(flag):
                  if(self.parent.data[currItem[1]].__dict__[handle] != None):
                    for entry in self.parent.data[currItem[1]].__dict__[handle]:
                      if(type(self.parent.data[currItem[1]].__dict__[handle][entry]) == type([])):
                        for item in self.parent.data[currItem[1]].__dict__[handle][entry]:
                          if(item.contains(event)[0]):
                            # found an item that has been clicked on
                            currIndex = -1
                      else:
                        if(self.parent.data[currItem[1]].__dict__[handle][entry].contains(event)[0]):
                          # found an item that has been clicked on
                          currIndex = -1
                if(flag2):
                  if((handle in self.parent.data[currItem[1]].handlesBothAxes) and (self.parent.data[currItem[1]].handlesBothAxes[handle] != None)):
                    for entry in self.parent.data[currItem[1]].handlesBothAxes[handle]:
                      if(type(self.parent.data[currItem[1]].handlesBothAxes[handle][entry]) == type([])):
                        for item in self.parent.data[currItem[1]].handlesBothAxes[handle][entry]:
                          if(item.contains(event)[0]):
                            # found an item that has been clicked on
                            currIndex = -1
                      else:
                        if(self.parent.data[currItem[1]].handlesBothAxes[handle][entry].contains(event)[0]):
                          # found an item that has been clicked on
                          currIndex = -1
                        
              # did we hit the item?
              if(currIndex == -1):
                self.parent.objectsarea.changeStyle(self.parent.data[currItem[1]], True, False, callButton=None)
            elif(currItem[0] == 'extra'):
              relevantHandles = ['handle', 'handle_div']
              # check for items on splitY
              if(self.splitY):
                # determine whether the event originated from the same axis as the object is on
                flag = False
                if((inaxes in [self.ax, self.ax_div]) and (not self.parent.extras[currItem[1]].onSecondAxes)):
                  flag = True
                if((inaxes in [self.ax2, self.ax2_div]) and self.parent.extras[currItem[1]].onSecondAxes):
                  flag = True
                # check whether we have item on both axes
                if(self.parent.extras[currItem[1]].onBothAxes and (not flag)):
                  for handle in relevantHandles:
                    if((handle in self.parent.extras[currItem[1]].handlesBothAxes) and (self.parent.extras[currItem[1]].handlesBothAxes[handle] != None) and (self.parent.extras[currItem[1]].handlesBothAxes[handle].contains(event)[0])):
                      # found an item that has been clicked on - break out of loop
                      currIndex = -1
              else:
                flag = True
              # normal click handling
              if(flag):
                for handle in relevantHandles:
                  if((self.parent.extras[currItem[1]].__dict__[handle] != None) and (self.parent.extras[currItem[1]].__dict__[handle].contains(event)[0])):
                    # found an item that has been clicked on - break out of loop
                    currIndex = -1
              if(currIndex == -1):
                self.parent.objectsarea.changeStyleExtra(currItem[1], callButton=None)
    
            if(currIndex == -1):
              # we hit an item, so don't fall through
              fallThrough = False
    
            # advance item
            currIndex -= 1
       
      # if still no item found, check for grid in the back, and then configure canvas
      if(fallThrough):
        # check grid lines that are in the back of the plot objects
        searchItems = {}
        axes = ['x', 'y']
        if(self.splitShow):
          axes += ['x2']
        if(self.isSecondAxesActive()):
          axes += ['y2']
        for axis in axes:
          for number, line in enumerate(self.gridLinesStore[axis]):
            if(self.gridOrder[axis] == 'back'):
              searchItems['gridli' + axis.upper() + '_' + str(number)] = line
        for axis in axes:
          for number, line in enumerate(self.gridMinorLinesStore[axis]):
            if(self.gridMinorOrder[axis] == 'back'):
              searchItems['minorGridli' + axis.upper() + '_' + str(number)] = line

        for itemType, item in searchItems.items():
          if(fallThrough and hasattr(item, 'contains') and item.contains(event)[0]):
            fallThrough = False
            self.menu = ConfigPlotMenu(self, item=item, itemType=itemType)
        
        if(fallThrough):
          self.menu = ConfigPlotMenu(self, item=self.ax2.patch, itemType='canvas')
        needStyling = True
        
      # is there a context menu that would require styling?
      if(needStyling):
        # apply styles to popup window
        if(QSTYLE != None):
          self.menu.setStyle(QSTYLE)
        if(QSTYLESHEET != None):
          self.menu.setStyleSheet(QSTYLESHEET)
        # first need to display QMenu to get reliable size (even sizeHint fails)
        menuPos = QtGui.QCursor.pos()
        self.menu.popup(menuPos)
        # now move window to new position
        menuX = menuPos.x() - self.menu.width() / 2
        menuX = max(menuX, 0)
        menuPos.setX(int(menuX))
        self.menu.move(menuPos)
            
    # disable double-click for toggling cross hair (due to config of plot elements on double click, see above)
    fallThrough = False
    if((event.button == 2) or fallThrough):
      # toggle cross hair on middle mouse button -- cannot use QtCore.Qt.MidButton as this equates to 4?!
      self.cursorVisible = not self.cursorVisible
      # check whether cursor already exists
      if(self.cursor == None):
        self.cursor = MyCursor(self.ax, useblit=True, color='black', linewidth=1)
        self.cursor.setParent(self)
        self.cursor.setAx2(self.ax2)
      if(self.cursor_div == None):
        if(self.splitShow):
          self.cursor_div = MyCursor(self.ax_div, useblit=True, color='black', linewidth=1)
          self.cursor_div.setParent(self)
          self.cursor_div.setAx2(self.ax2_div)
          
      # link cursors
      # as of matplotlib 3.1.2. we have to make sure that split axes are displayed before we can initialize the div cursor
      if(self.cursor_div != None):
        self.cursor.setTwin(self.cursor_div)
        self.cursor_div.setTwin(self.cursor)
      
      # deal with splity cursors
      if((self.cursor2 == None) and self.splitY):
        self.cursor2 = MyCursor(self.ax2, useblit=True, color='black', linewidth=1)
        self.cursor2.setParent(self)
        self.cursor2.setAx2(self.ax2)
      if((self.cursor2_div == None) and self.splitY and self.splitShow):
        self.cursor2_div = MyCursor(self.ax2_div, useblit=True, color='black', linewidth=1)
        self.cursor2_div.setParent(self)
        self.cursor2_div.setAx2(self.ax2_div)
    
      # link cursors
      if(self.cursor2 != None):
        self.cursor.setTwinVertical(self.cursor2)
        self.cursor2.setTwinVertical(self.cursor)
        self.cursor2.setTwin(self.cursor2_div)

      if(self.cursor2_div != None):
        self.cursor_div.setTwinVertical(self.cursor2_div)
        self.cursor2_div.setTwinVertical(self.cursor_div)
        self.cursor2_div.setTwin(self.cursor2)

      # toggle visibility (in correct order to get display of twins right)
      if(event.inaxes == self.ax):
        order = [self.cursor2_div, self.cursor2, self.cursor_div, self.cursor]
      elif(event.inaxes == self.ax_div):
        order = [self.cursor2, self.cursor2_div, self.cursor, self.cursor_div]
      elif(event.inaxes == self.ax2):
        order = [self.cursor_div, self.cursor2_div, self.cursor, self.cursor2]
      else:
        order = [self.cursor, self.cursor2, self.cursor_div, self.cursor2_div]
        
      if(self.splitY):
        for cursor in order:
          if(cursor != None):
            cursor.toggleVisibility(self.cursorVisible, event)
            
        # extra call to turn off surplus vertical lines
        if(order[0] != None):
          order[0].linev.set_visible(False)
          order[0].lineh.set_visible(False)
          order[0]._update()
      else:
        for cursor in order[-2:]:
          if(cursor != None):
            cursor.toggleVisibility(self.cursorVisible, event)
            
    # reset freehand toggle
    if(self.drawFreehand):
      self.drawFreehand = False
      self.parent.objectsarea.extrasCreateFreehandButton.setChecked(False)
      
  def mouseScrolled(self, event):
    # adjusts zoom level when Ctrl is pressed also
    modifiers = QtWidgets.QApplication.queryKeyboardModifiers()
    if((modifiers & QtCore.Qt.KeyboardModifier.ControlModifier) and (event.button in ['down', 'up'])):
      if(event.button == 'down'):
        self.incZoomLevel(1)
      else:
        self.incZoomLevel(-1)

  def destructAboutLogo(self):
    # destroys about logo
    counter = self.dataplotwidget.getDestructionCounter()
    if(counter >= 0):
      counter -= 1
      self.dataplotwidget.setDestructionCounter(np.max((counter, 0)))
      if((counter <= 0) and (len(self.handlesAbout))):
        for entry in self.handlesAbout:
          if(hasattr(entry, 'remove')):
            entry.remove()
        self.handlesAbout = []

  def drawAboutLogo(self, aspect=0, destructCounter=1):
    # draws program info on canvas
    # helper function that transforms coordinates according to axis settings
    def processCoordSet(coords, minX, maxX, modeX, minY, maxY, modeY, relWidth, relHeight, relOffsetX, relOffsetY):
      coords /= 100.0
      # process X coords
      if(modeX == 'linear'):
        coords[:,0] *= (maxX - minX) * relWidth
        coords[:,0] += minX + (maxX - minX) * relOffsetX
      else:
        minX, maxX = np.log(minX), np.log(maxX)
        coords[:,0] *= (maxX - minX) * relWidth
        coords[:,0] += minX + (maxX - minX) * relOffsetX
        coords[:,0] = np.exp(coords[:,0])
      # process Y coords
      if(modeY == 'linear'):
        coords[:,1] *= (maxY - minY) * relHeight
        coords[:,1] += minY + (maxY - minY) * relOffsetY
      else:
        minY, maxY = np.log(minY), np.log(maxY)
        coords[:,1] *= (maxY - minY) * relHeight
        coords[:,1] += minY + (maxY - minY) * relOffsetY
        coords[:,1] = np.exp(coords[:,1])
      return coords

    # helper function that transforms coordinates according to axis settings
    def processCoordSingle(minX, maxX, modeX, minY, maxY, modeY, relOffsetX, relOffsetY):
      # process X coords
      if(modeX == 'linear'):
        x = minX + (maxX - minX) * relOffsetX
      else:
        minX, maxX = np.log(minX), np.log(maxX)
        x = minX + (maxX - minX) * relOffsetX
        x = np.exp(x)
      # process Y coords
      if(modeY == 'linear'):
        y = minY + (maxY - minY) * relOffsetY
      else:
        minY, maxY = np.log(minY), np.log(maxY)
        y = minY + (maxY - minY) * relOffsetY
        y = np.exp(y)
      return [x, y]
    
    # check whether a previous logo is still displayed?
    if(not len(self.handlesAbout)):
      # settings
      zOffset = 2001 # should be in front of selectedData which are at 2000
      ubtCol = [0.051, 0.306, 0.702, 1.0]
      greyCol = [0.3, 0.3, 0.3, 1.0]
      blackCol = [0.0, 0.0, 0.0, 1.0]
      
      # retrieve axis info
      targetAxis = self.ax
      minX, maxX = self.minX, self.maxX
      minY, maxY = self.minY, self.maxY
      modeX, modeY = self.modeX, self.modeY

      # calculate draw dimensions (will depend on split axes and padding of plot)
      if(self.splitShow):
        if(modeX == 'linear'):
          # account for ratio of split axes
          deltaX = (maxX - minX) * (1 + self.splitFraction) / self.splitFraction
          # correct for padding in the middle
          try:
            deltaX /= (1 - self.padSize['left'] - (1 - self.padSize['right']) - self.splitPad)
            deltaX *= (1 - self.padSize['left'] - (1 - self.padSize['right']))
          except:
            pass
          maxX = minX + deltaX
        else:
          deltaX = (maxX / minX) ** ((1 + self.splitFraction) / self.splitFraction)
          try:
            factor = 1 / (1 - self.padSize['left'] - (1 - self.padSize['right']) - self.splitPad)
            factor *= (1 - self.padSize['left'] - (1 - self.padSize['right']))
            deltaX = deltaX ** factor
          except:
            pass
          maxX = minX * deltaX
      if(self.splitY):
        # account for ratio of split axes
        if(modeY == 'linear'):
          deltaY = (maxY - minY) * (1 + self.splitYFraction)
          # correct for padding in the middle
          try:
            deltaY /= (1 - self.padSize['bottom'] - (1 - self.padSize['top']) - self.splitYPad)
            deltaY *= (1 - self.padSize['bottom'] - (1 - self.padSize['top']))
          except:
            pass
          minY = maxY - deltaY
        else:
          deltaY = (maxY / minY) ** (1 + self.splitYFraction)
          try:
            factor = 1 / (1 - self.padSize['bottom'] - (1 - self.padSize['top']) - self.splitYPad)
            factor *= (1 - self.padSize['bottom'] - (1 - self.padSize['top']))
            deltaY = deltaY ** factor
          except:
            pass
          minY = maxY / deltaY
        
      # need to compensate for padding
      if(modeX == 'linear'):
        deltaX = (maxX - minX)
        minX, maxX = maxX - deltaX * (1 + self.padSize['left'] / (self.padSize['right'] - self.padSize['left'])),\
                      minX + deltaX * (1 + (1 - self.padSize['right']) / (self.padSize['right'] - self.padSize['left']))
      else:
        deltaX = (maxX / minX)
        minX, maxX = maxX / deltaX ** (1 + self.padSize['left'] / (self.padSize['right'] - self.padSize['left'])),\
                      minX * deltaX ** (1 + (1 - self.padSize['right']) / (self.padSize['right'] - self.padSize['left']))

      if(modeY == 'linear'):
        deltaY = (maxY - minY)
        minY, maxY = maxY - deltaY * (1 + self.padSize['bottom'] / (self.padSize['top'] - self.padSize['bottom'])),\
                      minY + deltaY * (1 + (1 - self.padSize['top']) / (self.padSize['top'] - self.padSize['bottom']))
      else:
        deltaY = (maxY / minY)
        minY, maxY = maxY / deltaY ** (1 + self.padSize['bottom'] / (self.padSize['top'] - self.padSize['bottom'])),\
                      minY * deltaY ** (1 + (1 - self.padSize['top']) / (self.padSize['top'] - self.padSize['bottom']))
      
      # no aspect ratio specified, calculate own (don't do on first call since widgets have not resized properly yet)      
      currWidth, currHeight = self.matplot.get_size_inches()
      if(not aspect):
        aspect = currWidth / currHeight
      scaleFont = 100.0 / self.matplot.get_dpi()
        
      # define coordinate sets for graphics ...
      coords0 = np.array([[0.0, 0.0], [0.0, 100.0], [100.0, 100.0], [100.0, 0.0]])
      # ... and process them
      scaleWidth = 1
      scaleHeight = 1 / aspect
      scaleMin = np.min((scaleWidth, scaleHeight))
      scaleHeight = np.min((2.5, scaleHeight))
      scaleHeight = np.max((1.0, scaleHeight))
      widthBox = 0.8 * scaleWidth
      heightBox = 0.5 * scaleHeight
      coords0 = processCoordSet(coords0, minX, maxX, modeX, minY, maxY, modeY, widthBox, heightBox, 0.1, (1 - heightBox) / 2.0)
  
      # calculate coords for text labels
      text0 = processCoordSingle(minX, maxX, modeX, minY, maxY, modeY, 0.5, 0.63)
      text1 = processCoordSingle(minX, maxX, modeX, minY, maxY, modeY, 0.5, 0.4)
      text2 = processCoordSingle(minX, maxX, modeX, minY, maxY, modeY, 0.5, 0.3)
      text3 = processCoordSingle(minX, maxX, modeX, minY, maxY, modeY, 0.82, 0.64)
      text4 = processCoordSingle(minX, maxX, modeX, minY, maxY, modeY, 0.84, 0.61)
  
      # draw the background
      currZoomLevel = self.matplot.get_dpi()
      linewidth = 1.0 * 100.0 / currZoomLevel
      polyPatch0 = matplotlib.patches.Polygon(coords0, closed=True, facecolor=[1.0, 1.0, 1.0, 0.7],\
        edgecolor = greyCol, linestyle='solid', linewidth=linewidth, zorder=zOffset)
      retv = targetAxis.add_patch(polyPatch0)
      retv.set_clip_on(False)
      self.handlesAbout.append(retv)
      zOffset += 1
      
      # determine scale factor for matplotlib
      MPL_SCALEFACTOR = myapp.devicePixelRatio()
      
      # print some information
      # according to docu, positional argument would be 'text' but Matplotlib wants 's' => better don't use positional arguments
      #retv = self.ax.text(x=self.minX + 0.5 * width, y=self.minY + 0.5 * height, s='Fit-o-mat',\
      #  horizontalalignment='center', zorder=zOffset)
      retv = targetAxis.text(text0[0], text0[1], 'Fit-o-mat',\
        horizontalalignment='center', zorder=zOffset, style='normal', fontsize=scaleFont * int(20 * SCALEFONT * MPL_SCALEFACTOR), fontweight='normal', color=ubtCol)
      self.handlesAbout.append(retv)
      zOffset += 1
      
      retv = targetAxis.text(text1[0], text1[1],\
        '\u00A9' + ' by A.M. 2017-2025 @MoeglichLab\nandreas.moeglich@uni-bayreuth.de\nGNU General Public License v3.0',\
        horizontalalignment='center', zorder=zOffset, style='normal', fontsize=scaleFont * int(10 * SCALEFONT * MPL_SCALEFACTOR), fontweight='normal', color=greyCol)
      self.handlesAbout.append(retv)
      zOffset += 1
      
      retv = targetAxis.text(text2[0], text2[1], 'version ' + VERSION, horizontalalignment='center', zorder=zOffset,\
        style='italic', fontsize=scaleFont * int(8 * SCALEFONT * MPL_SCALEFACTOR), fontweight='normal', color=greyCol)
      self.handlesAbout.append(retv)
      zOffset += 1
      
      retv = targetAxis.text(text3[0], text3[1], 'f', horizontalalignment='center', zorder=zOffset, family='serif',\
        style='italic', fontsize=scaleFont * int(20 * SCALEFONT * MPL_SCALEFACTOR), fontweight='normal', color=blackCol)
      self.handlesAbout.append(retv)
      zOffset += 1
      
      retv = targetAxis.text(text4[0], text4[1], 'x', horizontalalignment='center', zorder=zOffset, family='serif',\
        style='italic', fontsize=scaleFont * int(20 * SCALEFONT * MPL_SCALEFACTOR), fontweight='normal', color=blackCol)
      self.handlesAbout.append(retv)
      zOffset += 1
      
      # draw fit-o-mat logo
      logo = matplotlib.path.Path([[-0.05682998,  0.47677547], [-0.05682998,  0.68146271],
       [ 0.04833121,  0.84012554], [ 0.15428282,  0.99999958], [ 0.29107145,  0.99999958], [ 0.43181316,  0.99999958],
       [ 0.53697435,  0.84254788], [ 0.64213553,  0.68630738], [ 0.64213553,  0.47677547], [ 0.64213553,  0.26603271],
       [ 0.53855567,  0.1049474 ], [ 0.43576671, -0.05492643], [ 0.30055941, -0.05492643], [ 0.15586415, -0.05492643],
       [ 0.04912164,  0.10373591], [-0.05682998,  0.26239896], [-0.05682998,  0.47677547], [-0.05682998,  0.47677547],
       [-0.05682998,  0.47677547], [ 0.50930045,  0.47677547], [ 0.50930045,  0.60394799], [ 0.44604556,  0.69720786],
       [ 0.38279067,  0.79167887], [ 0.29581543,  0.79167887], [ 0.20646797,  0.79167887], [ 0.14163175,  0.69599673],
       [ 0.07758643,  0.60152573], [ 0.07758643,  0.46950853], [ 0.07758643,  0.34354735], [ 0.14242265,  0.24907614],
       [ 0.2080493 ,  0.15460492], [ 0.29581543,  0.15460492], [ 0.38120935,  0.15460492], [ 0.44525513,  0.2514984 ],
       [ 0.50930045,  0.34839189], [ 0.50930045,  0.47677547], [ 0.50930045,  0.47677547], [ 0.50930045,  0.47677547],
       [-0.6736831 ,  1.        ], [-0.08462276,  1.        ], [-0.12768349,  0.74262702], [-0.54674644,  0.74262702],
       [-0.62774753,  0.25816052], [-0.21066112,  0.25816052], [-0.25393759, -0.00224054], [-0.67003573, -0.00224054],
       [-0.82604952, -0.92878325], [-1.        , -0.92878325], [-0.6736831 ,  1.        ], [-0.6736831 ,  1.        ],
       [ 0.66980989,  0.99987871], [ 1.        , -0.92878325], [ 0.83870013, -0.92878325], [ 0.63849871,  0.24992411],
       [ 0.29407666, -1.        ], [-0.05129464,  0.25864427], [-0.25339365, -0.92878325], [-0.41469353, -0.92878325],
       [-0.08450342,  0.99987871], [ 0.29407666, -0.38085174], [ 0.66980989,  0.99987871], [ 0.66980989,  0.99987871]],
      [ 1,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
        2, 79,  1,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
        3,  3,  2, 79,  1,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 79,  1,
        2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 79])
      logo.vertices[:, 0] -= np.min(logo.vertices[:, 0])
      logo.vertices[:, 1] -= np.min(logo.vertices[:, 1])
      # multiply by 100 to counteract division by 100 in processCoordSet()
      logo.vertices *= 100.
      logo.vertices = processCoordSet(logo.vertices, minX, maxX, modeX, minY, maxY, modeY, 0.06 * scaleMin, 0.06 * scaleMin, 0.13, .61)
      polyPatch4 = matplotlib.patches.PathPatch(logo, facecolor='black', edgecolor=None, linewidth=0, zorder=zOffset)
      retv = targetAxis.add_patch(polyPatch4)
      retv.set_clip_on(False)
      self.handlesAbout.append(retv)
      zOffset += 1
    else:
      self.dataplotwidget.setDestructionCounter(1)
      self.destructAboutLogo()
      destructCounter = 0
      
    # refresh the plot to draw everything
    self.dataplotwidget.myRefresh()
    
    # set us up for ready destruction
    self.dataplotwidget.setDestructionCounter(destructCounter)

  def initLegend(self):
    # initializes legend (has to occur after function is drawn)
    self.setLegend(value=self.legendVisible, redraw=False, target='plot')
    self.setLegendPlacement(value=self.legendPlacement, redraw=False, target='plot')
    for prop in ['face', 'edge']:
      self.setLegendColor(value=self.legendColor[prop], prop=prop, redraw=False, target='plot')
    self.setLegendEdgeWidth(value=self.legendEdgeWidth, redraw=False, target='plot')
    self.setLegendShadow(value=self.legendShadow, redraw=False, target='plot')
    self.setLegendLabelColor(value=self.legendLabelColor, redraw=False, target='plot')
    self.setLegendLabelSize(value=self.legendLabelSize, redraw=False, target='plot')
    self.setLegendLabelBold(value=self.legendLabelWeight, redraw=False, target='plot')
    self.setLegendLabelItalic(value=self.legendLabelStyle, redraw=False, target='plot')
    self.setLegendLabelFont(value=self.legendLabelFont, redraw=False, target='plot')
    self.setLegendLabelLinespacing(value=self.legendLabelLinespacing, redraw=False, target='plot')
    self.setLegendMarkerFirst(value=self.legendMarkerFirst, redraw=False, target='plot')
    self.setLegendMarkerScale(value=self.legendMarkerScale, redraw=False, target='plot')
    self.setLegendNCol(value=self.legendNumCol, redraw=False, target='plot')
    self.toggleLegendFill(value=self.legendFill, redraw=False, target='plot')
    self.toggleLegendEdge(value=self.legendEdge, redraw=False, target='plot')
    self.setLegendMarkerNumPoints(value=self.legendNumPoints, redraw=False, target='plot')
    self.setLegendPadBorder(value=self.legendBorderPad, redraw=False, target='plot')
    self.setLegendPadRow(value=self.legendLabelSpacing, redraw=False, target='plot')
    self.setLegendPadCol(value=self.legendColumnSpacing, redraw=False, target='plot')
    self.setLegendPadLabel(value=self.legendLabelPad, redraw=False, target='plot')
   
  def changeAxisLimits(self, axis='x', target='plot', redraw=True):
    # check whether to operate on data or resid plot
    if(target in ['plot', 'resid']):
      # changes limits of axis
      if(axis == 'x'):
        # retrieve values from field
        upperx = self.upperLimitx.text()
        try:
          upperx = float(upperx)
        except:
          upperx = 0.0
  
        lowerx = self.lowerLimitx.text()
        try:
          lowerx = float(lowerx)
        except:
          lowerx = 0.0
        # check whether any value changed
        if((upperx != self.maxX) or (lowerx != self.minX)):
          # do some checks and update axis (don't redraw, plot functions will take care of)
          self.setAxisLimits(lower=lowerx, upper=upperx, axis='x', updateLabel=False, target='plot', redraw=False, updateGrid=True, preserveInverted=False)
          self.setAxisLimits(lower=lowerx, upper=upperx, axis='x', updateLabel=False, target='resid', redraw=False, updateGrid=True, preserveInverted=False)
          self.updateTickEntryField(axis='x')
          # replot function over current x-interval
          self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.plotFunction(fitobject=self.parent.fit[self.parent.activeFit], x=[],\
                         handlePlot=self.parent.fit[self.parent.activeFit].handlePlot, handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False)
          # copy in case split axes are shown
          curve = self.parent.fit[self.parent.activeFit]
          if(self.parent.plotArea.splitY and curve.onBothAxes):
            curve.duplicateForSplit()
          if(redraw):
            self.parent.plotArea.dataplotwidget.myRefresh()
          self.handleResidZero = self.plotResidZero(handleResidZero=self.handleResidZero, redraw=redraw)
      elif(axis == 'x2'):
        # retrieve values from field
        upperx = self.xSplitUpperLimitx.text()
        try:
          upperx = float(upperx)
        except:
          upperx = 0.0
  
        lowerx = self.xSplitLowerLimitx.text()
        try:
          lowerx = float(lowerx)
        except:
          lowerx = 0.0
        # check whether any value changed
        if((upperx != self.maxX_div) or (lowerx != self.minX_div)):
          # do some checks and update axis (don't redraw, plot functions will take care of)
          self.setAxisLimits(lower = lowerx, upper = upperx, axis = 'x2', updateLabel = False, target='plot', redraw=False, updateGrid=True, preserveInverted=False)
          self.setAxisLimits(lower = lowerx, upper = upperx, axis = 'x2', updateLabel = False, target='resid', redraw=False, updateGrid=True, preserveInverted=False)
          self.updateTickEntryField(axis='x2')
          # replot function over current x-interval
          self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.parent.plotArea.plotFunction(fitobject=self.parent.fit[self.parent.activeFit], x=[],\
                         handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div, handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True)
          # copy in case split axes are shown
          curve = self.parent.fit[self.parent.activeFit]
          if(self.parent.plotArea.splitY and curve.onBothAxes):
            curve.duplicateForSplit()
          if(redraw):
            self.parent.plotArea.dataplotwidget.myRefresh()
          self.handleResidZero_div = self.plotResidZero(handleResidZero=self.handleResidZero_div, redraw=redraw, splitX=True)
      elif(axis == 'y'):
        # retrieve values from field
        if(target == 'plot'):
          uppery = self.upperLimity.text()
          lowery = self.lowerLimity.text()
        else:
          uppery = self.upperLimitResidy.text()
          lowery = self.lowerLimitResidy.text()

        try:
          uppery = float(uppery)
        except:
          uppery = 0.0
  
        try:
          lowery = float(lowery)
        except:
          lowery = 0.0
        
        # check whether any value changed
        if(target == 'plot'):
          if((uppery != self.maxY) or (lowery != self.minY)):
            # do some checks and update axis
            self.setAxisLimits(lower=lowery, upper=uppery, axis='y', updateLabel=False, target=target, redraw=redraw, updateGrid=True, preserveInverted=False)
            self.updateTickEntryField(axis='y')
        else:
          if((uppery != self.maxResidY) or (lowery != self.minResidY)):
            # do some checks and update axis
            self.setAxisLimits(lower=lowery, upper=uppery, axis='y', updateLabel=False, target=target, redraw=redraw, updateGrid=True, preserveInverted=False)
            self.updateTickEntryField(axis='resid')
      elif(axis == 'y2'):
        # modify second y axis
        uppery = self.secondUpperLimit.text()
        lowery = self.secondLowerLimit.text()

        try:
          uppery = float(uppery)
        except:
          uppery = 0.0
  
        try:
          lowery = float(lowery)
        except:
          lowery = 0.0
        
        # check whether any value changed
        if((uppery != self.maxY2) or (lowery != self.minY2)):
          # do some checks and update axis
          self.setAxisLimits(lower=lowery, upper=uppery, axis='y2', updateLabel=False, target=target, redraw=redraw, updateGrid=True, preserveInverted=False)
          self.updateTickEntryField(axis='y2')
      
  def setAxisLimits(self, lower=0, upper=1, axis='x', updateLabel=False, target='plot', redraw=True, updateTicks=True, updateGrid=True, preserveInverted=True):
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        plotobject, axisobject, axisobject2 = self.dataplotwidget, self.ax, self.ax_div
        if(axis == 'x2'):
          axisobject = [self.ax_div, self.ax2_div]
      else:
        plotobject, axisobject, axisobject2 = self.residplotwidget, self.ax_resid, self.ax_resid_div
        if(axis == 'x2'):
          axisobject = [self.ax_resid_div]
    else:
      axis = 'abort'
    # ensure safe distance from largest positive/negative number
    # => otherwise matplotlib will crash when calculating tick locations
    MONSTER_FLOAT = 1.0e300
    lower, upper = min(lower, MONSTER_FLOAT), min(upper, MONSTER_FLOAT)
    lower, upper = max(lower, -MONSTER_FLOAT), max(upper, -MONSTER_FLOAT)
    # performs checks and then sets axis limits
    if(axis == 'x'):
      # check for inverted axis limits
      if(preserveInverted and (self.minX > self.maxX)):
        invertMe = True
      else:
        invertMe = False
      # have to prevent nan and inf here
      if(np.isfinite(lower)):
        self.minX = lower
      if(np.isfinite(upper)):
        self.maxX = upper
      # check current axis mode and take care of log values
      axis_mode = str(self.modeSelectorx.currentText())
      if (axis_mode == 'log'):
        if ((self.minX <= 0) or (self.maxX <= 0)):
          # look whether data are loaded
          data = self.parent.data[self.parent.activeData].value()
          if ('x' in data):
            xdata = np.array(data['x'])
            xdata = xdata[xdata > 0]
            if(len(xdata)):
              min_xdata = np.min(xdata); max_xdata = np.max(xdata)
              if((self.data_spacer > 0) and (len(xdata) > 1)):
                difference = np.log(max_xdata / min_xdata)
                min_xdata = np.exp(np.log(min_xdata) - self.data_spacer * difference)
                max_xdata = np.exp(np.log(max_xdata) + self.data_spacer * difference)
            else:
              min_xdata, max_xdata = 0, 0
            self.minX = np.max((self.EPSILON, self.minX, min_xdata))
            self.maxX = np.max((self.EPSILON, self.maxX, max_xdata))
          else:
            # ensure that min and max values are positive
            self.minX = np.max((self.EPSILON, self.minX))
            self.maxX = np.max((self.EPSILON, self.maxX))
          updateLabel = True

      if(invertMe):
        self.minX, self.maxX = np.max((self.minX, self.maxX)), np.min((self.minX, self.maxX))
        
      # update axis
      if(self.minX == self.maxX):
        if(axis_mode == 'log'):
          self.minX, self.maxX = self.minX / 1.1, self.maxX * 1.1
        else:
          self.minX, self.maxX = self.minX - self.EPSILON, self.maxX + self.EPSILON
        updateLabel = True

      # update labels?
      if(updateLabel):
        lowerText, upperText = self.parent.formatPairedNumbers(self.minX, self.maxX)
        # we should let the entry fields lose focus lest we get unintended autozoom updates afterwards
        self.upperLimitx.blockSignals(True)
        self.upperLimitx.clearFocus()
        self.upperLimitx.blockSignals(False)
        self.lowerLimitx.blockSignals(True)
        self.lowerLimitx.clearFocus()
        self.lowerLimitx.blockSignals(False)
        self.upperLimitx.setText(upperText)
        self.lowerLimitx.setText(lowerText)

      axisobject.set_xlim([self.minX, self.maxX])
      if(target == 'plot'):
        self.ax2.set_xlim([self.minX, self.maxX])
      # update push button
      self.flipAxisButtonX.setChecked(bool(self.maxX < self.minX))
      if(updateTicks):
        self.setTickOne4All(axis=axis, redraw=False, target=target)

      # do we have a grid to draw?
      if(updateGrid):
        for item in ('x', 'y', 'y2'):
          if(self.gridVisible[item]):
            if((item != 'y2') or self.isSecondAxesActive()):
              self.drawAxisGrid(axis=item, redraw=False, target=target)

      for axis in ['left', 'right']:
        if((axis_mode == 'log') and (self.axisPosition[axis] == 'data')):
          # only required for log plots which we had to cast to axes coords
          if(target == 'plot'):
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax, target=target, secondAxes=False)
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax_div, target=target, secondAxes=False, splitX=True)
          else:
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax_resid, target=target, secondAxes=False)
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax_resid_div, target=target, secondAxes=False, splitX=True)

      for spine in ['left2', 'right2']:
        if(self.axisPosition[spine] == 'data'):
          self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.matplot, axisobject=self.ax2, target='plot', secondAxes=True)
          self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.matplot, axisobject=self.ax2_div, target='plot', secondAxes=True, splitX=True)

      if(redraw):
        plotobject.myRefresh()
    elif(axis == 'x2'):
      # check for inverted axis limits
      if(preserveInverted and (self.minX_div > self.maxX_div)):
        invertMe = True
      else:
        invertMe = False
      # have to prevent nan and inf here
      if(np.isfinite(lower)):
        self.minX_div = lower
      if(np.isfinite(upper)):
        self.maxX_div = upper
      # check current axis mode and take care of log values
      axis_mode = str(self.xSplitModeSelectorx.currentText())
      if (axis_mode == 'log'):
        if ((self.minX_div <= 0) or (self.maxX_div <= 0)):
          # look whether data are loaded
          data = self.parent.data[self.parent.activeData].value()
          if ('x' in data):
            xdata = np.array(data['x'])
            xdata = xdata[xdata > 0]
            if(len(xdata)):
              min_xdata = np.min(xdata); max_xdata = np.max(xdata)
              if((self.data_spacer > 0) and (len(xdata) > 1)):
                difference = np.log(max_xdata / min_xdata)
                min_xdata = np.exp(np.log(min_xdata) - self.data_spacer * difference)
                max_xdata = np.exp(np.log(max_xdata) + self.data_spacer * difference)
            else:
              min_xdata, max_xdata = 0, 0
            self.minX_div = np.max((self.EPSILON, self.minX_div, min_xdata))
            self.maxX_div = np.max((self.EPSILON, self.maxX_div, max_xdata))
          else:
            # ensure that min and max values are positive
            self.minX_div = np.max((self.EPSILON, self.minX_div))
            self.maxX_div = np.max((self.EPSILON, self.maxX_div))
          updateLabel = True
      
      if(invertMe):
        self.minX_div, self.maxX_div = np.max((self.minX_div, self.maxX_div)), np.min((self.minX_div, self.maxX_div))
        
      # update axis
      if(self.minX_div == self.maxX_div):
        if(axis_mode == 'log'):
          self.minX_div, self.maxX_div = self.minX_div / 1.1, self.maxX_div * 1.1
        else:
          self.minX_div, self.maxX_div = self.minX_div - self.EPSILON, self.maxX_div + self.EPSILON
        updateLabel = True

      # update labels?
      if(updateLabel):
        lowerText, upperText = self.parent.formatPairedNumbers(self.minX_div, self.maxX_div)
        # we should let the entry fields lose focus lest we get unintended autozoom updates afterwards
        self.xSplitUpperLimitx.blockSignals(True)
        self.xSplitUpperLimitx.clearFocus()
        self.xSplitUpperLimitx.blockSignals(False)
        self.xSplitLowerLimitx.blockSignals(True)
        self.xSplitLowerLimitx.clearFocus()
        self.xSplitLowerLimitx.blockSignals(False)
        self.xSplitUpperLimitx.setText(upperText)
        self.xSplitLowerLimitx.setText(lowerText)

      for entry in axisobject:
        entry.set_xlim([self.minX_div, self.maxX_div])
      # update push button
      self.flipAxisButtonX2.setChecked(bool(self.maxX_div < self.minX_div))
      # should be able to use the same call which updates both regular x axis and divided one
      if(updateTicks):
        self.setTickOne4All(axis='x2', redraw=False, target=target)

      # do we have a grid to draw?
      if(updateGrid):
        for item in ('x2', 'y', 'y2'):
          if(self.gridVisible[item]):
            if((item != 'y2') or self.isSecondAxesActive()):
              self.drawAxisGrid(axis=item, redraw=False, target=target)

      '''
      for axis in ['left', 'right']:
        if((axis_mode == 'log') and (self.axisPosition[axis] == 'data')):
          # only required for log plots which we had to cast to axes coords
          if(target == 'plot'):
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax_div, target=target, secondAxes=False, splitX=True)
          else:
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax_resid_div, target=target, secondAxes=False, splitX=True)

      for spine in ['left2', 'right2']:
        if(self.axisPosition[spine] == 'data'):
          self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.matplot, axisobject=self.ax2_div, target='plot', secondAxes=True, splitX=True)
      '''
      
      if(redraw):
        plotobject.myRefresh()
    elif(axis == 'y'):
      # beware -- as we had to abandon make_axes_locatable, self.ax is no longer y-slaved to self.ax_div
      # => we need to do this ourselves
      if(target == 'plot'):
        # check for inverted axis limits
        if(preserveInverted and (self.minY > self.maxY)):
          invertMe = True
        else:
          invertMe = False
        # have to prevent nan and inf here
        if(np.isfinite(lower)):
          self.minY = lower
        if(np.isfinite(upper)):
          self.maxY = upper
        # check current axis mode and take care of log values
        axis_mode = str(self.modeSelectory.currentText())
        if ((axis_mode == 'log') and (target == 'plot')):
          if ((self.minY <= 0) or (self.maxY <= 0)):
            # look whether data are loaded
            data = self.parent.data[self.parent.activeData].value()
            if ('y' in data):
              ydata = np.array(data['y'])
              ydata = ydata[ydata > 0]
              if(len(ydata)):
                min_ydata = np.min(ydata); max_ydata = np.max(ydata)
                if((self.data_spacer > 0) and (len(ydata) > 1)):
                  difference = np.log(max_ydata / min_ydata)
                  min_ydata = np.exp(np.log(min_ydata) - self.data_spacer * difference)
                  max_ydata = np.exp(np.log(max_ydata) + self.data_spacer * difference)
              else:
                min_ydata, max_ydata = 0, 0
              self.minY = np.max((self.EPSILON, self.minY, min_ydata))
              self.maxY = np.max((self.EPSILON, self.maxY, max_ydata))
            else:
              # ensure that min and max values are positive
              self.minY = np.max((self.EPSILON, self.minY))
              self.maxY = np.max((self.EPSILON, self.maxY))
            updateLabel = True
        
        if(invertMe):
          self.minY, self.maxY = np.max((self.minY, self.maxY)), np.min((self.minY, self.maxY))

        # update axis
        if(self.minY == self.maxY):
          if(axis_mode == 'log'):
            self.minY, self.maxY = self.minY / 1.1, self.maxY * 1.1
          else:
            self.minY, self.maxY = self.minY - self.EPSILON, self.maxY + self.EPSILON
          updateLabel = True

        # update labels?
        if(updateLabel):
          lowerText, upperText = self.parent.formatPairedNumbers(self.minY, self.maxY)
          # we should let the entry fields lose focus lest we get unintended autozoom updates afterwards
          self.upperLimity.blockSignals(True)
          self.upperLimity.clearFocus()
          self.upperLimity.blockSignals(False)
          self.lowerLimity.blockSignals(True)
          self.lowerLimity.clearFocus()
          self.lowerLimity.blockSignals(False)
          self.upperLimity.setText(upperText)
          self.lowerLimity.setText(lowerText)

        axisobject.set_ylim([self.minY, self.maxY])
        axisobject2.set_ylim([self.minY, self.maxY])
        # update push button
        self.flipAxisButtonY.setChecked(bool(self.maxY < self.minY))
        if(updateTicks):
          self.setTickOne4All(axis=axis, redraw=False, target=target)

        # do we have a grid to draw?
        if(updateGrid):
          for item in ('x', 'x2', 'y'):
            if(self.gridVisible[item]):
              self.drawAxisGrid(axis=item, redraw=False, target=target)
  
        # adjust spine lines if needed
        for axis in ['top', 'bottom']:
          if(self.axisPosition[axis] == 'data'):
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax2, target=target, secondAxes=False)
            self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax2_div, target=target, secondAxes=False, splitX=True)

        if(redraw):
          plotobject.myRefresh()
      else:
        # don't check for inverted as this should not be possible for resid plot
        # have to prevent nan and inf here
        if(np.isfinite(lower)):
          self.minResidY = lower
        if(np.isfinite(upper)):
          self.maxResidY = upper

        # update axis
        if(self.minResidY == self.maxResidY):
          self.minResidY, self.maxResidY = self.minResidY - self.EPSILON, self.maxResidY + self.EPSILON
          updateLabel = True

        # update labels?
        if(updateLabel):
          lowerText, upperText = self.parent.formatPairedNumbers(self.minResidY, self.maxResidY)
          # we should let the entry fields lose focus lest we get unintended autozoom updates afterwards
          self.upperLimitResidy.blockSignals(True)
          self.upperLimitResidy.clearFocus()
          self.upperLimitResidy.blockSignals(False)
          self.lowerLimitResidy.blockSignals(True)
          self.lowerLimitResidy.clearFocus()
          self.lowerLimitResidy.blockSignals(False)
          self.upperLimitResidy.setText(upperText)
          self.lowerLimitResidy.setText(lowerText)

        axisobject.set_ylim([self.minResidY, self.maxResidY])
        axisobject2.set_ylim([self.minResidY, self.maxResidY])
        if(updateTicks):
          self.setTickOne4All(axis=axis, redraw=False, target=target)

        # do we have a grid to draw?
        if(updateGrid):
          for item in ('x', 'x2', 'y'):
            if(self.gridVisible[item]):
              self.drawAxisGrid(axis=item, redraw=False, target=target)

        if(redraw):
          plotobject.myRefresh()
    elif(axis == 'y2'):
      axisobject, axisobject2 = self.ax2, self.ax2_div
      # check for inverted axis limits
      if(preserveInverted and (self.minY2 > self.maxY2)):
        invertMe = True
      else:
        invertMe = False
      # have to prevent nan and inf here
      if(np.isfinite(lower)):
        self.minY2 = lower
      if(np.isfinite(upper)):
        self.maxY2 = upper
      # check current axis mode and take care of log values
      axis_mode = str(self.secondModeSelector.currentText())
      if(axis_mode == 'log'):
        if ((self.minY2 <= 0) or (self.maxY2 <= 0)):
          # look whether data are loaded
          data = self.parent.data[self.parent.activeData].value()
          if ('y' in data):
            ydata = np.array(data['y'])
            ydata = ydata[ydata > 0]
            if(len(ydata)):
              min_ydata = np.min(ydata); max_ydata = np.max(ydata)
              if((self.data_spacer > 0) and (len(ydata) > 1)):
                difference = np.log(max_ydata / min_ydata)
                min_ydata = np.exp(np.log(min_ydata) - self.data_spacer * difference)
                max_ydata = np.exp(np.log(max_ydata) + self.data_spacer * difference)
            else:
              min_ydata, max_ydata = 0, 0
            self.minY2 = np.max((self.EPSILON, self.minY2, min_ydata))
            self.maxY2 = np.max((self.EPSILON, self.maxY2, max_ydata))
          else:
            # ensure that min and max values are positive
            self.minY2 = np.max((self.EPSILON, self.minY2))
            self.maxY2 = np.max((self.EPSILON, self.maxY2))
          updateLabel = True
      
      if(invertMe):
        self.minY2, self.maxY2 = np.max((self.minY2, self.maxY2)), np.min((self.minY2, self.maxY2))

      # update axis
      if(self.minY2 == self.maxY2):
        if(axis_mode == 'log'):
          self.minY2, self.maxY2 = self.minY2 / 1.1, self.maxY2 * 1.1
        else:
          self.minY2, self.maxY2 = self.minY2 - self.EPSILON, self.maxY2 + self.EPSILON
        updateLabel = True

      # update labels?
      if(updateLabel):
        lowerText, upperText = self.parent.formatPairedNumbers(self.minY2, self.maxY2)
        # we should let the entry fields lose focus lest we get unintended autozoom updates afterwards
        self.secondUpperLimit.blockSignals(True)
        self.secondUpperLimit.clearFocus()
        self.secondUpperLimit.blockSignals(False)
        self.secondLowerLimit.blockSignals(True)
        self.secondLowerLimit.clearFocus()
        self.secondLowerLimit.blockSignals(False)
        self.secondUpperLimit.setText(upperText)
        self.secondLowerLimit.setText(lowerText)

      axisobject.set_ylim([self.minY2, self.maxY2])
      axisobject2.set_ylim([self.minY2, self.maxY2])
      # update push button
      self.flipAxisButtonY2.setChecked(bool(self.maxY2 < self.minY2))
      if(updateTicks):
        self.setTickOne4All(axis=axis, redraw=False, target=target)

      # do we have a grid to draw?
      if(updateGrid):
        for item in ('x', 'x2', 'y2'):
          if(self.gridVisible[item]):
            if((item != 'y2') or self.isSecondAxesActive()):
              self.drawAxisGrid(axis=item, redraw=False, target=target)

      # adjust spine lines if needed
      for axis in ['top', 'bottom']:
        if(self.axisPosition[axis] == 'data'):
          self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax2, target=target, secondAxes=False)
          self.setAxisPositionHelper(axis=axis, plotobject=plotobject, axisobject=self.ax2_div, target=target, secondAxes=False, splitX=True)

      if(redraw):
        plotobject.myRefresh()

  def drawAxisGrid(self, axis='x', target='plot', redraw=False):
    # draws custom grid lines
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        # draw grid on 1st set of axes
        plotobject = self.dataplotwidget; axisobject = [self.ax, self.ax_div]
        if(axis == 'x'):
          axisobject = [self.ax]
          if(self.splitY):
            axisobject += [self.ax2]
        elif(axis == 'x2'):
          axisobject =[self.ax2_div]
          if(self.splitY):
            axisobject += [self.ax_div]
        elif(axis == 'y2'):
          axisobject = [self.ax2, self.ax2_div]
        useVisible, useColor, useWidth, useStyle, useDashStyle, useGridOrder = self.gridVisible[axis], self.gridColor[axis], self.gridWidth[axis], self.gridStyle[axis], self.gridDashStyle[axis], self.gridOrder[axis]
      else:
        plotobject = self.residplotwidget; axisobject = [self.ax_resid, self.ax_resid_div]
        if(axis == 'x'):
          axisobject = [self.ax_resid]
        elif(axis == 'x2'):
          axisobject =[self.ax_resid_div]
        useVisible, useColor, useWidth, useStyle, useDashStyle, useGridOrder = self.gridVisible_resid[axis], self.gridColor_resid[axis], self.gridWidth_resid[axis], self.gridStyle_resid[axis], self.gridDashStyle_resid[axis], self.gridOrder_resid[axis]
      
      if(useGridOrder == 'back'):
        useZ = 1
      else:
        useZ = 500
        
      # remove previous lines for this axis
      if(target == 'plot'):
        for entry in self.gridLinesStore[axis]:
          entry.remove()
        self.gridLinesStore[axis] = []
      else:
        for entry in self.gridLinesStore_resid[axis]:
          entry.remove()
        self.gridLinesStore_resid[axis] = []

      # replace grid lines by custom lines
      allLineHandles = []
      if(useVisible):
        for targetAxis in axisobject:
          lines = []
          # determine positions for drawing new ones
          if(axis in ['x', 'x2']):
            ticks = np.array(targetAxis.xaxis.get_majorticklocs())
            if(axis == 'x2'):
              ticks = np.array(targetAxis.xaxis.get_majorticklocs())
            if(axis == 'x'):
              vmin, vmax = min(self.minX, self.maxX), max(self.minX, self.maxX)
            else:
              vmin, vmax = min(self.minX_div, self.maxX_div), max(self.minX_div, self.maxX_div)
            # convert ticks to axis coordinates
            ticks_norm = [targetAxis.transData.transform((i, 1))[0] for i in ticks]
            vmin_norm, vmax_norm = targetAxis.transData.transform((vmin, 1))[0], targetAxis.transData.transform((vmax, 1))[0]
            # filter for visible ticks
            cond1 = ticks >= vmin
            cond2 = ticks <= vmax
            cond3 = ~np.isclose(ticks_norm, vmin_norm)
            cond4 = ~np.isclose(ticks_norm, vmax_norm)
            ticks = ticks.compress(cond1 & cond2 & cond3 & cond4)
            # create plot coordinates
            if(target == 'resid'):
              minY, maxY = min(self.minResidY, self.maxResidY), max(self.minResidY, self.maxResidY)
            elif(targetAxis in [self.ax, self.ax_div]):
              minY, maxY = min(self.minY, self.maxY), max(self.minY, self.maxY)
            else:
              minY, maxY = min(self.minY2, self.maxY2), max(self.minY2, self.maxY2)
            for entry in ticks:
              lines.append([[entry, entry], [minY, maxY]])
          else:
            # don't draw axis lines for non-displayed second axes
            if((axis != 'y2') or self.isSecondAxesActive()):
              ticks = targetAxis.yaxis.get_majorticklocs()
              if(target == 'resid'):
                vmin, vmax = min(self.minResidY, self.maxResidY), max(self.minResidY, self.maxResidY)
              elif(axis == 'y'):
                vmin, vmax = min(self.minY, self.maxY), max(self.minY, self.maxY)
              else:
                vmin, vmax = min(self.minY2, self.maxY2), max(self.minY2, self.maxY2)
              # convert ticks to axis coordinates
              ticks_norm = [targetAxis.transData.transform((1, i))[1] for i in ticks]
              vmin_norm, vmax_norm = targetAxis.transData.transform((1, vmin))[1], targetAxis.transData.transform((1, vmax))[1]
              # filter for visible ticks
              cond1 = ticks >= vmin
              cond2 = ticks <= vmax
              cond3 = ~np.isclose(ticks_norm, vmin_norm)
              cond4 = ~np.isclose(ticks_norm, vmax_norm)
              ticks = ticks.compress(cond1 & cond2 & cond3 & cond4)
              # create plot coordinates
              if(targetAxis in [self.ax, self.ax2, self.ax_resid]):
                minX, maxX = min(self.minX, self.maxX), max(self.minX, self.maxX)
              else:
                minX, maxX = min(self.minX_div, self.maxX_div), max(self.minX_div, self.maxX_div)
              for entry in ticks:
                lines.append([[minX, maxX], [entry, entry]])
  
          # draw these lines
          for entry in lines:
            drawLine = matplotlib.lines.Line2D(entry[0], entry[1])
            allLineHandles.append(targetAxis.add_line(drawLine))
            # apply styles
            drawLine.set_color(useColor)
            drawLine.set_linewidth(useWidth)
            drawLine.set_linestyle(useStyle)
            drawLine.set_dash_capstyle(useDashStyle)
            drawLine.set_solid_capstyle(useDashStyle)
            drawLine.set_zorder(useZ)
            
        # store them for future manipulation
        if(target == 'plot'):
          self.gridLinesStore[axis].extend(allLineHandles)
        else:
          self.gridLinesStore_resid[axis].extend(allLineHandles)
          
      # call draw minor grid
      self.drawMinorAxisGrid(axis=axis, target=target, redraw=False)

      # now take care of axis shading
      if(target == 'plot'):
        for entry in self.gridRectStore[axis]:
          entry.remove()
        self.gridRectStore[axis] = []
      else:
        for entry in self.gridRectStore_resid[axis]:
          entry.remove()
        self.gridRectStore_resid[axis] = []
      
      useVisible, useColor = self.gridRectVisible[axis] and self.gridVisible[axis], self.gridRectColor[axis]
      if(axis in ['x', 'x2']):
        if(useVisible):
          if(target == 'plot'):
            ymin, ymax = min(self.minY, self.maxY), max(self.minY, self.maxY)
            if(axis == 'x'):
              candidateAxes = [self.ax]
            else:
              candidateAxes = [self.ax_div]
          else:
            ymin, ymax = min(self.minResidY, self.maxResidY), max(self.minResidY, self.maxResidY)
            if(axis == 'x'):
              candidateAxes = [self.ax_resid]
            else:
              candidateAxes = [self.ax_resid_div]
  
          for targetAxis in candidateAxes:
            ticks = np.array(targetAxis.xaxis.get_majorticklocs())
            allRectHandles = []
            for i in range(len(ticks))[::2]:
              if(i + 1 < len(ticks)):
                drawRect = matplotlib.patches.Rectangle(xy=(ticks[i], ymin), width=ticks[i + 1] - ticks[i], height=ymax - ymin)
                allRectHandles.append(targetAxis.add_patch(drawRect))
                drawRect.set_facecolor(useColor)
                drawRect.set_linewidth(0)
                drawRect.set_zorder(useZ + .1)
            if(target == 'plot'):
              self.gridRectStore[axis].extend(allRectHandles)
            else:
              self.gridRectStore_resid[axis].extend(allRectHandles)
            
          # consider split y axis
          if((target == 'plot') and self.splitY):
            ymin, ymax = min(self.minY2, self.maxY2), max(self.minY2, self.maxY2)
            if(axis == 'x'):
              candidateAxes = [self.ax2]
            else:
              candidateAxes = [self.ax2_div]
            
            for targetAxis in candidateAxes:
              ticks = np.array(targetAxis.xaxis.get_majorticklocs())
              allRectHandles = []
              for i in range(len(ticks))[::2]:
                if(i + 1 < len(ticks)):
                  drawRect = matplotlib.patches.Rectangle(xy=(ticks[i], ymin), width=ticks[i + 1] - ticks[i], height=ymax - ymin)
                  allRectHandles.append(targetAxis.add_patch(drawRect))
                  drawRect.set_facecolor(useColor)
                  drawRect.set_linewidth(0)
                  drawRect.set_zorder(useZ + .1)
              self.gridRectStore[axis].extend(allRectHandles)
      else:
        # y axes here
        # don't draw axis lines for non-displayed second axes
        if((axis != 'y2') or self.isSecondAxesActive()):
          if(useVisible):
            xmin, xmax = min(self.minX, self.maxX), max(self.minX, self.maxX)
            if(target == 'plot'):
              if(axis == 'y'):
                candidateAxes = [self.ax]
              else:
                candidateAxes = [self.ax2]
            else:
              candidateAxes = [self.ax_resid]
    
            for targetAxis in candidateAxes:
              ticks = np.array(targetAxis.yaxis.get_majorticklocs())
              allRectHandles = []
              for i in range(len(ticks))[::2]:
                if(i + 1 < len(ticks)):
                  drawRect = matplotlib.patches.Rectangle(xy=(xmin, ticks[i]), height=ticks[i + 1] - ticks[i], width=xmax - xmin)
                  allRectHandles.append(targetAxis.add_patch(drawRect))
                  drawRect.set_facecolor(useColor)
                  drawRect.set_linewidth(0)
                  drawRect.set_zorder(useZ + .1)
              if(target == 'plot'):
                self.gridRectStore[axis].extend(allRectHandles)
              else:
                self.gridRectStore_resid[axis].extend(allRectHandles)
              
            # consider split x axis
            if(self.splitShow):
              xmin, xmax = min(self.minX_div, self.maxX_div), max(self.minX_div, self.maxX_div)
              
              if(target == 'plot'):
                if(axis == 'y'):
                  candidateAxes = [self.ax_div]
                else:
                  candidateAxes = [self.ax2_div]
              else:
                candidateAxes = [self.ax_resid_div]
      
              for targetAxis in candidateAxes:
                ticks = np.array(targetAxis.yaxis.get_majorticklocs())
                allRectHandles = []
                for i in range(len(ticks))[::2]:
                  if(i + 1 < len(ticks)):
                    drawRect = matplotlib.patches.Rectangle(xy=(xmin, ticks[i]), height=ticks[i + 1] - ticks[i], width=xmax - xmin)
                    allRectHandles.append(targetAxis.add_patch(drawRect))
                    drawRect.set_facecolor(useColor)
                    drawRect.set_linewidth(0)
                    drawRect.set_zorder(useZ + .1)
                if(target == 'plot'):
                  self.gridRectStore[axis].extend(allRectHandles)
                else:
                  self.gridRectStore_resid[axis].extend(allRectHandles)

      # redraw
      if(redraw):
        plotobject.myRefresh()

  def drawMinorAxisGrid(self, axis='x', target='plot', redraw=False):
    # draws custom grid lines
    if(target in ['plot', 'resid']):
      if(target == 'plot'):
        # draw grid on 1st set of axes
        plotobject, axisobject = self.dataplotwidget, [self.ax, self.ax_div]
        if(axis == 'x'):
          axisobject = [self.ax]
          if(self.splitY):
            axisobject += [self.ax2]
        elif(axis == 'x2'):
          axisobject =[self.ax2_div]
          if(self.splitY):
            axisobject += [self.ax_div]
        elif(axis == 'y2'):
          axisobject = [self.ax2, self.ax2_div]
        useVisible, useColor, useWidth, useStyle, useDashStyle, useGridOrder = self.gridMinorVisible[axis], self.gridMinorColor[axis], self.gridMinorWidth[axis], self.gridMinorStyle[axis], self.gridMinorDashStyle[axis], self.gridMinorOrder[axis]
      else:
        plotobject, axisobject = self.residplotwidget, [self.ax_resid, self.ax_resid_div]
        if(axis == 'x'):
          axisobject = [self.ax_resid]
        elif(axis == 'x2'):
          axisobject =[self.ax_resid_div]
        useVisible, useColor, useWidth, useStyle, useDashStyle, useGridOrder = self.gridMinorVisible_resid[axis], self.gridMinorColor_resid[axis], self.gridMinorWidth_resid[axis], self.gridMinorStyle_resid[axis], self.gridMinorDashStyle_resid[axis], self.gridMinorOrder_resid[axis]
      
      if(useGridOrder == 'back'):
        useZ = 1
      else:
        useZ = 500
        
      # remove previous lines for this axis
      if(target == 'plot'):
        for entry in self.gridMinorLinesStore[axis]:
          entry.remove()
        self.gridMinorLinesStore[axis] = []
      else:
        for entry in self.gridMinorLinesStore_resid[axis]:
          entry.remove()
        self.gridMinorLinesStore_resid[axis] = []

      # replace grid lines by custom lines
      allLineHandles = []
      if(useVisible):
        for targetAxis in axisobject:
          lines = []
          # determine positions for drawing new ones
          if(axis in ['x', 'x2']):
            ticks = np.array(targetAxis.xaxis.get_minorticklocs())
            if(type(ticks) == type([])):
              ticks = np.array(ticks)
            if(axis == 'x2'):
              ticks = np.array(targetAxis.xaxis.get_minorticklocs())
            if(axis == 'x'):
              vmin, vmax = min(self.minX, self.maxX), max(self.minX, self.maxX)
            else:
              vmin, vmax = min(self.minX_div, self.maxX_div), max(self.minX_div, self.maxX_div)
            # convert ticks to axis coordinates
            ticks_norm = [targetAxis.transData.transform((i, 1))[0] for i in ticks]
            vmin_norm, vmax_norm = targetAxis.transData.transform((vmin, 1))[0], targetAxis.transData.transform((vmax, 1))[0]
            # filter for visible ticks
            cond1 = ticks >= vmin
            cond2 = ticks <= vmax
            cond3 = ~np.isclose(ticks_norm, vmin_norm)
            cond4 = ~np.isclose(ticks_norm, vmax_norm)
            ticks = ticks.compress(cond1 & cond2 & cond3 & cond4)
            # create plot coordinates
            if(target == 'resid'):
              minY, maxY = min(self.minResidY, self.maxResidY), max(self.minResidY, self.maxResidY)
            elif(targetAxis in [self.ax, self.ax_div]):
              minY, maxY = min(self.minY, self.maxY), max(self.minY, self.maxY)
            else:
              minY, maxY = min(self.minY2, self.maxY2), max(self.minY2, self.maxY2)
            for entry in ticks:
              lines.append([[entry, entry], [minY, maxY]])
          else:
            # don't draw axis lines for non-displayed second axes
            if((axis != 'y2') or self.isSecondAxesActive()):
              ticks = targetAxis.yaxis.get_minorticklocs()
              if(type(ticks) == type([])):
                ticks = np.array(ticks)
              if(target == 'resid'):
                vmin, vmax = min(self.minResidY, self.maxResidY), max(self.minResidY, self.maxResidY)
              elif(axis == 'y'):
                vmin, vmax = min(self.minY, self.maxY), max(self.minY, self.maxY)
              else:
                vmin, vmax = min(self.minY2, self.maxY2), max(self.minY2, self.maxY2)
              # convert ticks to axis coordinates
              ticks_norm = [targetAxis.transData.transform((1, i))[1] for i in ticks]
              vmin_norm, vmax_norm = targetAxis.transData.transform((1, vmin))[1], targetAxis.transData.transform((1, vmax))[1]
              # filter for visible ticks
              cond1 = ticks >= vmin
              cond2 = ticks <= vmax
              cond3 = ~np.isclose(ticks_norm, vmin_norm)
              cond4 = ~np.isclose(ticks_norm, vmax_norm)
              ticks = ticks.compress(cond1 & cond2 & cond3 & cond4)
              # create plot coordinates
              if(targetAxis in [self.ax, self.ax2, self.ax_resid]):
                minX, maxX = min(self.minX, self.maxX), max(self.minX, self.maxX)
              else:
                minX, maxX = min(self.minX_div, self.maxX_div), max(self.minX_div, self.maxX_div)
              for entry in ticks:
                lines.append([[minX, maxX], [entry, entry]])
  
          # draw these lines
          for entry in lines:
            drawLine = matplotlib.lines.Line2D(entry[0], entry[1])
            allLineHandles.append(targetAxis.add_line(drawLine))
            # apply styles
            drawLine.set_color(useColor)
            drawLine.set_linewidth(useWidth)
            drawLine.set_linestyle(useStyle)
            drawLine.set_dash_capstyle(useDashStyle)
            drawLine.set_solid_capstyle(useDashStyle)
            drawLine.set_zorder(useZ)
          
        # store them for future manipulation
        if(target == 'plot'):
          self.gridMinorLinesStore[axis].extend(allLineHandles)
        else:
          self.gridMinorLinesStore_resid[axis].extend(allLineHandles)

      # redraw
      if(redraw):
        plotobject.myRefresh()

  def updateTickEntryField(self, axis='x'):
    # updates tick entry field in graphics area when auto is selected
    if(axis in ['x', 'x2', 'y', 'y2', 'resid']):
      # check for existence of graphics tab (during initialization)
      if(axis == 'y2'):
        graphicsarea = 'graphicsarea2'
      else:
        graphicsarea = 'graphicsarea'
      if(hasattr(self.parent, graphicsarea)):
        if(axis == 'x'):
          tickAuto, tickEntry = self.parent.graphicsarea.configTickXAuto, self.parent.graphicsarea.configTickXEntry
        elif(axis == 'x2'):
          tickAuto, tickEntry = self.parent.graphicsarea.configInnerTickXAuto, self.parent.graphicsarea.configInnerTickXEntry
        elif(axis in ['y', 'y2']):
          tickAuto, tickEntry = self.parent.__dict__[graphicsarea].configTickYAuto, self.parent.__dict__[graphicsarea].configTickYEntry
        else:
          tickAuto, tickEntry = self.parent.graphicsarea.configTickResidYAuto, self.parent.graphicsarea.configTickResidYEntry
  
        if(tickAuto.isChecked()):
          nuTicks = self.getAxisTicks(axis)
          tickstr = self.parent.__dict__[graphicsarea].magicTickstring(nuTicks)
          tickEntry.setText(tickstr)

  def changeAxisMode(self, axis='x', redraw=True):
    # changes scale mode of axis
    if(axis == 'x'):
      # adjust axis
      axis_mode = str(self.modeSelectorx.currentText())
      if(axis_mode == 'linear'):
        # in this case, change axis scale first
        self.ax.set_xscale(axis_mode)
        self.ax2.set_xscale(axis_mode)
        self.ax_resid.set_xscale(axis_mode)
      self.setAxisLimits(lower = self.minX, upper = self.maxX, axis = 'x', updateLabel = False, target='plot', redraw=False, updateGrid=True)
      self.setAxisLimits(lower = self.minX, upper = self.maxX, axis = 'x', updateLabel = False, target='resid', redraw=False, updateGrid=True)
      if(axis_mode != 'linear'):
        # in this case, change axis scale last
        self.ax.set_xscale(axis_mode)
        self.ax2.set_xscale(axis_mode)
        self.ax_resid.set_xscale(axis_mode)
      self.modeX = axis_mode
      # trigger redrawing of fit function with new axis limits and minor ticks
      if(self.modeX == 'linear'):
        # reset minor ticks to 1 when changing to log scale
        if(self.ticksXMinor > 1):
          self.ticksXMinor, targetField = 2, self.parent.graphicsarea.configMinorTickX
          targetField.blockSignals(True)
          if(targetField.findText(str(1)) >= 0):
            targetField.setCurrentIndex(targetField.findText(str(1)))
          targetField.blockSignals(False)
        minorAutoticks = MyAutoMinorLocator(self.ticksXMinor)
        self.ax.xaxis.set_minor_locator(minorAutoticks)
        self.ax2.xaxis.set_minor_locator(minorAutoticks)
        self.ax_resid.xaxis.set_minor_locator(minorAutoticks)
        self.ax.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        self.ax2.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        self.ax_resid.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
      elif(self.modeX == 'log'):
        # reset minor ticks to 8 when changing to log scale
        if(self.ticksXMinor > 1):
          self.ticksXMinor, targetField = 9, self.parent.graphicsarea.configMinorTickX
          targetField.blockSignals(True)
          if(targetField.findText(str(8)) >= 0):
            targetField.setCurrentIndex(targetField.findText(str(8)))
          targetField.blockSignals(False)
        minorAutoticks = MyAutoMinorLocator(self.ticksXMinor)
        self.ax.xaxis.set_minor_locator(minorAutoticks)
        self.ax2.xaxis.set_minor_locator(minorAutoticks)
        self.ax_resid.xaxis.set_minor_locator(minorAutoticks)
        self.ax.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        self.ax2.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        self.ax_resid.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        
      self.parent.fit[self.parent.activeFit].handlePlot, self.parent.fit[self.parent.activeFit].handleBoot = self.plotFunction(fitobject=self.parent.fit[self.parent.activeFit], x=[],\
                     handlePlot=self.parent.fit[self.parent.activeFit].handlePlot, handleBoot=self.parent.fit[self.parent.activeFit].handleBoot, redraw=False)
      # copy in case split axes are shown
      curve = self.parent.fit[self.parent.activeFit]
      if(self.parent.plotArea.splitY and curve.onBothAxes):
        curve.duplicateForSplit()
      self.handleResidZero = self.plotResidZero(handleResidZero=self.handleResidZero, redraw=False)
      # adjust spine lines if needed
      for spine in ['left', 'right']:
        if(self.axisPosition[spine] == 'data'):
          self.setAxisPositionHelper(axis=spine, plotobject=self.matplot, axisobject=self.ax, target='plot', secondAxes=False)
          self.setAxisPositionHelper(axis=spine, plotobject=self.residplot, axisobject=self.ax_resid, target='resid', secondAxes=False)
      for spine in ['left2', 'right2']:
        if(self.axisPosition[spine] == 'data'):
          self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.matplot, axisobject=self.ax2, target='plot', secondAxes=True)
      
      # update axis grid
      for target in ('plot', 'resid'):
        for item in ('x', 'y', 'y2'):
          if(self.gridVisible[item]):
            if((item != 'y2') or self.isSecondAxesActive()):
              self.drawAxisGrid(axis=item, redraw=False, target=target)
      
      # update tick entry field
      self.updateTickEntryField(axis=axis)
      # redraw
      if(redraw):
        self.dataplotwidget.myRefresh()
        self.residplotwidget.myRefresh()
    elif(axis == 'x2'):
      # adjust axis
      axis_mode = str(self.xSplitModeSelectorx.currentText())
      if(axis_mode == 'linear'):
        # in this case, change axis scale first
        self.ax_div.set_xscale(axis_mode)
        self.ax2_div.set_xscale(axis_mode)
        self.ax_resid_div.set_xscale(axis_mode)
      self.setAxisLimits(lower = self.minX_div, upper = self.maxX_div, axis = 'x2', updateLabel = False, target='plot', redraw=False, updateGrid=True)
      self.setAxisLimits(lower = self.minX_div, upper = self.maxX_div, axis = 'x2', updateLabel = False, target='resid', redraw=False, updateGrid=True)
      if(axis_mode != 'linear'):
        # in this case, change axis scale last
        self.ax_div.set_xscale(axis_mode)
        self.ax2_div.set_xscale(axis_mode)
        self.ax_resid_div.set_xscale(axis_mode)
      self.modeX_div = axis_mode
      # trigger redrawing of fit function with new axis limits and minor ticks
      if(self.modeX_div == 'linear'):
        # reset minor ticks to 1 when changing to log scale
        if(self.ticksXMinor_div > 1):
          self.ticksXMinor_div, targetField = 2, self.parent.graphicsarea.configInnerMinorTickX
          targetField.blockSignals(True)
          if(targetField.findText(str(1)) >= 0):
            targetField.setCurrentIndex(targetField.findText(str(1)))
          targetField.blockSignals(False)
        minorAutoticks2 = MyAutoMinorLocator(self.ticksXMinor_div)
        self.ax2_div.xaxis.set_minor_locator(minorAutoticks2)
        self.ax_resid_div.xaxis.set_minor_locator(minorAutoticks2)
        self.ax2_div.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        self.ax_resid_div.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
      elif(self.modeX_div == 'log'):
        # reset minor ticks to 9 when changing to log scale
        if(self.ticksXMinor_div > 1):
          self.ticksXMinor_div, targetField = 9, self.parent.graphicsarea.configInnerMinorTickX
          targetField.blockSignals(True)
          if(targetField.findText(str(8)) >= 0):
            targetField.setCurrentIndex(targetField.findText(str(8)))
          targetField.blockSignals(False)
        minorAutoticks2 = MyAutoMinorLocator(self.ticksXMinor_div + 1)
        self.ax2_div.xaxis.set_minor_locator(minorAutoticks2)
        self.ax_resid_div.xaxis.set_minor_locator(minorAutoticks2)
        self.ax2_div.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        # need to also call null formatter on overlaid axis to prevent minor labels from showing
        self.ax_div.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        self.ax_resid_div.xaxis.set_minor_formatter(matplotlib.ticker.NullFormatter())
        
      self.parent.fit[self.parent.activeFit].handlePlot_div, self.parent.fit[self.parent.activeFit].handleBoot_div = self.plotFunction(fitobject=self.parent.fit[self.parent.activeFit], x=[],\
                     handlePlot=self.parent.fit[self.parent.activeFit].handlePlot_div, handleBoot=self.parent.fit[self.parent.activeFit].handleBoot_div, redraw=False, splitX=True)
      # copy in case split axes are shown
      curve = self.parent.fit[self.parent.activeFit]
      if(self.parent.plotArea.splitY and curve.onBothAxes):
        curve.duplicateForSplit()
      self.handleResidZero_div = self.plotResidZero(handleResidZero=self.handleResidZero_div, redraw=False, splitX=True)

      # adjust spine lines if needed
      for spine in ['left', 'right']:
        if(self.axisPosition[spine] == 'data'):
          self.setAxisPositionHelper(axis=spine, plotobject=self.matplot, axisobject=self.ax_div, target='plot', secondAxes=False, splitX=True)
          self.setAxisPositionHelper(axis=spine, plotobject=self.residplot, axisobject=self.ax_resid_div, target='resid', secondAxes=False, splitX=True)
      for spine in ['left2', 'right2']:
        if(self.axisPosition[spine] == 'data'):
          self.setAxisPositionHelper(axis=spine[:-1], plotobject=self.matplot, axisobject=self.ax2_div, target='plot', secondAxes=True, splitX=True)

      # update axis grid
      for target in ('plot', 'resid'):
        for item in ('x2', 'y', 'y2'):
          if(self.gridVisible[item]):
            if((item != 'y2') or self.isSecondAxesActive()):
              self.drawAxisGrid(axis=item, redraw=False, target=target)
      
      # update tick entry field
      self.updateTickEntryField(axis=axis)
      # redraw
      if(redraw):
        self.dataplotwidget.myRefresh()
        self.residplotwidget.myRefresh()
    elif(axis == 'y'):
      # adjust axis
      axis_mode = str(self.modeSelectory.currentText())
      if(axis_mode == 'linear'):
        # in this case, change axis scale first
        self.ax.set_yscale(axis_mode)
        self.ax_div.set_yscale(axis_mode)
      self.setAxisLimits(lower = self.minY, upper = self.maxY, axis = 'y', updateLabel = False, redraw=False, updateGrid=True)
      if(axis_mode != 'linear'):
        # in this case, change axis scale last
        self.ax.set_yscale(axis_mode)
        self